#ifndef SLMOTION_FACE_DETECTOR2
#define SLMOTION_FACE_DETECTOR2

#include "ViolaJonesFaceDetector.hpp"

namespace slmotion {
  /**
   * The default interpolated Viola-Jones-style face detector
   */
  class FaceDetector2 : public ViolaJonesFaceDetector {
  public:
    /**
     * Returns the component name
     */
    virtual std::string getComponentName() {
      return "Face Detector";
    }



    /**
     * Constructs a new face detector object
     *
     * @param faceDetectorOpts Options struct as defined above
     * @param blackBoard A pointer to a black board object. The board must 
     * outlive the face detector.
     * @param frameSource A pointer to a frame source object. The source 
     * must outlive the face detector.
     */
    explicit FaceDetector2(BlackBoard* blackBoard, FrameSource* frameSource,
                          const std::string& cascadeFilenameArgument = getDefaultHaarCascadeFile())
      : 
      ViolaJonesFaceDetector(blackBoard, frameSource, cascadeFilenameArgument)
    {
    }



    /**
     * Performs raw face detection in the given frame, without 
     * interpolation. However, scaling and translation are applied.
     */
    void process(frame_number_t framenumber);



    inline virtual property_set_t getProvided() const {
      return property_set_t { FACEDETECTOR_BLACKBOARD_ENTRY };
    }



    virtual std::string getShortDescription() const {
      return "Viola-Jones face detector";
    }



    virtual std::string getLongDescription() const {
      return "Viola-Jones face detector that interpolates any missing frames.";
    }



    virtual property_set_t getRequirements() const {
      return property_set_t();
    }

    virtual std::string getShortName() const {
      return "FaceDetector2";
    }

    virtual std::string getComponentName() const {
      return "Face Detector";
    }

    FaceDetector2(bool) : ViolaJonesFaceDetector(true) {}

    virtual void reset() { }

    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }

  private:
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;



    /**
     * Returns a copy of the original rectangle that has been scaled and
     * translated according to the object settings, with the exception of
     * INVALID_FACE which will remain invalid.
     */
    cv::Rect scaleAndTranslate(const cv::Rect& original);



    /**
     * Iterates over the associated frames and performs detection in each 
     * frame, the last frame number is exclusive. After detection has been
     * performed for all frames, any detections missing between two 
     * successful detections are linearly interpolated. Any unsuccessful 
     * detections before the first and after the last successful detections
     * are replaced by the first and last successful detections, 
     * respectfully.
     *
     * I.e. for i in [first,last): detect and interpolate
     *
     * @param first First frame to process (default 0)
     * @param last Last frame to process (default maximum). If the last 
     * frame exceeds the boundaries of the frame source, only frames up to 
     * the maximum are processed.
     */
    bool processRangeImplementation(frame_number_t first,
                                    frame_number_t last,
                                    UiCallback* uiCallback);



    /**
     * Sets the cascade filename in use, resets the cascade object, and checks 
     * that it was opened successfully, or throws an exception.
     */
    void setAndVerifyCascadeFilename(const std::string& filename);
    


    /**
     * Detects a face in the image using a Haar classifier
     *
     * @param frame Source frame; assumed to be a 3-channel BGR image.
     * @param facelist Output vector where found faces are stored
     * @param mask Should contain a skin filter mask or NULL if no mask is to be used.
     *
     * @return true if faces were found, sotring faces as Rect objects in the
     * facelist vector.
     */
    // bool detectFace(const cv::Mat& frame, std::vector<cv::Rect>& facelist, const cv::Mat* mask = NULL);

    /**
     * Performs a raw face detection and stores the result in the
     * cache, and returns the result. No interpolation is applied, and no 
     * other frames are considered; these are left for other functions. If 
     * multiple face candidates are detected, an INVALID_FACE is returned.
     *
     * @param img Input image. Assumed to be in BGR.
     *
     * @return The raw detection result, or INVALID_FACE in case of failure.
     */
    cv::Rect detectFace(const cv::Mat& img);



    /**
     * Reads in frames from the assembly line, optionally filling it
     * when necessary. Any frames before the first and after the last
     * possible detection, are considered equal to the first and last
     * detection, respectively. Any gaps between two successful
     * detections are filled in using linear interpolation.
     */
    cv::Rect getFaceAndInterpolate(const cv::Mat& currentFrame,
                                   size_t frnumber);



    /**
     * Interpolate missing detections for the given range
     */
    static void interpolate(std::vector<cv::Rect*>& rects);
  };
}
#endif
