#ifdef SLMOTION_ENABLE_LIBFLANDMARK
#include "FaceSkinAnomalyDetector.hpp"
#include "math.hpp"
#include "LimitedPriorityQueue.hpp"
#include "debug.hpp"
#include "util.hpp"

namespace slmotion {
  static FaceSkinAnomalyDetector DUMMY(true);

  typedef FaceSkinAnomalyDetector::Vec8f Vec8f;

  static void process1Neighbour(const std::list<std::pair<Vec8f, cv::Vec3f>>& sampleVectors,
                                  const Vec8f& current, cv::Mat& frame, int i, int j, 
                                  int normType) {
    const std::pair<Vec8f, cv::Vec3f>* closestNeighbour = nullptr;
    double closestNeighbourDistance = FLT_MAX;
    
    for (auto it = sampleVectors.cbegin(); it != sampleVectors.cend(); ++it) {
      if (cv::norm(it->first-current) < closestNeighbourDistance) {
        closestNeighbour = &(*it);
        closestNeighbourDistance = math::norm(closestNeighbour->first-current, normType);
      }
    }
    
    frame.at<cv::Vec3f>(i,j) = closestNeighbour->second;
  }

  static void processKNeighbours(size_t KNEIGHBOURS, const std::list<std::pair<Vec8f, cv::Vec3f> >& sampleVectors, const Vec8f& current, cv::Mat& frame, int i, int j, int normType) {
    auto farther = [&](const std::pair<Vec8f, cv::Vec3f>& lh, 
                       const std::pair<Vec8f, cv::Vec3f>& rh)->bool {
      return math::norm(lh.first-current, normType) > math::norm(rh.first-current, normType);
    };
    
    LimitedPriorityQueue<std::pair<Vec8f,cv::Vec3f>, decltype(farther)> queue(KNEIGHBOURS, farther);
    for (auto it = sampleVectors.cbegin(); it != sampleVectors.cend(); ++it)
      queue.insert(*it);

    for (auto it = queue.begin(); it != queue.end(); ++it)
      frame.at<cv::Vec3f>(i,j) +=  it->second*(1./KNEIGHBOURS);
  }


  void FaceSkinAnomalyDetector::process(frame_number_t frnumber) {
    assert(isTrained);
    assert(norm == 1 || norm == 2);

    int normType = norm == 1 ? cv::NORM_L1 : cv::NORM_L2;

    cv::Mat frame(getFrameSource()[frnumber].size(), CV_32FC3);
    BlackBoardPointer<cv::Mat> skinMask = getBlackBoard().get<cv::Mat>(frnumber, 
                                                                       SKINDETECTOR_BLACKBOARD_MASK_ENTRY);

    cv::Mat expectation(frame.size(), CV_32FC3, cv::Scalar::all(0));
    BlackBoardPointer<std::vector<cv::Point2f> > landmarks(getBlackBoard().get<std::vector<cv::Point2f>>(frnumber, FACIAL_LANDMARK_BLACKBOARD_ENTRY));
    assert(landmarks->size() == 8);

    timerOn();

    std::cerr << std::endl;
    for (int i = 0; i < frame.rows; ++i) {
      std::cerr << "\r" << i*frame.rows * (100./(frame.rows*frame.cols)) << "%";
      for (int j = 0; j < frame.cols; ++j) {
        if (skinMask->at<uchar>(i, j) > 0) {
          Vec8f current;
          for (int k = 0; k < 8; ++k)
            current[k] = cv::norm((*landmarks)[k] - cv::Point2f(j,i));

          if (kNeighbours == 1)
            process1Neighbour(sampleVectors, current, frame, i, j, normType);
          else
            processKNeighbours(kNeighbours, sampleVectors, current, frame, i, j, normType);
        }
      }
    }
    std::cerr << std::endl;

    printTime(std::cerr, timerOff());
    std::cerr << std::endl;



    
    cv::Mat frame2;
    cv::cvtColor(frame, frame2, CV_Lab2BGR);
    assert(frame2.type() == CV_32FC3);
    frame2 *= 255.;
    frame2.convertTo(frame, CV_8UC3);
    cv::imshow("", frame);
    cv::waitKey(0);
  }

  Component* FaceSkinAnomalyDetector::createComponentImpl(const boost::program_options::variables_map& opts, BlackBoard* blackBoard, FrameSource* frameSource) const {
    FaceSkinAnomalyDetector fsad(blackBoard, frameSource);
    if (opts.count("FaceSkinAnomalyDetector.kneighbours"))
      fsad.kNeighbours = opts["FaceSkinAnomalyDetector.kneighbours"].as<int>();
    if (opts.count("FaceSkinAnomalyDetector.sampledecimationfactor")) {
      std::string decimationFactorString = 
        opts["FaceSkinAnomalyDetector.sampledecimationfactor"].as<std::string>();
      if (matchRegex("[0-9]+", decimationFactorString))
        fsad.sampleDecimationFactor = boost::lexical_cast<int>(decimationFactorString);
      else if (matchRegex("auto\\([0-9]+\\)", decimationFactorString)) {
        fsad.sampleDecimationFactor = 0;
        fsad.autoDecimationTarget = boost::lexical_cast<int>(replaceRegex("auto\\(([0-9]+)\\)", "\\1", decimationFactorString));
      }
      else
        throw ConfigurationFileException("Invalid sample decimation factor value specified! Must be an integer > 1 or a string of the form auto(<int>)!");
    }
    if (opts.count("FaceSkinAnomalyDetector.norm"))
      fsad.norm = opts["FaceSkinAnomalyDetector.norm"].as<int>();

    if (fsad.kNeighbours < 1)
      throw ConfigurationFileException("Invalid kneighbours value specified (< 1)!");

    if (fsad.sampleDecimationFactor < 1)
      throw ConfigurationFileException("Invalid sample decimation factor value specified (< 1)!");

    if (fsad.norm < 1 || fsad.norm > 2)
      throw ConfigurationFileException("Invalid norm value specified (must be 1 or 2)!");
                                       
    return new FaceSkinAnomalyDetector(fsad);
  }



  bool FaceSkinAnomalyDetector::processRangeImplementation(frame_number_t first, frame_number_t last, UiCallback*) {
    if (!isTrained)
      train(first, last);

    return Component::processRangeImplementation(first, last, nullptr);
  }

  bool FaceSkinAnomalyDetector::train(frame_number_t first, 
                                      frame_number_t last) {
    if (debug == 1)
      std::cerr << "Training... ";

    for (frame_number_t i = first; i < last; ++i) {
      if (debug > 1) 
        std::cerr << "Training frame " << i << "..." << std::endl;

      BlackBoardPointer<std::list<BodyPart> > bodyParts(getBlackBoard().get<std::list<BodyPart>>(i, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY));
      const BodyPart* head = nullptr;
      for (auto it = bodyParts->cbegin(); it != bodyParts->cend(); ++it)
        if (it->getIdentity() == BodyPart::HEAD) 
          head = &*it;

      if (head == nullptr) {
        if (debug > 2)
          std::cerr << "FaceSkinAnomalyDetector: no head in frame " << i << std::endl;
        continue;
      }
      
      cv::Mat frame(getFrameSource()[i]);
      cv::Mat frame32f(frame.size(), CV_32FC3);
      frame.convertTo(frame32f, CV_32FC3);
      frame32f *= 1./255;

      cv::cvtColor(frame32f, frame, CV_BGR2Lab);
      assert(frame.type() == CV_32FC3);
      cv::Mat faceMask(head->getBlob().toMatrix(frame.size()));
      BlackBoardPointer<std::vector<cv::Point2f> > landmarks(getBlackBoard().get<std::vector<cv::Point2f>>(i, FACIAL_LANDMARK_BLACKBOARD_ENTRY));
      assert(landmarks->size() == 8);
      int decimator = 0; // a value in Z_sampleDecimationFactor; decimate the input sample if
                         // non-zero
      for (int u = 0; u < faceMask.rows; ++u) {
        for (int v = 0; v < faceMask.cols; ++v) {
          if (faceMask.at<uchar>(u,v) > 0 && 
              (decimator++ % sampleDecimationFactor) == 0) {
            Vec8f key;
            cv::Point2f current(v,u);
            for (int j = 0; j < 8; ++j)
              key[j] = cv::norm((*landmarks)[j] - current);
            sampleVectors.push_back(std::make_pair(key, frame.at<cv::Vec3f>(current)));
          }
        }
      }
    }

    if (autoDecimationTarget > 0) {
      int decimationFactor = std::ceil(static_cast<double>(autoDecimationTarget) / 
                                       sampleVectors.size());
      if (debug > 1)
        std::cerr << "Estimated decimation factor: " << decimationFactor << std::endl
                  << "Decimating...";

      int decimator = 0;
      auto it = sampleVectors.begin();
      while (it != sampleVectors.end()) {
        if ((decimator++ % decimationFactor) > 0)
          sampleVectors.erase(it++);
        else
          ++it;
      }
    }
    
    if (debug > 0)
      std::cerr << "OK" << std::endl;
    
    if (debug > 1)
      std::cerr << "Got " << sampleVectors.size() << " neighbour samples!" 
                << std::endl;

    isTrained = true;

    return true;
  }



  boost::program_options::options_description FaceSkinAnomalyDetector::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description desc;
    desc.add_options()
      ("FaceSkinAnomalyDetector.kneighbours",
       boost::program_options::value<int>()->default_value(1),
       "Number of closest neighbour pixels to consider")
      ("FaceSkinAnomalyDetector.sampledecimationfactor",
       boost::program_options::value<std::string>()->default_value("1"),
       "A factor by which training samples are decimated (i.e. given value n, "
       "only every nth value is preserved. Alternatively, the value can be "
       "specified as auto(<int>) where <int> is a target number of samples to "
       "preserve.")
      ("FaceSkinAnomalyDetector.norm",
       boost::program_options::value<int>()->default_value(2),
       "L-norm to use when selecting the nearest neighbour (may be 1 or 2)");
    return desc;
  }
}
#endif // SLMOTION_ENABLE_LIBFLANDMARK
