#ifdef SLMOTION_ENABLE_LIBFLANDMARK
#include "FacialLandmarkDetector.hpp"
#include "BlackBoardDumpWriter.hpp"
#include "util.hpp"
#include "BodyPartCollector.hpp"
#include "TrackingSkinDetector.hpp"
#include "VideoFileSource.hpp"
#include <fstream>

#define DUMP_DATA 0

namespace slmotion {
  extern int debug;
  static FacialLandmarkDetector DUMMY(true);

  FacialLandmarkDetector::FacialLandmarkDetector(bool) : 
    Component(true), model(NULL) { 
    BlackBoardDumpWriter::registerAnyWriter<std::vector<cv::Point2f> >();
  }

  extern BlackBoard* vvi_blackBoard;
  extern FrameSource* vvi_frameSource;

  FacialLandmarkDetector::~FacialLandmarkDetector() {
    if (model) // freeing NULL seems to cause a SEGFAULT
      flandmark_free(model);
  }

  Component::property_set_t FacialLandmarkDetector::getRequirements() const {
    property_set_t props { FACEDETECTOR_BLACKBOARD_ENTRY };
    if (enableConfidence)
      props.insert(SKINDETECTOR_BLACKBOARD_MASK_ENTRY);
    return props;
  }



  static char modelBuffer[PATH_MAX];
  const char* getLibFlandmarkModel() {
    std::string model =
#ifdef SLMOTION_LIBFLANDMARK_MODEL
      SLMOTION_LIBFLANDMARK_MODEL;
#else // SLMOTION_LIBFLANDMARK_MODEL
      getInstallPrefix() + "/share/flandmark/flandmark_model.dat";
#endif // SLMOTION_LIBFLANDMARK_MODEL
    std::ifstream ifs(model);
    if (!ifs.good())
      throw IOException("Could not open " + model);
    ifs.close();
    strcpy(modelBuffer, model.c_str());
    return modelBuffer;
  }



  std::vector<cv::Point2f> FacialLandmarkDetector::getPointsForFace(const cv::Rect& face, 
                                                                    const cv::Mat& inFrame) {
    if (face == ViolaJonesFaceDetector::INVALID_FACE)
      throw SLMotionException("Could not detect facial landmarks: INVALID_FACE supplied");

    std::vector<cv::Point2f> points;
    cv::Mat greyscale;
    cv::cvtColor(inFrame, greyscale, CV_BGR2GRAY);
    int bbox[] = { face.tl().x, face.tl().y, face.br().x, face.br().y };
    std::vector<double> detection(2*model->data.options.M);
    IplImage greyscaleIpl = IplImage(greyscale);
    flandmark_detect(&greyscaleIpl, bbox, model, &detection[0]);

    for (int i = 0; i < 2*model->data.options.M; i += 2) {
      // int px = int(detection[i]*sf[0] + face.tl().x);
      // int py = int(detection[i+1]*sf[1] + face.tl().y);
      points.push_back(cv::Point2f(detection[i], detection[i+1]));
    }

    return points;
  }


  void FacialLandmarkDetector::process(frame_number_t frnumber) {
    cv::Mat inFrame = getFrameSource()[frnumber];
    BlackBoardPointer<cv::Rect> facePtr = getBlackBoard().get<cv::Rect>(frnumber, FACEDETECTOR_BLACKBOARD_ENTRY);
    cv::Rect face = *facePtr;

    if (face == ViolaJonesFaceDetector::INVALID_FACE)
      throw SLMotionException("Could not detect facial landmarks: INVALID_FACE supplied");

    if (enableConfidence) {
      double confidence;
      BlackBoardPointer<std::list<BodyPart> > bodyPartPointer = getBlackBoard().get<std::list<BodyPart>>(frnumber, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY);
      const std::list<BodyPart>& bodyparts = *bodyPartPointer;
      auto it = bodyparts.cbegin();
      while (it != bodyparts.cend() && !it->isA(BodyPart::HEAD))
        ++it;

      if (it == bodyparts.cend())
        confidence = 0;
      else {
        cv::Mat skinMask = it->getBlob().toMatrix(getFrameSource()[frnumber].size());
        cv::Mat faceMask = skinMask(face);
        confidence = cv::countNonZero(faceMask);
        confidence /= (faceMask.cols * faceMask.rows);
      }
      getBlackBoard().set<double>(frnumber, 
                                  FACIAL_LANDMARK_CONFIDENCE_BLACKBOARD_ENTRY,
                                  confidence);
      if (confidence < confidenceThreshold) {
        // put an empty vector in and terminate
        getBlackBoard().set<std::vector<cv::Point2f>>(frnumber,
                                                      FACIAL_LANDMARK_BLACKBOARD_ENTRY,
                                                      std::vector<cv::Point2f>());
        return;
      }
    }

    std::vector<cv::Point2f> points = getPointsForFace(face, inFrame);

    getBlackBoard().set(frnumber, FACIAL_LANDMARK_BLACKBOARD_ENTRY, points);

    if (DUMP_DATA) {
      std::cout << std::endl
		<< face.tl().x << " " << face.tl().y << " "
		<< face.br().x << " " << face.br().y;
      for (auto i=points.begin(); i!=points.end(); i++)
	std::cout << "   " << i->x << " " << i->y;
      std::string label = getFrameSource().getFileName();
      size_t p = label.rfind('/');
      if (p!=std::string::npos)
	label.erase(0, p+1);
      p = label.rfind('.');
      if (p!=std::string::npos)
	label.erase(p);

      if (dynamic_cast<VideoFileSource*>(&getFrameSource())) {
	p = label.rfind(':');
	if (p==std::string::npos) {
	  std::stringstream ss;
	  ss << ":" << frnumber;
	  label += ss.str();
	}
      }
      std::cout << "   " << label << std::endl;
    }

  }

  boost::program_options::options_description FacialLandmarkDetector::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description desc;
    desc.add_options()
      ("FacialLandmarkDetector.enableconfidence", 
       boost::program_options::value<bool>()->default_value(false),
       "If enabled, a confidence value will be stored along with the landmark locations")
      ("FacialLandmarkDetector.confidencethreshold",
       boost::program_options::value<double>()->default_value(0.65),
       "Sets the minimum confidence before discarding the result")
      ("FacialLandmarkDetector.enablelowpassfilter",
       boost::program_options::value<bool>()->default_value(false),
       "If enabled, the detection results will be low pass filtered")
      ("FacialLandmarkDetector.lowpasscutoff",
       boost::program_options::value<double>()->default_value(8.0),
       "Sets the cut-off frequency for low-pass filter, defined as N/(value here)")
      ("FacialLandmarkDetector.pruneandinterpolate",
       boost::program_options::value<bool>()->default_value(false),
       "If enabled, the detection will be post-processed by rejecting detections in frames where their positions differ too much from typical. The rejected values are replaced by interpolation.")
;
    return desc;
  }

  Component* FacialLandmarkDetector::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const {
    FacialLandmarkDetector f(blackBoard, frameSource);
    if (configuration.count("FacialLandmarkDetector.enableconfidence"))
      f.enableConfidence = configuration["FacialLandmarkDetector.enableconfidence"].as<bool>();
    if (configuration.count("FacialLandmarkDetector.confidencethreshold"))
      f.confidenceThreshold = 
        configuration["FacialLandmarkDetector.confidencethreshold"].as<double>();
    if (configuration.count("FacialLandmarkDetector.enablelowpassfilter"))
      f.enableLowpassFilter = 
        configuration["FacialLandmarkDetector.enablelowpassfilter"].as<bool>();
    if (configuration.count("FacialLandmarkDetector.lowpasscutoff"))
      f.lowPassCutoff = 
        configuration["FacialLandmarkDetector.lowpasscutoff"].as<double>();
    if (configuration.count("FacialLandmarkDetector.pruneandinterpolate"))
      f.pruneandinterpolate = 
        configuration["FacialLandmarkDetector.pruneandinterpolate"].as<bool>();

    return new FacialLandmarkDetector(f);
  }

  Component::property_set_t FacialLandmarkDetector::getProvided() const {
    property_set_t props { FACIAL_LANDMARK_BLACKBOARD_ENTRY };
    if (enableConfidence)
      props.insert(FACIAL_LANDMARK_CONFIDENCE_BLACKBOARD_ENTRY);
    return props;
  }



  bool FacialLandmarkDetector::processRangeImplementation(frame_number_t first, 
                                                          frame_number_t last, 
                                                          UiCallback* uiCallback) {
    if (slmotion::debug > 0)
      std::cerr << "First pass (detection)..." << std::endl;
    for (size_t i = first; i < last; ++i) {
      if (slmotion::debug > 0) {
        if (slmotion::debug == 1)
          std::cerr << '\r';
        std::cerr << i + 1 - first << '/' << last - first;
        if (slmotion::debug > 1)
          std::cerr << std::endl;
      }
      this->process(i);
      if (uiCallback != nullptr && !(*uiCallback)(i*100./(last-first)))
        return false;
    }
    if (slmotion::debug == 1)
      std::cerr << std::endl;

    if (enableLowpassFilter) {
      if (slmotion::debug > 0)
        std::cerr << "Second pass (low-pass filter)..." << std::endl;
      lowpassfilter(first, last);
    }

    if(pruneandinterpolate){
      if (slmotion::debug > 0)
        std::cerr << "Pruning and interpolating..." << std::endl;
      pruneAndInterpolate(first, last);
    }

    return true;
  }



  /**
   * Converts a vector of points into an OpenCV column matrix
   */
  static cv::Mat pointVectorToColumnVector(const std::vector<cv::Point2f>& points) {
    cv::Mat vec(points.size()*2, 1, CV_32FC1, cv::Scalar::all(0));
    for (size_t i = 0; i < points.size(); ++i) {
      vec.at<float>(2*i, 0) = points[i].x;
      vec.at<float>(2*i+1, 0) = points[i].y;
    }
    return vec;
  }



  /**
   * Does the reverse
   */
  static std::vector<cv::Point2f> columnVectorToPointVector(const cv::Mat& vec) {
    assert(vec.rows % 2 == 0);
    assert(vec.type() == CV_32FC1);
    assert(vec.cols == 1);
    std::vector<cv::Point2f> points;
    for (int i = 0; i < vec.rows; i += 2)
      points.push_back(cv::Point2f(vec.at<float>(i,0), vec.at<float>(i+1,0)));
    return points;
  }



  void FacialLandmarkDetector::lowpassfilter(frame_number_t first, frame_number_t last) {
    assert(last > first);
    // replace missing detections with a 10-point moving average
    cv::Mat movingAverage(16, 1, CV_32FC1, cv::Scalar::all(0));
    cv::Mat values(16, last-first, CV_32FC1, cv::Scalar::all(0));
    std::list<cv::Mat> vecs;
    
    for (frame_number_t frame = first; frame < last; ++frame) {
      BlackBoardPointer<std::vector<cv::Point2f> > pointsPtr = 
        getBlackBoard().get<std::vector<cv::Point2f>>(frame, FACIAL_LANDMARK_BLACKBOARD_ENTRY);
      const std::vector<cv::Point2f>& points = *pointsPtr;
      cv::Mat vec;
      if (points.size() > 0)
        vec = pointVectorToColumnVector(points);
      else
        vec = movingAverage.clone();

      vecs.push_back(vec);
      if (vecs.size() > 10) {
        movingAverage = movingAverage - 0.1 * vecs.front();
        vecs.pop_front();
      }
      movingAverage = movingAverage + 0.1*vecs.back();

      for (int i = 0; i < values.rows; ++i)
        values.at<float>(i, frame-first) = vec.at<float>(i,0);
    }

    // apply low-pass filtering
    cv::Mat valuesDFT;
    cv::dft(values, valuesDFT, cv::DFT_ROWS);
    for (int j = 0; j < valuesDFT.cols; ++j)
      for (int i = 0; i < valuesDFT.rows; ++i)
        valuesDFT.at<float>(i,j) = 
          valuesDFT.at<float>(i,j) * (1./(1.+std::pow(j/(valuesDFT.cols/8.), 2)));

    cv::dft(valuesDFT, values, cv::DFT_ROWS + cv::DFT_INVERSE + cv::DFT_SCALE);

    // then re-store values to the black board
    for (frame_number_t frame = first; frame < last; ++frame) 
      getBlackBoard().set(frame, FACIAL_LANDMARK_BLACKBOARD_ENTRY,
                          columnVectorToPointVector(values.colRange(frame-first,
                                                                    frame-first+1)));
                          
  }

  void FacialLandmarkDetector::pruneAndInterpolate(frame_number_t first, frame_number_t last) {
    assert(last > first);

   localBlackBoardType bb; 

   // this is a bit dirty...

   vvi_blackBoard=&(getBlackBoard());
   vvi_frameSource=&(getFrameSource());

   for(int f=first;f<(int)last;f++){

 BlackBoardPointer<std::vector<cv::Point2f> > pointsPtr = 
        getBlackBoard().get<std::vector<cv::Point2f>>(f, FACIAL_LANDMARK_BLACKBOARD_ENTRY);
 //      const std::vector<cv::Point2f>& points = *pointsPtr;

 bb[std::pair<int,std::string>(f,"faciallandmarks")]=*pointsPtr;

      //     bb[std::pair<int,std::string>(f,"faciallandmarks")]=

       //vvi_blackBoard->get<std::vector<cv::Point2f> >(f, FACIAL_LANDMARK_BLACKBOARD_ENTRY);
   }


   interpolateLandmarksToBB(first,last-1,bb);

   for (frame_number_t frame = first; frame < last; ++frame) 
     //     getBlackBoard().set(frame, FACIAL_LANDMARK_BLACKBOARD_ENTRY,
     //			 *boost::any_cast<BlackBoardPointer<std::vector<cv::Point2f> > >(bb[std::pair<int,std::string>(frame,"faciallandmarks")]));
//           getBlackBoard().set(frame, FACIAL_LANDMARK_BLACKBOARD_ENTRY,
//      			 *boost::any_cast<BlackBoardPointer<std::vector<cv::Point2f> > >(bb[std::pair<int,std::string>(frame,"faciallandmarks")]));
        getBlackBoard().set(frame, FACIAL_LANDMARK_BLACKBOARD_ENTRY,
			 *boost::any_cast<std::vector<cv::Point2f> >(&bb[std::pair<int,std::string>(frame,"faciallandmarks")]));

  }

}
#endif
