#include "Filter.hpp"
#include "TrackingSkinDetector.hpp"

using cv::Mat;
using cv::Scalar;
using cv::Point;
using cv::Size;
using std::vector;

namespace slmotion {
  PostProcessFilter::~PostProcessFilter() { }



  void PostProcessCropFilter::apply(const cv::Mat&, cv::Mat& outMask) const {
    // crop
    const int magicx = cropArea.tl().x;
    const int magicy = cropArea.tl().y;
    const int magicw = cropArea.width;
    const int magich = cropArea.height;
    if (magicx != -1 || magicy != -1 || magicw != 0 || magich != 0) {
      Mat cropmask = Mat(outMask.size(), CV_8UC1, Scalar::all(0));
      rectangle(cropmask, Point(magicx, magicy),
                Point(magicx + magicw, magicy + magich), Scalar::all(255),
                CV_FILLED);
      Mat croptemp(outMask.size(), CV_8UC1, Scalar::all(0));
      outMask.copyTo(croptemp, cropmask);
      outMask = croptemp;
    }
  }



  void PostProcessHoleFillerFilter::apply(const Mat&, Mat& m) const {
    assert(m.type() == CV_8UC1);
    vector< vector<Point> > contours;
    cv::findContours(m, contours, CV_RETR_EXTERNAL, CV_CHAIN_APPROX_SIMPLE);

    m = Scalar::all(0);
    drawContours(m, contours, -1, Scalar::all(255), CV_FILLED);
  }



  void PostProcessMorphologicalTransformationFilter::apply(const Mat&, Mat& m) const {
    switch(this->type) {
    case Type::OPEN:
      cv::morphologyEx(m, m, cv::MORPH_OPEN, 
                       cv::getStructuringElement(cv::MORPH_RECT, Size(3, 3)), 
                       Point(-1,-1), this->iterations);
      break;
    case Type::CLOSE:
      cv::morphologyEx(m, m, cv::MORPH_CLOSE, 
                       cv::getStructuringElement(cv::MORPH_RECT, Size(3, 3)), 
                       Point(-1,-1), this->iterations);
      break;

    case Type::DILATE:
      cv::dilate(m, m, cv::getStructuringElement(cv::MORPH_RECT,Size(3, 3)),
                 Point(-1,-1), this->iterations);
      break;

    case Type::ERODE:
      cv::erode(m, m, cv::getStructuringElement(cv::MORPH_RECT,Size(3, 3)),
                Point(-1,-1), this->iterations);
      break;
    }
  }



  void PostProcessWatershedFilter::apply(const cv::Mat& inColour, cv::Mat& target) const {
    assert(inColour.type() == CV_8UC3);
    assert(target.type() == CV_8UC1);
    assert(inColour.size() == target.size());

    Mat dilatedTarget;
    dilate(target, dilatedTarget, Mat(), Point(-1,-1), this->dilateAmount);

    vector< vector<Point> > contours;
    cv::findContours(target, contours, CV_RETR_EXTERNAL, CV_CHAIN_APPROX_SIMPLE);
    target = Scalar::all(0);

    Mat markers(target.size(), CV_32SC1, Scalar::all(1));
    markers.setTo(Scalar::all(0), dilatedTarget);

    for (size_t i = 0; i < contours.size(); ++i) {
      drawContours(markers, contours, i, Scalar::all(i+2), -1);
    }
    cv::watershed(inColour, markers);

    for (int i = 0; i < markers.rows; ++i) {
      for (int j = 0; j < markers.cols; ++j) {
        int mark = markers.at<int>(i,j);
        if (mark > 1 && mark < static_cast<int>(contours.size()+2))
          target.at<uchar>(i,j) = 255;
        else
          target.at<uchar>(i,j) = 0;
      }
    }
  }

    void SmallComponentRemoverFilter::apply(const Mat&, Mat& m) const {
    assert(m.type() == CV_8UC1);

    Mat ulbl;
    labelConnectedComponents(m,ulbl);

    std::map<uchar,size_t> counts;

    for(int x=0;x<ulbl.cols;x++)
      for(int y=0;y<ulbl.rows;y++)
	counts[ulbl.at<uchar>(y,x)]++;

    for(int x=0;x<ulbl.cols;x++)
      for(int y=0;y<ulbl.rows;y++){
	uchar l=ulbl.at<uchar>(y,x);
	if(l && counts[l]<sizeThreshold)
	  m.at<uchar>(y,x)=0;
      }

  }
}
