#ifndef SLMOTION_FRAMESOURCE
#define SLMOTION_FRAMESOURCE

#include <unordered_map>
#include <list>
#include <iterator>
#include <stdexcept>
#include <memory>
#ifdef SLMOTION_THREADING
#include <thread>
#endif

#include "opencv.hpp"

#include "exceptions.hpp"

namespace slmotion {
  /**
   * This simple class represents the track type etc.
   */
  struct TrackInfo {
    /**
     * Track type
     */
    enum class Type {
      BGR_IMAGE, ///< 8-bit BGR image (the default track should be of this type)
      DEPTH_DATA, ///< Depth data (Kinect etc.)
      META_TRACK, ///< This value is for combined frame sources
      PCM_SOUND, ///< Ordinary PCM-encoded wave sound
      RAW_VECTOR, ///< Raw one-dimensional data
      RAW_MATRIX ///< Raw two-dimensional data
    } type;
    std::string trackName; ///< Name of the track (can be used to identify it)
  };



  /**
   * This abstract class presents an object that provides random access to
   * an ordered sequence of frames, i.e. the video.
   */
  class FrameSource {
  public:
    class Iterator;

    typedef size_t frame_number_type;
    typedef ptrdiff_t difference_type;
    typedef Iterator const_iterator;
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator;

    class Iterator {
    public:
      typedef ptrdiff_t difference_type;
      typedef std::random_access_iterator_tag iterator_category;
      typedef const cv::Mat value_type;
      typedef value_type* pointer;
      typedef value_type& reference;

      /**
       * Prefix increment
       */
      inline Iterator& operator++() {
        ++currentFrame;
        return *this;
      }



      /**
       * Postfix increment
       */
      Iterator operator++(int);



      /**
       * Prefix decrement
       */
      inline Iterator& operator--() {
        --currentFrame;
        return *this;
      }



      /**
       * Postfix decrement
       */
      Iterator operator--(int);


      /**
       * Ordinary addition operator for FrameSource iterators.
       *
       * @amount Amount to increment
       *
       * @param A copy of the lhs iterator, incremented by the given amount
       */
      Iterator operator+(difference_type amount) const;



      /**
       * Ordinary substraction operator for FrameSource iterators.
       *
       * @amount Amount to increment
       *
       * @param A copy of the lhs iterator, decremented by the given amount
       */
      Iterator operator-(difference_type amount) const;



      /**
       * A difference operation: returns the number of iterations
       * required to reach the rhs iterator from lhs, i.e. the x in
       * lhs + x = rhs
       *
       * @return The number of required iterations
       */
      difference_type operator-(const Iterator& rhs) const;



      // non-member friend declarations (for rhs addition/substraction)
      friend FrameSource::Iterator operator+(FrameSource::difference_type, 
                                             const FrameSource::Iterator&);
      friend Iterator operator-(difference_type, const Iterator&);

      // comparison operators
      /**
       * The standard equality operator. Compares member-by-member.
       *
       * @param other What to compare with
       */
      bool operator==(const Iterator& other) const {
        return other.currentFrame == currentFrame &&
          other.frameSource == frameSource;
      }



      /**
       * The standard inequality operator. The exact opposite of ==
       *
       * @param other What to compare with
       */
      bool operator!=(const Iterator& other) const {
        return !(*this == other);
      }



      /**
       * Returns true if and only if both iterators refer to the same 
       * source object and the LHS iterator has a current frame number
       * that is smaller than that of the RHS iterator
       *
       * @param other RHS iterator
       */
      inline bool operator<(const Iterator& other) const {
        return frameSource == other.frameSource &&
          currentFrame < other.currentFrame;
      }



      /**
       * Returns true if and only if both iterators refer to the same 
       * source object and the LHS iterator has a current frame number
       * that is smaller than or equal to that of the RHS iterator
       *
       * @param other RHS iterator
       */
      inline bool operator<=(const Iterator& other) const {
        return frameSource == other.frameSource &&
          currentFrame <= other.currentFrame;
      }



      /**
       * Returns true if and only if both iterators refer to the same 
       * source object and the LHS iterator has a current frame number
       * that is greater than that of the RHS iterator
       *
       * @param other RHS iterator
       */
      inline bool operator>(const Iterator& other) const {
        return frameSource == other.frameSource &&
          currentFrame > other.currentFrame;
      }



      /**
       * Returns true if and only if both iterators refer to the same 
       * source object and the LHS iterator has a current frame number
       * that is greater than or equal to that of the RHS iterator
       *
       * @param other RHS iterator
       */
      inline bool operator>=(const Iterator& other) const {
        return frameSource == other.frameSource &&
          currentFrame >= other.currentFrame;
      }



      /**
       * Ordinary constructor. Sets values.
       *
       * @param associatedSource Pointer to the associated source object
       * @param initialFrame Initial frame number
       *
       */
      explicit Iterator(FrameSource* associatedSource = NULL,
                        size_t initialFrame = 0) :
        currentFrame(initialFrame), frameSource(associatedSource) {}



      /**
       * The ordinary copy constructor
       *
       * @param other Another iterator that is to be copied
       */
      Iterator(const Iterator& other) :
        currentFrame(other.currentFrame),
        frameSource(other.frameSource) {}



      /**
       * Regular copy-assignment
       */
      Iterator& operator=(const Iterator& other) {
        if(&other != this) {
          this->currentFrame = other.currentFrame;
          this->frameSource = other.frameSource;
        }
        return *this;
      }



      /**
       * Additive assignment. Increments the iterator by the given 
       * amount.
       *
       * @param amount Amount to increment
       */
      inline Iterator& operator+=(difference_type amount) {
        currentFrame += amount;
        return *this;
      }



      /**
       * Subtractive assignment. Decrements the iterator by the given 
       * amount.
       *
       * @param amount Amount to decrement
       */
      inline Iterator& operator-=(difference_type amount) {
        currentFrame -= amount;
        return *this;
      }



      // dereferencing
      pointer operator->() const {
        return &(*frameSource)[currentFrame];
      }



      /**
       * The dereferencing operator. Returns a const reference to the frame.
       *
       * It is up to the caller to make sure that the data is not modified.
       */
      const reference operator*() const {
        return (*frameSource)[currentFrame];
      }



      /**
       * Returns the specified frame. The actual frame is computed relative 
       * to the current frame. I.e. if current frame number is x and the 
       * relative frame number is (+/-)y, then the retrieved frame would be
       * x(+/-)y
       *
       * @param relativeFramenumber A signed integer, the relative number of 
       * frames between current frame and the desired frame
       */
      inline const reference operator[](difference_type relativeFramenumber) const {
        if (relativeFramenumber < 0 && 
            static_cast<frame_number_type>(std::abs(relativeFramenumber)) > 
            currentFrame)
          throw std::out_of_range("Cannot get a negative frame");
        return (*frameSource)[currentFrame + relativeFramenumber];
      }


      
      /**
       * Gets the current frame number
       *
       * @return The current frame number
       */    
      inline frame_number_type getCurrentFramenumber() {
        return currentFrame;
      }



    private:
      size_t currentFrame; ///< Internal frame counter
      FrameSource* frameSource; ///< Associated source object
    };



    // iterator functions
    inline const_iterator begin() {
      return const_iterator(this, 0);
    }



    inline const_iterator end() {
      return const_iterator(this, size());
    }



    /**
     * C++0x style const_iterator cbegin function
     */
    inline const_iterator cbegin() {
      return const_iterator(this, 0);
    }



    /**
     * C++0x style const_iterator cend function
     */
    inline const_iterator cend() {
      return const_iterator(this, size());
    }



    const_reverse_iterator rbegin() {
      return const_reverse_iterator(end());
    }



    const_reverse_iterator rend() {
      return const_reverse_iterator(begin());
    }



    /**
     * C++0x style crbegin()
     */
    const_reverse_iterator crbegin() const; 



    /**
     * C++0x style crend()
     */
    const_reverse_iterator crend() const;



    /**
     * Pure virtual destructor to prevent instantiation
     */
    virtual ~FrameSource() = 0;



    /**
     * Constructs the framesource with the given cache size
     *
     * @param cacheSize Desired maximum cache size
     */
    explicit FrameSource(size_t cacheSize = SIZE_MAX) :
      maxCacheSize(cacheSize),
      aspect(-1),
      targetScale(-1, -1),
      trackInfo(TrackInfo { 
          TrackInfo::Type::BGR_IMAGE, ""
            })
    {}



    /**
     * Constructs the framesource with the given cache size
     *
     * @param cacheSize Desired maximum cache size
     */
    explicit FrameSource(size_t cacheSize, 
                         TrackInfo trackInfo) :
      maxCacheSize(cacheSize),
      aspect(-1),
      targetScale(-1, -1),
      trackInfo(trackInfo)
    {}
    


    inline void setTrackInfo(const TrackInfo& newInfo) {
      trackInfo = newInfo;
    }



    /**
     * Returns a const reference to a copy of the frame at the given
     * frame number on the default track. WARNING: It is still
     * possible to manipulate the data of the matrix through the data
     * member pointer; even though the matrix itself is const,
     * constness only applies to the pointer -- the data is not
     * const. Therefore, care should be taken not to corrupt the
     * internal cache because frames in the cache may be retrieved and
     * replaced at any given time.
     *
     * @param frameNumber 
     */
    virtual const cv::Mat& operator[](frame_number_type frameNumber) = 0;
 


    /**
     * Returns the total number of frames
     */
    virtual frame_number_type size() const = 0;



    inline void setScale(const cv::Size& newScale) {
      targetScale = newScale;
    }



    inline void setAspect(double d) {
      aspect = d;
    }



    /**
     * Returns the total number of tracks for the given frame source
     */
    virtual size_t getNumberOfTracks() const {
      return 1;
    }



    /**
     * Returns the frame source corresponding to the given track number
     *
     * @param trackNumber A 0-based track number
     * @return A reference to the corresponding frame source object
     */
    virtual FrameSource& getTrack(size_t trackNumber) {
      if (trackNumber >= getNumberOfTracks())
        throw std::out_of_range("The given track number exceeds the number of tracks.");

      return *this;
    }



    /**
     * Returns the number of the default track (should be a BGR track)
     */
    virtual size_t getDefaultTrackNumber() const {
      return 0;
    }



    /**
     * Returns the default track
     */
    inline FrameSource& getDefaultTrack() {
      return getTrack(getDefaultTrackNumber());
    }



    /**
     * Returns information about the track
     */
    inline TrackInfo getTrackInfo() const {
      return trackInfo;
    }

    /**
     * Returns information about the label
     */
    inline const std::string& getLabel() const {
      return label;
    }

    /**
     * Sets information about the label
     */
    inline void setLabel(const std::string& l) {
      label = l;
    }

    /**
     * Returns information about the file name
     */
    virtual const std::string& getFileName() const {
      static std::string empty;
      return empty;
    }

    /**
     * Returns the number of frames per second which can be used to
     * compute the timecodes. Timecodes are reported in milliseconds.
     */
    virtual double getFps() const = 0;

    /**
     * Converts the given timecode (in ms) to a frame number
     */
    inline frame_number_type timecodeToFrameNumber(double timecode) const {
      return std::round((timecode/1000.)*getFps());
    }

    /**
     * Converts the given frame number to a timecode (in ms)
     */
    inline double timecodeToFrameNumber(frame_number_type frnumber) const {
      return (frnumber/getFps())*1000.;
    }



  protected:
    /**
     * Returns true if the frame is in cache
     */
    inline bool frameInCache(frame_number_type frameNumber) {
#ifdef SLMOTION_THREADING
    std::lock_guard<std::mutex> lk(this->mutex);
#endif
      return frameCache.count(frameNumber);
    }



    /**
     * Obtains a frame from the cache
     */
    inline const cv::Mat& getFrameFromCache(frame_number_type frameNumber) {
      assert(frameInCache(frameNumber));
#ifdef SLMOTION_THREADING
    std::lock_guard<std::mutex> lk(this->mutex);
#endif
      return *frameCache[frameNumber];
    }



    /**
     * Puts the frame into cache
     */
    void cacheFrame(frame_number_type frameNumber, const cv::Mat& frame);

  //   void init(size_t cacheSize) {
  //     this->trackInfo = TrackInfo { 
  //       TrackInfo::Type::BGR_IMAGE, ""
  //     };
  //     this->maxCacheSize = cacheSize;
  //     this->aspect = -1;
  //     this->targetScale = cv::Size(-1, -1);
  // }



  private:
    /**
     * Returns a scaled version of the image
     */
    cv::Mat scale(const cv::Mat& src);



    std::unordered_map<frame_number_type, std::shared_ptr<cv::Mat>> frameCache; ///< Internal frame cache
    frame_number_type maxCacheSize; ///< Maximum size for the cache
    std::list<frame_number_type> fifoIndices; ///< First-In First-Out indices of frames in the order they were first retrieved; to make it possible that the frame cache will be emptied beginning from the oldest frames
    double aspect; ///< The target aspect ratio. If negative, no correction is performed.
    cv::Size targetScale; ///< Scaling parameter. The video is scaled to the target resolution, unless either member of the pair is negative, in which case no scaling is performed at all.
    TrackInfo trackInfo; ///< Information about the track
    std::string label;   ///< Label of the object if from PicSOM database

#ifdef SLMOTION_THREADING
    std::mutex mutex; ///< If threaded access is used, this object will be used for mutual exclusion
#endif
  }; // class FrameSource



  /**
   * Ordinary addition operator for FrameSource iterators.
   *
   * @amount Amount to increment
   * @rhs Right-hand-sided iterator
   *
   * @param A copy of the rhs iterator, incremented by the given amount
   */
  FrameSource::Iterator operator+(FrameSource::difference_type amount,
                                  const FrameSource::Iterator& rhs);



  /**
   * Ordinary substraction operator for FrameSource iterators.
   *
   * @amount Amount to decrement
   * @rhs Right-hand-sided iterator
   *
   * @param A copy of the rhs iterator, decremented by the given amount
   */
  FrameSource::Iterator operator-(FrameSource::difference_type amount, 
                                  const FrameSource::Iterator& rhs);



  /**
   * This class represents a simple exception when dealing with frames
   */
  class FrameSourceException : public SLMotionException {
  public:
    explicit FrameSourceException(const char* msg) :
      SLMotionException(msg)
    {}

    explicit FrameSourceException(const std::string& msg) :
      SLMotionException(msg.c_str())
    {}

    virtual ~FrameSourceException() throw() {}
  };
}
#endif
