#include "GaussianForegroundClassifier.hpp"

using std::vector;
using std::cerr;
using std::endl;
using cv::Scalar;
using cv::Mat;

namespace slmotion {
  void GaussianForegroundClassifier::train(const vector<vector<float> > & /* negSamples*/,
					   const vector<vector<float> > &  posSamples,
					   const vector<bool> *dimMask,
					   const vector<float>* /*negWeights*/,
					   const vector<float>* /*posWeights*/) {



   // NEGATIVE SAMPLES ARE IGNORED

   vector<vector<size_t> > componentLabels(2);


   size_t dim=posSamples[0].size();
   vector<size_t> activeDim;

   for(size_t d=0;d<dim;d++)
     if(dimMask==NULL || (*dimMask)[d]) activeDim.push_back(d);
   
   dim=activeDim.size();


   cerr << "active dimensionality: " << dim << endl;
   
   cerr << "active components: " << endl;
   
   for(size_t d=0;d<dim;d++){
     cerr << activeDim[d]<< " ";
   }
   cerr << endl;

   
   // compute the covariance matrix
    
   // determine class means

   mean=Mat(dim,1,CV_32FC1,Scalar::all(0));

   for(size_t i=0;i<posSamples.size();i++){
     for(size_t d=0;d<dim;d++)
       mean.at<float>(d,0)+=posSamples[i][activeDim[d]];
   }

   for(size_t d=0;d<dim;d++)
     mean.at<float>(d,0) /= posSamples.size();
 
    
   cerr << "determined mean:" << endl;
   cerr << "  m0= ";
   for(size_t d=0;d<dim;d++){
     cerr << mean.at<float>(d,0);    
     cerr << " ";
   }
   cerr << endl;
  
   // then determine the sample covariance (divide by N instead of N-1)

   Mat covarMat=Mat(dim,dim,CV_32FC1,Scalar::all(0));

   for(size_t i=0;i<posSamples.size();i++){
     for(size_t d1=0;d1<dim;d1++)
       for(size_t d2=0;d2<=d1;d2++)
	 covarMat.at<float>(d1,d2)+=
	   (posSamples[i][activeDim[d1]]-mean.at<float>(d1,0)) *
	   (posSamples[i][activeDim[d2]]-mean.at<float>(d2,0));
   }
   
   for(size_t d1=0;d1<dim;d1++)
     for(size_t d2=0;d2<=d1;d2++){
       covarMat.at<float>(d1,d2) /= posSamples.size();
       if(d1!=d2)
	 covarMat.at<float>(d2,d1) = covarMat.at<float>(d1,d2);
     }

   // covariance matrix ready at this point
  
   inverseCovariance = covarMat.inv(cv::DECOMP_SVD);
   covDetSqrt = sqrt(determinant(covarMat));
   if (covDetSqrt == 0) {
     cerr << "WARNING: The determinant of a covariance matrix was zero. This is most likely due to axes without any variance. Please be advised, the value has been replaced by a bogus value for computation purposes." << endl;
     covDetSqrt = 1;
     
   }
 
   predictioncache.clear();

 }

  float GaussianForegroundClassifier::predict(const std::vector<float>  &sample,
					      const vector<bool> *dimMask){

    const vector<float> *sptr=&sample;
    size_t dim=sample.size();

    vector<float> newSample;
    
    if(dimMask){
      for(size_t d=0;d<dim;d++)
	if((*dimMask)[d])
	  newSample.push_back(sample[d]);
      sptr=&newSample;
    }


    if(predictioncache.count(sample)==0){
      size_t dim=sptr->size();

      assert(dim==(size_t)mean.rows);
      
      float l=0;

      // determine the likelihoods (scaled with the factor 2*pi^dim/2)
    
      double maxval = -DBL_MAX;
      
      Mat xMinusM;
      mean.copyTo(xMinusM);
      for(size_t d=0;d<dim;d++)
	xMinusM.at<float>(d,0) -= (*sptr)[d];
      Mat result = xMinusM.t() * inverseCovariance * xMinusM;
	
      maxval = std::max(maxval, -0.5 * result.at<float>(0,0));
      l=covDetSqrt*exp(maxval);
      
      predictioncache[sample]= l;
    }

    return predictioncache[sample];

  }
}
