#ifndef SLMOTION_GAUSSIAN_SKIN_DETECTOR
#define SLMOTION_GAUSSIAN_SKIN_DETECTOR

#include "SkinDetector.hpp"
#include "FaceDetector.hpp"
#include "GaussianMixture.hpp"

namespace slmotion {
  /**
   * Gaussian Skin Detector, reworked to work with the new framework
   */
  class GaussianSkinDetector : public SkinDetector {
  public:
    explicit GaussianSkinDetector(BlackBoard* blackBoard, 
                                  FrameSource* frameSource,
                                  const ColourSpace& c = *ColourSpace::HSV) :
      SkinDetector(blackBoard, frameSource, c),
      classificationThreshold(-10000),
      totalColourClasses(2), relevantColourClasses(2),
      weights(false)
    { }



    void reset();



    /**
     * Trains the model using the given samples.
     *
     * @param samples Skin samples as colour vectors, one per row. Must be
     * of type CV_32FC1.
     * @param responses Response vectors. Ignored in this version.
     * @param update Sets whether the model should be updated or replaced.
     * Ignored in this version.
     */
    void train(const cv::Mat& samples, const cv::Mat& responses,
               bool update = false);



    /**
     * Reads the image file using the given filename, converts it into skin
     * pixel samples assuming that each non-zero pixel in the image is skin,
     * then trains the model using that particular training data by calling
     * train() as declared above
     *
     * @param filename Image file name
     */
    void train(const std::string& filename);



    /**
     * Reads in frames using the given IO object until the first successful
     * face detection has been performed using the given face detector. Then
     * selects pixels from an elliptical area within the face detection 
     * results, and trains the model.
     *
     * @param faceDetector Pointer to a face detector object that should be
     * used for detection.
     *
     * @return True if the model was trained. False if it was not (e.g. no
     * faces were found).
     */
    bool trainByFaceDetector();



    inline void setClassificationThreshold(double d) {
      classificationThreshold = d;
    }


    
    inline void setWeights(bool b) {
      weights = b;
    }



    /**
     * Sets the number of colour distributions to consider.
     * @param k K (> 0) total number of clusters
     * @param r R (0 < R <= K) the number of relevant distributions used 
     * when classifying pixels
     */
    void setNumberOfDistributions(size_t k, size_t r);



    inline int getNumberOfRelevantDistributions() {
      return relevantColourClasses;
    }



    virtual std::string getShortDescription() const {
      return "Detects skin coloured pixels using a simple Gaussian model";
    }

    virtual std::string getLongDescription() const {
      return "Detects skin coloured pixels using a simple Gaussian model";
    }

    virtual property_set_t getRequirements() const {
      if (trainFilename.length() == 0)
        return property_set_t { FACEDETECTOR_BLACKBOARD_ENTRY };
      return property_set_t();
    }

    virtual std::string getShortName() const {
      return "GaussianSkinDetector";
    }

    virtual std::string getComponentName() const {
      return "Gaussian Skin Detector";
    }

    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;

    // dummy constructor
    GaussianSkinDetector(bool) : SkinDetector(true) {}



    /**
     * Constructs the instance from the given configuration, and passes the
     * configuration also to superclasses
     */
    GaussianSkinDetector(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource);


    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }

  private:
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;

    /**
     * Checks the type of the input matrix and invokes the proper template detect<T>()
     */
    void detect(const cv::Mat& inFrame, cv::Mat& outMask);



    /**
     * Returns the pixel log classification decision from the cache, if 
     * available. Otherwise computes the log probability and stores the 
     * decision in the cache for future use.
     */
    bool getPixelValue(const std::vector<float>& x) {
#ifdef SLMOTION_THREADING
      std::lock_guard<std::mutex> lock(this->mutex);
#endif

      auto it = pixelValueCache.find(x);
      if (it != pixelValueCache.end()) 
        return it->second;

      if (weights)
        return pixelValueCache[x] = (gaussMix.computeLogMixPdf(x) > 
                                     classificationThreshold);
      else
        return pixelValueCache[x] = (gaussMix.computeLogMaxPdf(x) > 
                                     classificationThreshold);

      // return (pixelValueCache[x] = (computeLogProbability(x) > classificationThreshold));
    }


    template<typename T>
    void detect(const cv::Mat& inFrame,
                cv::Mat& outMask) {
      // since
      // p = 1/((2pi)^(k/2) |S|^(1/2)) exp {-(x-m)^T*S^-1*(x-m)/2}
      // examine the corresponding log probability density
      //
      // pl = -0.5 * (x-m)^T*S^-1*(x-m) - (2pi)^(/2) |S|^(1/2)
      
      std::vector<float> values(inFrame.channels());
      
      for (int i = 0; i < outMask.rows; ++i) {
        const T* framePtr = inFrame.ptr<T>(i);
        for (int j = 0; j < outMask.cols; ++j) {
          for (int k = 0; k < inFrame.channels(); ++k) 
            values[k] = *framePtr++;
          outMask.at<uchar>(i,j) = getPixelValue(values) ? 255 : 0;
        }
      }
    }



    GaussianMixture gaussMix;
    double classificationThreshold;
    size_t totalColourClasses;
    size_t relevantColourClasses;
    std::map<std::vector<float>, bool> pixelValueCache; ///< cached classification decisions
    bool weights;
    std::string trainFilename;

#ifdef SLMOTION_THREADING
    std::mutex mutex;
#endif
   
  };
}

#endif
