#ifndef SLMOTION_HOG_DESCRIPTOR
#define SLMOTION_HOG_DESCRIPTOR

#include <opencv2/opencv.hpp>

namespace slmotion {
  /**
   * This class provides a more accessible interface to OpenCV HOG descriptors
   */
  class HOGDescriptor {
  public:
    struct Cell {
      cv::Size size;
      cv::Point tl;
      cv::Point br;
      std::vector<float> bins;
    };

    struct Block {
      cv::Size size;
      cv::Point tl;
      cv::Point br;
      std::vector<Cell> cells;
    };

    // identifies individual values in the HOG vector
    class ValueIdentifier {
    public:
      int blockXIdx;
      int blockYIdx;
      int cellIdx;
      cv::Point tl; // cell tl
      cv::Point br; // cell br
      int binNumber;
      double angle;
      cv::Point2d centroid;

      double distanceTo(const ValueIdentifier& that, double angleWeight) const;
    };



    /**
     * Returns the value identification of given vector index
     */
    ValueIdentifier identify(int i) const;



    /**
     * Computes the HOG descriptors for the given matrix. The
     * dimensions of the matrix must be divisible by 16.
     */
    explicit HOGDescriptor(const cv::Mat& m);



    /**
     * Returns the raw descriptor values
     */
    inline const std::vector<float>& getRawValues() const {
      return hogValues;
    }



    /**
     * Returns a vector whose data corresponds to the content of the
     * raw values.
     */
    inline cv::Mat getHogVector() {
      return cv::Mat(hogValues.size(), 1, CV_32FC1, &hogValues[0]).clone();
    }



    /**
     * Returns the raw descriptor size
     */
    inline int getDescriptorSize() const {
      return hog.nbins * getCellsPerBlock() * getNBlocks();
    }



    /**
     * Returns the block by given indices
     */
    Block getBlock(int x, int y) const;
    


    inline int getCellsPerBlock() const {
      return cellsPerBlock;
    }

    inline int getNBlocks() const {
      return nBlocks;
    }

    inline int getNBlocksX() const {
      return nBlocksX;
    }

    inline int getNBlocksY() const {
      return nBlocksY;
    }

    inline cv::Size getBlockSize() const {
      return blockSize;
    }

    inline cv::Size getCellSize() const {
      return cellSize;
    }

    /**
     * Draws oriented gradients as lines into the image at the given position
     */
    static void drawOrientedGradients(cv::Mat& m, const cv::Point2d& p, 
                                      const std::vector<float>& grads,
                                      double scaleFactor = 1.0);



    /**
     * Visualises the HOG on the input image. The image should be of
     * equal size as the image from which the HOG was computed.
     */
    void visualise(cv::Mat& m) const;



    inline int getNBins() const {
      return nBins;
    }



    /**
     * Returns a symmetric n×n matrix where each value x_ij
     * corresponds to the distance between elements i and j in the HOG
     * vector. The distance d(i,j) is defined as follows:
     *
     * let p_i be the centre point of the cell to which the value i belongs
     *
     * let p_j be likewise the centre point of the cell of j
     *
     * let theta_i be the angle of the bin of i (supposing the bin
     * number is k = i % nBins, then theta_i = k*pi/nBins + pi/2)
     *
     * let theta_j be the corresponding angle of j
     *
     * then d(i,j) = ||p_i-p_j||_2 + w (1-cos^2(theta_i - theta_j))
     * where w is the weight of the angular component of the distance.
     * This represents the fact that the two bins are considered "similar"
     * when the _smaller_ angle between the two vectors is small. Ie. the 
     * vectors can be considered "two-sided". Furthermore, depending on the
     * weight, the relative importance between the euclidean distance of the
     * position and the angle can be defined. At distances >> w, the euclidean
     * distance dominates, whereas when the distance is small, the angular
     * difference becomes dominant.
     */
    cv::Mat generateDistanceMatrix(double w) const;



    /**
     * Supposing the dimensionality of this HOG is n, given another
     * HOG with dimensionality m, returns an m×n matrix where each
     * value x_ij corresponds to the distance between element i in
     * this HOG to the element j in the HOG vector of the other
     * matrix. The distance d(i,j) is defined as above. Similarly, w
     * is the weight of the angular component of the distance.
     */
    cv::Mat generateDistanceMatrix(double w, const HOGDescriptor& that) const;



    /**
     * Given two HOGs, selects *only* those bins in the histograms
     * whose cumulative value (computed in descending order) exceeds
     * the threshold which is between 0 and 1.0. For 0, zero-vectors
     * are produced. For 1.0, behavior is almost identical to above,
     * but bins of zero importance will be trimmed. For values higher
     * than 1.0, the behavior is exactly identical. The remaining
     * components are stored to the weight vectors, and a
     * corresponding distance matrix is computed.
     *
     * maxSize provides an alternative limit for the output vectors by
     * imposing a maximum size that the vectors can have (in terms of
     * bin count).
     *
     * Optionally, a pointer can be passed to store the indices
     * corresponding to the preserved weights.
     */
    void generateTrimmedDistanceMatrix(double w, double threshold, 
                                       size_t maxSize,
                                       const HOGDescriptor& that, 
                                       cv::Mat& outDm,
                                       cv::Mat& outThisWeights,
                                       cv::Mat& outThatWeights,
                                       std::vector<int>* indicesThis = nullptr,
                                       std::vector<int>* indicesThat = nullptr
                                       ) const;



    inline int getValuesPerBlock() const {
      return valuesPerBlock;
    };



    /**
     * Returns a block-normalised copy of the descriptor
     */
    HOGDescriptor getBlockNormalisedCopy() const;



  private:
    std::vector<Cell> constructCells(const std::vector<float>& vals,
                                     const cv::Point& blockTl) const;



    cv::HOGDescriptor hog;
    std::vector<float> hogValues;
    int valuesPerBlock; // how many values in the vector above there are per block
    int nBlocksY;
    int nBlocksX;
    int nBlocks;
    int cellsPerBlock;
    int nBins;
    cv::Size blockSize;
    cv::Size cellSize;
  };
}

#endif
