#ifndef HAND_VISUALISER
#define HAND_VISUALISER

#ifdef SLMOTION_ENABLE_LIBHAND
#include "HandConfiguration.hpp"
#include <hand_renderer.h>

namespace slmotion {
  /**
   * This class serves as an abstraction that will allow LibHand to
   * render the given hand configuration
   */
  class HandVisualiser {
  public:
    /**
     * Renders the given hand pose.
     *
     * @param r Camera distance from the origin
     * @param theta Camera theta (yaw)
     * @param phi Camera phi (pitch)
     * @pram tilt Camera tilt (roll)
     * @param frameSize Requested frame size
     *
     * @return An image of the hand (BGR)
     */
    cv::Mat render(const libhand::FullHandPose& pose, 
                   double r, double theta = 0, double phi = 0, double tilt = 0,
                   int tx = 0, int ty = 0);

    inline cv::Mat render(const libhand::FullHandPose& pose, 
                          double r, double theta, double phi, double tilt,
                          const cv::Size& frameSize) {
      renderer.SetRenderSize(frameSize.width, frameSize.height);
      return render(pose, r, theta, phi, tilt);
    }

    inline cv::Mat render(const HandConfiguration& pose, 
                          double r, double theta = 0, double phi = 0, 
                          double tilt = 0, int tx = 0, int ty = 0) {
      pose.apply(bogusHandPose);
      return render(bogusHandPose, r, theta, phi, tilt, tx, ty);
    }

    inline cv::Mat render(const HandConfiguration& pose, 
                          double r, double theta, double phi, 
                          double tilt, const cv::Size& frameSize) {
      pose.apply(bogusHandPose);
      return render(bogusHandPose, r, theta, phi, tilt, frameSize);
    }

    /**
     * Initialises the visualiser
     *
     * @param sceneSpec Scene specification, a yaml file that
     * specifies the hand scene
     */
    inline explicit HandVisualiser(const std::string& sceneSpec, const cv::Size frameSize = cv::Size(0,0)) {
      init(sceneSpec, frameSize);
    }

    /**
     * Initialises the visualiser with default specs
     */
    inline explicit HandVisualiser(const cv::Size frameSize = cv::Size(0,0)) {
      init(getDefaultSceneSpec(), frameSize);
    }

  private:
    void init(const std::string& sceneSpec, const cv::Size& frameSize = cv::Size(0,0));

    static std::string getDefaultSceneSpec();

    libhand::HandRenderer renderer;
    libhand::SceneSpec sceneSpec;
    // this is here to prevent the need of reinstantiating
    libhand::FullHandPose bogusHandPose; 
  };
}

#endif
#endif
