#ifndef SLMOTION_KINECT_EXAMPLE_COMPONENT
#define SLMOTION_KINECT_EXAMPLE_COMPONENT

#include "Component.hpp"

namespace slmotion {
  /**
   * This class is a very simple example component that can process
   * Kinect data.  The component simply weighs each pixel in the input
   * frames with the corresponding depth, and saves the frame as
   * "KinectEntry" on the blackboard.
   */
  class KinectExampleComponent : public Component { 
  public:
    // All components inherit from the Component class.  The Component
    // class acts as a simple interface that allows the Analyser class
    // to give similar treatment to each independent component. Each
    // component is associated with a FrameSource object that provides
    // input data (such as video frames or Kinect depth data) and a
    // BlackBoard. These two are, generally, shared by all
    // components. The BlackBoard is used to store the results
    // generated by the component and can be used to access any
    // results generated by a component preceeding the current
    // component in the processing chain.

    /**
     * This is a very basic constructor. Just passes the arguments to
     * the superclass constructor.
     *
     * @param blackBoard The black board associated with the component
     * @param frameSource The frame source associated with the component
     */
    KinectExampleComponent(BlackBoard* blackBoard, FrameSource* frameSource) :
      Component(blackBoard, frameSource), constant(1) {
      // these conditions are expected of the kinect source:
      // - two tracks
      // - track number zero contains the BGR video
      // - track number one contains the depth data
      FrameSource& fs = getFrameSource();
      if (fs.getNumberOfTracks() != 2 ||
          fs.getTrack(0).getTrackInfo().type != TrackInfo::Type::BGR_IMAGE ||
          fs.getTrack(1).getTrackInfo().type != TrackInfo::Type::DEPTH_DATA)
        throw SLMotionException("Invalid frame source! A Kinect source is expectd!");
    }



    /**
     * This is a special-purpose constructor. Most importantly, it
     * will call the Component(bool) constructor and cause the
     * component to be registered. Also, the constructor will register
     * any configuration file options required.
     */
    KinectExampleComponent(bool) : Component(true) {
      // Calling Component(bool) will cause the component to be registered
      // The constructor serves no other purpose
    }

    

    /**
     * C++11 allows copy constructors to be explicitly defaulted. This
     * class has no members that need special treatment, so the
     * default behaviour should do just fine.
     */
    KinectExampleComponent(const KinectExampleComponent&) = default;



    /**
     * C++11 allows copy-assignment operators to be explicitly
     * defaulted. This class has no members that need special
     * treatment, so the default behaviour should do just fine.
     */
    KinectExampleComponent& operator=(const KinectExampleComponent&) = default;



    /**
     * This is a move constructor, introduced in C++11. The move
     * constructor is different from the copy constructor in that,
     * instead of a const-reference, it takes in an rvalue
     * reference. Theoretically, this may speed up some operations,
     * although in the case of this class, this is quite unlikely, but
     * in some cases, such as moving a vector, this can lead to
     * significant speedups. They too can be defaulted, and the
     * default behaviour is perfectly acceptable in this case.
     */
    KinectExampleComponent(KinectExampleComponent&&) = default;



    /**
     * The move-assignment is the move equivalent of a copy-assignment
     * operator. As of GCC 4.5.1, default move-assignment operators
     * have not been implemented yet.
     */
    // KinectExampleComponent& operator=(KinectExampleComponent&&) = default;



    /**
     * This is a mandatory function, required to be implemented by all
     * Components. The function is called with a valid frame number
     * that can be fetched from the FrameSource and the component may
     * perform any processing it needs to do.
     *
     * In this case, it simply reads in the input frame and the
     * corresponding depth frame, and weighs each pixel by the depth
     * value, normalised between 0 and 1, and multiplies by a constant
     * (1 by default).
     */
    virtual void process(frame_number_t frameNumber);



    /**
     * Simple setter
     */
    inline void setConstant(double d) {
      constant = d;
    }



    /**
     * This is a function called upon assignment. It cleans up any
     * internal state the component may have (internal caches etc.).
     *
     * This component has no internal state so the default behaviour
     * is perfectly acceptable (i.e. NOP)
     */
    virtual void reset() { }



    /**
     * Returns the long human-readable name of the component
     * (which can be seen in the user interface)
     */
    inline virtual std::string getComponentName() const {
      return "Kinect Example Component";
    }



    /**
     * This is the short command-line identifier. These names should be
     * unique.
     */
    inline virtual std::string getShortName() const {
      return "KinectExample";
    }



    /**
     * Each component knows what properties or entries they expect to
     * have on the black board before processing, possibly depending
     * on the component settings. In this case, the default behavior
     * is adequate, i.e. return an empty set.
     */
    virtual property_set_t getRequirements() const {
      return property_set_t();
    }



    /**
     * Likewise, the component will tell what entries it shall store
     * or modify on the black board
     */
    inline virtual property_set_t getProvided() const {
      return property_set_t { "KinectEntry" };
    }



    /**
     * This description appears in the list printed out when slmotion is called
     * with the option ``--components help''
     */
    virtual std::string getShortDescription() const {
      return "This is an example component that takes in Kinect data, and "
        "weighs each pixel by the corresponding depth value with respect to "
        "minimum and maximum depth values in the given frame, and multiplies "
        "by a constant.";
    }



    /**
     * This description appears in the automatically generated
     * configuration file.
     */
    virtual std::string getLongDescription() const {
      return "This is an example component that takes in Kinect data, and "
        "weighs each pixel by the corresponding depth value with respect to "
        "minimum and maximum depth values in the given frame, and multiplies "
        "by a constant.";
    }



    /**
     * This function returns a description of options for the configuration
     * file used by this particular component, including with default
     * values. These descriptions are called when the configuration file
     * is parsed or a default configuration file is created.
     */
    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;



    /**
     * This function works just as above but defines command line parameters
     * instead.
     */
    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }


  private:
    /**
     * This is the implementation for the
     * Component::createComponent. This function creates a new
     * instance of the class, configures it, and then returns a
     * pointer, or NULL if the configuration is invalid.
     */
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;


    /**
     * In the case of some components, processing single frames is not
     * an option. In this case, a function called processRange is used
     * instead. The function Component::processRange guarantees to
     * call this function, which serves as the implementation, with a
     * valid range of frames. The first frame is inclusive, the last
     * frame exclusive, i.e. processing just one frame would be
     * implied by last - first == 1.
     *
     * This function is optional. The default implementation simply calls
     * process for each frame in the range.
     */
    // virtual bool processRangeImplementation(frame_number_t first, 
    //                                         frame_number_t last, 
    //                                         UiCallback* uiCallback);



    double constant;
  };
}

#endif
