#ifndef SLMOTION_LIMITED_PRIORITY_QUEUE
#define SLMOTION_LIMITED_PRIORITY_QUEUE

#include <deque>
#include <functional>
#include <cstdlib>
#include <algorithm>

namespace slmotion {
  /**
   * This template class acts as a size-limited priority queue, the size of which is determined
   * at compilation time. Any items that compare less than the lowest item in a container of 
   * maximum size, are discarded.
   *
   * @tparam T Type of item
   * @tparam COMPARE A predicate that takes in two T's and returns true if the left-hand 
   * argument should compare "less" than the "right" hand argument.
   * @tparam IMPLEMENTATION An underlying container type. Should be sortable by std::sort, and
   * have the following member functions: size(), front(), back(), push_back(), push_front(),
   * pop_front().
   * The implementation should also be default-constructible.
   */
  template <typename T, typename COMPARE = std::less<T>,
            typename IMPLEMENTATION = std::deque<T>>
  class LimitedPriorityQueue {
  public:
    explicit LimitedPriorityQueue(size_t MAXSIZE) : MAXSIZE(MAXSIZE) {}

    LimitedPriorityQueue(size_t MAXSIZE, const COMPARE& comparator) : MAXSIZE(MAXSIZE), 
                                                                      comparator(comparator) {}

    /**
     * Inserts the given item.
     */
    void insert(const T& item) {
      if (container.size() > 0) {
        if (container.size() < MAXSIZE && comparator(item, container.front()))
          container.push_front(item);
        else if (comparator(container.back(), item))
          container.push_back(item);
        else if (comparator(container.front(), item)) {
          container.push_front(item);
          std::sort(container.begin(), container.end(), comparator);
        }

        while (container.size() > MAXSIZE)
          container.pop_front();  
      }
      else 
        container.push_back(item);
    }

    /**
     * Returns the number of elements.
     */
    size_t size() const {
      return container.size();
    }

    /**
     * Returns the top element
     */
    T top() const {
      return container.back();
    }

    /**
     * Returns the bottom element
     */
    T bottom() const {
      return container.front();
    }

    // iterators

    typename IMPLEMENTATION::iterator begin() {
      return container.begin();
    }

    typename IMPLEMENTATION::iterator end() {
      return container.end();
    }

    typename IMPLEMENTATION::const_iterator begin() const {
      return container.begin();
    }

    typename IMPLEMENTATION::const_iterator end() const {
      return container.end();
    }

    typename IMPLEMENTATION::const_iterator cbegin() const {
      return container.begin();
    }

    typename IMPLEMENTATION::const_iterator cend() const {
      return container.end();
    }

  private:
    size_t MAXSIZE;
    COMPARE comparator;
    IMPLEMENTATION container;
  };
}

#endif

