#ifndef SLMOTION_LINE_POINT_ITERATOR
#define SLMOTION_LINE_POINT_ITERATOR

#include <opencv2/opencv.hpp>

namespace slmotion {
  /**
   * In many respects, this class is similar to cv::LineIterator, but it 
   * returns the point instead of the value.
   *
   * Remark: the implementation is based on the version of Bresenham's 
   * algorithm presented in Wikipedia.
   */
  class LinePointIterator {
  public:
    /**
     * Creates a new iterator for the given point range
     */
    LinePointIterator(const cv::Point& startPoint,
                      const cv::Point& endPoint);



    /**
     * The default copy-constructor
     */
    LinePointIterator(const LinePointIterator&) = default;



    /**
     * The default copy-assignment
     */
    LinePointIterator& operator=(const LinePointIterator&) = default;



    /**
     * Returns the current point
     */
    inline cv::Point operator*() const {
      return currentPoint;
    }



    /**
     * Returns the end point
     */
    inline cv::Point end() const {
      return endPoint;
    }



    /**
     * Returns the start point
     */
    inline cv::Point start() const {
      return startPoint;
    }



    /**
     * Returns the total path length in pixels
     */
    inline size_t length() const {
      return std::max(std::abs(endPoint.x-startPoint.x),
                      std::abs(endPoint.y-startPoint.y)) + 1;
    }



    /**
     * Prefix increment
     */
    LinePointIterator& operator++();


    
    /**
     * Postfix increment
     */
    inline LinePointIterator operator++(int) {
      LinePointIterator temp(*this);
      ++*this;
      return temp;
    }
    
  private:
    cv::Point startPoint;
    cv::Point endPoint;
    cv::Point currentPoint;
    int dx, dy;
    int sx, sy;
    int err;
  };
}

#endif
