#ifndef SLMOTION_MULTI_FRAME_SOURCE
#define SLMOTION_MULTI_FRAME_SOURCE

#include <algorithm>
#include "FrameSource.hpp"

namespace slmotion {
  /**
   * This class is a meta-framesource that embeds several other frame 
   * sources within itself, providing access to each track in a unified 
   * manner.
   */
  class MultiFrameSource : public FrameSource {
  public:
    explicit MultiFrameSource(size_t cacheSize = SIZE_MAX) :
      FrameSource(cacheSize, TrackInfo { TrackInfo::Type::META_TRACK, 
            "MultiFrameSourceMetaTrack"} ),
      defaultTrackNumber(0), cachedFps(DBL_MAX) { }



    /**
     * The size of the multi source is defined as the minimum size between
     * each track
     */
    inline frame_number_type size() const {
      if (tracks.size() == 0)
        return 0;

      size_t minSize = SIZE_MAX;
      for (auto it = tracks.begin(); it != tracks.end(); ++it)
        minSize = std::min(minSize, (*it)->size());
      return minSize;
    }



    inline const cv::Mat& operator[](frame_number_type frameNumber) {
      return getDefaultTrack()[frameNumber];
    }



    inline void addTrack(std::shared_ptr<FrameSource> newTrack) {
      tracks.push_back(newTrack);
      cachedFps = DBL_MAX;
    }



    /**
     * Returns the number of the default track (should be a BGR track)
     */
    virtual size_t getDefaultTrackNumber() const {
      return defaultTrackNumber;
    }



    inline void setDefaultTrackNumber(size_t newDefault) {
      defaultTrackNumber = newDefault;
    }



    /**
     * Returns the frame source corresponding to the given track number
     *
     * @param trackNumber A 0-based track number
     * @return A reference to the corresponding frame source object
     */
    virtual FrameSource& getTrack(size_t trackNumber) {
      if (trackNumber >= getNumberOfTracks())
        throw std::out_of_range("The given track number exceeds the number of tracks.");

      return *tracks[trackNumber];
    }



    /**
     * Returns the total number of tracks for the given frame source
     */
    virtual size_t getNumberOfTracks() const {
      return tracks.size();
    }



    /**
     * Returns the fps but only if each track agrees with the fps
     */
    double getFps() const {
      if (tracks.size() == 0)
        return 0;

      if (cachedFps == DBL_MAX) {
        for (const auto& t : tracks) {
          if (cachedFps == DBL_MAX)
            cachedFps = t->getFps();
          else if (cachedFps != t->getFps())
            throw FrameSourceException("Disagreement between frame sources on fps!");
        }
      }
      return cachedFps;
    }



  private:
    std::deque<std::shared_ptr<FrameSource>> tracks;
    size_t defaultTrackNumber;
    mutable double cachedFps;
  };
}

#endif
