#ifndef SLMOTION_OPENNI_ONI_FILE_SOURCE
#define SLMOTION_OPENNI_ONI_FILE_SOURCE

#ifdef SLMOTION_ENABLE_OPENNI

#ifdef __linux__
#ifndef linux
#define linux __linux__
#endif
#endif

#include <XnCppWrapper.h>

#include "FrameSource.hpp"

namespace slmotion {
  /**
   * Sets the OpenNI environment variables to precompiled defaults.
   */
  void setOpenNiEnvironment();

  /**
   * This class can be used to read in OpenNI ONI recordings (kinect).
   *
   * NOTE: An environment variable OPEN_NI_INSTALL_PATH must be set in case
   * the library is not installed in the default location, e.g.
   * export OPEN_NI_INSTALL_PATH=/share/imagedb/slmotion/openni
   */
  class OpenNiOniFileSource : public FrameSource {
  public:
    virtual ~OpenNiOniFileSource() {}

    virtual frame_number_type size() const;

    explicit OpenNiOniFileSource(const std::string& filename,
                                 size_t cacheSize = SIZE_MAX);

    virtual const cv::Mat& operator[](frame_number_type frameNumber);

    // no copies (until copy constructors have been implemented)
    OpenNiOniFileSource(const OpenNiOniFileSource&) = delete;
    OpenNiOniFileSource& operator=(const OpenNiOniFileSource&) = delete;



    virtual size_t getNumberOfTracks() const {
      return skeletons.empty() ? 2 : 3;
    }



    virtual FrameSource& getTrack(size_t trackNumber);

    virtual double getFps() const;

  private:
    size_t getCurrentFrame();
    size_t getCurrentDepthFrame();
    void seekToFrame(frame_number_type frameNumber);


    class OniTrack;
    class OniImageTrack;
    class OniDepthTrack;
    class OniSkeletonTrack;

    xn::Context context; ///< Oni context
    xn::Player player;
    xn::ImageGenerator imageGenerator;
    std::string imageGeneratorNodeName;
    xn::ImageMetaData imageGeneratorMetadata;
    xn::DepthMetaData depthGeneratorMetadata;
    xn::DepthGenerator depthGenerator;
    std::string depthGeneratorNodeName;
    cv::Mat skeletons; ///< Skeleton points; empty if no skeleton points exist
#ifdef SLMOTION_THREADING
    mutable std::mutex mutex; ///< If threaded access is used, this object will be used for mutual exclusion
#endif

    mutable uint32_t imageFrameCount; ///< Image frame count
    mutable uint32_t depthFrameCount; ///< Depth frame count

    std::shared_ptr<FrameSource> imageTrack;
    std::shared_ptr<FrameSource> depthTrack;
    std::shared_ptr<FrameSource> skeletonTrack;
  };
}

#endif

#endif


