#ifndef SLMOTION_PDM_NEW
#define SLMOTION_PDM_NEW

#include "opencv.hpp"

#include <vector>

namespace slmotion {
  // forward declarations
  class Blob;

  /**
   * This class represents a point distribution model that is created from
   * Blob objects. Landmarks are placed evenly around blob boundaries.
   */
  class Pdm {
  public:
    bool operator==(const Pdm&) const;
    inline bool operator!=(const Pdm& other) const {
      return !(*this == other);
    }


    inline const cv::Mat& getEigenVectors() const {
      return eigenVectors;
    }

    inline const cv::Mat& getEigenValues() const {
      return eigenValues;
    }


    /**
     * Instances of this class represent the pose parameters
     */
    struct PoseParameter {
      double theta; ///< Orientation
      double scale; ///< Scale
      double tx; ///< Translation along the X axis
      double ty; ///< Translation along the Y axis
    };

    const static PoseParameter IDENTITY;



    /**
     * Constructs the model by placing nLandmarks landmarks around the 
     * contour of each blob, then aligns the blobs with one another and 
     * performs PCA.
     */
    Pdm(const std::vector<Blob>& blobs, size_t nLandmarks);



    /**
     * Explicitly construct a PDM (beware!)
     */
    Pdm(size_t nLandmarks, const cv::Mat& meanShape, 
        const cv::Mat& eigenVectors, const cv::Mat& eigenValues,
        const cv::Point2d& meanAnchor) :
      nLandmarks(nLandmarks), mean(meanShape.clone()),
      eigenVectors(eigenVectors.clone()), eigenValues(eigenValues.clone()),
      meanAnchor(meanAnchor) {
      assert(mean.rows % 2 == 0);
      assert(mean.rows > 0);
      assert(static_cast<size_t>(mean.rows) == 2*nLandmarks);
      assert(mean.cols == 1);
      assert(mean.type() == CV_64FC1);
      assert(mean.type() == eigenValues.type());
      assert(eigenValues.type() == eigenVectors.type());
      assert(eigenValues.size() == mean.size());
      assert(eigenVectors.rows == mean.rows &&
             eigenVectors.cols == mean.rows);
    }



    /**
     * Generates a new plausible shape vector given a parameter vector b.
     *
     * @param b A parameter column vector, that is linear weights for
     * eigenvectors
     *
     * @return A 2N column vector, formed by computing mean(x) + Pb
     * where b is the shape parameter vector, and P is the
     * transformation matrix composed of eigenvectors (principal components)
     */
    cv::Mat generateShape(const cv::Mat& b) const;



    /**
     * Like above, but take in a an approximate displacement vector dx,
     * and return the corresponding approximate low-dimensional component
     * displacement db with t components.
     */
    cv::Mat generateApproximateShapeDifference(const cv::Mat& dx,
                                               size_t t, 
                                               double maxShapeParameterDeviation) const;



    /**
     * The copy constructor. Constructs a deep copy.
     */
    inline Pdm(const Pdm& other) :
      nLandmarks(other.nLandmarks), mean(other.mean.clone()),
      eigenVectors(other.eigenVectors.clone()), 
      eigenValues(other.eigenValues.clone()), 
      meanAnchor(other.meanAnchor) {}



    /**
     * Copy-assignment. A deep copy is created.
     *
     * @param other The object that is to be copied
     */
    Pdm& operator=(const Pdm& other);



    inline cv::Point2d getMeanAnchor() const {
      return meanAnchor;
    }



  private:
    size_t nLandmarks; ///< Number N of landmarks
    cv::Mat mean; ///< Mean shape
    cv::Mat eigenVectors; ///< Transformation eigenvectors as columns
    /**
     * Eigenvalues as a column vector, in descending order
     */
    cv::Mat eigenValues; 
    /**
     * Mean absolute anchor location for the shapes used in training
     */
    cv::Point2d meanAnchor; 
  };



  inline bool operator==(const Pdm::PoseParameter& lh, 
                  const Pdm::PoseParameter& rh) {
    return lh.theta == rh.theta && lh.scale == rh.scale &&
      lh.tx == rh.tx && lh.ty == rh.ty;
  }



  /**
   * Aligns the given landmark vector x2 with respect to the target
   * vector x1 as well as possible, using least squares method (as in
   * Cootes et al., 1992) without a weight term.
   *
   * Both x1 and x2 should have an equal amount of landmarks. This
   * operation can be performed in-place.
   *
   * Returns an error figure, and optionally outputs the minimised
   * parameters.
   *
   * @param x1 Reference vector. Should be a column vector with an
   * even number of double elements.
   * @param x2 The alignee, have the same properties as x1
   * @param out Optional output vector, NULL if not required.  
   * @param s Optional scaling parameter output, NULL if not required 
   * @param theta Optional rotation angle output, NULL if not required
   * @param tx Optional horizontal translation, NULL if not required 
   * @param ty Optional vertical translation, NULL if not required
   * @return Sum of squared differences between the original x2 and the 
   * realigned vector.
   * @return Sum of absolute differences with respect to parameters 
   * required for identity transformation (i.e. |tx| + |ty| + |a1 - 1| + 
   * |a2|) that can be used for testing for convergence.
   */
  double align(const cv::Mat& x1, const cv::Mat& x2, cv::Mat* out = NULL, 
               double* out_s = NULL, double* out_theta = NULL, 
               double* out_tx = NULL, double* out_ty = NULL);



  /**
   * Same as above, but store the new pose parameters into a pose parameter
   * object.
   */
  double align(const cv::Mat& x1, const cv::Mat& x2,
               Pdm::PoseParameter& newPose, cv::Mat* out = NULL);



  /**
   * Applies the new pose to the PDM instance
   */
  cv::Mat transform(const cv::Mat& input, const Pdm::PoseParameter& pose);




  /**
   * Draws landmarks given an anchor, i.e. translates the shape so that
   * landmark zero is at the anchor location
   */
  void drawLandmarks(cv::Mat& img, const cv::Mat& landmarkVector,
                     const cv::Point& anchor, const cv::Scalar& colour);



  /**
   * Same as above but without translation
   */
  void drawLandmarks(cv::Mat& img, const cv::Mat& landmarkVector,
                     const cv::Scalar& colour);



  /**
   * Goes through the blobs and creates corresponding landmark vectors
   * by placing the number of landmarks evenly around the blob
   * contour. Landmark order is adjusted so that the bottommost point
   * corresponds to the first landmark.
   *
   * @param blobs The blobs 
   * @param nLandmarks The number N of landmark
   * points
   *
   * @outBlobLandmarks Will store a vector of landmark vectors. The
   * std::vector will contain exactly the same number of matrices as
   * there are blobs, each corresponding to the blob at the same
   * index. The matrices will be column vectors of type CV_64FC1 with
   * dimensions of 2N×1. The data is arranged as follows: [x0, y0, x1,
   * y1, ..., xN-1, yN-1]^T
   *
   * @outMeanAnchor Will store the mean anchor location if given
   */
  void getBlobLandmarks(const std::vector<Blob>& blobs, 
                        size_t nLandmarks, 
                        std::vector<cv::Mat>& outBlobLandmarks,
                        cv::Point2d* outMeanAnchor = NULL);
}
#endif

#if 0

// THIS OLD IMPLEMENTATION IS DEPRECATED AND IS PRESERVED HERE ONLY AS A 
// REMINDER

namespace slmotion {
  /**
   * WARNING
   * THIS CLASS IS DEPRECATED
   *
   * This class represents a point distribution model that is created from
   * Blob objects. Landmarks are placed evenly around blob boundaries.
   * 
   * Objects of this class are used both as mean approximates, and as training
   * samples, from which means are created.
   */
  class PDM {
  public:
    /**
     * A simple structure to set the required parameters for fitting the ASM
     */
    struct AsmFittingContext {
      int sobelApertureSize; ///< Aperture size for the Sobel operator when computing the intensity gradient map for ASM fitting

      int maxLookDistance; ///< Maximum distance allowed for target landmarks when performing ASM fitting.

      double convergenceThreshold; ///< Convergence threshold for the fitASM function

      double intensityFactor; ///< Intensity factor for the fitASM function

      double minIntensity; ///< Minimum absolute value for the new target gradient when scanning for new targets in the intensity gradient map. If negative, use mean.
    
      int maxIterations; ///< Maximum number of iterations allowed when performing ASM fitting

      int pcaComponentCount; ///< Number of PCA components to use when adjusting the model

      std::pair <double, double> scaleApproximationFactor; ///< Factors for initial pose scale approximation

      double maxShapeDeviation; ///< The absolute values of the particular shape vector may never exceed this parameter times the square root of the associated eigenvalue

      bool blackout; ///< If true, black out non-important pixels in the gradient map when doing ASM fitting

      enum {
	PROBE_ONLY,
	POSE_HISTORY,
	POSE_AND_SHAPE_HISTORY
      } initialPoseEstimationMethod; ///< sets the method for initial pose estimation

      double initialPoseEstimationGoodnessThreshold; ///< Sets the threshold for initial pose estimation methods that require historical data

      bool equaliseHistogram; ///< If enabled, perform histogram equalisation on the greyscale image from which intensity gradients are computed

      /**
       * Sets the way the new target is selected
       */
      enum TargetType {
	ABSOLUTE,
	NEGATIVE,
	POSITIVE
      } targetType;

      /**
       * Specifies the method used for choosing new target landmarks
       */
      enum targetMethod {
	GRADIENT,
	DIRECTIONAL,
	CANNY
      } targetMethod;

      /**
       * Sets the convolution kernel used for gradient computation
       */
      enum GradientType {
	SOBEL, 
	ROBERTS,
	PREWITT,
	ROBINSON,
	LAPLACIAN,
	LAPLACIAN4,
	LAPLACIAN8,
	KIRSCH
      } gradientMethod;

      double cannyThreshold1; ///< Threshold for the Canny edge detector
      double cannyThreshold2; ///< Threshold for the Canny edge detector

      /**
       * A simple constructor to set default values
       */
      AsmFittingContext() :
	sobelApertureSize(5),
	maxLookDistance(30),
	convergenceThreshold(1.0),
	intensityFactor(5.0),
	minIntensity(-1),
	maxIterations(20),
	pcaComponentCount(3),
	scaleApproximationFactor(std::pair<double,double>(1.0,1.0)),
	maxShapeDeviation(3.0),
	blackout(true),
	initialPoseEstimationMethod(POSE_AND_SHAPE_HISTORY),
	initialPoseEstimationGoodnessThreshold(0.7),
	equaliseHistogram(false),
	targetType(ABSOLUTE),
	targetMethod(DIRECTIONAL),
	gradientMethod(SOBEL),
	cannyThreshold1(150),
	cannyThreshold2(100)
      {}
    };



    /**
     * Copy constructor, creates a deep copy
     *
     * @param x The object to be copied
     */
    PDM(const PDM& x) :
      landmarks(x.landmarks),
      prevS(1),
      prevT(0)
    {
      // Needed because the Mat copy constructor only makes a shallow copy
      x.transformation.copyTo(transformation);
      x.eigenvalues.copyTo(eigenvalues);
    }



    /**
     * Assignment operator, creates a deep copy
     *
     * @param x The object to be copied
     */
    inline PDM& operator= (const PDM& x) {
      if (&x != this) {
	PDM temp(x);
	swap(landmarks, temp.landmarks);
	swap(transformation, temp.transformation);
	swap(eigenvalues, temp.eigenvalues);
      }
      return *this;
    }



    /**
     * Constructs a new point distribution model object. Landmarks are stored
     * in reference to the anchor point.
     *
     * @param img A binary image where non-zero pixels should represent the
     * blob
     * @param anchor An anchor point that serves as a point of reference for
     * the points in the model
     * @param nlandmarks The number of landmark points
     */
    PDM(const cv::Mat& img, const cv::Point& anchor, size_t nlandmarks);



    /**
     * Constructs a new point distribution model object
     * @param landmarks Set of landmarks
     */
    explicit PDM(const std::vector<cv::Point>& landmarks) :
      landmarks(landmarks),
      prevS(1),
      prevT(0)
    {}



    /**
     * Constructs a new point distribution model object with a given
     * transformation matrix.
     *
     * @param landmarks Set of landmarks
     * @param transformation 2N×2N transformation matrix where N is the
     * number of landmarks
     * @param eigenvalues 2N row vector storing eigenvalues in descending
     * order
     */
    PDM(const std::vector<cv::Point>& landmarks, const cv::Mat& transformation,
	const cv::Mat& eigenvalues);



    /**
     * Default constructor. Creates an empty PDM (not very useful).
     */
    PDM() :
      prevS(1),
      prevT(0)
    {}



    /**
     * Creates an approximately aligned mean PDM from a vector of sample
     * vectors
     *
     * @param vectors PDM vectors that need pairwise coalignment
     * @param convThresh Convergence threshold
     * @param maxIterations Maximum number of iterations
     * @param outMean Mean shape output
     */
    static void approximateAlign(const std::deque<PDM>& vectors,
				 double convThresh, 
				 unsigned int maxIterations,
				 PDM& outMean);



    /**
     * Returns the mean PDM
     *
     * @param vectors PDM vectors whose mean is to be computed
     * @return The mean PDM
     */
    static PDM mean(const std::vector<PDM>& vectors);



    /**
     * Translates landmarks so that their mean is at the origin
     */
    void zeroMeanify();



    /**
     * Converts the PDM to a binary matrix by regarding landmarks as angles of
     * a polygon. Each point within the polygon will be marked non-zero.
     *
     * The matrix will be of type CV_8UC1
     *
     * @param size Size of the matrix
     * @param anchor Anchor point
     * 
     * @return A new matrix
     */
    cv::Mat toFilledMatrix(const cv::Size& size, const cv::Point& anchor) const;



    /**
     * Draws landmarks as circles in the image
     * @param img Target image
     * @param radius Circle radius
     * @param thickness Circle line thickness
     * @param colour Colour of the circle
     * @param anchor Anchor point (i.e. where the first landmark should be positioned)
     */
    // void drawLandmarks(cv::Mat& img, int radius, int thickness, const cv::Scalar& colour, const cv::Point& anchor) const;



    /**
     * Creates an approximate PDM instance from a mean PDM with the given
     * shape parameter.
     *
     * That is, assuming a mean vector MEAN, MEAN.createApproximate(SHAPE)
     * would return a PDM that corresponds to a vector
     * MEAN + (transformation matrix of MEAN) * SHAPE
     *
     * @param shape A column vector of dimension 0 < M <= 2N where N is the
     * number of landmarks.
     *
     * @return The new approximated PDM object
     */
    PDM createApproximate(const cv::Mat& shape) const;



    /**
     * Returns the eigenvalue corresponding to ith component
     *
     * @param i Dimensional component index (0-based)
     * 
     * @return The corresponding eigenvalue
     */
    inline double getEigenValue(int i) {
      assert(eigenvalues.cols >= i);
      return eigenvalues.at<double>(0, i);
    }



    /**
     * Fits an ASM in the image. The object is modified by storing scaling and
     * orientation information for future reference.
     *
     * @param target Target BGR image
     * @param mask A binary mask of the target image, used for guessing the
     * initial pose. Must be of equal dimensions with target and of type
     * CV_8UC1.
     * @param anchorGuess Guessed anchor point position
     * @param bestFit Output parameter for the shape best fit
     * @param anchor Output parameter for the best anchor position found
     * @param fittingContext Used to convey options
     * @param blackOutMask An optional binary mask (much like the mask
     * parameter), that is used to black out unnecessary areas in the gradient
     * image. I.e. every pixel corresponding to a zero-pixel in the mask will
     * be blacked out. If set NULL, this parameter is not used.
     */
    void fitASM(const cv::Mat& target, const cv::Mat& mask, const cv::Point& anchorGuess,
		PDM& bestFit, cv::Point& anchor,
		const AsmFittingContext& fittingContext,
		const cv::Mat* blackOutMask);



    /**
     * Applies the set pose parameters where for each point (x,y), the
     * corresponding point in would become
     *
     * M(s, theta)(x, y)^T + (tx, ty)^T
     *
     * where
     *               (s cos theta   -s sin theta)
     * M(s, theta) = (                          )
     *               (s sin theta    s cos theta)
     *
     * @param s Scale parameter
     * @param theta The rotational parameter
     * @param tx Horizontal translation
     * @param ty Vertical translation
     *
     * @return The transformed PDM
     */
    inline PDM transform(double s, double theta, double tx, double ty) const {
      std::vector<cv::Point> newLandmarks(landmarks.size());
      for (size_t i = 0; i < landmarks.size(); i++) {
	double x = landmarks[i].x;
	double y = landmarks[i].y;
	newLandmarks[i] = cv::Point(s*cos(theta)*x - s*sin(theta)*y + tx,
				s*sin(theta)*x + s*cos(theta)*y + ty);
      }

      if (transformation.empty())
	return PDM(newLandmarks);
      else
	return PDM(newLandmarks, transformation, eigenvalues);
    }



    /**
     * Like above, but perform the transformation to a vector representation
     * of the PDM.
     *
     * @param vect Vector to be transformed. Should be a column vectorof type
     * CV_64FC1, with each xi at index 2i and yi at 2i+1, e.g. like one
     * computed using toVector()-function.
     * @param s Scaling parameter
     * @param theta Rotational parameter
     * @param tx Horizontal translation
     * @param ty Vertical translation
     *
     * @return Transformed vector of the same type and size.
     */
    inline static cv::Mat transform(const cv::Mat& vect, double s, double theta, double tx, double ty) {
      assert(vect.type() == CV_64FC1 && vect.cols == 1 && vect.rows % 2 == 0);
      cv::Mat result(vect.rows, vect.cols, vect.type());

      for (int i = 0; i < vect.rows / 2; i++) {
	result.at<double>(2*i, 0) = s*cos(theta)*vect.at<double>(2*i,0) - 
	  s*sin(theta)*vect.at<double>(2*i+1,0) + tx;

	result.at<double>(2*i+1, 0) = s*sin(theta)*vect.at<double>(2*i,0) +
	  s*cos(theta)*vect.at<double>(2*i+1,0) + ty;
      }
  
      return result;
    }



    /**
     * Returns a 2N-column-vector representation of the PDM with for each
     * landmark i, the x component xi is at index 2i, and the corresponding y
     * component is at 2i + 1. The returned matrix is of the type CV_64FC1
     */

    inline cv::Mat toVector() const {
      cv::Mat vector(2*landmarks.size(), 1, CV_64FC1);
      for (size_t i = 0; i < landmarks.size(); i++) {
	vector.at<double>(2*i, 0) = landmarks[i].x;
	vector.at<double>(2*i+1, 0) = landmarks[i].y;
      }
      return vector;
    } 


  
    /**
     * Returns a 2N-column-vector representation of the PDM with t most
     * significant PCA components, using the transformation matrix stored in
     * the mean PDM
     */
    inline cv::Mat toVector(const PDM& mean, int t) const {
      cv::Mat vector(2*landmarks.size(), 1, CV_64FC1);

      // Calculate bt such that
      // x = mean(x) + Pt*bt
      // For Pt, SHOULD be Pt^-1 = Pt^T
      // so
      // bt = Pt^T (x-mean(x))

      for (size_t i = 0; i < landmarks.size(); i++) {
	vector.at<double>(2*i, 0) = landmarks[i].x - mean.landmarks[i].x;
	vector.at<double>(2*i+1, 0) = landmarks[i].y - mean.landmarks[i].y;
      }
      return mean.transformation.colRange(0,t).t() * vector;
    }



    /**
     * Returns a reference to the landmark vector
     *
     * @return A reference to the internal vector containing the landmarks
     */
    inline const std::vector<cv::Point>& getLandmarks() const {
      return landmarks;
    }



    /**
     * Draws a line around the PDM in the image in the given colour
     *
     * If the PDM is empty, this function does nothing.
     *
     * @param img Target image to draw the PDM in
     * @param anchor Anchor point
     * @param colour The colour to draw the line in
     * @param thickness Line thickness (use CV_FILLED to fill the polygon)
     */
    void drawLine(cv::Mat& img, const cv::Point& anchor, const cv::Scalar& colour,
		  int thickness = 1) const; 



    /**
     * Computes the intensity gradient map for the given image
     *
     * @param src Source image
     * @param apertureSize Sobel operator aperture size
     * @param equaliseHistogram If true, perform histogram equalisation on the
     * grayscale image
     * @param gradientType Selects the convolution kernel for gradient
     * approximation
     *
     * @return The computed intensity gradient map
     */
    static cv::Mat computeGradientMap(const cv::Mat& src, int apertureSize, bool equaliseHistogram, AsmFittingContext::GradientType gradientType);



    /**
     * Computes intensity gradient maps for the given image for both x- and
     * y-axes separately
     *
     * @param src Source image
     * @param outX Output X matrix
     * @param outY Output Y matrix
     * @param apertureSize Sobel operator aperture size
     * @param equaliseHistogram If true, perform histogram equalisation on the
     * grayscale image
     * @param gradientType Selects the convolution kernel for gradient
     * approximation
     * 
     */
    static void computeGradientMap2d(const cv::Mat& src, cv::Mat& outX, cv::Mat& outY, int apertureSize, bool equaliseHistogram, AsmFittingContext::GradientType gradientType);



    /**
     * Computes the intersection of pixels enclosed within the two PDMs, and
     * returns the number of such pixels
     *
     * @param a PDM a
     * @param b PDM b
     * @param anchA Anchor for PDM a
     * @param anchB Anchor for PDM b
     * @param size The size of the image in which the pixels of the PDMs 
     * should be considered valid (i.e. the frame size)
     *
     * @return The number of pixels in the intersection of a and b
     */
    static int intersectionArea(const PDM& a, const PDM& b,
                                const cv::Point& anchA, 
                                const cv::Point& anchB);



    /**
     * Returns the centroid of the PDM with respect to point p
     *
     * @param p Anchor
     *
     * @return The location of the centroid with respect to p. If the PDM has
     * no landmarks, (-1, -1) is returned instead.
     */
    cv::Point2d computeCentroid(const cv::Point& p) const;



    /**
     * Computes orientation angle for the PDM, using the first landmark
     * and the landmark that is farthest away as reference
     *
     * @return An angle. If the PDM has no landmarks, then 0 is returned.
     */
    double computeOrientation() const;



    /**
     * Returns true if the two PDMs are memberwise equal
     */
    bool operator==(const PDM& other) const;



  private:
    /**
     * Creates a new PDM that is defined as x2 aligned as well as possible to
     * match x1 using least squares method (as in Cootes et al., 1992)
     * without a weight term
     *
     * Both x1 and x2 should have an equal amount of landmarks. This operation
     * can be performed in-place.
     *
     * @param x1 Reference vector
     * @param x2 The alignee, should have an equal amount of landmarks as x1
     * @param out Optional output PDM, NULL if not required.
     * @param s Optional scaling parameter output, NULL if not required
     * @param theta Optional rotation angle output, NULL if not required
     * @param tx Optional horizontal translation, NULL if not required
     * @param ty Optional vertical translation, NULL if not required
     * @return Sum of absolute differences with respect to parameters required
     * for identity transformation (i.e. |tx| + |ty| + |a1 - 1| + |a2|) that
     * can be used for testing for convergence.
     *
     */
    static double align(const PDM& x1, const PDM& x2, PDM* out,
			double* s = NULL, double* theta = NULL,
			double* tx = NULL, double* ty = NULL);



    std::vector<cv::Point> landmarks;
    cv::Mat transformation; ///< Transformation matrix for PCA
    cv::Mat eigenvalues; ///< Eigenvalues (for PCA)
    double prevS;
    double prevT;
    cv::Mat prevBt;



    /**
     * Returns a unit normal corresponding to the landmark at index i,
     * computed using left and right neighbours with the exception of the
     * first and the last points
     */
    inline cv::Point2d normal(int i) const {
      const cv::Point& right = landmarks[std::min(i + 1, static_cast<int>(landmarks.size()) - 1)];
      const cv::Point& left = landmarks[std::max(0, i - 1)];

      int dx = right.x - left.x;
      int dy = left.y - right.y;

      return cv::Point2d(dy, dx) * (1/sqrt(dx*dx + dy*dy));
    }



    /**
     * Returns the centroid of the PDM
     *
     * @return The centroid
     */
    inline cv::Point2d getCentroid() {
      cv::Point sum(0, 0);
      for (std::vector<cv::Point>::const_iterator it = landmarks.begin();
	   it != landmarks.end(); it++) {
	sum += *it;
      }
      sum *= 1./landmarks.size();
      return sum;
    }



    /**
     * Finds new target landmarks and stores them in a vector.
     * Can be performed in-place.
     *
     * @param src Source landmarks
     * @param newTargets New targets, corresponding to each index in source
     * @param gradImg Greyscale image of type CV_16SC1, should contain
     * intensity gradient information (obtained e.g. by using the Sobel
     * function)
     * @param intensityFactor A factor by which the intensity must differ, so
     * that the new target landmarks is of clearly higher intensity
     * @param maxLookDistance Maximum distance beyond which target landmarks should not be looked for
     * @param minIntensity Minimum allowed absolute value for the intensity
     * gradient at the new target landmark. If < 0, use the mean computed over
     * the whole gradient image.
     * @param type Selecting criterion for the new landmark
     */
    void findTargetLandmarks(const std::vector<cv::Point>& src, std::vector<cv::Point>& newTargets, const cv::Mat& gradImg, double intensityFactor, int maxLookDistance, double minIntensity, AsmFittingContext::TargetType type) const;


  
    /**
     * Attempts to find a best estimate for the rotational parameter theta by
     * comparing the number of pixels in the mask enclosed within a polygon
     * generated by the landmark points with different values of theta
     *
     * Once the best value for theta has been found, perform a similar
     * approximation for the scaling parameter s.
     *
     * If the mask contains only non-zero elements, this function outputs
     * values 0 and 1 for theta and s, respectively.
     *
     * @param mask The mask that should correspond to the body part blob
     * @param anchorGuess Guessed anchor position (usually the lowest point in
     * the blob)
     * @param theta A reference to the variable where the approximated value
     * for theta should be stored
     * @param s A reference to the variable where the approximated value for
     * s should be stored
     * @param scaleFactor A pair of doubles that represents the factor by
     * which the PDM scaled in its initial pose may exceed the dimensions of
     * the object in the reference frame
     */
    void findInitialThetaS(const cv::Mat& mask, const cv::Point& anchorGuess,
			   double& theta, double& s,
			   std::pair<double, double> scaleFactor) const;



    /**
     * Evaluates goodness of the pose by considering the fraction of the
     * overlapping pixels versus the number of pixels in the image with
     * largest number of pixels (either the reference or the corresponding
     * filled polygon image for the PDM with the set pose parameters applied)
     *
     * @param reference Reference image
     * @param s Scale
     * @param theta Rotation
     * @param anchor Anchor point
     */
    inline double evaluatePoseGoodness(const cv::Mat& reference, double s,
				       double theta, const cv::Point& anchor) const {
      assert(reference.type() == CV_8UC1);
      PDM tempPDM = this->transform(s, theta, 0, 0);
      cv::Mat temp(reference.size(), CV_8UC1, cv::Scalar(0));
      cv::Mat temp2 = tempPDM.toFilledMatrix(temp.size(), anchor);
      min(temp2, reference, temp);
      double goodness = static_cast<double>(countNonZero(temp)) /
	static_cast<double>(std::max(countNonZero(reference), countNonZero(temp2)));
      return goodness;
    }



    /**
     * Computes the dimensions of the PDM
     *
     * @return The size
     */
    cv::Size computeSize() const;



    /**
     * Finds new target landmarks in the source image
     *
     * @param srcPoints Source points that need to be fitted in the image
     * @param newTargets Output parameter for the new found targets
     * @param gradX X-directional gradient image (of type CV_16SC1)
     * @param gradY Y-directional gradient image (of type CV_16SC1)
     * @param fittingContext Used to convey options
     */
    void findDirectionalTargetLandmarks(const std::vector<cv::Point>& srcPoints, std::vector<cv::Point>& newTargets, const cv::Mat& gradX, const cv::Mat& gradY, const AsmFittingContext& fittingContext) const;



    /**
     * Finds new target landmarks in the source image using the Canny edge
     * detector
     *
     * @param canny Binary Canny image
     * @param srcPoints Source points that need to be fitted in the image
     * @param newTargets Output parameter for the new found targets
     * @param fittingContext Used to convey options
     */
    void findCannyTargetLandmarks(const cv::Mat& canny, const std::vector<cv::Point>& srcPoints, std::vector<cv::Point>& newTargets, const AsmFittingContext& fittingContext) const;
  };



  /**
   * A simple exception class for signaling invalid PDMs.
   */
  class PDMException : public SLMotionException {
  public:
    /**
     * A constructor
     *
     * @param msg The message
     */
    PDMException(const char* msg) throw() :
      SLMotionException(msg)
    {}
  };
}

#endif
