#include "PropagationSkinDetector.hpp"
#include "ELMClassifier.hpp"
#include "Blob.hpp"
#include <boost/lexical_cast.hpp>
#include <numeric>

using cv::Mat;
using cv::TermCriteria;
using boost::lexical_cast;
using cv::Rect;
using cv::Size;
using cv::Vec3b;
using cv::Vec3f;
using cv::Scalar;
using std::string;
using std::vector;
using std::cerr;
using std::endl;



namespace slmotion {


  extern int debug;

  void PropagationSkinDetector::detect(const cv::Mat& /* inFrame*/,  cv::Mat& /*outMask*/)
  {
    assert(false); // detection cannot be performed for frame-by-frame
  }


  bool PropagationSkinDetector::processRangeImplementation(frame_number_t first, 
							   frame_number_t last, 
							   UiCallback* uiCallback){

    //    cerr << "PropagationSkinDetector called for range ["<<first<<","<<last<<")"<<endl;

    // loop and process the range in comfortable size chunks

    int minlen=50;
    int minfacecount=6;
 
    int firstframe=first,lastframe; 

   
    do{

      int facecount=0;

      for(lastframe=firstframe;lastframe<(int)last&&(lastframe-firstframe+1<=minlen || facecount<minfacecount);lastframe++){
	//	cerr << "trying firstframe,lastframe="<<firstframe<<","<<lastframe<<" facecount="<<facecount<<endl; 
	if(!getBlackBoard().has(lastframe, FACEDETECTOR_BLACKBOARD_ENTRY)) continue;
	if(*getBlackBoard().get<Rect>(lastframe, FACEDETECTOR_BLACKBOARD_ENTRY)== FaceDetector::INVALID_FACE)
	  continue;
	facecount++;
      }

      lastframe--;
      
      if(facecount<minfacecount) // not enough faces in the chunk (because we hit the end)
	for(;firstframe>=(int)first&&facecount<minfacecount;firstframe--){

	  if(!getBlackBoard().has(firstframe, FACEDETECTOR_BLACKBOARD_ENTRY)) continue;
	  if(*getBlackBoard().get<Rect>(firstframe, FACEDETECTOR_BLACKBOARD_ENTRY)== FaceDetector::INVALID_FACE)
	    continue;
	  facecount++;
	  
	}
      

      if(firstframe<(int)first) firstframe=first;

      if(facecount<minfacecount)
	throw string("Could not find enough faces for reliable skin detection");

      if(uiCallback!=NULL)
	if(!(*uiCallback)(firstframe)) return false;
      propagateRange(firstframe,lastframe,firstframe==(int)first);
      // train the skin color model only for the first chunk


      firstframe=lastframe+1;
    } while(firstframe < (int)last);

    return true;

  }

  void PropagationSkinDetector::propagateRange(int firstframe,int lastframe, bool trainColourModel){
    if (slmotion::debug > 1)
      cerr << "propagateRange("<<firstframe<<","<<lastframe<<","<<(int)trainColourModel<<endl;

    localBlackBoardType localbb;

    vector<size_t> fdframes;

    // store scale all frames to local bb
    // scale also the face detections from the global bb


    int orgrows,orgcols;

    for(int frnumber=firstframe;frnumber <=lastframe;frnumber++){

      //	cerr << "Copying frames to blackboard." << endl;
   
      Mat frame=getFrameSource()[frnumber];

      size_t maxdim=fmax(frame.rows,frame.cols);
      float factor=750.0/maxdim;

      orgrows=frame.rows;
      orgcols=frame.cols;

      if(factor>1) factor=1; // do not scale up

      Mat minFrame;
	 
      //	 downsampleFrame(frame,minFrame,factor);
      cv::resize(frame,minFrame, cv::Size(0,0), factor, factor, CV_INTER_AREA );
      //	 minFrame=frame;
	 
      //      cerr << "scaled frame to size " << minFrame.cols << " x " << minFrame.rows << endl;

      localbb[std::pair<int,std::string>(frnumber,"frame")]=minFrame.clone();
      
      if(getBlackBoard().has(lastframe, FACEDETECTOR_BLACKBOARD_ENTRY)){

	Rect faceLocation = *getBlackBoard().get<Rect>(frnumber, FACEDETECTOR_BLACKBOARD_ENTRY);
	if (faceLocation != FaceDetector::INVALID_FACE) {         
	  // scale the rectangle

	  faceLocation.x *= factor;
	  faceLocation.y *= factor;
	  faceLocation.width *= factor;
	  faceLocation.height *= factor;

	  localbb[std::pair<int,std::string>(frnumber,"facerect")]=faceLocation;
	  fdframes.push_back(frnumber);
	  
// 	  for(int i=faceLocation.y;i<faceLocation.y+faceLocation.height;i++)
// 	    for(int j=faceLocation.x;j<faceLocation.x+faceLocation.width;j++){
// 	      //	      cerr << "setting pixel "<<i<<","<<j<<" minFrame.size="<<minFrame.rows<<" x " << minFrame.cols << endl;
// 	      minFrame.at<Vec3b>(i,j)=Vec3b(128,0,0);
// 	    }
	}
      }

     //  cv::imshow("scaled frame", minFrame);
//       cv::waitKey(0);      

    }


    // frames and face detections now on blackboard

    estimateBackgroundWatershed(firstframe,lastframe,localbb);

    // place potential propagation seed mask on blackboard

    string tgtid("diffmask");

    //extractMovementMask(localbb,tgtid,firstframe,lastframe);
    findNonBackgroundBlocks(firstframe,lastframe,localbb,tgtid);


    FeatureExtractor fe;


    if(trainColourModel){

      // otherwise assume that the colour classifier has already been trained

      delete cls;
      cls=new ELMClassifier;

      vector<size_t> sampledframes;

       size_t nrFramesToSample=20;

       vector<vector<float> > negSamples,posSamples;
       vector<float> negWeights,posWeights;
       
       FeatureExtractor fe;

       if(nrFramesToSample>=fdframes.size())
	 sampledframes=fdframes;
       else{
	 float tgtdelta=fdframes.size();
	 tgtdelta /= nrFramesToSample+1;
	 
	 float tgtsampleidx=tgtdelta*0.5;
	 
	 while(tgtsampleidx<=fdframes.size()-1){
	   sampledframes.push_back(fdframes[(size_t)tgtsampleidx]);
	   tgtsampleidx += tgtdelta;
	 }
       }

       sampleFrames(localbb,"dummyid","",false,sampledframes,negSamples,posSamples,negWeights,posWeights,fe,false,false,true,false,false,true,true);

       size_t samplesize=5000;

       if(posSamples.size()>samplesize) posSamples.resize(samplesize);
       if(negSamples.size()>samplesize) negSamples.resize(samplesize);

       // at the moment this truncation is a way to prevent numerical
       // problems in ELM training (should eventually fix properly)

       // hard coded: cls uses feature 0

       vector<bool> dimMask=fe.getFeatureMask(0);
       cls->train(negSamples,posSamples,&dimMask,&negWeights,
		 &posWeights);

       //       cerr << "...classifier trained" << endl;

       bool adaptiveThresholds=false;

      
       if(adaptiveThresholds){

	 // estimate adaptively the thresholds from the training examples

	 vector<float> posResponses;

	 size_t sampletgt=50000;
	 int sampleincr=posSamples.size()/sampletgt;
      
	 if(sampleincr<1) sampleincr=1;
      
	 for(size_t i=0;i<posSamples.size();i+=sampleincr)
	   posResponses.push_back(cls->predict(posSamples[i],&dimMask));

	 // sort the list

	 sort(posResponses.begin(),posResponses.end());

	 float corefraction=0.9;
	 float outerfraction=0.95;

	 corethreshold=0.9*posResponses[(1-corefraction)*posResponses.size()];
	 outerthreshold=0.65*posResponses[(1-outerfraction)*posResponses.size()];
	 if(slmotion::debug > 1)
	   cerr << "setting propagation thresholds: core="<<corethreshold<<
	     " outer"<<outerthreshold<<endl;
       }

    } // trainColourModel

    // from now on, assume that cls points to a valid colour classifier

    // seed the skin detection by confirming skin in areas specified by the seed mask

   
    vector<float> samplevec;
    size_t maxseedcount=0;
    size_t maxseedframe=firstframe;
 

    size_t seedstep=1;
    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){
	
	size_t frameseedcount=0;
	//	cerr << "seeding frame " << frnumber << endl;
	
	Mat  frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"frame")]);

	localbb[std::pair<int,std::string>(frnumber,"skinpmap")]=Mat(frame.rows,frame.cols,cv::DataType<float>::type,cv::Scalar::all(0));
	localbb[std::pair<int,std::string>(frnumber,"skinpropagationprogress")]=
	  Mat(frame.rows,frame.cols,cv::DataType<uchar>::type,cv::Scalar::all(0));

	Mat  pmap=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinpmap")]);
	Mat  progressmap=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinpropagationprogress")]);

	// bit 7 ( 128) -> confirmed core skin pixel
	// bit 6 ( 64 ) -> skin probability evaluated in pmap
	// bit 5 ( 32 ) -> pixel inserted in list to be tested
	// bit 4 ( 16 ) -> core membership of the pixel has been determined
	// bit 3 ( 8 )  -> confirmed outer pixel 

	Mat  diffmask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"diffmask")]);
	
	if(frnumber%seedstep==0){


	for(int i = 0; i < frame.rows; i++)
	  for(int j = 0; j < frame.cols; j++)
	    if(diffmask.at<float>(i,j)>0){
	      if(propagationTestBall(i,j,frnumber,corethreshold,3,localbb,*cls,fe)){
		progressmap.at<uchar>(i,j)|=128; // confirmed core
		frameseedcount++;
	      }
	      progressmap.at<uchar>(i,j)|=16; // core membership tested
	    }
	    
// 	cv::imshow("seeding result", pmap);
// 	cv::imshow("seeding result: progressmap", progressmap);
// 	cv::waitKey(0);     

	if(frameseedcount>maxseedcount){
	  maxseedcount=frameseedcount;
	  maxseedframe=frnumber;
	}
	}
    }
    
     
    // then perform the actual propagation in slices	 

    size_t framesinslice=15;

    int slicestart=firstframe;
    int sliceend;

    // process first the slice that includes the frame w/ max number of seeds
    
    while(maxseedframe-slicestart>framesinslice) slicestart += framesinslice;
    int initfirst=slicestart;
    sliceend=slicestart+framesinslice-1;
    if(sliceend>lastframe) sliceend=lastframe;
    
    propagateSkin(slicestart, sliceend,firstframe,lastframe,
	       localbb,"skinpropagationprogress",
		  corethreshold,outerthreshold,cls[0],fe);

    // then process the slices before the initial slice

    for(slicestart=initfirst-framesinslice;slicestart>=(int)firstframe; slicestart -= framesinslice){
      sliceend=slicestart+framesinslice-1;
      if(sliceend>lastframe) sliceend=lastframe;
      
      propagateSkin(slicestart, sliceend,firstframe,lastframe,
		    localbb,"skinpropagationprogress",
		    corethreshold,outerthreshold,cls[0],fe);
    }
    
    // and the slices after the initial
    
    for(slicestart=initfirst+framesinslice;slicestart<=lastframe; slicestart += framesinslice){
      sliceend=slicestart+framesinslice-1;
      if(sliceend>lastframe) sliceend=lastframe;
      
      propagateSkin(slicestart, sliceend,firstframe,lastframe,
		    localbb,"skinpropagationprogress",
		    corethreshold,outerthreshold,cls[0],fe);
    }

    // visualise the propagation results
    // and produce skin detection mask
 
    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){
      
      Mat  frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"frame")]);
      Mat  pmap=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinpmap")]);
      Mat  progressmap=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinpropagationprogress")]);
      Mat  mask(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));
      
      Mat visframe=frame.clone();

      Vec3b white,gray;

      white[0]=white[1]=white[2]=255;
      gray[0]=gray[1]=gray[2]=127;
      
      // overlay pmap with the watersheds
      
      //   Mat segmmask(frame.rows,frame.cols, CV_32SC1,Scalar::all(0));
      //    watershedSegmentation(frame,segmmask,20);
      
      for(int i = 0; i < frame.rows; i++)
	for(int j = 0; j < frame.cols; j++){
	  if(progressmap.at<uchar>(i,j)&128){
	    visframe.at<Vec3b>(i,j)=white;
	    mask.at<uchar>(i,j)=255;
	  }
	  else if(progressmap.at<uchar>(i,j)&8){
	    visframe.at<Vec3b>(i,j)=gray;
	    mask.at<uchar>(i,j)=255;
	  }
	  else{
	    mask.at<uchar>(i,j)=0;
	    pmap.at<float>(i,j)=0;
	  }
	  //  if(segmmask.at<int>(i,j)<0)
	  // 	 visframe.at<Vec3b>(i,j)=white;
	}

// 	cv::imshow("propagation progress",visframe);
// 	cv::imshow("pmap",pmap);
// 	cv::imshow("skin mask",mask);
	

// 	cv::waitKey(0);

	localbb[std::pair<int,std::string>(frnumber,"skinmask")]=mask.clone();
    }
 
    // perform temporal smoothing
    temporalSmoothPmap(localbb,"skinpmap","skinmask",firstframe,lastframe);
    // detectShadows(firstframe, lastframe,localbb,"skinmask","shadowmask");

    // perform the speciefied post-processing steps provided by the skindetector class
    // to the skin mask and 
    // transfer the results to the global localbb

    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      Mat  frame=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"frame")]);
      Mat  pmap=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinpmap")]);
      Mat  mask=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"skinmask")]);


      Mat restoredmask;

      cv::resize(mask,restoredmask, cv::Size(orgcols,orgrows), 0, 0, CV_INTER_AREA );

      Mat orgframe=getFrameSource()[frnumber].clone();

      postprocess(orgframe,restoredmask);

      bool displayFrames=slmotion::debug > 2;

      if(displayFrames){

      for(int i=0;i<orgrows;i++)
	for(int j=0;j<orgcols;j++)
	  if(restoredmask.at<uchar>(i,j))
	    orgframe.at<Vec3b>(i,j)=Vec3b(128,128,0);
	

  //     cv::imshow("skin after post-processing",mask);
       cv::imshow("detected skin",orgframe);

       cv::waitKey(0);
      }


      getBlackBoard().set(frnumber, SKINDETECTOR_BLACKBOARD_MASK_ENTRY, 
			  restoredmask);
    }
    

  }

  // helper functions

 void floatimagesc(const string &str, cv::Mat f){
    
    Mat norm=f.clone();

    float maxval;

    for(int i=0;i<f.rows;i++)
      for(int j=0;j<f.cols;j++)
	if(f.at<float>(i,j) > maxval) maxval=f.at<float>(i,j);

    for(int i=0;i<f.rows;i++)
      for(int j=0;j<f.cols;j++)
	norm.at<float>(i,j)=f.at<float>(i,j)/maxval;

    cv::imshow(str, norm);   
  }

  void extractMovementMask(localBlackBoardType &blackboard, 
			   string &targetid,
			 int firstframe, int lastframe){

    localBlackBoardType localbb;

    // assume that blackboard holds frames with id "frame" for 
    // the range [firstframe,lastframe]

    // store the colour difference magnitudes between consecutive frames

    //    cerr << "starting to generate movement masks" << endl;

       double dsum=0,dsqrsum=0;
       int count=0;


       
       for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){

	 Mat  frame=*boost::any_cast<cv::Mat>(&blackboard[std::pair<int,std::string>(frnumber,"frame")]);
	 Mat  prevframe=*boost::any_cast<cv::Mat>(&blackboard[std::pair<int,std::string>(frnumber-1,"frame")]);


	 Mat diffframe(frame.rows,frame.cols,cv::DataType<float>::type);

	 for(int i = 0; i < frame.rows;i++)
	   for(int j = 0; j < frame.cols;j++){

	     cv::Vec3b diff=frame.at<cv::Vec3b>(i,j)-prevframe.at<cv::Vec3b>(i,j);
	       float d=sqrt(diff.dot(diff));
	       diffframe.at<float>(i,j)=d;
	       dsum += d;
	       dsqrsum += d*d;
	       count++;
	   }

	 localbb[std::pair<int,std::string>(frnumber,"diffmagn")]=diffframe.clone();

       }

       float aved=dsum/count;
       float vard=dsqrsum/count-aved*aved;
       float stdevd=sqrt(vard);

       //       cerr << "difference magnitudes stored" << endl;
       

       // suppress differences below aved+coeffd*stdevd

       // loosen the constraint until at least 0.7% of pixels gets included

       float coeffd=1.0;

       bool enoughmarked=false;

       size_t tgtcount=0.007*(lastframe-firstframe)*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(firstframe+1,"diffmagn")])->rows*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(firstframe+1,"diffmagn")])->cols;

       while(!enoughmarked){

	 // cerr << "trying coefficient " << coeffd << endl;

	 size_t count=0;

       for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){
	 
	 Mat  diffframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"diffmagn")]);
	 
	 for(int i = 0; i < diffframe.rows;i++)
	   for(int j = 0; j < diffframe.cols;j++){
	     if(diffframe.at<float>(i,j)<aved+coeffd*stdevd)
	       diffframe.at<float>(i,j)=0;
	     else{
	       count++;
	     }
	   }	 

       }

       //       cerr << "count="<<count<<" (tgt:"<<tgtcount<<")"<<endl;

       if(count>tgtcount)
	 coeffd *= 1.07;
       else
	 enoughmarked=true;
       }

       //       cerr << "selected threshold coefficient " << coeffd << endl; 

     

    

       for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){
	 
	 Mat  diffframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"diffmagn")]);
	 
	 cv::GaussianBlur(diffframe,diffframe,Size(0,0),3);

       }

       // threshold blurred differences
       
       size_t nonzerocount=0;

       float maxval=0;

       for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){
	 
	 Mat  diffframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"diffmagn")]);
	 
	 for(int i = 0; i < diffframe.rows;i++)
	   for(int j = 0; j < diffframe.cols;j++)
	     if(diffframe.at<float>(i,j)>0){
	       nonzerocount++;
	       if(diffframe.at<float>(i,j)>maxval)
		 maxval=diffframe.at<float>(i,j);
	     }
	 
       }
       


       bool thresholdfound=false;
       float thr=0.05*maxval;

       while(!thresholdfound){
	 size_t pixelsleft=0;

	 //	 cerr << "trying threshold " << thr << endl;
	 
	 for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){
	 
	   Mat  diffframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"diffmagn")]);

	   for(int i = 0; i < diffframe.rows;i++)
	     for(int j = 0; j < diffframe.cols;j++)
	       if(diffframe.at<float>(i,j)>thr)
		 pixelsleft++;
	       else
		 diffframe.at<float>(i,j)=0;
	 }
	 if(pixelsleft>0.45*nonzerocount)
	   thr += 0.05*maxval;
	 else
	   thresholdfound=true;

       }

       // suppress temporally isolated changes

       size_t maxdist=6;


        for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){
	 
	   Mat  diffframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"diffmagn")]);


	   Mat suppressed=diffframe.clone();

	   for(int i = 0; i < diffframe.rows;i++)
	     for(int j = 0; j < diffframe.cols;j++)
	       if(diffframe.at<float>(i,j)>0){

		   int minx=j-maxdist;
		   if(minx<0) minx=0;
		   int maxx=j+maxdist;
		   if(maxx>=diffframe.cols) maxx=diffframe.cols-1;

		   int miny=i-maxdist;
		   if(miny<0) miny=0;
		   int maxy=i+maxdist;
		   if(maxy>=diffframe.rows) maxy=diffframe.rows-1;

		   bool supportnext=false;
		   bool supportprev=false;
		 
		   if(frnumber>firstframe+1){
		     Mat  prevframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber-1,"diffmagn")]);

		     for(int ii=miny;!supportprev && ii<=maxy;ii++)
		       for(int jj=minx;!supportprev && jj<=maxx;jj++)
			 if(prevframe.at<float>(ii,jj)>0)
			   supportprev=true;
		   }
		   else
		     supportprev=true;

		   if(frnumber<lastframe){
		     Mat  nextframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber+1,"diffmagn")]);

		     for(int ii=miny;!supportnext && ii<=maxy;ii++)
		       for(int jj=minx;!supportnext && jj<=maxx;jj++)
			 if(nextframe.at<float>(ii,jj)>0)
			   supportnext=true;
		   }
		   else
		     supportnext=true;
		  
		   if(supportprev==false||supportnext==false)
		     suppressed.at<float>(i,j)=0;
		   else
		     suppressed.at<float>(i,j)=1;
	       }

	 localbb[std::pair<int,std::string>(frnumber,"suppdiff")]=suppressed.clone();
	}


	// another suppression pass;

  for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){
	 
	   Mat  diffframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"suppdiff")]);


	   Mat suppressed=diffframe.clone();

	   for(int i = 0; i < diffframe.rows;i++)
	     for(int j = 0; j < diffframe.cols;j++)
	       if(diffframe.at<float>(i,j)>0){

		   int minx=j-maxdist;
		   if(minx<0) minx=0;
		   int maxx=j+maxdist;
		   if(maxx>=diffframe.cols) maxx=diffframe.cols-1;

		   int miny=i-maxdist;
		   if(miny<0) miny=0;
		   int maxy=i+maxdist;
		   if(maxy>=diffframe.rows) maxy=diffframe.rows-1;

		   bool supportnext=false;
		   bool supportprev=false;
		 
		   if(frnumber>firstframe+1){
		     Mat  prevframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber-1,"suppdiff")]);

		     for(int ii=miny;!supportprev && ii<=maxy;ii++)
		       for(int jj=minx;!supportprev && jj<=maxx;jj++)
			 if(prevframe.at<float>(ii,jj)>0)
			   supportprev=true;
		   }
		   else
		     supportprev=true;

		   if(frnumber<lastframe){
		     Mat  nextframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber+1,"suppdiff")]);

		     for(int ii=miny;!supportnext && ii<=maxy;ii++)
		       for(int jj=minx;!supportnext && jj<=maxx;jj++)
			 if(nextframe.at<float>(ii,jj)>0)
			   supportnext=true;
		   }
		   else
		     supportnext=true;
		  
		   if(supportprev==false||supportnext==false)
		     suppressed.at<float>(i,j)=0;
		   else
		     suppressed.at<float>(i,j)=1;
	       }


	   size_t motionmargin=4;

	   cv::dilate(suppressed, suppressed, 
      		 cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(motionmargin, motionmargin)));


	 localbb[std::pair<int,std::string>(frnumber,"suppdiff2")]=suppressed.clone();
	}


  // finally, suppress spatially isolated blobs

  size_t nbrradius=15;


  // calculate density images and form histograms

  size_t density_bincount=200;
  size_t density_nonzerocount=0;

  vector<size_t> densityhistogram(density_bincount,0);

  for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){
	 
	   Mat  diffframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"suppdiff2")]);

	   // form integral image

	   Mat intimage(diffframe.size()+cv::Size(1,1),cv::DataType<float>::type, cv::Scalar::all(0));

// 	   cerr << "forming integral image of frame " << 
// 	     frnumber << "..." << endl;

	   for(int i = 0; i < diffframe.rows;i++)
	     for(int j = 0; j < diffframe.cols;j++)
	       intimage.at<float>(i+1,j+1) =  intimage.at<float>(i+1,j) +
		 intimage.at<float>(i,j+1)-intimage.at<float>(i,j)+diffframe.at<float>(i,j);

// 	   cerr << "...done" << endl;

// 	   floatimagesc("diffframe",diffframe);
// 	   floatimagesc("intimage",intimage);
// 	   cv::waitKey(0);



	   Mat densityimage(diffframe.size(),cv::DataType<float>::type, cv::Scalar::all(0));

	   for(int i = 0; i < diffframe.rows;i++)
	     for(int j = 0; j < diffframe.cols;j++)
	       if(diffframe.at<float>(i,j)>0){

		 int minx=j-nbrradius;
		 if(minx<0) minx=0;
		 int maxx=j+nbrradius;
		 if(maxx>=diffframe.cols) maxx=diffframe.cols-1;
		 
		 int miny=i-nbrradius;
		 if(miny<0) miny=0;
		 int maxy=i+nbrradius;
		 if(maxy>=diffframe.rows) maxy=diffframe.rows-1;

		 size_t nbr_size=(maxx-minx+1)*(maxy-miny+1);

		 densityimage.at<float>(i,j)=intimage.at<float>(maxy+1,maxx+1)-intimage.at<float>(miny,maxx+1)-
		   intimage.at<float>(maxy+1,minx)+intimage.at<float>(miny,minx);

		 densityimage.at<float>(i,j) /= nbr_size;

		 size_t binidx=densityimage.at<float>(i,j)*density_bincount;
		 if(binidx>=density_bincount) binidx=density_bincount-1;

		 densityhistogram[binidx]++;
		 density_nonzerocount++;

	       }


	   localbb[std::pair<int,std::string>(frnumber,"diffdensity")]=densityimage.clone();

  }


  // select a density threshold that suppresses fixed fraction
  // of lowest-density pixels

  size_t cumcount=0;
  float densitysuppressfraction=0.12;
  size_t thridx=0;

  do{
    cumcount += densityhistogram[thridx++];
  } while(cumcount < densitysuppressfraction*density_nonzerocount);

  float densitythreshold=thridx;
  densitythreshold /= density_bincount;

 //  cerr << "determined density threshold " << densitythreshold <<
//     "(thridx="<<thridx<<") for suppressing " << cumcount << " low-density pixels out of "
//        << density_nonzerocount << endl;

  for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){
	 
    Mat  diffdensity=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"diffdensity")]);
    Mat  diffframe=*boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"suppdiff2")]);
   
    Mat before=diffframe.clone();

    for(int i = 0; i < diffframe.rows;i++)
      for(int j = 0; j < diffframe.cols;j++)
	if(diffdensity.at<float>(i,j)<densitythreshold)
	  diffframe.at<float>(i,j)=0;

  //   floatimagesc("before",before);
//     floatimagesc("after",diffframe);
//     cv::waitKey(0);

  }

  // copy the result to the bb passed as parameter

  blackboard[std::pair<int,std::string>(firstframe,targetid)]=boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(firstframe+1,"suppdiff2")])->clone();

  for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++)
    blackboard[std::pair<int,std::string>(frnumber,targetid)]=boost::any_cast<cv::Mat>(&localbb[std::pair<int,std::string>(frnumber,"suppdiff2")])->clone();

  }


  bool propagationTestBall(size_t row, size_t col, size_t frame, float pthreshold, size_t testballradius, localBlackBoardType &bb, BinaryClassifier &cls, FeatureExtractor &fe){

    Mat  pmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,"skinpmap")]);

    Mat  progressmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,"skinpropagationprogress")]);

    Mat *potentialfg=NULL;

    if(bb.count(std::pair<int,std::string>(frame,"potentialfgmask")))
      potentialfg=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frame,"potentialfgmask")]);

    // cerr << "testing ball at " << row <<","<<col<<","<<frame<< "using radius " << testballradius <<endl;
    
    if(row<testballradius || row+testballradius >= (size_t)pmap.rows
       || col<testballradius || col+testballradius >= (size_t)pmap.cols)
      return false;

    vector<float> samplevec;

    for(int dy=-(int)testballradius;dy<=(int)testballradius;dy++)
      for(int dx=-(int)testballradius;dx<=(int)testballradius;dx++){

	//	cerr << "trying dx=" << dx << " dy="<<dy<<endl;


	if(dx*dx+dy*dy > (int)(testballradius*testballradius))
	  continue;

	//	cerr << "dx=" << dx << " dy="<<dy<<endl;

	// ensure that the skin probability of that pixel 
	// is on pmap


	if(((progressmap.at<uchar>(row+dy,col+dx))&64)==0){
	  fe.extractFeatures(samplevec,row+dy,col+dx,frame,bb);

	  // hard-coded: perform detection w/ cls[0]
	  vector<bool> dimMask=fe.getFeatureMask(0);

	  pmap.at<float>(row+dy,col+dx)=cls.predict(samplevec,&dimMask);
	  if(potentialfg&&potentialfg->at<uchar>(row+dy,col+dx)==0)
	    pmap.at<float>(row+dy,col+dx)=0;
	  progressmap.at<uchar>(row+dy,col+dx) |= 64;

	}

// 	cerr << "testing location " << row+dy <<","<<col+dx<<","<<frame<<endl;

// 	cerr << "pmap="<<pmap.at<float>(row+dy,col+dx) << endl;

	if(pmap.at<float>(row+dy,col+dx)<pthreshold)
	  return false; // short-circuiting logic
      }

    return true; // all pvalues must reach the threshold

  }

  void downsampleFrame(Mat &src, Mat &dst, float /*factor*/){

    dst=Mat(src.rows/2, src.cols/2, CV_8UC3);
    Vec3b newvec;

    for(int i=0;i<dst.rows;i++)
      for(int j=0;j<dst.cols;j++){
	for(size_t d=0;d<3;d++){
	  float v=src.at<Vec3b>(2*i,2*j)[d]+src.at<Vec3b>(2*i+1,2*j)[d]+
				 src.at<Vec3b>(2*i,2*j+1)[d]+src.at<Vec3b>(2*i+1,2*j+1)[d];
	  v *= 0.25;
	  dst.at<Vec3b>(i,j)[d]=v;
	}
      }


  }


 void propagateSkin(int firstframe, int lastframe,
		    int videofirst, int videolast,
		     localBlackBoardType &bb, const string &progressid,
		     float corethreshold, float outerthreshold,
		     BinaryClassifier &cls, FeatureExtractor &fe){

   // function that invokes the three phases of skin propagation


//    cerr << "applying propagation for frame interval ["
// 	<<firstframe<<","<<lastframe<<"]"<<endl;

   // propagate the core

   size_t ballradius=2;

   doSkinPropagation(firstframe,lastframe,videofirst,videolast,
		     bb,progressid,corethreshold,ballradius,cls,fe);


   // expand the core region in the catchment basin using the outer threshold

 //   cerr << "showing results of first pass" << endl;

//     for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

//  	Mat  pmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"skinpmap")]);
//  	Mat  progressmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid)]);

// 	Mat  frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);
	
// 	Mat visframe=frame.clone();

// 	Vec3b white,gray;

// 	white[0]=white[1]=white[2]=255;
// 	gray[0]=gray[1]=gray[2]=127;
      
// 	for(int i = 0; i < frame.rows; i++)
// 	  for(int j = 0; j < frame.cols; j++){
// 	    if(progressmap.at<uchar>(i,j)&128){
// 	    visframe.at<Vec3b>(i,j)=white;
// 	    }
// 	    else if(progressmap.at<uchar>(i,j)&8){
// 	      visframe.at<Vec3b>(i,j)=gray;
// 	    }
// 	  }

// 	cv::imshow("propagation progress",visframe);
//  	cv::imshow("pmap", pmap);
//  	cv::imshow("progressmap", progressmap);
//  	cv::waitKey(0);     


//     }


   vector<float> samplevec;

   for(int frnumber=firstframe-1;frnumber<=lastframe+1;frnumber++){

     if(frnumber<videofirst) continue;
     if(frnumber>videolast) continue;

     //     cerr << "calculating watershed for frame " << frnumber << endl;
	

   Mat  frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);
   Mat  pmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"skinpmap")]);
   Mat  progressmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid)]);

   Mat  progressmap2=progressmap.clone();

   Mat mask(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));
   
   for(int i = 0; i < frame.rows; i++)
     for(int j = 0; j < frame.cols; j++)
       if(progressmap.at<uchar>(i,j)&128){ // core pixel
	 mask.at<uchar>(i,j)=255;
       } else{
	 mask.at<uchar>(i,j)=0;
	 //progressmap2.at<uchar>(i,j) = 0; 
	 progressmap2.at<uchar>(i,j) &= ~(32+16+8); 
	 // reset the pixel to unexamined status
       }
   
   // apply morphological filtering to the mask to remove
   // thin structures
   
   cv::morphologyEx(mask, mask, cv::MORPH_OPEN, 
		    cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(3, 3)), 
		    cv::Point(-1,-1), 1);
   Mat vismask=mask.clone();
   

//    cv::imshow("after first opening", mask);
//    cv::waitKey(0);     
   
   // prepare seeds for the watershed expansion
   
   vector<Blob> outBlobs = 
     Blob::extractBlobsFromMask(mask,0,0);
   
   for (size_t blobidx=0;blobidx<outBlobs.size();blobidx++)
     outBlobs[blobidx].draw(mask, Scalar::all(blobidx+1));
   
   int backgroundmargin = 40;
   
   Mat flippedMask=mask.clone();
   
   for(int i = 0; i < frame.rows;i++)
     for(int j = 0; j < frame.cols;j++)
       flippedMask.at<uchar>(i,j)=(flippedMask.at<uchar>(i,j)==0)?255:0;
   
   Mat distMat;
   distanceTransform(flippedMask, distMat, CV_DIST_L2,3);
   
   for(int i = 0; i < frame.rows; i+=6)
     for(int j = 0; j < frame.cols; j+=6)
       if(distMat.at<float>(i,j)>backgroundmargin)
	 mask.at<uchar>(i,j)=255;
   
   Mat blobmask=Mat::zeros(frame.size(), CV_32SC1);
   
   for(int i = 0; i < frame.rows; i++)
     for(int j = 0; j < frame.cols; j++)
       blobmask.at<int>(i,j)=(int)mask.at<uchar>(i,j);
   
   // perform watershed
   
   cv::watershed(frame,blobmask);
   
   for(int i = 0; i < frame.rows; i++)
     for(int j = 0; j < frame.cols; j++)
       if(blobmask.at<int>(i,j)>0&&blobmask.at<int>(i,j)!=255)
	 mask.at<uchar>(i,j)= 255;
       else
	 mask.at<uchar>(i,j)= 0;
   
   // morplogically fill small gaps
   
   cv::morphologyEx(mask, mask, cv::MORPH_CLOSE, 
		    cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(3, 3)), 
		    cv::Point(-1,-1), 1);
   
   // mask now holds the watershed expansion

   bb[std::pair<int,std::string>(frnumber,"corewatershed")]=mask.clone();
   bb[std::pair<int,std::string>(frnumber,progressid+"2")]=progressmap2.clone();

//    cv::imshow("corewatershed", mask);
//    cv::imshow("progressmap2", progressmap2);
//    cv::waitKey(0);     

   }


   // continue propagation within the watershed using the lower threshold
   
   string restrictionid("corewatershed");
   
   doSkinPropagation(firstframe,lastframe,videofirst,videolast,
		     bb,progressid+"2",outerthreshold,ballradius,
		     cls,fe,&restrictionid);

 //   cerr << "showing results of second pass" << endl;

//     for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

//  	Mat  pmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"skinpmap")]);
//  	Mat  progressmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid+"2")]);

// 	Mat  frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);
	
// 	Mat visframe=frame.clone();

// 	Vec3b white,gray;

// 	white[0]=white[1]=white[2]=255;
// 	gray[0]=gray[1]=gray[2]=127;
      
// 	for(int i = 0; i < frame.rows; i++)
// 	  for(int j = 0; j < frame.cols; j++){
// 	    if(progressmap.at<uchar>(i,j)&128){
// 	    visframe.at<Vec3b>(i,j)=white;
// 	    }
// 	    else if(progressmap.at<uchar>(i,j)&8){
// 	      visframe.at<Vec3b>(i,j)=gray;
// 	    }
// 	  }

// 	cv::imshow("propagation progress",visframe);

//  	cv::imshow("pmap", pmap);
//  	cv::imshow("progressmap", progressmap);
//  	cv::waitKey(0);     


//     }

  

   // refine the results within the ballradius from detected skin 
   // inside the catchment basin

   // prepare the masks

   for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

     Mat ws=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"corewatershed")]);


     Mat  progressmap2=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid+"2")]);

     Mat mask(ws.rows,ws.cols,CV_8UC1,cv::Scalar::all(0));
     
     for(int i = 0; i < mask.rows; i++)
       for(int j = 0; j < mask.cols; j++)
	 if(progressmap2.at<uchar>(i,j)&128)
	   mask.at<uchar>(i,j)=255;
     
     cv::dilate(mask, mask, 
		cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(ballradius, ballradius+3)));
     
     for(int i = 0; i < mask.rows; i++)
       for(int j = 0; j < mask.cols; j++){
	 if(ws.at<uchar>(i,j)!=0)
	   ws.at<uchar>(i,j)=mask.at<uchar>(i,j); // re-use the 
	 // "corewatershed" id for the new mask

	 if((progressmap2.at<uchar>(i,j)&128)==0)
	   // not a propagated skin pixel
	   progressmap2.at<uchar>(i,j) &= ~(32+16+8); 
       }


   }

   doSkinPropagation(firstframe,lastframe,videofirst,videolast,
		     bb,progressid+"2",outerthreshold,0,
		     cls,fe,&restrictionid);


 // map the new propagationresults to the original map

   for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){
	

     Mat  pmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"skinpmap")]);
     Mat  progressmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid)]);
     Mat  progressmap2=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid+"2")]);
   

     for(int i = 0; i < progressmap.rows; i++)
       for(int j = 0; j < progressmap.cols; j++)
	 if(progressmap2.at<uchar>(i,j)&128){ // detected skin pixel
	   if((progressmap.at<uchar>(i,j)&128)==0) 
	     progressmap.at<uchar>(i,j)|=8;
	 } else{

	   pmap.at<float>(i,j)=0;
	 }
   }

 //   cerr << "showing results of third pass" << endl;

//     for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

//  	Mat  pmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"skinpmap")]);
//  	Mat  progressmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid+"2")]);

// 	Mat  frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);
	
// 	Mat visframe=frame.clone();

// 	Vec3b white,gray;

// 	white[0]=white[1]=white[2]=255;
// 	gray[0]=gray[1]=gray[2]=127;
      
// 	for(int i = 0; i < frame.rows; i++)
// 	  for(int j = 0; j < frame.cols; j++){
// 	    if(progressmap.at<uchar>(i,j)&128){
// 	    visframe.at<Vec3b>(i,j)=white;
// 	    }
// 	    else if(progressmap.at<uchar>(i,j)&8){
// 	      visframe.at<Vec3b>(i,j)=gray;
// 	    }
// 	  }

// 	cv::imshow("propagation progress",visframe);

//  	cv::imshow("pmap", pmap);
//  	cv::imshow("progressmap", progressmap);
//  	cv::waitKey(0);     


//     }

//    cerr << "skinPropagation returning()" << endl;

 }
		 
  void detectShadows(int firstframe, int lastframe,
		     localBlackBoardType &bb, const string &objectmaskid,
		     const string &shadowmaskid){

    // assume that frames are on blackboard
    // assume that the object mask is of type uchar, and to be 
    // interpreted binary

    // detect shadowed and and unshadowed backgroung

    // construct background image 

    // ( currently use the same background image for the whole sequence

    Mat firstframemat=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,"frame")]);

    Mat bg(firstframemat.rows,firstframemat.cols,CV_32FC3,cv::Scalar::all(0));
    Mat bguchar(firstframemat.rows,firstframemat.cols,CV_8UC3);
    Mat counts(firstframemat.rows,firstframemat.cols,cv::DataType<int>::type,cv::Scalar::all(0));
    Mat bgint(firstframemat.rows,firstframemat.cols,cv::DataType<float>::type,cv::Scalar::all(0));


    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);
      Mat mask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,objectmaskid)]);


      for(int i=0;i<frame.rows;i++)
      	for(int j=0;j<frame.cols;j++)
	  if(mask.at<uchar>(i,j)==0){
	    counts.at<int>(i,j)++;
	    bg.at<Vec3f>(i,j)[0] += frame.at<Vec3b>(i,j)[0]; 
	    bg.at<Vec3f>(i,j)[1] += frame.at<Vec3b>(i,j)[1]; 
	    bg.at<Vec3f>(i,j)[2] += frame.at<Vec3b>(i,j)[2]; 
	  }
	  
    }


    for(int i=0;i<bg.rows;i++)
      for(int j=0;j<bg.cols;j++){
	if(counts.at<int>(i,j)!=0){
	  bg.at<Vec3f>(i,j)[0] /= counts.at<int>(i,j);
	  bg.at<Vec3f>(i,j)[1] /= counts.at<int>(i,j);
	  bg.at<Vec3f>(i,j)[2] /= counts.at<int>(i,j);

	}
	  bguchar.at<Vec3b>(i,j)[0] = bg.at<Vec3f>(i,j)[0];
	  bguchar.at<Vec3b>(i,j)[1] = bg.at<Vec3f>(i,j)[1];
	  bguchar.at<Vec3b>(i,j)[2] = bg.at<Vec3f>(i,j)[2];

      }

 //    cv::imshow("globally estimated background",bguchar);
//     cv::waitKey(0);

    Mat bghsv=bguchar.clone();

    cvtColor(bguchar,bghsv,CV_BGR2HSV);

    for(int i=0;i<bg.rows;i++)
      	for(int j=0;j<bg.cols;j++)
	  for(int d=0;d<3;d++)
	    bgint.at<float>(i,j) += bg.at<Vec3f>(i,j)[d];

    //    floatimagesc("bging",bgint);
    //    cv::waitKey(0);

    float timeconstant=0.9;
    
    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);
      Mat mask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,objectmaskid)]);
      
      Mat shadowmask=mask.clone();

      Mat frint(firstframemat.rows,firstframemat.cols,cv::DataType<float>::type,cv::Scalar::all(0));

      Mat frhsv;

      cvtColor(frame,frhsv,CV_BGR2HSV);

      for(int i=0;i<frame.rows;i++)
      	for(int j=0;j<frame.cols;j++){
	  if(mask.at<uchar>(i,j)==0){
	  for(int d=0;d<3;d++){
	    frint.at<float>(i,j) += frame.at<Vec3b>(i,j)[d];
	    bg.at<Vec3f>(i,j)[d]=timeconstant*bg.at<Vec3f>(i,j)[d]
	      + (1-timeconstant)*frame.at<Vec3b>(i,j)[d];
	  }
	  bgint.at<float>(i,j)=timeconstant*bgint.at<float>(i,j)
	    + (1-timeconstant)*frint.at<float>(i,j);
	  }


	  bguchar.at<Vec3b>(i,j)[0] = bg.at<Vec3f>(i,j)[0];
	  bguchar.at<Vec3b>(i,j)[1] = bg.at<Vec3f>(i,j)[1];
	  bguchar.at<Vec3b>(i,j)[2] = bg.at<Vec3f>(i,j)[2];
	  
	}

      cv::imshow("bkgndframe",bguchar);
      cv::waitKey(0);
      
      for(int i=0;i<frame.rows;i++)
      	for(int j=0;j<frame.cols;j++){
	  shadowmask.at<uchar>(i,j)=0;
	  if(mask.at<uchar>(i,j)!=0){
	    
	    // flag only shadows that are detected as the object
	    
	    int inttolerance=10;

	    if(frint.at<float>(i,j)>bgint.at<float>(i,j)+inttolerance) continue; // intensity constraint failed


	    // require that average hue in a neighbourhood is not very 
	    // different from the background image

	    int deltah=fabs(bghsv.at<Vec3b>(i,j)[0]-frhsv.at<Vec3b>(i,j)[0]);

	    if(frint.at<float>(i,j)>10 && deltah>90) deltah=180-deltah;

	    int huethreshold=7;

	    if(deltah>huethreshold) continue;

	    // require that colour components retain their ordering

	    size_t f1,f2,b1,b2;
	    
	    f1 = (frame.at<Vec3b>(i,j)[0]>frame.at<Vec3b>(i,j)[1])?0:1;
	    f2 = 1-f1;
	    if(frame.at<Vec3b>(i,j)[f1]<frame.at<Vec3b>(i,j)[2]){
	      f2=f1;
	      f1=2;
	    } else{
	      if(frame.at<Vec3b>(i,j)[2]>frame.at<Vec3b>(i,j)[f2])
		f2=2;
	    }

	    b1 = (bg.at<Vec3b>(i,j)[0]>bg.at<Vec3b>(i,j)[1])?0:1;
	    b2 = 1-b1;
	    if(bg.at<Vec3b>(i,j)[b1]<bg.at<Vec3b>(i,j)[2]){
	      b2=b1;
	      b1=2;
	    } else{
	      if(bg.at<Vec3b>(i,j)[2]>bg.at<Vec3b>(i,j)[b2])
		b2=2;
	    }

	    // if(!(b1==f1&&b2==f2)) continue;

	    // evaluate the linearity of intensities 
	    // in the neighbourhood of the pixel

	    float er=0,eb=0,et=0;

	    int nbrradius=1;

	    if(i<nbrradius || i>frint.rows-nbrradius) continue;
	    if(j<nbrradius || j>frint.cols-nbrradius) continue;

	    for(int di=-nbrradius;di<=nbrradius;di++)
	      for(int dj=-nbrradius;dj<=nbrradius;dj++){
		size_t ii=i+di;
		size_t jj=j+dj;
		
		er += frint.at<float>(ii,jj)*bgint.at<float>(ii,jj);
		eb += bgint.at<float>(ii,jj)*bgint.at<float>(ii,jj);
		et += frint.at<float>(ii,jj)*frint.at<float>(ii,jj);
	      }
	    
	    float ccr_threshold=0.99;

	    float ccr=er*er/(eb*et);

	    // cerr << "er: " << er << " eb:"<<eb<<" et:"<<et<<" -> ccr: " << ccr << endl;
	    
	    if(ccr<ccr_threshold) continue;


	    //finally, require that LBP signature does not differ very 
	    // much from bkg

	    
	    float lbptolerance=6;

	    int hamdist=0;
	    
	    for(int di=-nbrradius;di<=nbrradius;di++)
	      for(int dj=-nbrradius;dj<=nbrradius;dj++){
		
		size_t ii=i+di;
		size_t jj=j+dj;

		float deltaf=frint.at<float>(ii,jj)-frint.at<float>(i,j);
		float deltab=bgint.at<float>(ii,jj)-bgint.at<float>(i,j);
		  
		if(deltaf*deltab<0 && fmin(fabs(deltaf),fabs(deltab))>lbptolerance)
		  hamdist++;
	  }

	    int hamthr=4;

	    if(frint.at<float>(i,j)>10 && hamdist>hamthr) continue;



	    shadowmask.at<uchar>(i,j)=255;
	  }
	  
	}
      bb[std::pair<int,std::string>(frnumber,shadowmaskid)]=shadowmask.clone();
    }
  

 // display detected shadows

    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){
      
      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);
      Mat shadowmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,shadowmaskid)]);
      Mat mask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,objectmaskid)]);

      Mat snf=frame.clone();

      for(int i=0;i<frame.rows;i++)
      	for(int j=0;j<frame.cols;j++)
	  if(shadowmask.at<uchar>(i,j))
	     snf.at<Vec3b>(i,j)=Vec3b(0,255,0);
	  else if(mask.at<uchar>(i,j))
	    snf.at<Vec3b>(i,j)=Vec3b(0,0,255);
	
      cv::imshow("frame",frame);
      cv::imshow("potential shadow",shadowmask);
      cv::imshow("frame with shadows",snf);
      cv::waitKey(0);
    }
  } 

  void temporalSmoothPmap(localBlackBoardType &bb, const string &pmapid, 
			  const string &skinmaskid,int firstframe,
			  int lastframe){

	cv::Mat f=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,"frame")]);
	cv::Mat m=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,skinmaskid)]);
	
	float dsum=0,dsqrsum=0;
	size_t dcount=0;
	
	// for convenience, ignore some border pixels of the matrix
	
	for(int i = 1; i < f.rows-1; i++)
	  for(int j = 0; j < f.cols-1; j++){
	    if(m.at<uchar>(i,j)){
	      if(m.at<uchar>(i+1,j)){
		cv::Vec3b diff=f.at<cv::Vec3b>(i,j)-f.at<cv::Vec3b>(i+1,j);
		float d=sqrt(diff.dot(diff));
		dsum += d;
		dsqrsum += d*d;
		dcount++;
	      }
	      
	      if(m.at<uchar>(i-1,j+1)){
		cv::Vec3b diff=f.at<cv::Vec3b>(i,j)-f.at<cv::Vec3b>(i-1,j+1);
		float d=sqrt(diff.dot(diff));
		dsum += d;
		dsqrsum += d*d;
		dcount++;
	      }
	      
	      if(m.at<uchar>(i,j+1)){
		cv::Vec3b diff=f.at<cv::Vec3b>(i,j)-f.at<cv::Vec3b>(i,j+1);
		float d=sqrt(diff.dot(diff));
		dsum += d;
		dsqrsum += d*d;
		dcount++;
	      }
	      
	      if(m.at<uchar>(i+1,j+1)){
		cv::Vec3b diff=f.at<cv::Vec3b>(i,j)-f.at<cv::Vec3b>(i+1,j+1);
		float d=sqrt(diff.dot(diff));
		dsum += d;
		dsqrsum += d*d;
		dcount++;
	      }
	      
	    }
	  }
	
	dsum /= dcount;
	dsqrsum /= dcount;
	
	float dstdev=sqrt(dsqrsum-dsum*dsum);
	
	
// 	  cerr << "Estimated the between-skin-pixel std "<<dstdev << endl;
// 	 cerr << "from " << dcount << " pixel pairs" << endl;

       // now propagate detected skin pixels temporally if pixel intensity 
       // stays about constant

       size_t changecount;

       do{

	 changecount=0;

	 // forward pass

	 cv::Mat frame,prevframe,mask,prevmask,nextmask;

	 prevframe=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,"frame")]);
	 prevmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,skinmaskid)]);

	 for(int frnumber=firstframe+1;frnumber<=lastframe;frnumber++){

	   frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);
	   mask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,skinmaskid)]);
	   if(frnumber != lastframe)
	     nextmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber+1,skinmaskid)]);


	   for(int i = 0; i < frame.rows; i++)
	     for(int j = 0; j < frame.cols; j++){

	       if(prevmask.at<uchar>(i,j)>0 && mask.at<uchar>(i,j)==0 && (frnumber==lastframe||nextmask.at<uchar>(i,j)>0)){

	       // previous & next frame contains skin in this 
	       // location but this frame not

	       // 
		 cv::Vec3b diff=frame.at<cv::Vec3b>(i,j)
		   -prevframe.at<cv::Vec3b>(i,j);

		 float d=sqrt(diff.dot(diff));
	       
		 //		 cerr << "candidate for propagation: d=" << d << endl;
	       //	       cerr << "prev pixel: (" << (*pfit)[0] << " "<<(*pfit)[1]<<" "<<(*pfit)[2]<<")" << endl;
	       //	       cerr << "current pixel: (" << (*fit)[0] << " "<<(*fit)[1]<<" "<<(*fit)[2]<<")" << endl;
	       
		 if(d<2*dstdev){
		   mask.at<uchar>(i,j)= prevmask.at<uchar>(i,j);
		   float p=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber-1,pmapid)])->at<float>(i,j);

		   if(frnumber<lastframe)
		     p= 0.5*(p+boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber+1,pmapid)])->at<float>(i,j));


		   boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,pmapid)])->at<float>(i,j)=p;

		   changecount++;
		 }
	       }

	       if(prevmask.at<uchar>(i,j)==0 && mask.at<uchar>(i,j)>0 && (frnumber<lastframe&& nextmask.at<uchar>(i,j)==0) 
&& boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,pmapid)])->at<float>(i,j)<0.5 ){

		 cv::Vec3b diff=frame.at<cv::Vec3b>(i,j)
		   -prevframe.at<cv::Vec3b>(i,j);

		 float d=sqrt(diff.dot(diff));
		 if(d<2*dstdev){
		   mask.at<uchar>(i,j)=0;
		   boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,pmapid)])->at<float>(i,j)=0;
		   changecount++;
		 }

	       }

	       

	     }
	   
	   prevframe=frame;
	   prevmask=mask;

	 }

	 //	   cerr << changecount << "new skin pixels found in forward pass " << endl;

	 // backward pass

	 for(int frnumber=lastframe-1;frnumber>=(int)firstframe;frnumber--){

	   //	   cerr << "backward pass: framenr=" << frnumber << endl;

	   frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);
	   mask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,skinmaskid)]);
	   if(frnumber != firstframe)
	     nextmask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber-1,skinmaskid)]);


	   //	   cv::imshow("backward pass mask", mask);
	   //	   cv::waitKey(0);

	   for(int i = 0; i < frame.rows; i++)
	     for(int j = 0; j < frame.cols; j++){

	       if(prevmask.at<uchar>(i,j)>0 && mask.at<uchar>(i,j)==0
		  && (frnumber==firstframe||nextmask.at<uchar>(i,j)>0)){

	       // previous frame contains skin in this 
	       // location but this frame not

	       // 
		 cv::Vec3b diff=frame.at<cv::Vec3b>(i,j)
		   -prevframe.at<cv::Vec3b>(i,j);

		 float d=sqrt(diff.dot(diff));
	       
		 //		 cerr << "candidate for propagation: d=" << d << endl;
	       //	       cerr << "prev pixel: (" << (*pfit)[0] << " "<<(*pfit)[1]<<" "<<(*pfit)[2]<<")" << endl;
	       //	       cerr << "current pixel: (" << (*fit)[0] << " "<<(*fit)[1]<<" "<<(*fit)[2]<<")" << endl;
	       
		 if(d<2*dstdev){
		   mask.at<uchar>(i,j)= prevmask.at<uchar>(i,j);
		   
		   float p=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber+1,pmapid)])->at<float>(i,j);

		   if(frnumber>firstframe)
		     p= 0.5*(p+boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber-1,pmapid)])->at<float>(i,j));


		   boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,pmapid)])->at<float>(i,j)=p;



		   changecount++;
		 }
	       }
	       if(prevmask.at<uchar>(i,j)==0 && mask.at<uchar>(i,j)>0 && (frnumber>firstframe && nextmask.at<uchar>(i,j)==0) 
		  && boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,pmapid)])->at<float>(i,j)<0.5 ){

		 cv::Vec3b diff=frame.at<cv::Vec3b>(i,j)
		   -prevframe.at<cv::Vec3b>(i,j);

		 float d=sqrt(diff.dot(diff));
		 if(d<2*dstdev){
		   mask.at<uchar>(i,j)=0;
		   boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,pmapid)])->at<float>(i,j)=0;
		   changecount++;
		 }

	       }



	     }
	   
	   prevframe=frame;
	   prevmask=mask;


	 }

	 //	   cerr << changecount << "new skin pixels in total after backward pass " << endl;



	 
       } while (changecount>0);


//        cerr << "showing masks after temporal smoothing" << endl;

//        for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){
	
// 	cerr << "frame " << frnumber << endl;

// 	Mat  frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);

// 	Mat  pmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,pmapid)]);

// 	Mat framewskin=frame.clone();

// 	Mat mask=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,skinmaskid)]);


// 	Mat segmmask(frame.rows,frame.cols, CV_32SC1,Scalar::all(0));
// 	watershedSegmentation(frame,segmmask,20);


// 	  for(int i = 0; i < frame.rows; i++)
// 	    for(int j = 0; j < frame.cols; j++){

// 	      if(mask.at<uchar>(i,j)){ // detected skin pixel
// 		float p=pmap.at<float>(i,j);
// 		Vec3b org=framewskin.at<Vec3b>(i,j);
// 		framewskin.at<Vec3b>(i,j)=Vec3b(255*p+(1-p)*org[0],
// 						  255*p+(1-p)*org[1],
// 						  (1-p)*org[2]);
		
// 	      }
// // 	      if(segmmask.at<int>(i,j)<0)
// // 		framewskin.at<Vec3b>(i,j)=Vec3b(255,255,255);
// 	    }
// 	  cv::imshow("frame",frame);
// 	  cv::imshow("frame with propagated skin",framewskin);
// 	  cv::imshow("skin mask",mask);

// 	  cv::waitKey(0);
//        }

	

  }


  void sampleFrames(localBlackBoardType &bb,const string &skinmaskid, 
		    const string &pmapid,bool weightsamples,
		    const vector<size_t> &framesToSample,
		    vector<vector<float> > &negSamples,
		    vector<vector<float> > &posSamples,
		    vector<float> &negWeights,
		    vector<float> &posWeights,
		    FeatureExtractor &fe,
		    bool underfacetriangle,bool allmask, bool facearea,
		    bool negblockface, bool negblockmask, bool negblockfacelike,
		    bool negimportancesampling){


    size_t skinpixelcount=0;
    size_t backgroundpixelcount=0;

    bool aprioriweighting=true;

    negSamples.clear();
    posSamples.clear();
    negWeights.clear();
    posWeights.clear();
   
    for(size_t sampleidx=0;sampleidx<framesToSample.size();sampleidx++) {
      size_t framenr=framesToSample[sampleidx];
      //      cerr << "Sampling from frame " << framenr << endl;

      cv::Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr,"frame")]);
   
      Mat *mask=NULL;
      if(bb.count(std::pair<int,std::string>(framenr,skinmaskid))>0)
	mask=boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr,skinmaskid)]);
      else
	aprioriweighting=true; // skin fraction can't be estimated

      Mat pmap;
      if(weightsamples)
	pmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(framenr,pmapid)]);
      
      cv::Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(framenr,"facerect")]);

      // visualise the colour histogram similarity of all the segments with the face rectangle

      Mat segmmask(frame.rows,frame.cols, CV_32SC1,cv::Scalar::all(0));

      watershedSegmentation(frame,segmmask,20);

//       cv::imshow("watershed segmentation",segmmask );
	
//       cv::waitKey(0);

      vector<int> facehist;
      vector<vector<int> > segmhist;
      float maxdist=0;

      extractColourHistogram(frame,faceLocation,facehist);
      extractColourHistograms(frame,segmmask,segmhist);

      int n_segm=segmhist.size();
	
      vector<float> dist(n_segm);
      vector<int> pixelsoutsideface(n_segm,0);

      for(int i=0;i<n_segm;i++){
	dist[i]=chisqrdist(facehist,segmhist[i]);
	if(dist[i]>maxdist) maxdist=dist[i];
      }

      for(int i = 0; i < segmmask.rows; i++)
	for(int j = 0; j < segmmask.cols; j++){
	  
	  if(i>=faceLocation.y && 
	     i < faceLocation.y + faceLocation.height &&
	     j>=faceLocation.x && 
	     j < faceLocation.x + faceLocation.width) continue;
	  
	  int lbl=segmmask.at<int>(i,j);
	  if(lbl>0)
	    pixelsoutsideface[lbl]++;
	}
    
      int totalarea=segmmask.rows*segmmask.cols-faceLocation.width*faceLocation.height;

      vector<int> idx=sortedordering(dist);

      vector<float> cumarea(n_segm);

      float a_cum=0;
      
      for(int i=0;i<n_segm;i++){
	float a_new=pixelsoutsideface[idx[i]];
	a_new /= totalarea;
	a_cum += a_new;
	cumarea[idx[i]]=a_cum;
      }

 //      cerr << "dumping cumareas:" << endl;
//       for(int i=0;i<n_segm;i++)
// 	cerr << "cumarea["<<i<<"]="<<cumarea[i]<<endl;
    


      // first collect positive samples

      // scan through the triangle under the face

      if(mask){
	for(int i = faceLocation.br().y; i < frame.rows; i++){
	
	  int deltay=i-faceLocation.tl().y;
	  int minx,maxx;

	  minx=maxx=faceLocation.tl().x+faceLocation.br().x;
	  minx -= 1.5*deltay;
	  maxx += 1.5*deltay;
	  
	  minx /= 2;
	  maxx /= 2;

	  if(minx<0) minx=0;
	  if(maxx>mask->cols) maxx=mask->cols;
	  
	  for(int j = minx; j < maxx; j++){
	    
	    if(mask->at<uchar>(i,j)){
	      if(underfacetriangle){
		vector<float> samplevec;
		fe.extractFeatures(samplevec,i,j,framenr,bb);
		posSamples.push_back(samplevec);
		posWeights.push_back(weightsamples ? pmap.at<float>(i,j) : 1);
	      }
	      skinpixelcount++;
	    }
	    else backgroundpixelcount++;
	    
	  }
	}
      }
	

      if(allmask){

	assert(mask!=NULL);

	// use all the initially detected skin pixels as training samples


	for(int i = 0; i < mask->rows; i++){
	  for(int j = 0; j < mask->cols; j++){
	  
	    if(mask->at<uchar>(i,j)){
	      vector<float> samplevec;
	      fe.extractFeatures(samplevec,i,j,framenr,bb);
	      posSamples.push_back(samplevec);
	      posWeights.push_back(weightsamples ? pmap.at<float>(i,j) : 1);
	    }
	    
	  }

	}
      }

      // use the center part of the face location as positive sample
      if(facearea){

	int margin=2+0.15*faceLocation.width;

	float aspectratio=faceLocation.width/faceLocation.height;
	float maxdist=0.5*faceLocation.width-margin;

	maxdist *= maxdist;


	for(int i=faceLocation.tl().y+margin;i<=faceLocation.br().y-margin;i++)
	  for(int j=faceLocation.tl().x+margin;j<=faceLocation.br().x-margin;j++){

	    //
	    float dx=j-faceLocation.tl().x-0.5*faceLocation.width;
	    float dy=i-faceLocation.tl().y-0.5*faceLocation.height;
	    
	    if(dx*dx+dy*dy*aspectratio*aspectratio > maxdist) continue;
	    // restrict to the ellipse bounded by the box

	    vector<float> samplevec;
	    fe.extractFeatures(samplevec,i,j,framenr,bb);
	    posSamples.push_back(samplevec);
	    posWeights.push_back(weightsamples ? pmap.at<float>(i,j) : 1);
	  }
      }

      // then collect negative samples

      // expand the skin mask to introduce don't care margins around the detected skin pixels

      //       Mat dontcare=Mat::zeros(frame.rows,frame.cols,cv::DataType<uchar>::type);

      Mat dontcare(frame.rows,frame.cols, CV_8UC1,
                            Scalar::all(0));;

      if(negblockmask){

	assert(mask!=NULL);

	size_t dontcaremargin=8;
	
	// perform morphological dilation of the mask
	
	cv::dilate(*mask, dontcare, 
		   cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(dontcaremargin, dontcaremargin)));
      }

      if(negblockface){
	
	// exclude the face are from sampling
	
	for(int i=faceLocation.tl().y;i<=faceLocation.br().y;i++)
	  for(int j=faceLocation.tl().x;j<=faceLocation.br().x;j++)
	    dontcare.at<uchar>(i,j)=255;
      }

      if(negblockfacelike){

	//	  cerr << "blocking face-like areas" << endl; 
	
	float likenessthreshold=0.1;
	
	for(int i = 0; i < segmmask.rows; i++){
	  for(int j = 0; j < segmmask.cols; j++){
	    int lbl=segmmask.at<int>(i,j);
	    if(lbl>0){
	      if(cumarea[lbl]<=likenessthreshold)
		dontcare.at<uchar>(i,j)=255;
	    }
	  }
	  
	}
      }
	
	
    
//       cv::imshow("dontcare",dontcare );

//       cv::waitKey(0);
	
      float negmultiplier=4;
      size_t tgtn=negmultiplier*posSamples.size();

      //      cerr << "tgtn="<<tgtn<<endl;

      float mincumarea=1;
      
      if(negimportancesampling)
	for(int i = 0; i < segmmask.rows; i++)
	  for(int j = 0; j < segmmask.cols; j++)
	    if(dontcare.at<uchar>(i,j)==0){
	      int lbl=segmmask.at<int>(i,j);
	      if(lbl>0 && cumarea[lbl]<mincumarea)
		mincumarea=cumarea[lbl];
	    }
      
      while(negSamples.size()<tgtn){

	// sample within the frame randomly
	
	int i,j;

	do{
	  float r=rand();
	  r /= RAND_MAX;
	  i=frame.rows*r;
	} while(i>=frame.rows);

	do{
	  float r=rand();
	  r /= RAND_MAX;
	  j=frame.cols*r;
	} while(i>=frame.cols);

	if(negimportancesampling && mincumarea<1.0){
	   int lbl=segmmask.at<int>(i,j);
	   if(lbl<=0) continue;
	   float p=cumarea[lbl]-mincumarea;
	   p /= 1.0-mincumarea;

	   // now p in range [0,1];

	   // could do some monotonic transform of p here

	   p *= RAND_MAX;
	   if(rand()>=p) continue;

	}


	if(dontcare.at<uchar>(i,j)==0){
	  vector<float> samplevec;
	  fe.extractFeatures(samplevec,i,j,framenr,bb);
	  negSamples.push_back(samplevec);
	  negWeights.push_back(1);
	}
      }

    } // for frames
  
       
//     cerr << "collected " << (posSamples.size()+negSamples.size()) << 
//       " samples in total." << endl;
//     cerr << "positive: " << posSamples.size() 
// 	 << " negative: " << negSamples.size() << endl;


    //    cerr << "dumping posweights" << endl;
//     for(int i=0;i<posWeights.size();i++)
//       cerr << posWeights[i] << endl;

  
    // set the sample weighting to reflect the estimate 
    // of skin probability under the face triangle

    float poswsum=0;
    for(int i=0;i<(int)posWeights.size();i++)
      poswsum += posWeights[i];
      
    float posw;


    if(aprioriweighting)
      posw=negSamples.size()*0.05/poswsum;
    else{
      posw=negSamples.size()*skinpixelcount;
      posw /= poswsum*backgroundpixelcount;
    }

    //     cerr << "weighting positives wit wpos=" << posw << endl;

     for(int i=0;i<(int)posWeights.size();i++)
      posWeights[i] *= posw;

  }




void doSkinPropagation(int firstframe, int lastframe,
		    int videofirst, int videolast,
		     localBlackBoardType &bb, const string &progressid,
		     float threshold, size_t ballradius,
		       BinaryClassifier &cls, 
			FeatureExtractor &fe, const string *restrictionid){

  
 //  for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){
//     cv::imshow("incoming progressmap", *boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid)]));
//     if(restrictionid)
//       cv::imshow("incoming restriction", *boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,*restrictionid)]));
//     cv::waitKey(0);
//   }

   // propagate the skin areas from the seeds

   // key to progress map

   // bit 7 ( 128) -> confirmed core skin pixel
   // bit 6 ( 64 ) -> skin probability evaluated in pmap
   // bit 5 ( 32 ) -> pixel inserted in list to be tested
   // bit 4 ( 16 ) -> core membership of the pixel has been determined
   // bit 3 ( 8 )  -> confirmed outer pixel 

  std::deque<propagationRecord> propstack;

   int addcount=0; // for showing visualisations after a fixed
                    // number of pixels added

   vector<size_t> addhist(lastframe+1,0); 
   // to keep track to which frame the added pixels belong so that 
   // the frame w/ most added pixels can be shown


   int maxtime;

   // fill the stack with neighbours of the seeds

   for(int frnumber=firstframe-1;frnumber<=lastframe+1;frnumber++){

     if(frnumber<videofirst) continue;
     if(frnumber>videolast) continue;

     propagationRecord toadd;

     toadd.orgTime=0;

     Mat  pmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"skinpmap")]);
     Mat progressmap=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid)]);

     maxtime=sqrt(pmap.rows*pmap.cols)/6;

     for(int i = 0; i < pmap.rows; i++)
       for(int j = 0; j < pmap.cols; j++)
	 if(progressmap.at<uchar>(i,j)&128){ // confirmed core

	 // enumerate all the spatial neighbours

	 int xincr[]={1,1,0,-1,-1,-1,0,1,0};
	 int yincr[]={0,1,1,1,0,-1,-1,-1,0};

	 for(size_t dir=0;dir<9;dir++){
	   int ii=i+yincr[dir];
	   int jj=j+xincr[dir];


	   if(ii>=0&&ii<pmap.rows && jj>=0 && jj<pmap.cols){

	     toadd.row=ii;
	     toadd.col=jj;

	     if(frnumber>=firstframe && frnumber<= lastframe && (boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid)])->at<uchar>(ii,jj) & (32+16)) == 0){
	       toadd.frame=frnumber;
	       propstack.push_back(toadd);
	       boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,progressid)])->at<uchar>(ii,jj) |= 32;

	     } else{
	     }


	  
// 	if(frnumber>firstframe && (boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber-1,progressid)])->at<uchar>(ii,jj)&(32+16))==0){
// 	       toadd.frame=frnumber-1;
// 	       propstack.push_back(toadd);
// 	       boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber-1,progressid)])->at<uchar>(ii,jj)|=32;
// 	     } 

// 	if(frnumber<lastframe && (boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber+1,progressid)])->at<uchar>(ii,jj)&(32+16)) ==0){
// 	       toadd.frame=frnumber+1;
// 	       propstack.push_back(toadd);
// 	       boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber+1,progressid)])->at<uchar>(ii,jj)|=32;
// 	     }
	   }
	 }
	 }
   }


   //   cerr << "inserted" << propstack.size() << "neighbours of seed pixels" << endl;
 
 propagationRecord toadd;

 while(propstack.size()>0){

   propagationRecord loc=propstack.front();
   propstack.pop_front();

   //   cerr << "stack size after pop:"<<propstack.size()<<endl;

   if( ((boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame,progressid)])->at<uchar>(loc.row,loc.col))&16)==0){

     // the core membership of this pixel has not yet been determined

     //  cerr << "about to test pixel " << loc.col<<","<<loc.row<<endl;

     size_t testballradius=ballradius;
     float pthreshold=threshold;

     boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame,progressid)])->at<uchar>(loc.row,loc.col) |= 16;

     // the core membership will be determined after the test below

     if((restrictionid==NULL ||  boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame,*restrictionid)])->at<uchar>(loc.row,loc.col) != 0)&&propagationTestBall(loc.row,loc.col,loc.frame,pthreshold,testballradius,bb,cls,fe)){



	  // pixel belongs to core, mark it as a core pixel

       boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame,progressid)])->at<uchar>(loc.row,loc.col) |= 128;
       
	addcount++;
	addhist[loc.frame]++;
     
	// insert neighbours of the pixel to the stack

	if(loc.orgTime<maxtime){

	  toadd.orgTime=loc.orgTime+1;

	  // enumerate all the spatial neighbours

	  Mat pmap= *boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame,"skinpmap")]);
	
	  int xincr[]={1,1,0,-1,-1,-1,0,1,0};
	  int yincr[]={0,1,1,1,0,-1,-1,-1,0};

	  for(size_t dir=0;dir<9;dir++){
	  int ii=loc.row+yincr[dir];
	  int jj=loc.col+xincr[dir];

	  if(ii>=0&&ii<pmap.rows && jj>=0 && jj<pmap.cols){

	    toadd.row=ii;
	    toadd.col=jj;
	    
	    if((boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame,progressid)])->at<uchar>(ii,jj)&(32+16))==0){
	       toadd.frame=loc.frame;
	       propstack.push_back(toadd);
	       boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame,progressid)])->at<uchar>(ii,jj)|=32;
	     }
	  
	   //  if(loc.frame>firstframe && (boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame-1,progressid)])->at<uchar>(ii,jj)&(32+16))==0){
// 	       toadd.frame=loc.frame-1;
// 	       propstack.push_back(toadd);
// 	       boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame-1,progressid)])->at<uchar>(ii,jj)|=32;
// 	     } 

// 	    if(loc.frame<lastframe && (boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame+1,progressid)])->at<uchar>(ii,jj)&(32+16))==0){
// 	       toadd.frame=loc.frame+1;
// 	       propstack.push_back(toadd);
// 	       boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(loc.frame+1,progressid)])->at<uchar>(ii,jj)|=32;
// 	     }
	  }
	}
     }
     }
   }
   

//    int visinterval=30000;

//    if(addcount>=visinterval){


//      //     cerr << " stack size now " << propstack.size() << endl;

//      // go through the addition histogram to determine which 
//      // frame to show

//      size_t maxcount=addhist[0];
//      size_t maxind=0;

//      for(int bin=1;bin<=lastframe;bin++)
//        if(addhist[bin]>maxcount){
// 	 maxind=bin;
// 	 maxcount=addhist[bin];
//        }

//      //     cerr << "showing frame " << maxind << " with " << maxcount <<
//      //       " added pixels" << endl;

// //      cv::imshow("progress map", *boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(maxind,progressid)]));
// //      cv::waitKey(0);


//      addcount=0;
//      for(int bin=0;bin<=lastframe;bin++)
//        addhist[bin]=0;

//    }

 } // while propstack

 // cerr << "doSkinPropagation() returning" << endl;

}


  void watershedSegmentation(Mat &srcimg,Mat &dstmask,size_t gridspacing){

    dstmask=Mat(srcimg.rows, srcimg.cols, CV_32SC1,Scalar::all(0));

    int lbl=1;

    for(int i=0;i<srcimg.rows;i+=gridspacing)
      for(int j=0;j<srcimg.cols;j+=gridspacing)
	dstmask.at<int>(i,j)=lbl++;

    //    Mat vismask;
    //    falseColour32BitImg(dstmask,vismask);

 //    cv::imshow("seeds", vismask);
//     cv::waitKey(0);


    cv::watershed(srcimg,dstmask);


  }

  void falseColour32BitImg(Mat &src, Mat &dst){

    int maxind=0;

    dst=Mat(src.rows, src.cols, CV_8UC3,Scalar::all(0));

    for(int i=0;i<src.rows;i++)
      for(int j=0;j<src.cols;j++)
	if(src.at<int>(i,j)>maxind) maxind = src.at<int>(i,j);

    //    cerr << "maximum index is " << maxind  << endl;

    vector<Vec3b> colourtable(maxind+1);
    
    colourtable[0]=Vec3b(0,0,0);

    for(int i=1;i<=maxind;i++)
      colourtable[i]=Vec3b(rand()&255,rand()&255,rand()&255);

    for(int i=0;i<src.rows;i++)
      for(int j=0;j<src.cols;j++)
	if(src.at<int>(i,j)>0)
	  dst.at<Vec3b>(i,j)=colourtable[src.at<int>(i,j)];


  }

  void extractColourHistograms(cv::Mat &srcimg,cv::Mat &lblimg,vector<vector<int> > &histograms){
    int histbins=256;

    int maxind=0;

    for(int i=0;i<lblimg.rows;i++)
      for(int j=0;j<lblimg.cols;j++)
	if(lblimg.at<int>(i,j)>maxind) maxind = lblimg.at<int>(i,j);

    //    cerr << "determined maxind=" << maxind << endl;

    histograms=vector<vector<int> >(maxind+1,vector<int>(histbins,0));

    for(int i=0;i<lblimg.rows;i++)
      for(int j=0;j<lblimg.cols;j++){
	int binidx=quantiseBGR(srcimg.at<Vec3b>(i,j));
	int lbl=lblimg.at<int>(i,j);

	if(lbl>0){

// 	  cerr << "accumulating histogram for ("<<i<<","<<j<<"): binidx="
// 	       << binidx << " lbl=" << lbl << endl;

	  histograms[lbl][binidx]++;
	}
      }
    

  }
  
  void extractColourHistogram(cv::Mat &srcimg,cv::Rect &roi, vector<int> &histogram){
    int histbins=256;

    histogram=vector<int>(histbins,0);

    // take only the ellipse inside roi

    float aspectratio=roi.width/roi.height;
    float maxdist=0.5*roi.width;

    maxdist *= maxdist;
	      
    for(int y = roi.y; y < roi.y + roi.height; y++)
      for(int x = roi.x; x < roi.x + roi.width; x++){

	float dx=x-roi.tl().x-0.5*roi.width;
	float dy=y-roi.tl().y-0.5*roi.height;
	    
	if(dx*dx+dy*dy*aspectratio*aspectratio > maxdist) continue;
	// restrict to the ellipse bounded by the box

	//	cerr << "face pixel" << x <<","<< y << endl;
	histogram[quantiseBGR(srcimg.at<Vec3b>(y,x))]++;
      }

  }

  int quantiseBGR(Vec3b &bgr){

    // use the same quantisation as the MPEG-7 SC descriptor

    // assume input in [0,255] for all channels
    
    float h=bgr[2]/255.0;
    float s=bgr[1]/255.0;
    float v=bgr[0]/255.0;

    rgb_to_hsv(h,s,v);

    int hidx=(int)(h*16);
    if(hidx==16) hidx=15;
    int sidx=(int)(s*4);
    if(sidx==4) sidx=3;
    int vidx=(int)(v*4);
    if(vidx==4) vidx=3;

    return (hidx<<4) + (sidx<<2) + vidx;


  }

  void rgb_to_hsv(float &r, float &g, float &b) {
      // note that hue coordinate represents angle in
      // hsv cone. discontinuity artifact at h=1/0

      float h,s,v;

      float cmax = fmax(r,fmax(g,b));
      float cmin = fmin(r,fmin(g,b));

      v = cmax;
      if (v == 0.0) {
	s = h = 0.0;   // pixel is black
      }
      else {
	s = (cmax-cmin) / cmax;

	if(s==0) h=0;
	else{
	  float cdelta = cmax - cmin;
	  float rc = (cmax-r) / cdelta;
	  float gc = (cmax-g) / cdelta;
	  float bc = (cmax-b) / cdelta;
	  if (r == cmax)
	    h = bc-gc;
	  else
	    if (g == cmax)
	      h = 2.0 + rc-bc;
	    else
	      h = 4.0 + gc-rc;
	  h /= 6.0;
	  if (h < 0.0)
	    h += 1.0;
	}
      }
  
      r=h; g=s; b=v;
    }

  float chisqrdist(vector<int> &h1, vector<int> &h2){
    float d=0;

    // histograms are interpreted as summing to unity 

    int bins=h1.size();

    int sum1=std::accumulate(h1.begin(),h1.end(),0);
    int sum2=std::accumulate(h2.begin(),h2.end(),0);

    if(sum1==0 || sum2==0) return 0;

    for(int i=0;i<bins;i++){
      if(h1[i]+h2[i]){
	float n1=h1[i]; n1 /= sum1;
	float n2=h2[i]; n2 /= sum2;
	float diff=n1-n2;
	diff *= diff;
	diff /= n1+n2;
	d += diff;
      }
    }

    return d;

  }

  void displayHist(vector<int> &h){
    int n=h.size();

    float maxval=0;
    for(int i=0;i<n;i++)
      if(h[i]>maxval) maxval=h[i];

    int rows=200;

    Mat v(rows, n, CV_8UC1,Scalar::all(0));

    for(int i=0;i<n;i++)
      for(int r=rows-1;r>rows*(1.0-h[i]/maxval);r--){
	// cerr << "setting pixel" << r << "," << i << endl;
	v.at<uchar>(r,i)=255;
      }

    cv::imshow("histogram",v);
    cv::waitKey(0);


  }

  void dumpHist(vector<int> &h){
    int n=h.size();

    for(int i=0;i<n;i++)
      cerr << "h["<<i<<"]="<<h[i]<<endl;



  }


  float *floatvalues;
  
  int indirectIntComp(const void* _a, const void* _b)
   {
      // you've got to explicitly cast to the correct type
      const int* a = (const int*) _a;
      const int* b = (const int*) _b;

      if(floatvalues[*a] > floatvalues[*b]) return 1;              // first item is bigger than the second one -> return 1
      else
         if(floatvalues[*a] == floatvalues[*b]) return  0;         // equality -> return 0
         else         return -1;         // second item is bigger than the first one -> return -1
   }




  vector<int> sortedordering(vector<float> v){
    
    // returns the ordering of elements of v that would 
    // produce a non-decreasing list

    vector<int> idx(v.size());
    for(int i=0;i<(int)v.size();i++)
      idx[i]=i;

    floatvalues=&(v[0]);

    qsort((void*) &(idx[0]), /*number of items*/ v.size(), /*size of an item*/ sizeof(int), /*comparison-function*/ indirectIntComp);

    return idx;

  }

  void estimateBackgroundWatershed(int firstframe, int lastframe, 
				   localBlackBoardType &bb){

    
    Mat firstframemat=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(firstframe,"frame")]);
    Mat bg(firstframemat.rows,firstframemat.cols,CV_32FC3,cv::Scalar::all(0));
    Mat bguchar(firstframemat.rows,firstframemat.cols,CV_8UC3,cv::Scalar::all(0));
    Mat counts(firstframemat.rows,firstframemat.cols,cv::DataType<int>::type,cv::Scalar::all(0));

    // find the cumulative face area colour histogram

    int nbins=256;

    vector<int> facehist(nbins,0);

    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){
      if(bb.count(std::pair<int,std::string>(frnumber,"facerect"))>0){
	Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);    
	cv::Rect faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(frnumber,"facerect")]);

	// use only central part of the face for the estimation


	int xtrim=0.2*faceLocation.width;
	int ytrim=0.2*faceLocation.height;

	faceLocation.x += xtrim;
	faceLocation.width -= 2*xtrim;
	faceLocation.y += ytrim;
	faceLocation.height -= 2*ytrim;

	

	vector<int> temphist;
	extractColourHistogram(frame,faceLocation,temphist);
	for(int i=0;i<nbins;i++)
	  facehist[i] += temphist[i];
      }
    }

    


    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);    

      Mat segmmask(frame.rows,frame.cols, CV_32SC1,Scalar::all(0));
      watershedSegmentation(frame,segmmask,20);

      cv::Rect faceLocation;
      bool faceinframe=bb.count(std::pair<int,std::string>(frnumber,"facerect"))>0;
      if(faceinframe)
	faceLocation=*boost::any_cast<cv::Rect>(&bb[std::pair<int,std::string>(frnumber,"facerect")]);

      vector<vector<int> > segmhist;
      float maxdist=0;
      extractColourHistograms(frame,segmmask,segmhist);
      int n_segm=segmhist.size();
      vector<float> dist(n_segm);
      vector<int> pixelsoutsideface(n_segm,0);

      for(int i=0;i<n_segm;i++){
	dist[i]=chisqrdist(facehist,segmhist[i]);
	if(dist[i]>maxdist) maxdist=dist[i];
      }

      for(int i = 0; i < segmmask.rows; i++)
	for(int j = 0; j < segmmask.cols; j++){
	  
	  if(faceinframe &&
	     i>=faceLocation.y && 
	     i < faceLocation.y + faceLocation.height &&
	     j>=faceLocation.x && 
	     j < faceLocation.x + faceLocation.width) continue;
	  
	  int lbl=segmmask.at<int>(i,j);
	  if(lbl>0)
	    pixelsoutsideface[lbl]++;
	}
    
      int totalarea=segmmask.rows*segmmask.cols;
      if(faceinframe)
	totalarea -= faceLocation.width*faceLocation.height;

      vector<int> idx=sortedordering(dist);

      vector<float> cumarea(n_segm);

      float a_cum=0;
      
      for(int i=0;i<n_segm;i++){
	float a_new=pixelsoutsideface[idx[i]];
	a_new /= totalarea;
	a_cum += a_new;
	cumarea[idx[i]]=a_cum;
      }

      float threshold=(faceinframe)?0.1:0.25;

      for(int i=0;i<frame.rows;i++)
      	for(int j=0;j<frame.cols;j++){
	  int lbl=segmmask.at<int>(i,j);
	  if(lbl>0 && cumarea[lbl]>threshold){
	    counts.at<int>(i,j)++;
	    bg.at<Vec3f>(i,j)[0] += frame.at<Vec3b>(i,j)[0]; 
	    bg.at<Vec3f>(i,j)[1] += frame.at<Vec3b>(i,j)[1]; 
	    bg.at<Vec3f>(i,j)[2] += frame.at<Vec3b>(i,j)[2]; 
	  }
	}
    }

    for(int i=0;i<bg.rows;i++)
      for(int j=0;j<bg.cols;j++){
	if(counts.at<int>(i,j)>2){ // allow two outliers
	  bg.at<Vec3f>(i,j)[0] /= counts.at<int>(i,j);
	  bg.at<Vec3f>(i,j)[1] /= counts.at<int>(i,j);
	  bg.at<Vec3f>(i,j)[2] /= counts.at<int>(i,j);

	}
	else
	  bg.at<Vec3f>(i,j)=Vec3f(0,0,0);
	bguchar.at<Vec3b>(i,j)[0] = bg.at<Vec3f>(i,j)[0];
	bguchar.at<Vec3b>(i,j)[1] = bg.at<Vec3f>(i,j)[1];
	bguchar.at<Vec3b>(i,j)[2] = bg.at<Vec3f>(i,j)[2];
	  
      }

//     cv::imshow("estimated background",bguchar);
//     cv::waitKey(0);

    bb[std::pair<int,std::string>(-1,"background")]=bg.clone();
    bb[std::pair<int,std::string>(-1,"backgroundcounts")]=counts.clone();

  }


  void findNonBackgroundBlocks(int firstframe, int lastframe, 
			       localBlackBoardType &bb, const string &fgid){

    float colourdifftol=40;
    int blocksize=18;

    int mindifferent=160;
    
    Mat bgmat=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(-1,"background")]);
    Mat counts=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(-1,"backgroundcounts")]);


    //cerr << "got background and counts matrices" << endl;

    for(int frnumber=firstframe;frnumber<=lastframe;frnumber++){

      //      cerr << "processing frame" << frnumber << endl;

      Mat frame=*boost::any_cast<cv::Mat>(&bb[std::pair<int,std::string>(frnumber,"frame")]);    
      Mat fgmask(frame.rows,frame.cols,CV_8UC1,cv::Scalar::all(0));
      for(int i=0;i<frame.rows;i+=blocksize)
	for(int j=0;j<frame.cols;j+=blocksize){
	  int diffcount=0;

	  //	  cerr <<"i="<<i<<" j="<<j<<endl;

	  for(int ii=0;ii<blocksize;ii++){
	    if(i+ii>=frame.rows) continue;
	    for(int jj=0;jj<blocksize;jj++){
	      //	      cerr << "i+ii=" << i+ii << " j+jj="<<j+jj<<endl;
	      if(j+jj>=frame.cols) continue;
	      if(counts.at<int>(i+ii,j+jj)<3){
		diffcount++;
		fgmask.at<uchar>(i+ii,j+jj)=255;
		continue;
	      }

	      // cerr << "i+ii=" << i+ii << " j+jj="<<j+jj<<endl;

	      float dsqrsum=0;
	      for(int c=0;c<3;c++){
		float d=frame.at<Vec3b>(i+ii,j+jj)[c]-bgmat.at<Vec3f>(i+ii,j+jj)[c];
		dsqrsum += d*d;
	      }
	      if(dsqrsum>=colourdifftol*colourdifftol){
		diffcount++;
		fgmask.at<uchar>(i+ii,j+jj)=255;
	      }
	    }
	  }
	}

      bb[std::pair<int,std::string>(frnumber,"potentialfgmask")]=fgmask.clone();

      //      cv::imshow("mask before opening",fgmask);
      //             cv::waitKey(0);

      cv::morphologyEx(fgmask, fgmask, cv::MORPH_OPEN, 
		       cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(3, 3)), 
		       cv::Point(-1,-1), 3);

      //      cv::imshow("mask after opening",fgmask);
      //      cv::waitKey(0);


      for(int i=0;i<frame.rows;i+=blocksize)
	for(int j=0;j<frame.cols;j+=blocksize){
	  int diffcount=0;
	  for(int ii=0;ii<blocksize;ii++){
	    if(i+ii>=frame.rows) continue;
	    for(int jj=0;jj<blocksize;jj++){
	      //	      cerr << "i+ii=" << i+ii << " j+jj="<<j+jj<<endl;
	      if(j+jj>=frame.cols) continue;
	      if(fgmask.at<uchar>(i+ii,j+jj))
		diffcount++;
	    }
	  }

	  if(diffcount<mindifferent)
	    for(int ii=0;ii<blocksize;ii++){
	      if(i+ii>=frame.rows) continue;
	      for(int jj=0;jj<blocksize;jj++){
		if(j+jj>=frame.cols) continue;
		fgmask.at<uchar>(i+ii,j+jj)=0;
	      }
	    }


	}

  //      cv::imshow("frame",frame);
//        cv::imshow("potential non-background pixels",fgmask);
//        cv::waitKey(0);

      // convert the mask data type


      Mat fgfloatmask(frame.rows,frame.cols,cv::DataType<float>::type,cv::Scalar::all(0));
      for(int i=0;i<frame.rows;i++)
	for(int j=0;j<frame.cols;j++)
	  if(fgmask.at<uchar>(i,j)) fgfloatmask.at<float>(i,j)=1.0;

      bb[std::pair<int,std::string>(frnumber,fgid)]=fgfloatmask.clone();
    
    }


  }





}

