#include "PythonComponent.hpp"
#include "PythonComponentBase.hpp"

namespace bp = boost::python;

namespace slmotion {
  static PythonComponent DUMMY(true);

  void PythonComponent::process(frame_number_t frameNumber) {
    bp::object processFunction = instance.attr("process");
    processFunction(frameNumber);
  }

  boost::program_options::options_description PythonComponent::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description desc;
    desc.add_options()
      ("PythonComponent.class", 
       boost::program_options::value<std::string>()->default_value(""),
       "Class of the actual Python implementation");
    return desc;
  }



  Component* PythonComponent::createComponentImpl(const boost::program_options::variables_map& vm, BlackBoard* blackBoard, FrameSource* frameSource) const {
    PythonComponent pc(blackBoard, frameSource);
    if (!vm.count("PythonComponent.class"))
      throw ConfigurationFileException("Cannot construct a Python Component: "
                                       "mandatory option \"class\" is missing!");

    // create a corresponding dictionary
    bp::dict opts;
    for (auto it = vm.cbegin(); it != vm.cend(); ++it) {
      // remove the prefix, ignore irrelevant options
      if (it->first.find("PythonComponent.") == 0) {
        std::string optName = it->first.substr(strlen("PythonComponent."));
        boost::any value = it->second.value();
        if (value.type() == typeid(std::string))
          opts[optName] = bp::str(boost::any_cast<std::string>(value));
        else if (value.type() == typeid(double))
          opts[optName] = bp::object(boost::any_cast<double>(value));
        else if (value.type() == typeid(int))
          opts[optName] = bp::object(boost::any_cast<int>(value));
        else if (value.type() == typeid(bool))
          opts[optName] = bp::object(boost::any_cast<bool>(value));
        else
          throw ConfigurationFileException("Unknown value type " + 
                                           string_demangle(value.type().name()) +
                                           "encountered while constructing a "
                                           "PythonComponent!");
      }
    }

    // call the python construction function
    pc.className = vm["PythonComponent.class"].as<std::string>();
    bp::dict nameSpace = bp::extract<bp::dict>(bp::scope());
    bp::object classObject = nameSpace[pc.className];
    pc.instance = classObject(opts);
    

    return new PythonComponent(pc);
  }



  bool PythonComponent::processRangeImplementation(frame_number_t first, 
                                                   frame_number_t last,
                                                   UiCallback* uiCallback) {
    bp::extract<python::PythonComponentBase*> extractor(instance);
    if (extractor.check()) {
      python::PythonComponentBase* pcb = extractor();
      pcb->setUiCallback(uiCallback);
    }

    return instance.attr("processRange")(first, last);
  }


  std::string PythonComponent::getShortDescription() const {
    if (className != "") {
      std::string s = bp::extract<std::string>(instance.attr("getShortDescription")());
      return s;
    }
    return "A Python meta-component that can be used to instantiate components written in Python";
  }

  std::string PythonComponent::getLongDescription() const {
    if (className != "") {
      std::string s = bp::extract<std::string>(instance.attr("getLongDescription")());
      return s;
    }
    return "A Python meta-component that can be used to instantiate components written in Python";
  }

  Component::property_set_t PythonComponent::getRequirements() const {
    property_set_t props;
    if (className != "") {
      bp::object requirementSet = instance.attr("getRequirements")();
      while (bp::len(requirementSet) > 0) {
        std::string s = bp::extract<std::string>(requirementSet.attr("pop")());
        props.insert(s);
      }
    }
    return props;
  }

  Component::property_set_t PythonComponent::getProvided() const {
    property_set_t props;
    if (className != "") {
      bp::object providedSet = instance.attr("getProvided")();
      while (bp::len(providedSet) > 0) {
        std::string s = bp::extract<std::string>(providedSet.attr("pop")());
        props.insert(s);
      }
    }
    return props;
  }

  std::string PythonComponent::getShortName() const {
    std::string s;
    if (className != "") 
      s = bp::extract<std::string>(instance.attr("getShortName")());
    else
      s = "PythonComponent";
    return s;
  }

  std::string PythonComponent::getComponentName() const {
    std::string s;
    if (className != "") 
      s = bp::extract<std::string>(instance.attr("getComponentName")());
    else
      s = "Python Component";
    return s;
  }
}
