#include "PythonEnvironment.hpp"
#include "PythonComponentBase.hpp"
#include "PythonBlackBoard.hpp"
#include "PythonFrameSource.hpp"
#include "PythonException.hpp"
#include "Analyser.hpp"
#include "AnalysisController.hpp"
#include "util.hpp"
#include <boost/python/stl_iterator.hpp>
#include <boost/python/suite/indexing/vector_indexing_suite.hpp>
#include <signal.h>
#include <cstdlib>

namespace bp = boost::python;

using std::string;
using std::vector;

namespace slmotion {
  namespace python {
    static boost::python::object mainModule;
    static boost::python::object mainBuiltins;
    static boost::python::object mainBuiltinsInt;
    static boost::python::object mainNamespace;
    static boost::python::object numpy;
    static boost::python::object numpyZeros;
    static boost::python::object numpyArray;
    static boost::python::object numpyEmpty;
    static boost::python::object numpyFromString;
    static boost::python::object numpyNdArray;
    static boost::python::object numpyNdArrayResize;
    static boost::python::object numpyFloat;
    static boost::python::object numpyInt;
    static std::map<int, boost::python::object> cvTypeToNumpyDtype;

    const char* const PythonEnvironment::SLMOTION_PYTHON_ENVIRONMENT_VARIABLE = "_slmotion_env";

    // a kludge, simply to hold a pointer
    struct PointerHolder {
      PythonEnvironment* env;
    };
    

    void PythonEnvironment::initGlobals() {
      globalNamespace.clear();
      // add built-ins to the global namespace
      globalNamespace["__builtins__"] = mainNamespace["__builtins__"];

      // add a pointer to the environment
      globalNamespace[SLMOTION_PYTHON_ENVIRONMENT_VARIABLE] = bp::object(PointerHolder{ this });
    }



    void PythonEnvironment::processScript(const std::string& scriptfile) {
      // Python overrides the default SIGINT handler (to pass C-c as
      // exception to the interpreter); however, we would like to
      // return the old handler after this function is done
      static struct sigaction pythonSigintHandler;
      static bool pythonHandlerStored = false;
      if (!pythonHandlerStored) {
        sigaction(SIGINT, nullptr, &pythonSigintHandler);
        pythonHandlerStored = true;
      }

      sigaction(SIGINT, &pythonSigintHandler, nullptr);

      try {
        initGlobals();
        bp::scope s(globalNamespace);
        boost::python::exec_file(boost::python::str(scriptfile), globalNamespace, globalNamespace);
      }
      catch(boost::python::error_already_set const &) {
        std::cerr << "Python error occurred: " << std::endl;
        PyErr_Print();
        throw;
      }

      // restore the old handler
      struct sigaction act;
      act.sa_handler = nullptr;
      act.sa_flags = 0;
      sigemptyset(&act.sa_mask);

      sigaction(SIGINT, &act, nullptr);

      // this does not work with boost_python at the moment
      // Py_Finalize();
    }



    PythonEnvironment* getEnv() {
      auto scope = bp::scope();
      auto envv = scope.operator[](PythonEnvironment::SLMOTION_PYTHON_ENVIRONMENT_VARIABLE);
      PointerHolder p((bp::extract<PointerHolder>(envv)));
      return p.env;
    }



    /**
     * This is the Python-API-specific variant that creates the components from
     * Python representations
     */
    static std::vector<std::shared_ptr<Component> > 
    createComponents(const std::vector<ComponentSpecification>& componentChain, 
                     FrameSource* frameSource, BlackBoard* blackBoard) {
      std::vector<std::shared_ptr<Component> > comps;
      std::set<BlackBoard::property_key_t> providedProperties;
      for (auto it = componentChain.cbegin(); it != componentChain.cend(); ++it) {
        Component* c = COMPONENT_LIST->createComponent(it->componentName, it->opts,
                                                       getEnv()->getCommandLineParameters(),
                                                       blackBoard, frameSource);        
        if (c == nullptr)
          throw PythonException("Error: The component \"" + 
                                it->componentName + "\" could not be created!");

        Component::property_set_t requirements = c->getRequirements();
        for (auto it = requirements.cbegin(); it != requirements.cend(); ++it) {
          if (!providedProperties.count(*it)) {
            std::vector<std::string> providers = Component::findProviders(*it);
            std::string s = "Component '" + c->getComponentName() + "' requires "
              "a property called '" + *it + "' to be provided by some other "
              "component onto the black board before the component is called. "
              "However, no preceeding component in the component chain says it "
              "provides this property.";
            if (providers.size() > 0) {
              s += " The following components say they do provide the required property: ";
              int count = 0;
              for (auto jt = providers.begin(); jt != providers.end(); ++jt) {
                if (count > 0)
                  s += ", ";
                s += *jt;
                ++count;
              } 
            }
            throw PythonException(s);
          }
        }
        auto provided = c->getProvided();
        providedProperties.insert(provided.cbegin(), provided.cend());
        comps.push_back(std::shared_ptr<Component>(c));
      }
      return comps;
    }



    static cv::Mat numpyNdArrayToMatrix(const bp::object& object) {
      // ndarray is the numpy equivalent of a matrix
      // so construct a matrix
      std::string dtypeName = boost::python::extract<std::string>(boost::python::str(object.attr("dtype")));
      int nDim = boost::python::extract<int>(object.attr("ndim"));
      boost::python::tuple shape = boost::python::extract<boost::python::tuple>(object.attr("shape"));
      int rows = boost::python::extract<int>(shape[0]);
      // for vectors, there is only one dimension
      int cols = nDim > 1 ? boost::python::extract<int>(shape[1]) : 1;

      // prefer row vectors
      if (nDim == 1)
        std::swap(cols, rows);

      int channels = nDim == 3 ? boost::python::extract<int>(shape[2]) : 
        nDim == 2 ? 1 : 0;
      int type = 0;
      if (dtypeName == "uint8" && nDim == 2)
        type = CV_8UC1;
      else if (dtypeName == "uint8" && nDim == 3 && channels == 3)
        type = CV_8UC3;
      else if (dtypeName == "int16" && (nDim == 2 || nDim == 1))
        type = CV_16SC1;
      else if (dtypeName == "int32" && (nDim == 2 || nDim == 1))
        type = CV_32SC1;
      else if (dtypeName == "float32" && (nDim == 2 || nDim == 1))
        type = CV_32FC1;
      else if (dtypeName == "float64" && (nDim == 2 || nDim == 1))
        type = CV_64FC1;
      else if (dtypeName == "int32" && (nDim == 3))
        type = CV_32SC1;
      else if (dtypeName == "float32" && (nDim == 3))
        type = CV_32FC1;
      else if (dtypeName == "int64" && (nDim == 2)) {
        type = CV_32SC1; // downcast 64 bit arrays
        cols <<= 1; // copy faster by misinterpretation
      }
      else
        throw PythonException("Tried to convert a numpy ndarray to a cv::Mat, "
                              "but no known OpenCV matrix type corresponds to "
                              "dtype " + dtypeName + " with " + 
                              boost::lexical_cast<std::string>(nDim) + 
                              " dimensions and " +
                              boost::lexical_cast<std::string>(channels) +
                              " channels!");

      cv::Mat m;
      if (nDim < 3 || type == CV_8UC3)
        m = cv::Mat(rows, cols, type, 
                    bp::extract<char*>(object.attr("tostring")())
                    ).clone();
      else if (nDim >= 3) {
        int sizes[nDim];
        for (int i = 0; i < nDim; ++i)
          sizes[i] = bp::extract<int>(shape[i]);
        m = cv::Mat(nDim, sizes, type, 
                    bp::extract<char*>(object.attr("tostring")())).clone();

        if (dtypeName == "int64" && nDim == 2 && type == CV_32SC1) {
          // this is the actual, endianness-dependendt downcast
          static const union {
            uint32_t i;
            char c[4];
          } endiannessUnion { 0x01020304 };
          
          cv::Mat temp(rows, cols, type);

          for (int i = 0; i < rows; ++i)
            for (int j = (endiannessUnion.c[0] == 1); j < cols; j += 2)
              temp.at<int>(i,j>>1) = m.at<int>(i,j);
        }
      }
      return m;
    }



    template<typename T>
    T objectToCppType(const bp::object& object) {
      return bp::extract<T>(object);
    }

    template<>
    cv::Rect objectToCppType(const bp::object& object) {
      if (bp::len(object) == 4) {
        bp::object ints[4];

        for (int i = 0; i < 4; ++i) {
          bp::object o = object[i];
          ints[i] = mainBuiltinsInt(o);
        }
        return cv::Rect(bp::extract<int>(ints[0]),
                        bp::extract<int>(ints[1]),
                        bp::extract<int>(ints[2]),
                        bp::extract<int>(ints[3]));
      }
      else
        throw PythonException("cv::Rects must be constructed from an object "
                              "that behaves as if it were a 4-tuple of "
                              "ints!");
      return cv::Rect();
    }



    template<>
    cv::Point objectToCppType(const bp::object& object) {
      if (bp::len(object) == 1 && bp::len(object[0]) == 2) {
        bp::object ints[2];

        for (int i = 0; i < 2; ++i) {
          bp::object o = object[0][i];
          ints[i] = mainBuiltinsInt(o);
        }
        return cv::Point(bp::extract<int>(ints[0]),
                         bp::extract<int>(ints[1]));
      }
      else
        throw PythonException("cv::Points must be constructed from an object "
                              "that behaves as if it were a 1-tuple containing "
                              "a 2-tuple of ints!");
      return cv::Point(); // bogus
    }



    template<>
    cv::Point2f objectToCppType(const bp::object& object) {
      if (bp::len(object) == 1 && bp::len(object[0]) == 2) 
        return cv::Point2f(bp::extract<float>(object[0][0]),
                           bp::extract<float>(object[0][1]));
      else
        throw PythonException("cv::Point2fs must be constructed from an object "
                              "that behaves as if it were a 1-tuple containing "
                              "a 2-tuple of floats!");
      return cv::Point2f(); // bogus
    }



    // this does the type specific stuff
    template <typename T>
    static boost::any pythonListToVector(bp::list list) {
      std::vector<T> vector(bp::len(list));
      for (int i = 0; i < bp::len(list); ++i)
        vector[i] = objectToCppType<T>(list[i]);

      return vector;
    }



    /**
     * Attempts to convert a Python list to a suitable vector. If the
     * list is empty, an empty std::vector<int> is returned.
     */
    static boost::any pythonListToVector(bp::list list) {
      if (bp::len(list) == 0)
        return boost::any(std::vector<int>());

      std::string className = boost::python::extract<std::string>(list[0].attr("__class__")
                                                                  .attr("__name__"));
      // check that all elements have the same type
      for (int i = 0; i < bp::len(list); ++i) {
        std::string itemClassName = 
          boost::python::extract<std::string>(list[i].attr("__class__")
                                              .attr("__name__"));
        if (className != itemClassName)
          throw PythonException("Cannot convert a list with multiple types to a"
                                " vector! Every element must have the same "
                                "type.");
      }

      if (className != "ndarray")
        throw PythonException("Unknown conversion from a list of " + className 
                              + "s to a vector requested.");
      return pythonListToVector<cv::Mat>(list);
    }
   


    /**
     * Attempts to convert a Python object to a boost::any object
     * containing a valid C++ object of a similar type
     */
    boost::any objectToAny(const boost::python::object& object) {
      std::string className = boost::python::extract<std::string>(object.attr("__class__")
                                                                  .attr("__name__"));
      if (className == "ndarray") {
        return boost::any(numpyNdArrayToMatrix(object));
      }
      else if (className == "int") {
        int d = bp::extract<int>(object);
        return boost::any(d);
      }
      else if (className == "bool") {
        bool d = bp::extract<bool>(object);
        return boost::any(d);
      }
      else if (className == "float") {
        double d = bp::extract<double>(object);
        return boost::any(d);
      }
      else if (className == "str") {
        std::string s = bp::extract<std::string>(object);
        return boost::any(s);
      }
      else if (className == "list")
        return pythonListToVector(bp::extract<bp::list>(object));
      else
        throw PythonException("No conversion of a Python object of type " + 
                              className + " to a C++ type is known!");
      return boost::any();
    }



    boost::any objectToAnyAs(const boost::python::object& object,
                             const std::string& typeSpec) {
      std::string className = boost::python::extract<std::string>(object.attr("__class__")
                                                                  .attr("__name__"));
      boost::any any;
      if (typeSpec == "Rect") 
        any = objectToCppType<cv::Rect>(object);
      else if (typeSpec == "Point") 
        any = objectToCppType<cv::Point>(object);
      else if (typeSpec == "Point2f") 
        any = objectToCppType<cv::Point2f>(object);
      else if (typeSpec == "vector<Point>") {
        // assume 3-dimensional ndarray input
        int dims = bp::extract<int>(object.attr("ndim"));
        if (className != "ndarray" || 
            dims != 3)
          throw PythonException("vector<Point> expects 3-dimensional ndarray input!");
        bp::tuple shape = bp::extract<bp::tuple>(object.attr("shape"));
        if (bp::extract<int>(shape[1]) != 1 || bp::extract<int>(shape[2]) != 2)
          throw PythonException("Expected dimensions to be as follows: "
                                "(n-points,1,2) but got (" + 
                                boost::lexical_cast<std::string>(bp::extract<int>(shape[0])) +
                                "," +
                                boost::lexical_cast<std::string>(bp::extract<int>(shape[1])) +
                                "," +
                                boost::lexical_cast<std::string>(bp::extract<int>(shape[2])) +
                                ")");
        std::string dtypeName = boost::python::extract<std::string>(boost::python::str(object.attr("dtype")));
        if (dtypeName != "int" && dtypeName != "int64" && dtypeName != "int32")
          throw PythonException("Expected dtype to be one of the following: "
                                "int, or int32, but got " + dtypeName +
                                " instead.");
        cv::Mat m = numpyNdArrayToMatrix(object);
        if (m.type() != CV_32SC1)
          throw PythonException("Expected to create a CV_32SC1 matrix but got"
                                " " + cvMatTypeToString(m.type()) + 
                                "instead");
          
        std::vector<cv::Point> vec(m.size[0]);
        for (int i = 0; i < m.size[0]; ++i)
          vec[i] = cv::Point(m.at<int>(i,0,0), m.at<int>(i,0,1));
        any = vec;
      }
      else if (typeSpec == "vector<Point2f>") {
        // assume 3-dimensional ndarray input
        int dims = bp::extract<int>(object.attr("ndim"));
        if (className != "ndarray" || 
            dims != 3)
          throw PythonException("vector<Point2f> expects 3-dimensional ndarray input!");
        bp::tuple shape = bp::extract<bp::tuple>(object.attr("shape"));
        if (bp::extract<int>(shape[1]) != 1 || bp::extract<int>(shape[2]) != 2)
          throw PythonException("Expected dimensions to be as follows: "
                                "(n-points,1,2) but got (" + 
                                boost::lexical_cast<std::string>(bp::extract<int>(shape[0])) +
                                "," +
                                boost::lexical_cast<std::string>(bp::extract<int>(shape[1])) +
                                "," +
                                boost::lexical_cast<std::string>(bp::extract<int>(shape[2])) +
                                ")");
        std::string dtypeName = boost::python::extract<std::string>(boost::python::str(object.attr("dtype")));
        if (dtypeName != "float32")
          throw PythonException("Expected dtype to be one of the following: "
                                "float32, but got " + dtypeName +
                                " instead.");
        cv::Mat m = numpyNdArrayToMatrix(object);
        if (m.type() != CV_32FC1)
          throw PythonException("Expected to create a CV_32FC1 matrix but got"
                                " " + cvMatTypeToString(m.type()) + 
                                "instead");
          
        std::vector<cv::Point2f> vec(m.size[0]);
        for (int i = 0; i < m.size[0]; ++i)
          vec[i] = cv::Point2f(m.at<float>(i,0,0), m.at<float>(i,0,1));
        any = vec;
      }
      else if (typeSpec == "vector<Rect>") {
        if (className == "list")
          any = pythonListToVector<cv::Rect>(bp::extract<bp::list>(object));
        else
          throw PythonException("vector<Rect> conversion expects a list of ndarrays as input!");
      }
      else if (typeSpec == "long") {
        any = (long) bp::extract<long>(object);
      }
      else if (typeSpec == "unsigned long") 
        any = (unsigned long) bp::extract<unsigned long>(object);
      else if (typeSpec == "unsigned int") 
        any = (unsigned int) bp::extract<unsigned int>(object);
      else if (typeSpec == "unsigned short") 
        any = (unsigned short) bp::extract<unsigned short>(object);
      else if (typeSpec == "short") 
        any = (short) bp::extract<short>(object);
      else if (typeSpec == "float") 
        any = (float) bp::extract<float>(object);
      else
        throw PythonException("\"" + typeSpec + "\" is not a recognised type "
                              "specifier!");
      return any;
    }



    std::vector<Analyser> 
    static createAnalysersForPython(const std::vector<slmotion::configuration::Job>& jobs, 
                                    const configuration::SLMotionConfiguration& globalConfiguration,
                                    const std::vector<ComponentSpecification>& componentChain,
                                    std::shared_ptr<FrameSource> pythonMainFrameSource) {
      std::deque<configuration::SLMotionConfiguration> configs;
      std::vector<Analyser> anals;

      for (size_t inputIdx = 0; inputIdx < jobs.size(); ++inputIdx) {
        const std::string& inputName = jobs[inputIdx].filenames.size() > 0 ? 
          jobs[inputIdx].filenames.front() : "";
        configs.push_back(globalConfiguration);
        configuration::SLMotionConfiguration& config = configs.back();
        config.inputFilename = inputName;

        std::shared_ptr<SLIO> slio(new SLIO(inputName));
        std::shared_ptr<Visualiser> visualiser(new Visualiser(slio));

        std::shared_ptr<FrameSource> frameSource = pythonMainFrameSource.get() ?
          pythonMainFrameSource : createFrameSourceForJob(jobs[inputIdx],
                                                          config.frameCacheSize,
                                                          config.videoBackend);

#ifdef SLMOTION_THREADING
        Thread::setMaxThreads(config.maxThreads);
#endif
        applyConfiguration(config, *visualiser);
        applyConfiguration(config, *slio);
        applyConfiguration(config, *frameSource);

        auto env = getEnv();

        if (env->getBlackBoard() == nullptr) {
#ifdef SLMOTION_ENABLE_PICSOM
          env->setBlackBoard(std::shared_ptr<BlackBoard>(new BlackBoard(jobs[inputIdx].picsom,
                                                                        jobs[inputIdx].picsom_db)));
#else
          env->setBlackBoard(std::shared_ptr<BlackBoard>(new BlackBoard()));
#endif // SLMOTION_ENABLE_PICSOM
        }

	env->getMainFrameSource()->setLabel(jobs[inputIdx].label);

        std::vector<std::shared_ptr<slmotion::Component>> comps = 
          createComponents(componentChain, 
                           env->getMainFrameSource().get(), 
                           env->getBlackBoard().get());

        anals.push_back(Analyser(slio, frameSource, visualiser, 
                                 env->getBlackBoard()));
        anals.back().addPreprocessComponents(std::move(comps));
      }

      return anals;
    }



    /**
     * A wrapper to call the ``analyseVideofiles'' function of the analysis
     * controller.
     */
    static void process(size_t skip, size_t framesMax) {
      PythonEnvironment* env = getEnv();
      auto opts = env->getOpts();
      auto jobs = env->getJobs();

      // Python overrides the default SIGINT handler (to pass C-c as
      // exception to the interpreter); however, for slmotion, this is
      // problematic because this function is computationally very
      // intensive, so override the Python handler for the time being
      struct sigaction act;
      act.sa_handler = nullptr;
      act.sa_flags = 0;
      
      // there are some differences between GNU and Apple platforms
      sigemptyset(&act.sa_mask);

      struct sigaction oldact;
      sigaction(SIGINT, &act, &oldact);

      assert(opts != nullptr);
      assert(jobs != nullptr);
      
      const std::vector<ComponentSpecification>& componentChain = env->getComponentChain();
      std::vector<Analyser> analysers = createAnalysersForPython(*jobs, *opts, 
                                                                 componentChain,
                                                                 env->getMainFrameSource());
      assert(analysers.size() == 1);
      AnalysisController analCon(analysers.front().getVisualiser(), 
                                 analysers.front().getSlio(),
                                 opts->pauseMode);
      analCon.analyseVideofiles(analysers.front(), skip, framesMax,
                                false, // opts->storeImages,
                                // opts->disableAnnotationBar,
                                "",  // elan output is disabled here because typically the feature vector is not done yet
                                opts->annotationTemplateFilename,
                                opts->inputFilename,
                                // opts->visualiserCombinationStyle,
                                env,
                                env->getUiCallback());

      sigaction(SIGINT, &oldact, &act);
    }



    static void outputVideo(size_t first = 0, size_t last = SIZE_MAX) {
      PythonEnvironment* env = getEnv();
      auto opts = env->getOpts();
      if (opts->outputFilename.length() > 0) {
        auto fs = env->getMainFrameSource();
        auto bb = env->getBlackBoard();
        first = std::min(first, fs->size());
        last = std::min(last, fs->size());
        std::shared_ptr<SLIO> slio(new SLIO(opts->inputFilename));
        applyConfiguration(*opts, *slio);
        slio->setOutFps(fs->getFps());
        Visualiser visualiser(slio);
        visualiser.setComplexVisualisation(opts->complexVisualisation);
        
        auto uiCallback = env->getUiCallback();
        uiCallback->setCurrentComponentName("Video output");
        uiCallback->setCurrentComponentIndex(0);
        uiCallback->setNComponents(1);
        
        for (size_t f = first; f < last; ++f) {
          cv::Mat frame = visualiser.visualise(*fs, *bb, f);
          // frame.resize(frame.rows + ANNBARHEIGHT);
          // frame(cv::Rect(0, frame.rows-ANNBARHEIGHT, 
          //                frame.cols, ANNBARHEIGHT)).setTo(cv::Scalar::all(0));
          // visualiser.addAnnotationBar(frame, f, fs->getLabel(), *bb);
          slio->storeImage(frame, f, fs->getLabel(), *bb);
          (*uiCallback)((f+1-first)*100./(last-first));
        }
      }
      else
        std::cerr << "Warning: outputVideo() called but no video output file "
                  << "defined; not doing anything."
                  << std::endl;
    }



    static void setAnnotationTextFormat(const std::string& format) {
      getEnv()->getOpts()->annotationFormat = format;
    }



    static void setComponents(const boost::python::list& components) {
      std::vector<ComponentSpecification>* componentChain = getEnv()->getComponentChainPtr();
      componentChain->clear();
      for (boost::python::stl_input_iterator<boost::python::object> it(components), end;
           it != end; ++it) {
        ComponentSpecification c = boost::python::extract<ComponentSpecification>(*it);
        componentChain->push_back(c);
      }
    }



    static void verbose(int level) {
      slmotion::debug = level;
    }



    static void setVisualisation(const std::string& script) {
      getEnv()->getOpts()->complexVisualisation = script;
    }



    template<typename T>
    static bp::object convertToPythonObject(const T& t) {
      return bp::object(t);
    }


    /**
     * Converts an std::vector to a python list
     */
    template <typename T>
    static bp::list vectorToPythonList(const std::vector<T>& v) {
      bp::list outList;
      for (const T& e : v) 
        outList.append(convertToPythonObject(e));
      return outList;
    }



    template<typename T>
    static bp::object convertToPythonObject(const cv::Point_<T>& p) {
      bp::list l;
      l.append(p.x);
      l.append(p.y);
      bp::list l2;
      l2.append(l);
      return numpyArray(l2);
    }



    template<>
    bp::object convertToPythonObject(const cv::Rect& r) {
      boost::python::list data;
      std::vector<int> row{r.x, r.y, r.width, r.height};
      for (auto it = row.cbegin(); it != row.cend(); ++it)
        data.append(*it);
      return boost::python::object(numpyArray(data, 
                                              cvTypeToNumpyDtype[CV_32S]));
    }

    template<typename T>
    static bp::object convertToPythonObject(const std::vector<T>& v) {
      return vectorToPythonList<T>(v);
    }

    template<>
    bp::object convertToPythonObject(const FeatureVector& fv) {
      auto deque = fv.toDoubleVectorDeque();
      bp::list l;
      for (auto it = deque.cbegin(); it != deque.end(); ++it)
        l.append(vectorToPythonList<double>(*it));

      return l;
    }

    bp::object matrixToNumpyArray(const cv::Mat& m) {
      assert(m.isContinuous());
      boost::python::object* dataType = nullptr;
      if (cvTypeToNumpyDtype.count(m.depth()))
        dataType = &cvTypeToNumpyDtype[m.depth()];
      else
        throw PythonException("No known Python conversion for the OpenCV "
                              "element depth " + 
                              boost::lexical_cast<std::string>(m.depth()));
      
      assert(m.channels() == 3 || m.channels() == 1);

      bp::tuple shape;
      if (m.channels() == 3 && m.dims == 2 ) // bgr images
        shape = bp::make_tuple(m.rows, m.cols, m.channels());
      else if (m.dims == 3) 
        shape = bp::make_tuple(m.size[0], m.size[1], m.size[2]);
      else if (m.rows == 1 || m.cols == 1) // vectors
        shape = bp::make_tuple(std::max(m.rows, m.cols));
      else
        shape = bp::make_tuple(m.rows, m.cols);
        
      bp::str data((const char*)m.data, m.total()*m.elemSize());
      bp::object array = numpyFromString(data, *dataType);
      array.attr("resize")(shape);
      return array;
    }



    /**
     * Attempts to convert a boost::any object (from the black board)
     * to a suitable Python object representation
     */
    boost::python::object anyToObject(const boost::any& any) {
      if (any.type() == typeid(std::string))
        return boost::python::str(boost::any_cast<std::string>(any));
      else if (any.type() == typeid(cv::Rect)) 
        return convertToPythonObject(boost::any_cast<cv::Rect>(any));
      else if (any.type() == typeid(cv::Point))
        return convertToPythonObject(boost::any_cast<cv::Point>(any));
      else if (any.type() == typeid(cv::Point2f))
        return convertToPythonObject(boost::any_cast<cv::Point2f>(any));
      else if (any.type() == typeid(cv::Mat)) 
        return matrixToNumpyArray(boost::any_cast<cv::Mat>(any));
      else if (any.type() == typeid(std::vector<cv::Point2f>)) {
        const std::vector<cv::Point2f>& v = boost::any_cast<const std::vector<cv::Point2f>&>(any);
        int sizes[] = { static_cast<int>(v.size()), 1, 2 };
        cv::Mat m(3, sizes, CV_32FC1);
        for (size_t i = 0; i < v.size(); ++i) {
          m.at<float>(i, 0, 0) = v[i].x;
          m.at<float>(i, 0, 1) = v[i].y;
        }
        return matrixToNumpyArray(m);
      }
      else if (any.type() == typeid(std::vector<cv::Point>)) {
        const std::vector<cv::Point>& v = boost::any_cast<const std::vector<cv::Point>&>(any);
        int sizes[] = { static_cast<int>(v.size()), 1, 2 };
        cv::Mat m(3, sizes, CV_32SC1);
        for (size_t i = 0; i < v.size(); ++i) {
          m.at<int>(i, 0, 0) = v[i].x;
          m.at<int>(i, 0, 1) = v[i].y;
        }
        return matrixToNumpyArray(m);
      }
      else if (any.type() == typeid(std::vector<cv::Rect>)) {
        return vectorToPythonList<cv::Rect>(boost::any_cast<const std::vector<cv::Rect>&>(any));
      }
      else if (any.type() == typeid(std::vector<float>))
        return vectorToPythonList<float>(boost::any_cast<const std::vector<float>& >(any));
      else if (any.type() == typeid(int))
        return bp::object(boost::any_cast<int>(any));
      else if (any.type() == typeid(unsigned int))
        return bp::object(boost::any_cast<unsigned int>(any));
      else if (any.type() == typeid(short))
        return bp::object(boost::any_cast<short>(any));
      else if (any.type() == typeid(unsigned short))
        return bp::object(boost::any_cast<unsigned short>(any));
      else if (any.type() == typeid(long))
        return bp::object(boost::any_cast<long>(any));
      else if (any.type() == typeid(unsigned long))
        return bp::object(boost::any_cast<unsigned long>(any));
      else if (any.type() == typeid(float))
        return bp::object(boost::any_cast<float>(any));
      else if (any.type() == typeid(double))
        return bp::object(boost::any_cast<double>(any));
      else if (any.type() == typeid(FeatureVector))
        return convertToPythonObject(boost::any_cast<FeatureVector>(any));
      else if (any.type() == typeid(std::vector<std::vector<cv::Point2f> >))
        return convertToPythonObject(boost::any_cast<const std::vector<std::vector<cv::Point2f> >& >(any));
               
      else
        throw PythonException("No conversion of a C++ object of type " + 
                              string_demangle(any.type().name()) + " to a "
                              "Python type is known!");

      return boost::python::object();
    }



    static void constructFeatureVector(const bp::list& definitionList, 
                                       size_t firstFrame, size_t lastFrame) {
      auto env = getEnv();
      if (lastFrame == SIZE_MAX)
        lastFrame = env->getMainFrameSource()->size() - 1;

      std::vector<SOM_PAK_Component> components;
      std::vector<std::string> propertyNames; // these are corresponding property names
      std::vector<std::string> interpretations;
      vector<size_t> sizes;

      for (bp::stl_input_iterator<boost::python::dict> it(definitionList), end;
           it != end; ++it) {
        if (!it->has_key("name"))
          throw PythonException("'name' -- a compulsory element for a feature"
                                " vector entry -- is missing.");

        std::string name = bp::extract<std::string>(it->get("name"));

        if (!it->has_key("interpretation"))
          throw PythonException("'interpretation' -- a compulsory element for a feature"
                                " vector entry -- is missing.");
        std::string interpretation = bp::extract<std::string>(it->get("interpretation"));

        if (!it->has_key("property"))
          throw PythonException("'property' -- a compulsory element for a feature"
                                " vector entry -- is missing.");
        std::string property = bp::extract<std::string>(it->get("property"));

        double min = -DBL_MAX, max = DBL_MAX;
        if (it->has_key("minimum"))
          min = bp::extract<double>(it->get("minimum"));

        if (it->has_key("maximum"))
          max = bp::extract<double>(it->get("maximum"));

        SOM_PAK_Component::ValueType type = SOM_PAK_Component::ValueType::REAL;
        if (it->has_key("type")) {
          std::string typeString = bp::extract<std::string>(it->get("type"));
          if (typeString == "integer")
            type = SOM_PAK_Component::ValueType::INTEGER;
          else if (typeString == "boolean")
            type = SOM_PAK_Component::ValueType::BOOLEAN;
          else if (typeString == "real")
            type = SOM_PAK_Component::ValueType::REAL;
          else
            throw PythonException("Unknown value type: " + typeString);
        }

        std::string unit = "1";
        if (it->has_key("unit"))
          unit = bp::extract<std::string>(it->get("unit"));

        // relevant for expandable components; the number of elements in each expanded component
        size_t componentSize = it->has_key("size") ? bp::extract<size_t>(it->get("size")) : 1;

        switch (type) {
        case SOM_PAK_Component::ValueType::INTEGER:
          // the default values would overflow
          components.push_back(SOM_PAK_Component(components.size(), name, 
                                                 (int)(min == -DBL_MAX ? 
                                                       INT_MIN : min), 
                                                 (int)(max == DBL_MAX ?
                                                       INT_MAX : max), type, 
                                                 unit));
          break;

        case SOM_PAK_Component::ValueType::REAL:
          components.push_back(SOM_PAK_Component(components.size(), name, (double)min, 
                                                 (double)max, type, unit));
          break;

        case SOM_PAK_Component::ValueType::BOOLEAN:
          components.push_back(SOM_PAK_Component(components.size(), name, false, 
                                                 true, type, unit));
          break;
        }
        propertyNames.push_back(property);

        if (interpretation == "Value" || interpretation == "Expand")
          interpretations.push_back(interpretation);
        else
          throw PythonException(interpretation + " is not a valid interpretation!");

        sizes.push_back(componentSize);

        // expand the components by the desired value
        if (interpretation == "Expand") {
          components.back().setName(name + ":" + boost::lexical_cast<string>(0));
          size_t n = 0;
          while (++n < componentSize) {
            components.push_back(components.back());
            components.back().setName(name + ":" + boost::lexical_cast<string>(n));
            propertyNames.push_back(property);
            interpretations.push_back(interpretation);
            sizes.push_back(componentSize);
          }
        }
      }

      assert(components.size() == interpretations.size());
      assert(interpretations.size() == propertyNames.size());
      assert(sizes.size() == propertyNames.size());
        

      auto blackboard = env->getBlackBoard();
      auto framesource = env->getMainFrameSource();
      
      // each element in this vector corresponds to one "component track"
      std::vector<std::vector<boost::any> > featureTracks;
      for (size_t i = 0; i < components.size(); ++i) {
        std::vector<boost::any> track;
        for (size_t f = firstFrame; f <= lastFrame; ++f) {
          boost::any interpretedValue;
          if (propertyNames[i] == "framenr")
            interpretedValue = boost::any((int)f);
          else if (propertyNames[i] == "timecode")
            interpretedValue = boost::any((double)(f*1000./framesource->getFps()));
          else if (interpretations[i] == "Value") {
            // pass the value verbatim
            interpretedValue = *blackboard->get(f, propertyNames[i]);
          }
          else if (interpretations[i] == "Expand") {
            // these are handled separately
            interpretedValue = boost::any();             
          }
          else
            assert(false && "This shouldn't happen: an interpretation was "
                   "accepted but undefined");
          track.push_back(interpretedValue);
        }
        featureTracks.push_back(track);
      }

      // then do a second pass and take care of expandable values
      for (size_t i = 0; i < components.size(); ++i) {
        if (interpretations[i] == "Expand") {
          auto vt = components[i].getValueType();
          typedef SOM_PAK_Component::ValueType VT;
          for (size_t k = i; k < i + sizes[i]; ++k) {
            for (size_t f = firstFrame; f <= lastFrame; ++f) {
              vector<float> floats = toFloatVector(*blackboard->get(f, propertyNames[i]));
              if (floats.size() != sizes[i])
                throw PythonException("Expected to find a vector "
                                      "interpretation of length " + 
                                      boost::lexical_cast<string>(sizes[i]) + 
                                      " but got a vector of size " + 
                                      boost::lexical_cast<string>(floats.size()) +
                                      " instead for component " + components[i].getName());
              featureTracks[k][f - firstFrame] = boost::any(vt == VT::INTEGER ? int(floats[k]) :
                                                            vt == VT::REAL ? double(floats[k]) :
                                                            bool(floats[k]));
            }
          }
          i += sizes[i];
        }
      }
      // finally, fix indices (expand may break them)
      for (size_t i = 0; i < components.size(); ++i)
        components[i].setIndex(i);

      blackboard->set(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY,
                      FeatureVector(components,
                                    featureTracks,
                                    firstFrame,
                                    lastFrame+1));
    }



    // stores the feature data on the disk
    static void storeFeatureData() {
      auto env = getEnv();
      SLIO slio(env->getMainFrameSource()->getFileName());
      if (env->getOpts()->featureOutputFilename.length() > 0) {
        slio.setFeatureOutFile(env->getOpts()->featureOutputFilename);
        if (env->getBlackBoard()->has(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY)) {
          slio.storeFeatureData(*env->getBlackBoard()->get<FeatureVector>(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY),
                                0, env->getMainFrameSource()->getLabel(), 
                                *env->getBlackBoard());
        }
        else
          std::cerr << "Warning: No feature vector on the black board; not storing anything." << std::endl;
      }
      else
        std::cerr << "Warning: No feature output filename specified; not storing anything." << std::endl;
    }

    static void storeElanAnnotations() {
      CsvToEafConverter converter;
      auto env = getEnv();
      if (env->getBlackBoard()->has(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY)) {
        BlackBoardPointer<FeatureVector> fv = env->getBlackBoard()->get<FeatureVector>(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY);
        auto opts = env->getOpts();
        if (opts->annotationTemplateFilename.length() > 0 && 
            opts->elanOutputFilename.length() > 0) 
          converter.convert(fv->toDoubleVectorDeque(), 
                            opts->annotationTemplateFilename, 
                            fv->getFieldNameMap(), opts->elanOutputFilename,
                            env->getMainFrameSource()->getFileName());
        else {
          if (opts->annotationTemplateFilename.length() == 0)
            std::cerr << "Warning: No elan output filename given; not doing "
                      << "anything."
                      << std::endl;
          if (opts->elanOutputFilename.length() == 0)
            std::cerr << "Warning: No annotation template specified; not doing"
                      << " anything."
                      << std::endl;        
        }
      }
      else
        std::cerr << "Warning: No feature vector on the black board; cannot "
          "create annotations." << std::endl;
    }



    BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS(pbbHasOverloads, has, 1, 2);
    BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS(pbbGetOverloads, get, 1, 2);
    BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS(pbbSetOverloads, set, 2, 4);
    BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS(pbbSetAsOverloads, setAs, 3, 5);
    BOOST_PYTHON_MEMBER_FUNCTION_OVERLOADS(pbbDeleteOverloads, delete_, 1, 2);

    BOOST_PYTHON_MODULE(slmotion) {
      boost::python::def("process", &process, 
                         (boost::python::arg("skip")=0, 
                          boost::python::arg("framesMax")=SIZE_MAX));
      boost::python::def("verbose", &verbose, boost::python::arg("level")=1);
      boost::python::def("setAnnotationTextFormat", &setAnnotationTextFormat);
      boost::python::def("setComponents", &setComponents);
      boost::python::def("setVisualisation", &setVisualisation);
      boost::python::class_<ComponentSpecification> componentClass("Component",
                                                                   boost::python::init<const std::string&, 
                                                                                       boost::python::dict>());
      componentClass.def(boost::python::init<const std::string&>());
      boost::python::class_<PythonBlackBoard> blackBoardClass("BlackBoard", boost::python::no_init);
      // this kludge is there to select the overloaded versions
      // typedef bool (PythonBlackBoard::*PbbGlobalHasPtr)(BlackBoard::frame_number_t, 
      //                                                   const BlackBoard::property_key_t&);
      // PbbGlobalHasPtr pbbghp = &PythonBlackBoard::has;
      // blackBoardClass.def("has", pbbghp);
      blackBoardClass.def("has", (bool(PythonBlackBoard::*)
                                  (BlackBoard::property_key_t,
                                   BlackBoard::frame_number_t))nullptr, 
                          pbbHasOverloads());
      blackBoardClass.def("get", (boost::python::object(PythonBlackBoard::*)
                                  (BlackBoard::property_key_t,
                                   BlackBoard::frame_number_t))nullptr, 
                          pbbGetOverloads());
      blackBoardClass.def("set", (void(PythonBlackBoard::*)
                                  (boost::python::object,
                                   BlackBoard::property_key_t,
                                   int, int))nullptr, 
                          pbbSetOverloads());
      blackBoardClass.def("setAs", (void(PythonBlackBoard::*)
                                    (const boost::python::object&,
                                     const std::string&,
                                     const BlackBoard::property_key_t&,
                                     int, int))nullptr, 
                          pbbSetAsOverloads());
      blackBoardClass.def("delete", 
                          (void(PythonBlackBoard::*)
                           (const std::string&,
                            int))nullptr, 
                          pbbDeleteOverloads());
      blackBoardClass.def("deleteAll", &PythonBlackBoard::deleteAll);
      blackBoardClass.def("clear", &PythonBlackBoard::clear);
      blackBoardClass.attr("COMPRESS_WITHOUT_REFERENCES") = PythonBlackBoard::COMPRESS_WITHOUT_REFERENCES;
      blackBoardClass.attr("NONE") = PythonBlackBoard::NONE;
      boost::python::scope().attr("blackboard") = boost::python::object(PythonBlackBoard(/*pythonBlackBoard*/));

      boost::python::class_<PythonFrameSource> frameSourceClass("FrameSource", boost::python::no_init);
      frameSourceClass.def("__getitem__", &PythonFrameSource::getFrame);
      frameSourceClass.def("__len__", &PythonFrameSource::len);
      frameSourceClass.def("getNumberOfTracks", &PythonFrameSource::getNumberOfTracks);
      frameSourceClass.def("getTrack", &PythonFrameSource::getTrack);
      boost::python::scope().attr("framesource") = bp::object(PythonFrameSource());

      bp::class_<std::vector<uchar> >("PyVecUchar")
        .def(bp::vector_indexing_suite<std::vector<uchar> >());

      bp::def("constructFeatureVector", &constructFeatureVector,
              (bp::arg("definitions"),
               boost::python::arg("firstFrame")=0, 
               boost::python::arg("lastFrame")=SIZE_MAX));

      bp::def("storeFeatureData", &storeFeatureData);
      bp::def("storeElanAnnotations", &storeElanAnnotations);
      bp::def("outputVideo", &outputVideo, 
              (boost::python::arg("first")=0, 
               boost::python::arg("last")=SIZE_MAX));

      bp::register_exception_translator<PythonNotImplementedException>(&PythonNotImplementedException::translate);
      bp::class_<PythonComponentBase, boost::noncopyable>("PythonComponentBaseBase", bp::no_init);
      bp::class_<PythonComponentBaseWrapper, 
                 boost::python::bases<PythonComponentBase> >("PythonComponentBase")
        .def("processRange", &PythonComponentBaseWrapper::processRange)
        .def("process", &PythonComponentBaseWrapper::process)
        .def("callback", &PythonComponentBaseWrapper::callback)
        .def("getShortDescription", &PythonComponentBaseWrapper::getShortDescription)
        .def("getLongDescription", &PythonComponentBaseWrapper::getLongDescription)
        .def("getRequirements", &PythonComponentBaseWrapper::getRequirements)
        .def("getProvided", &PythonComponentBaseWrapper::getProvided)
        .def("getShortName", &PythonComponentBaseWrapper::getShortName)
        .def("getComponentName", &PythonComponentBaseWrapper::getComponentName);
    }  



    PythonEnvironment::PythonEnvironment(std::vector<slmotion::configuration::Job>* jobsPtr,
                                         slmotion::configuration::SLMotionConfiguration* optsPtr,
                                         const boost::program_options::variables_map* commandLineParams,
                                         std::shared_ptr<FrameSource> frameSource,
                                         std::shared_ptr<BlackBoard> bb,
                                         UiCallback* uiCallback) :
      opts(optsPtr),
      jobs(jobsPtr),
      commandLineParameters(commandLineParams),
      pythonMainFrameSource(frameSource),
      callback(uiCallback) {
      if (bb.get() == nullptr) {
#ifdef SLMOTION_ENABLE_PICSOM
        pythonBlackBoard = std::shared_ptr<BlackBoard>(new BlackBoard(jobs->front().picsom,
                                                                      jobs->front().picsom_db));
#else // SLMOTION_ENABLE_PICSOM
        pythonBlackBoard = std::shared_ptr<BlackBoard>(new BlackBoard());
#endif // SLMOTION_ENABLE_PICSOM
      }
      else 
        pythonBlackBoard = bb;

      try {
        static bool interpreterInitialised = false;
        if (!interpreterInitialised) {
          // this if for mac:
          // check if site-packages needs to be set
          if (checkIfFileExists(getInstallPrefix() + "/lib/python2.7/site-packages/cv.py")) {
            const char* currentPythonPath = getenv("PYTHONPATH");
            std::string newPath = "PYTHONPATH=" + getInstallPrefix() + "/lib/python2.7/site-packages/";
            if (currentPythonPath)
              newPath += std::string(":") + currentPythonPath;
            static char newPathCharArray[1024];
            strcpy(newPathCharArray, newPath.c_str());
            
            if (putenv(newPathCharArray)) {
              fprintf(stderr, "FATAL ERROR: Could not putenv: %s\n", newPath.c_str());
              exit(EXIT_FAILURE);
            }
          }

          Py_Initialize();

          numpy = boost::python::import("numpy");
          boost::python::import("cv2");
        
          initslmotion();

          mainModule = boost::python::import("__main__");
          mainNamespace = mainModule.attr("__dict__"); 
          mainBuiltins = mainModule.attr("__builtins__");
          mainBuiltinsInt = mainBuiltins.attr("int");

          numpyZeros = numpy.attr("zeros");
          numpyArray = numpy.attr("array");
          numpyEmpty = numpy.attr("empty");
          numpyFloat = numpy.attr("float");
          numpyInt = numpy.attr("int");
          numpyFromString = numpy.attr("fromstring");
          numpyNdArray = numpy.attr("ndarray");
          cvTypeToNumpyDtype[CV_32F] = numpy.attr("dtype")("float32");
          cvTypeToNumpyDtype[CV_64F] = numpy.attr("dtype")("float64");
          cvTypeToNumpyDtype[CV_16S] = numpy.attr("dtype")("int16");
          cvTypeToNumpyDtype[CV_32S] = numpy.attr("dtype")("int32");
          cvTypeToNumpyDtype[CV_8U] = numpy.attr("dtype")("uint8");
          boost::python::numeric::array::set_module_and_type("numpy", "ndarray");
          interpreterInitialised = true;

          // initialise environment class
          bp::class_<PointerHolder>("PointerHolder", bp::no_init);
        }

        // add current working directory to path (seems to be missing by default)
        char* currentDir = getcwd(nullptr,0);
        bp::object sys = bp::import("sys");
        sys.attr("path").attr("insert")(0, bp::str(std::string(currentDir)));
        free(currentDir);
      }
      catch(boost::python::error_already_set const &) {
        std::cerr << "Python error occurred: " << std::endl;
        PyErr_Print();
        throw;
      }
    }
  }
}
