#include "RuleSkinDetector.hpp"
#include "util.hpp"

using cv::Vec3b;

namespace slmotion {
  static RuleSkinDetector DUMMY(true);



  /**
   * Ruleset A is as follows:
   *  pixel x = (R,G,B) is skin iff
   *   R > 95 AND G > 40 AND B > 20 AND max{R,G,B} - min{R,G,B} > 15 AND
   *   |R-G| > 15 AND R > G AND R > B
   *
   * @param bgrVector colour vector to classify
   * @return Returns true if the constraints are satisfied
   */
  static bool ruleA(const Vec3b& bgrVector) {
    uchar B = bgrVector[0];
    uchar G = bgrVector[1];
    uchar R = bgrVector[2];

    return R > 95 && G > 40 && B > 20 &&
      (std::max(std::max(R, G), B) - std::min(std::min(R, G), B)) > 15 &&
      abs(R-G) > 15 && abs(R > G) && R > B;

  }



  /**
   * Ruleset B is as follows:
   * pixel x = (R,G,B) is skin iff
   *   R > 220 AND G > 210 AND B > 170 AND |R-G| <= 15 AND R > B AND 
   *   G > B
   */
  static bool ruleB(const Vec3b& bgrVector) {
    uchar B = bgrVector[0];
    uchar G = bgrVector[1];
    uchar R = bgrVector[2];

    return R > 220 && G > 210 && B > 170 && std::abs(R-G) <= 15 && R > B &&
      G > B;
  }



  void RuleSkinDetector::detect(const cv::Mat& inFrame, cv::Mat& outMask) {
    decltype(&ruleA) ruleFunction = NULL;
    switch (ruleset) {
    case Ruleset::A:
      ruleFunction = &ruleA;
      break;
    case Ruleset::B:
      ruleFunction = &ruleB;
      break;
    }

    for(int y = 0; y < inFrame.rows; ++y) {
      for (int x = 0; x < inFrame.cols; ++x) {
        const cv::Vec3b& e = inFrame.at<cv::Vec3b>(y,x);
        // Set the value 255 if all rules evaluate as true
        // 0 otherwise
        outMask.at<uchar>(y,x) = ruleFunction(e) ? 255 : 0;
      }
    }
  }



  RuleSkinDetector::RuleSkinDetector(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) :
    SkinDetector(configuration, blackBoard, frameSource) {
    std::string s = configuration["RuleSkinDetector.ruleset"].as<std::string>();
    if (s == "a")
      ruleset =  Ruleset::A;
    else if (s == "b")
      ruleset =  Ruleset::B;
    else
      throw ConfigurationFileException("Invalid ruleset");
  }



  boost::program_options::options_description RuleSkinDetector::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()
      ("RuleSkinDetector.ruleset", 
       boost::program_options::value<std::string>()->default_value("a"),
       "Selects which fixed set of rules from Kovač et al. should be used. "
       "Possible values are a and b.");
    return opts;
  }



  Component* RuleSkinDetector::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const {
    RuleSkinDetector* r = new RuleSkinDetector(configuration, blackBoard, frameSource);
    r->setColourSpace(*ColourSpace::BGR);
    return r;
  }
}
