#include "SomPakHeaderDocument.hpp"
#include <fstream>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string.hpp>
#include "SLIO.hpp"
#include "util.hpp"

#ifdef SLMOTION_ENABLE_PICSOM
#include <DataBase.h>
#endif // SLMOTION_ENABLE_PICSOM

using std::string;
using std::cerr;
using std::endl;
using std::vector;
using std::cout;
using std::list;
using std::unique_ptr;

using boost::lexical_cast;

using cv::Size;
using cv::Mat;
using cv::imread;


namespace {
  using slmotion::Annotation;

  /**
   * A trivial class used as a functor to determine if the annotation 
   * matches the frame number.
   */
  struct InAnnRange {
    size_t frnumber;

    explicit InAnnRange(size_t frnumber) : frnumber(frnumber) {}

    bool operator()(const Annotation& ann) {
      return ann.timerefs.first <= static_cast<int>(frnumber) &&
        ann.timerefs.second >= static_cast<int>(frnumber);
    }
  };
}



namespace slmotion {
  extern int debug;


  // Define static members
  unique_ptr<SLIO::FormattedOutput> SLIO::debugOut(new SLIO::FormattedOutput);
  unique_ptr<SLIO::FormattedOutput> SLIO::infoOut(new SLIO::FormattedOutput);
  unique_ptr<SLIO::FormattedOutput> SLIO::warnOut(new SLIO::FormattedOutput);



  /**
   * Converts something into a string.
   * 
   * @param x Whatever needs to be converted
   * @param width Field width
   * @param fill Fill character in case width > the width of x
   */
  template<typename T> static inline std::string toString(const T& x, int width = 0, char fill = ' ') {
    std::stringstream s;
    if (width > 0) {
      s.width(width);
      s.fill(fill);
    }
    s << x;
    return s.str();
  }



  string SLIO::getAnnotationText(size_t frnumber) {
    InAnnRange inRange(frnumber);
    if (currentAnnotation != annotations.end() && 
        inRange(*currentAnnotation))
      return currentAnnotation->text;
    
    currentAnnotation = find_if(annotations.begin(),
                                annotations.end(),
                                inRange);
    if (currentAnnotation != annotations.end() && 
        inRange(*currentAnnotation))
      return currentAnnotation->text;    
    return "";
  }



  void SLIO::FormattedOutput::flush() {
    os << prefix << buffer.str();
    buffer.str("");
  }



  void SLIO::setDebugFormat(const std::string& msg, std::ostream& out) {
    debugOut.reset(new FormattedOutput(out, msg));
  }

  

  void SLIO::setInfoFormat(const std::string& msg, std::ostream& out) {
    infoOut.reset(new FormattedOutput(out, msg));
  }



  void SLIO::setWarningFormat(const std::string& msg, std::ostream& out) {
    warnOut.reset(new FormattedOutput(out, msg));
  }



  SLIO::FormattedOutput& SLIO::FormattedOutput::operator<< (std::ostream&(*manipulator)(std::ostream&)) {
    buffer << manipulator;
    writeln();
    return *this;
  }
  


  // void SLIO::resetCapture() {
  //   capture.release();
  //   currentFrameNumber = 0;
  //   capture = cv::VideoCapture(videoFilename);

  //   if (!capture.isOpened())
  //     throw SLIOException("Capture not initialised. Does the file exist?");
  // }



  bool SLIO::loadAnnFile(const string& filename) {
    std::ifstream annFile(filename.c_str());
    if (!annFile.is_open()) {
      cerr << "slmotion: WARNING: Failed to open " << filename <<
        " for reading." << endl;
      return false;
    }

    // read in annotations
    annotations.clear();
    Annotation ann;
    while(annFile >> ann) 
      annotations.push_back(ann);

    annotations.sort();
    currentAnnotation = annotations.begin();
    return true;
  }



  void SLIO::storeFeatureData(const FeatureVector& features,
                              size_t fileNumber,
			      const std::string& label,
			      const BlackBoard& blackBoard) {
    if (debug > 1)
      std::cerr << "slmotion: SLIO::storeFeatureData()" << std::endl
                << "slmotion: SLIO: parseFeatureOutFilename" << std::endl;
    string filename = parseFeatureOutFilename(featureOutFilename, fileNumber,
					      label, blackBoard);
    // If no filename has been specified, do nothing.
    if (filename.length() > 1) {
      size_t len = filename.length();
      string extension = len > 3 ? filename.substr(len-3) : "";// file extension
      boost::to_lower(extension);
      if (extension == "dat") {
        if (debug > 1)
          std::cerr << "slmotion: SLIO: storeFeatureDataInSomPak" << std::endl;
        storeFeatureDataInSomPak(filename, features, label);
      }
      else if (extension == "csv") {
        if (debug > 1)
          std::cerr << "slmotion: SLIO: storeFeatureDataInCsv" << std::endl;
        storeFeatureDataInCsv(filename, features, label);
      }
      else {
        cerr << "WARNING: Expected either 'dat' or 'csv' as the filename extension for the feature output file. However, '" + extension + "' was provided. Assuming CSV format." << endl;
        if (debug > 1)
          std::cerr << "slmotion: SLIO: storeFeatureDataInCsv" << std::endl;
        storeFeatureDataInCsv(filename, features, label);
      }
    }
  }



  void SLIO::storeFeatureDataInCsv(const std::string& filename,
                                   const FeatureVector& features,
				   const std::string& /*label*/) {
    std::ofstream ofs(filename);

    // write field names
    auto comps = features.getComponents();
    vector<string> fieldNames;
    for (auto it = comps.begin(); it != comps.end(); ++it)
      fieldNames.push_back(it->getName());
    storeFeatureEntry(ofs, fieldNames);

    auto fullMotionDescriptors = features.toDoubleVectorDeque();
    for (auto it = fullMotionDescriptors.cbegin();
         it != fullMotionDescriptors.cend(); ++it)
      storeFeatureEntry(ofs, *it);
  }


  static std::string solveInputDir(const std::string& label,
				   const BlackBoard& blackBoard) {
    string inputDir = ".";

#ifdef SLMOTION_ENABLE_PICSOM
    if (label!="") {
      picsom::DataBase *db = blackBoard.getPicSOMdataBase();
      std::string fname = db->SolveObjectPath(label);
      size_t p = fname.rfind('/');
      if (p!=string::npos)
	inputDir = fname.substr(0, p);
    }
#else
    inputDir = (void*)&label<(void*)&blackBoard ? inputDir : inputDir;
#endif // SLMOTION_ENABLE_PICSOM

    return inputDir;
  }


  std::string SLIO::parseFeatureOutFilename(const std::string& filename,
					    size_t fileNumber,
					    const std::string& label,
					    const BlackBoard& blackBoard) {
    string rtext = filename;
    string inputDir = solveInputDir(label, blackBoard);
    string stdDest = inputDir+"/features/"+label+"-slmotion.dat";

    boost::algorithm::replace_all(rtext, "%f", 
                                  boost::lexical_cast<string>(fileNumber));
    boost::algorithm::replace_all(rtext, "%l", label); 
    boost::algorithm::replace_all(rtext, "%b", inputDir);
    boost::algorithm::replace_all(rtext, "%B", stdDest);

    return rtext;
  }


#if 0
  bool SLIO::getNextFrame(Mat& img, size_t* frameNumber, size_t skip) {
    for (size_t i = 0; i < skip; i++) {
      if ( !capture.grab())
        return false;
      if (slmotion::debug > 0)
        cerr << '\r' << "Skipping frames: " << i + 1 << "/" << skip;
    }
    currentFrameNumber += skip;

    // update the pointer
    if (frameNumber)
      *frameNumber = currentFrameNumber++;
    
    if ( !capture.grab())
      return false;
    
    // in case no scaling or aspect ratio correction needs to be applied,
    // just return the frame as-is
    if (aspect < 0 && (scale.width < 0 || scale.height < 0))
      return capture.retrieve(img);

    // perform scaling and aspect ratio correction as necessary
    Mat temp;
     
    if (!capture.retrieve(temp))
      return false;
      
    Size newSize = scale.width > 0 && scale.height > 0 ? scale : temp.size();
    if (aspect > 0)
      newSize.width = newSize.height * aspect;

    if (debug > 1)
      cerr << "Converting video to " << newSize.width 
           << "x" << newSize.height << "pixels" << endl;
      
    resize(temp, img, newSize);

    return true;
  }
#endif


  static std::string parseOutFilename(const std::string& outFilename,
                                      size_t frnumber,
                                      const std::string& label,
				      const BlackBoard& blackBoard) {
    string filename = outFilename;
    string frameNrString = toString(frnumber, 6, '0');
    string inputDir = solveInputDir(label, blackBoard);

    boost::algorithm::replace_all(filename, "%f", frameNrString);
    boost::algorithm::replace_all(filename, "%l", label);
    boost::algorithm::replace_all(filename, "%b", inputDir);
    boost::algorithm::replace_all(filename, "%%", "%");
    return filename;
  }

  void SLIO::storeInVideofile(const Mat& frame, size_t frnumber, 
                              const std::string& label,
			      const BlackBoard& blackBoard) {
    if (!writer.isOpened()) {
      std::string parsedName = parseOutFilename(outFilename, frnumber,
						 label, blackBoard);
      if (debug > 1)
        cerr << "Trying to open a new video writer for <"
	     << parsedName << "> ... ";
    
      writer.open(parsedName,
		  CV_FOURCC(fourcc[0], fourcc[1], fourcc[2], fourcc[3]),
		  outFps, frame.size());

      if (!writer.isOpened()) {
        if (debug > 1)
          cerr << "FAILED!" << endl;
        throw SLIOException("Could not open video writer.");
      }
      else if (debug > 1)
        cerr << "Done." << endl;
    }
    writer << frame;
  }


  string SLIO::parseAnnFormat(const string& annFormat, size_t frNumber,
			      const std::string& lab,
			      const BlackBoard& blackBoard) const {
    string frnumberText = toString(frNumber);
    size_t q = lab.find(':');
    if (q!=string::npos)
      frnumberText = "";
    
    // If the current frame number is greater than the last frame for the
    // current annotation, read lines until a valid file is encountered and
    // the file handle remains open

    string annText = ((SLIO*)this)->getAnnotationText(frNumber);        
    
    // Replace magic markers according to their meaning
    // Then, finally, replace %%
    string frtext = annFormat;
    boost::algorithm::replace_all(frtext, "%f", frnumberText);
    boost::algorithm::replace_all(frtext, "%t", annText);
    boost::algorithm::replace_all(frtext, "%l", lab);
    boost::algorithm::replace_all(frtext, "%%", "%");

#ifdef SLMOTION_ENABLE_PICSOM
    using namespace picsom;

    for (;;) {
      size_t p = frtext.find("%T{");
      if (p==string::npos)
	break;
      size_t q = frtext.find('}', p);
      if (q!=string::npos) {
	string f = frtext.substr(p+3, q-p-3);
	frtext.erase(p, q-p+1);
	string s;
	q = f.find(':');
	if (q!=string::npos) {
	  s = f.substr(q+1);
	  f.erase(q);
	}
	picsom::DataBase *db = blackBoard.getPicSOMdataBase();
	int idx = db->LabelIndex(lab);
	string iname = db->DefaultTextIndex();
	string txt = db->TextIndexData(idx, iname, f);
	if (s=="1w") {
	  size_t q1 = txt.find(','), q2 = txt.find(';');
	  q = q1<q2 ? q1 : q2;
	  if (q!=string::npos)
	    txt.erase(q);
	}
	frtext.insert(p, txt);
      }
    }

    for (;;) {
      size_t p = frtext.find("%c{");
      if (p==string::npos)
	break;

      size_t q = frtext.find('}', p);
      if (q!=string::npos) {
	string clsall = frtext.substr(p+3, q-p-3);
	frtext.erase(p, q-p+1);
	picsom::DataBase *db = blackBoard.getPicSOMdataBase();
	list<string> clslist = db->SplitClassNames(clsall);
	string label = lab;
	if (frnumberText!="")
	  label += ":"+frnumberText;
	int idx = db->LabelIndex(label);
	int parent = db->ParentObject(idx);

	bool found = false;
	for (list<string>::const_iterator i=clslist.begin();
	     i!=clslist.end(); i++) {
	  const string& cls = *i;
	  picsom::ground_truth gt = db->GroundTruthExpression(cls);
	  bool hit = gt.index_ok(idx) && gt[idx]==1;
	  if (parent>=0 && gt.index_ok(parent) && gt[parent]==1)
	    hit = true;
	  if (hit) {
	    if (found)
	      frtext.insert(p++, " ");
	    frtext.insert(p, cls);
	    p += cls.size();
	    found = true;
	  }
	}
      }
    }

    for (;;) {
      size_t p = frtext.find("%s{");
      if (p==string::npos)
	break;

      size_t q = frtext.find('}', p);
      if (q!=string::npos) {
	string clsall = frtext.substr(p+3, q-p-3);
	frtext.erase(p, q-p+1);
	picsom::DataBase *db = blackBoard.getPicSOMdataBase();
	string label = lab;
	if (frnumberText!="")
	  label += ":"+frnumberText;
	string seglabel = "seg:"+label+"_lh";
	int segidx = db->LabelIndexGentle(seglabel);
	if (segidx>=0) {
	  frtext.insert(p, "lh");
	  p += 2;
	  if (false) {
	    Index *idxptr = db->FindIndex("contourfd", "", true, false);
	    VectorIndex *vecidxptr = dynamic_cast<VectorIndex*>(idxptr);
	    vecidxptr->ReadDataFile(false, false);
	    vecidxptr->SetDataSetNumbers(false, false);
	    const FloatVector *vec = vecidxptr->DataVector(segidx);
	    vec->Dump();
	  }
 	}
	seglabel = "seg:"+label+"_rh";
	segidx = db->LabelIndexGentle(seglabel);
	if (segidx>=0) {
	  frtext.insert(p, "rh");
	  p += 2;
	}
      }
    }

    for (;;) {
      size_t p = frtext.find("%F{");
      if (p==string::npos)
	break;

      size_t q = frtext.find('}', p);
      if (q!=string::npos) {
	string fname = "slmotion";
	string fspec = frtext.substr(p+3, q-p-3);
	frtext.erase(p, q-p+1);
	size_t np = fspec.find_first_of("0123456789");
	if (np!=0) {
	  np = fspec.find('[');
	  if (np!=string::npos) {
	    fname = fspec.substr(0, np);
	    fspec.erase(0, np+1);
	  }
	}
	int fcomp = atoi(fspec.c_str());
	picsom::DataBase *db = blackBoard.getPicSOMdataBase();
	string label = lab;
	if (frnumberText!="")
	  label += ":"+frnumberText;
	if (fname=="hsdet")
	  label = "seg:"+label+"_lh";
	int idx = db->LabelIndexGentle(label);
	if (idx>=0) {
	  Index *idxptr = db->FindIndex(fname, "", true, false);
	  VectorIndex *vecidxptr = dynamic_cast<VectorIndex*>(idxptr);
	  vecidxptr->ReadDataFile(false, false);
	  vecidxptr->SetDataSetNumbers(false, false);
	  const FloatVector *vec = vecidxptr->DataVector(idx);
	  if (vec) {
	    // vec->Dump();
	    frtext.insert(p, ToStr((*vec)[fcomp]));
	  } else
	    frtext.insert(p, "no");
 	}
      }
    }
#else
    long z = (long)&blackBoard;
    z = z+z;
#endif // SLMOTION_ENABLE_PICSOM

    // get something from black board
    for (;;) {
      size_t p = frtext.find("%b{");
      if (p==string::npos)
	break;
      size_t q = frtext.find('}', p);
      if (q!=string::npos) {
	string f = frtext.substr(p+3, q-p-3);
	frtext.erase(p, q-p+1);
        boost::any some = blackBoard.get(frNumber, f);
	frtext.insert(p, anyToString(some));
      }
    }

    
    return frtext;
  }



  void SLIO::storeInImagefile(const Mat& frame, size_t frnumber,
                              const std::string& label,
			      const BlackBoard& blackBoard) const {
    cv::imwrite(parseOutFilename(outFilename, frnumber, label, blackBoard),
		frame);
  }



  void SLIO::storeImage(const Mat& frame, size_t frnumber, 
                        const std::string& label,
			const BlackBoard& blackBoard) {
    if (outFilename.length() > 0) {
      if (videoOut)
        storeInVideofile(frame, frnumber, label, blackBoard);
      else
        storeInImagefile(frame, frnumber, label, blackBoard);
    }
    else
      throw SLIOException("Tried to store an image file into a file, but no"
                          " output filename is set.");
  }



  void SLIO::loadImages(const std::vector<std::string>& inFilenames, std::list<cv::Mat>& outList) {
    if (debug > 2)
      cerr << "Reading images" << endl;
    for (vector<string>::const_iterator it = inFilenames.begin();
         it != inFilenames.end(); it++) {
      outList.push_back(cv::imread(*it));
      if (outList.back().empty()) {
        cerr << "slmotion: WARNING: Read an empty matrix." << endl;
        cerr << "Are you sure \"" << *it << "\" is a valid image file?" << endl;
        outList.pop_back();
      }
    }
  }
 
 
 
  void SLIO::storeFeatureEntry(std::ofstream& ofs,
                               const vector<string>& strings) {
    if (strings.size() > 0) {
      std::ostringstream ss;
      
      for (size_t i = 0; i < strings.size() - 1; i++)
        ss << strings[i] << ',';
      ss << strings.back();
      ofs << ss.str() << '\n';
    }
  }



  void SLIO::storeFeatureEntry(std::ofstream& ofs,
                               const vector<double>& v) {
    assert(v.size() > 0);
    string s;
    for(auto it = v.cbegin(); it < v.cend() - 1; ++it)
      s += lexical_cast<string>(*it) + ",";
    s += lexical_cast<string>(v.back());
    ofs << s << endl;
  }



  void SLIO::setVideoFourcc(const std::string& newFourcc) {
    if (newFourcc.length() != 4)
      throw SLIOException("FourCC must be exactly four bytes long!");

    for (int i = 0; i < 4; ++i)
      fourcc[i] = newFourcc[i];
  }



  namespace {
    void writeSpaceSeparatedVectorRow(std::ofstream& ofs, 
                                      const std::vector<double>& v) {
      for(size_t i = 0; i < v.size(); ++i)
        ofs << lexical_cast<string>(v[i]) << " ";
    }
  }



  void SLIO::storeFeatureDataInSomPak(const std::string& filename, 
                                      const FeatureVector& features,
				      const std::string& label) {
    if (debug > 1)
      std::cerr << "slmotion: SLIO::storeFeatureDataInSomPak()" << std::endl
                << "slmotion: SLIO: construct SomPakHeaderDocument" 
                << std::endl;
    SomPakHeaderDocument header("slmotion", "slmotion feature", "all features produced by SLMotion in one long vector", "video" /* or should it be image? */);
  
    if (debug > 1)
      std::cerr << "slmotion: header.addComponents()" << std::endl;
    header.addComponents(features.getComponents());

    if (!header.validate())
      throw SLIOException("Invalid header");

    if (debug > 1)
      std::cerr << "slmotion: Open ofstream to \"" << filename << "\"" 
                << std::endl;
    std::ofstream ofs(filename);

    if (debug > 1)
      std::cerr << "slmotion: Open ostringstream" << std::endl;
    std::ostringstream oss;

    if (debug > 1)
      std::cerr << "slmotion: Write header to ostringstream" << std::endl;
    oss << header;

    if (debug > 1)
      std::cerr << "slmotion: Open istringstream" << std::endl;
    std::istringstream iss(oss.str());

    std::string s;


    if (debug > 1)
      std::cerr << "slmotion: Copy lines from istringstream to ofstream" 
                << std::endl;
    while (getline(iss, s))
      ofs << "# " << s << std::endl;

    if (debug > 1)
      std::cerr << "slmotion: Write feature component count" 
                << std::endl;
    ofs << lexical_cast<string>(features.getComponents().size()) << endl;

    if (debug > 1)
      std::cerr << "slmotion: Set label" << std::endl;
    string lab = label!="" ? label : videoFilename;

    if (debug > 1)
      std::cerr << "slmotion: Construct feature double vector deque" 
                << std::endl;
    auto fullFeatureVector = features.toDoubleVectorDeque();

    if (debug > 1)
      std::cerr << "slmotion: Write space separated vector rows" 
                << std::endl;
    for (size_t i = 0; i < fullFeatureVector.size(); ++i) {
      writeSpaceSeparatedVectorRow(ofs, fullFeatureVector[i]);
      ofs << lab << ":" << i << endl;
    }
    if (debug > 1)
      std::cerr << "slmotion: Done writing! Closing ofstream..." 
                << std::endl;
  }
}
