#ifndef SLMOTION_SLIO
#define SLMOTION_SLIO

#include <deque>
#include <list>
#include <memory>
#include <string>
#include "opencv.hpp"
#include "Annotation.hpp"
#include "exceptions.hpp"
#include "SOM_PAK_Component.hpp"
#include "FeatureVector.hpp"
#include "BlackBoard.hpp"



namespace slmotion {
  /**
   * This class should serve as a level of abstraction for I/O operations
   */
  class SLIO {
  public:
    /**
     * Creates a new I/O object that manages video capturing from the given
     * file, and also video output to a window or image files.
     *
     * @param filename Input filename
     * @param windowName Name of the namedWindow object
     * @param aspectCorrection If set, performs aspect ratio correction
     * @param aspect Desired aspect ratio (e.g. 1.33)
     * @param outFilename Output filename
     * @param videoOutput If true, output video rather than still images
     */
    explicit SLIO(const std::string& filename) :
      videoFilename(filename),
      //      windowName(windowName),
      videoOut(true),
      currentAnnotation(annotations.end()),
      outFps(25),
      currentFrameNumber(0)
    {
      //      resetCapture();
      setVideoFourcc("MJPG");
    }



    /**
     * Sets the FourCC code for output video files. OpenCV determines the
     * correct encoder using this piece of information.
     *
     * @param fourcc Input fourcc. Must be exactly four characters long.
     */
    void setVideoFourcc(const std::string& fourcc);



    /**
     * Parses the annotation formatting string
     * 
     * @param annFormat The annotation formatting parameter
     * @param frnumber Current frame number (%f)
     *
     * @return Parsed output
     */
    std::string parseAnnFormat(const std::string& annFormat, size_t frnumber,
			       const std::string& label,
			       const BlackBoard& blackBoard) const;



    /**
     * Sores the image in an image file, or passes it to a video encoder,
     * assuming a valid output filename is set.
     *
     * @param frame Current frame
     * @param frnumber Current frame number
     * @param label PicSOM label
     */
    void storeImage(const cv::Mat& frame, size_t frnumber,
                    const std::string& label,
		    const BlackBoard& blackBoard);



    /**
     * Captures the next frame from the input video stream.
     * Grabs the next frame using whatever method OpenCV uses by default.
     *
     * @param img Output parameter for a matrix to store the image in
     * @param frameNumber Optional pointer to a variable used to store the 
     * frame number
     * @param skip Optional parameter that specifies how many frames are to be
     * skipped before attempting to store the result.
     *
     * @return True if successful
     */
    // bool getNextFrame(cv::Mat& img, size_t* frameNumber = NULL, 
    //                   size_t skip = 0);



    /**
     * Reads in the content of the annotation file
     *
     * @param filename The new filename
     *
     * @return True if the file was opened successfully
     */
    bool loadAnnFile(const std::string& filename);



    /**
     * Sets the feature output file
     *
     * @param filename The new filename     
     */
    inline void setFeatureOutFile(const std::string& filename) {
     featureOutFilename = filename;
    }



    /**
     * Goes through all filenames in the vector, loads the files, and stores
     * them as matrices in outList
     *
     * @param inFilenames A vector of input filenames
     * @param outList Output list of matrices
     */
    void loadImages(const std::vector<std::string>& inFilenames, std::list<cv::Mat>& outList);



    /**
     * Sets the fourcc for the video writer
     *
     * @param s A string, the first four letters of which are considered.
     */
    inline void setFourcc(const std::string& s) {
      for (unsigned int i = 0; i < 4; i++) {
	if (i < s.length())
	  fourcc[i] = s.at(i);
	else
	  fourcc[i] = ' ';
      }
    }



    /**
     * Sets the output FPS for video writer
     *
     * @param d New value
     */
    inline void setOutFps(double d) {
      outFps = d;
    }



    inline void setOutFilename(const std::string& filename) {
      outFilename = filename;
    }



    class FormattedOutput; // forward declaration

    /**
     * Returns an instance to the formatted debug output
     *
     * @param The debug output instance
     */
    static FormattedOutput& getDebugOut() {
      return *debugOut;
    }



    /**
     * Returns an instance to the formatted warning output
     *
     * @param The warning output instance

     */
    static FormattedOutput& getWarningOut() {
      return *warnOut;
    }



    /**
     * Returns an instance to the formatted info output
     *
     * @param The info output instance
     */
    static FormattedOutput& getInfoOut() {
      return *infoOut;
    }



    /**
     * Sets debug message format and output stream
     */
    static void setDebugFormat(const std::string& msg = "", std::ostream& out = std::cerr);



    /**
     * Sets warning message format and output stream
     */
    static void setWarningFormat(const std::string& msg = "", std::ostream& out = std::cerr);



    /**
     * Sets info message format and output stream
     */
    static void setInfoFormat(const std::string& msg = "", std::ostream& out = std::cout);



    /**
     * A helper class to provide formatted output to a desired stream.
     *
     * Since different applications may want to use different types of debug
     * output, this class should be instantiated in the application
     * implementation file (the same file as the main function).
     */
    class FormattedOutput {
    public:
      /**
       * Sets the current instance according to the parameters
       *
       * @param os Reference to an output stream
       * @param format Formatting prefix that should be attached to the
       * beginning of each line
       */
      static void set(std::ostream& os = std::cerr,
		      const std::string& format = "");



      /**
       * Writes data to the output stream.
       */
      template <typename T> FormattedOutput& operator<< (const T& msg) {
	buffer << msg;
	writeln();
	return *this;
      }



      /**
       * Operator overloading to pass manipulator functions (such as
       * std::endl) to the internal buffer stream.
       *
       * @param manipulator A function pointer to a manipulator function
       *
       * @return A reference to this
       */
      FormattedOutput& operator<< (std::ostream&(*manipulator)(std::ostream&));



      /**
       * Writes a prefix and any remaining data from the buffer, then empties
       * the buffer.
       */
      void flush();



      /**
       * A destructor that flushes the buffer upon destruction.
       */
      ~FormattedOutput() {
	if (buffer.str().length() > 0)
	  flush();
      }

      /**
       * Creates the instance.
       *
       * @param os Sets the output stream for outputting the debug content
       * @param format Sets the output format. Currently, this is simply added
       * as a prefix to messages.
       */
      explicit FormattedOutput(std::ostream& os = std::cerr, 
			       const std::string& format = "") :
	os(os), prefix(format) {}



    private:
      /**
       * Writes lines if necessary
       */
      inline void writeln() {
	size_t n = buffer.str().find('\n');
	while (n != std::string::npos) {
	  os << prefix << buffer.str().substr(0, n+1);
	  buffer.str(buffer.str().substr(n+1));
	  buffer.seekp(0, std::ios_base::end);
	  n = buffer.str().find('\n');
	}
      }



      /**
       * Internal buffer
       */
      std::ostringstream buffer;

      std::ostream& os;
      std::string prefix;

      // Prevent copying and assignment
      FormattedOutput(const FormattedOutput&);
      FormattedOutput& operator= (const FormattedOutput&);
    };



    inline void setVideoOut(bool b) {
      videoOut = b;
    }



    void storeFeatureData(const FeatureVector& features,
                          size_t fileNumber,
			  const std::string& label,
			  const BlackBoard& blackBoard);



  private:
    /**
     * Stores the feature vector data in SOM_PAK format
     */
    void storeFeatureDataInSomPak(const std::string& filename,
				  const FeatureVector& features,
				  const std::string& label);



    /**
     * Stores the feature vector data in CSV format
     */
    void storeFeatureDataInCsv(const std::string& filename,
			       const FeatureVector& features,
			       const std::string& label);



    /**
     * Parses the given feature output filename, expanding any variables
     * necessary.
     *
     * @param filename Input filename template
     * @param fileNumber Current file number
     *
     * @return The proper filename
     */
    static std::string parseFeatureOutFilename(const std::string& filename,
                                               size_t fileNumber,
					       const std::string& label,
					       const BlackBoard& blackBoard);



    /**
     * Stores an entry in the following format:
     * d0,d1,d2,d3,d4...,dn\n
     * @param v Vector to be stored in CSV format
     */
    static void storeFeatureEntry(std::ofstream& ofs,
                                  const std::vector<double>& v);



    /**
     * Stores an entry in the following format:
     * string1,string2,string3...,stringn\n
     *
     * Useful for storing MD field identifiers
     *
     * @param strings A vector of strings
     **/
    static void storeFeatureEntry(std::ofstream& ofs,
                                  const std::vector<std::string>& strings);



    /**
     * Stores the given frame in an image file
     *
     * @param frame Image to be stored
     * @param frnumber Current frame number (possibly used to generate the
     * filename)
     */
    void storeInImagefile(const cv::Mat& frame, size_t frnumber,
                          const std::string& label,
			  const BlackBoard& blackBoard) const;



    /**
     * Passes the frame to the video writer. The writer is initialised on the
     * first call to this function.
     *
     * @param frame Frame to be stored
     * @param frnumber Current frame number
     */
    void storeInVideofile(const cv::Mat& frame, size_t frnumber, 
                          const std::string& label,
			  const BlackBoard& blackBoard);



    /**
     * Stores a single string in the MD output file. A newline is appended.
     *
     * @param s The string that needs to be stored.
     */
    // static void storeMDEntry(std::ofstream& ofs,
    //                          const std::string& s);



    /**
     * Formatted debug output
     */
    static std::unique_ptr<FormattedOutput> debugOut;



    /**
     * Formatted warning output
     */
    static std::unique_ptr<FormattedOutput> warnOut;



    /**
     * Formatted info output
     */
    static std::unique_ptr<FormattedOutput> infoOut;



    /**
     * Returns an annotation matching the frame number
     */
    std::string getAnnotationText(size_t frnumber);



    // cv::VideoCapture capture; ///< An OpenCV video capture object
    cv::VideoWriter writer; ///< An OpenCV video writer object
    std::string videoFilename; ///< Input video filename
    // std::string windowName; ///< Video frame window name
    std::string outFilename; ///< Output filename
    bool videoOut; ///< If set, produce a video file as output instead of still images
    std::string featureOutFilename; ///< Media descriptor output file stream

    std::list<Annotation> annotations;
    std::list<Annotation>::const_iterator currentAnnotation;

    char fourcc[4]; ///< The fourcc code for the video writer
    double outFps; ///< Output fps for the video writer
    size_t currentFrameNumber; ///< Used to keep track of current frame
  };



  /**
   * A very simple class for signaling I/O errors
   */
  class SLIOException : public SLMotionException {
  public:
    /**
     * Constructs the exception
     *
     * @param msg An explanatory message
     */
    SLIOException(const char* msg) throw() :
      SLMotionException(msg) 
    {}
  };
}
#endif
