#ifndef SLMOTION_SOM_PAK_COMPONENT
#define SLMOTION_SOM_PAK_COMPONENT

#include <cstddef>
#include <string>

namespace slmotion {
  class SOM_PAK_Component {
  public:
    /**
     * These are pseudo-types. True C++ types should be obtained using the
     * template below.
     */
    enum class ValueType {
      INTEGER, 
      REAL, 
      BOOLEAN 
    }; 
 
    template <typename T>
    T getMinValue() const {
      switch(valueType) {
      default:
      case ValueType::INTEGER:
        return min.i;
        break;

      case ValueType::REAL:
        return min.d;
        break;

      case ValueType::BOOLEAN:
        return min.b;
        break;
      }
    }



    template <typename T>
    T getMaxValue() const {
      switch(valueType) {
      default:
      case ValueType::INTEGER:
        return max.i;
        break;

      case ValueType::REAL:
        return max.d;
        break;

      case ValueType::BOOLEAN:
        return max.b;
        break;
      }
    }



    /**
     * The C++ end type is provided through specialisations of this class. 
     * It can be accessed by specifying SOM_PAK_Component::CppType::type as 
     * the variable type.
     *   When done correctly, won't break even if the underlying 
     * implementation is altered.
     *
     * @param unit Real-life unit that the component is supposed to present,
     * e.g. "m/s", "Hz", or "pixel/rad" etc.
     */
    template<ValueType V> struct CppType;

    template<typename T>
    SOM_PAK_Component(size_t index, const std::string& name, T minimum, 
                      T maximum, ValueType type, const std::string& unit) :
      index(index), name(name), valueType(type), unit(unit) {
      setMinMax(minimum, maximum);
    }



    inline size_t getIndex() const {
      return index;
    }



    inline void setIndex(size_t i) {
      index = i;
    }



    inline std::string getName() const {
      return name;
    }



    inline void setName(const std::string& newName) {
      name = newName;
    }



    inline std::string getUnit() const {
      return unit;
    }



    inline ValueType getValueType() const {
      return valueType;
    }



    /**
     * Returns true if the two components are equal memberwise
     */
    bool operator==(const SOM_PAK_Component& other) const {
      if (!(index == other.index && name == other.name && 
            valueType == other.valueType && unit == other.unit))
        return false;

      switch(valueType) {
      case ValueType::INTEGER:
        return min.i == other.min.i && max.i == other.max.i;
        break;
        
      case ValueType::REAL:
        return min.d == other.min.d && max.d == other.max.d;
        break;

      case ValueType::BOOLEAN:
        return min.b == other.min.b && max.b == other.max.b;
        break;
      }

      return false;
    }



  private:
    union MinMaxUnion {
      double d;
      int i;
      bool b;
    };

    template<typename T>
    void setMinMax(T minimum, T maximum) {
      switch(valueType) {
      case ValueType::INTEGER:
        min.i = minimum;
        max.i = maximum;
        break;

      case ValueType::REAL:
        min.d = minimum;
        max.d = maximum;
        break;

      case ValueType::BOOLEAN:
        min.b = minimum;
        max.b = maximum;
        break;
      }
    }

    size_t index;
    std::string name;
    MinMaxUnion min;
    MinMaxUnion max;
    ValueType valueType;
    std::string unit; ///< Real-life units that the component is supposed to present
  };



  template<>
  struct SOM_PAK_Component::CppType<SOM_PAK_Component::ValueType::INTEGER>  {
    typedef int type;
  };
  
  template<>
  struct SOM_PAK_Component::CppType<SOM_PAK_Component::ValueType::REAL> {
    typedef double type;
  };
  
  template<> 
  struct SOM_PAK_Component::CppType<SOM_PAK_Component::ValueType::BOOLEAN>  {
    typedef bool type;
  };
}

#endif
