#include "SkinDetector.hpp"
#include "math.hpp"
#include "util.hpp"

using std::vector;
using std::string;

using boost::lexical_cast;

using cv::Mat;
using cv::Range;
using cv::Point;
using cv::Scalar;
using cv::MatConstIterator_;
using cv::Vec3b;
using cv::Rect;
using cv::Size;
using cv::TermCriteria;
using std::cerr;
using std::endl;
using boost::program_options::value;



namespace {
  const int MINIMUM_BLOB_DISTANCE = 100; ///< The minimum number of pixels that blob centroids must  be apart
}



namespace slmotion {
  extern int debug;



  int findBestK2(const Mat& samples, double elbowCriterion, int maxK) {
    Mat tempLabels;
    // OpenCV 2.2 version:
    // double oldComp = cv::kmeans(samples, 1, tempLabels, TermCriteria(TermCriteria::MAX_ITER + TermCriteria::EPS, 10,1.0), 10, cv::KMEANS_RANDOM_CENTERS, NULL);
    // OpenCV 2.3 version:
    double oldComp = cv::kmeans(samples, 1, tempLabels, 
                                TermCriteria(TermCriteria::MAX_ITER + 
                                             TermCriteria::EPS, 10,1.0), 10, 
                                cv::KMEANS_RANDOM_CENTERS, cv::noArray());

    double comp; // compactness figure
    for (int i = 2; i < maxK; ++i) {
      comp = kmeans(samples, i, tempLabels,
                    TermCriteria(TermCriteria::MAX_ITER + TermCriteria::EPS, 10,
                                 1.0), 10, cv::KMEANS_RANDOM_CENTERS, 
                    cv::noArray());

      // OpenCV 2.2 version:
      // comp = kmeans(samples, i, tempLabels,
      //               TermCriteria(TermCriteria::MAX_ITER + TermCriteria::EPS, 10,
      //                            1.0), 10, cv::KMEANS_RANDOM_CENTERS, NULL);

      if ((oldComp - comp) / oldComp < elbowCriterion)
        return i-1;

      oldComp = comp;
    }
    return 0;
  }



  void SkinDetector::process(size_t framenr) {
    if (!getBlackBoard().has(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY))
      getBlackBoard().set(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY,
                          Mat());
    BlackBoardPointer<Mat> outMask = getBlackBoard().get<Mat>(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);
    const Mat& inFrame = getFrameSource()[framenr];

    // Make sure that the mask is not empty
    *outMask = Mat(inFrame.size(), CV_8UC1, cv::Scalar::all(0));

    // preprocessing: colour space conversion
    Mat temp;
    colourSpace->convertColour(inFrame, temp);

    // detection
    detect(temp, *outMask);

    // post processing
    postprocess(inFrame, *outMask);

    getBlackBoard().set(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY, 
                        *outMask);

    // if (debug == 1) {
    //   cv::imshow("output", outMask);
    //   cv::waitKey(0);
    // }
  }



  // dummy implementations
  SkinDetector::~SkinDetector() { }
  void SkinDetector::train(const std::vector<cv::Mat>&, 
                           const std::vector<cv::Mat>&,
                           bool) { }



  int findBestK(const cv::Mat& samples, int maxK) {
    assert(samples.type() == CV_32FC1);

    // Extract validation set by randomly selecting one fourth of vectors
    // create a matrix from a certain range
    Mat range(samples.rows, 1, CV_32SC1);

    for (int i = 0; i < samples.rows; ++i)
      range.at<int>(i,0) = i;

    randShuffle(range);

    range = range(Range(0, range.rows/4), Range::all());

    cv::sort(range, range, CV_SORT_EVERY_COLUMN | CV_SORT_ASCENDING);

    Mat trainSet(samples.rows - range.rows, samples.cols, samples.type());
    Mat validSet(range.rows, samples.cols, samples.type());

    // std::cout << range << std::endl;

    {
      cv::MatConstIterator_<int> it = range.begin<int>();
      int trainI = 0;
      int validI = 0;
      for (int i = 0; i < samples.rows; ++i) {
        if (it != range.end<int>() && i == *it) {
          for (int j = 0; j < samples.cols; ++j) 
            validSet.at<float>(validI, j) = samples.at<float>(i,j);
          ++validI;
          ++it;
        }
        else {
          for (int j = 0; j < samples.cols; ++j) 
            trainSet.at<float>(trainI, j) = samples.at<float>(i,j);
          ++trainI;
        }
      }
    }

    // std::cout << "Training set:" << std::endl;
    // std::cout << trainSet;
    // std::cout << "Validation set:" << std::endl;
    // std::cout << validSet;

    Mat tempCenters;
    Mat tempLabels;

    double error, bestError;
    error = bestError = DBL_MAX;
    int bestK = 0;
    
    for(int i = 1; i < maxK; ++i) {
      std::cout << 
        // OpenCV 2.2:
        // kmeans(trainSet, i, tempLabels,
        //        TermCriteria(TermCriteria::MAX_ITER +
        //                     TermCriteria::EPS, 10, 1.0), 10,
        //        cv::KMEANS_RANDOM_CENTERS, &tempCenters)
        kmeans(trainSet, i, tempLabels,
               TermCriteria(TermCriteria::MAX_ITER +
                            TermCriteria::EPS, 10, 1.0), 10,
               cv::KMEANS_RANDOM_CENTERS, tempCenters)
                << std::endl;

      // compute errors 
      error = kMeansValidationError(validSet, tempCenters);

      // Akaike information criterion'ish approach
      error = 2*i + validSet.rows * log(error);

      if (error < bestError) {
        bestError = error;
        bestK = i;
      }
    }

    return bestK;
  }



  double kMeansValidationError(const Mat& samples, const Mat& clusterMeans) {
    // for each sample t
    // samples are assumed to be row vectors, as are cluster means
    assert(samples.cols == clusterMeans.cols);
    assert(samples.type() == CV_32FC1);
    assert(clusterMeans.type() == CV_32FC1);
    double error = 0;
    for (int t = 0; t < samples.rows; ++t) {
      // select minimal error
      double d = DBL_MAX;
      for (int i = 0; i < clusterMeans.rows; ++i) {
        double e = 0; // error for each mean separately
        for (int j = 0; j < clusterMeans.cols; ++j) {
          double f = samples.at<float>(t,j) - clusterMeans.at<float>(i,j);
          e += f*f;
        }
        if (e < d)
          d = e;
      }
      error += d; // sum
    }
    error /= samples.rows;
    return clusterMeans.rows * error;
  }



  void getBiggestClusters(const Mat& samples, vector<Mat>& clusters, Mat& means,
                          int k, size_t totalClusters) {
    assert(samples.type() == CV_32FC1);
    Mat labels;
    Mat meansTemp;

    // Mat invCoVar; // this matrix will be used to hold the square root of the inverse covariance matrix, for normalisation purposes
    // cv::calcCovarMatrix(samples, invCoVar, meansTemp, CV_COVAR_NORMAL | 
    //                     CV_COVAR_SCALE | CV_COVAR_ROWS, CV_32FC1);
    // Mat P, D, P1, Dd;
    // diagonalise(invCoVar, P, D, P1);
    // Dd = D.diag();
    // cv::pow(Dd, -0.5, Dd);
    // Mat samples2 = samples * P * D * P1;

    // // In some cases, the normalisation may fail, e.g. if there is no 
    // // variance in some axes (e.g. all zeros). In such a case, if any sample
    // // is a NaN, use the unnormalised samples.
    // for (auto it = samples2.begin<float>(); it != samples2.end<float>();
    //      ++it) {
    //   if (*it != *it) {
    //     cerr << "WARNING: NaN values were found among normalised samples "
    //          << "while training skin detector. This could be due to axes "
    //          << "with no variance, e.g. the colour axes of black-and-white "
    //          << "footage. Be advised: the skin detecter will be trained "
    //          << "with unnormalised values. This may or may not be what you "
    //          << "expect, and the detector will perform very poorly in all "
    //          << "likelihood." << endl;
    //     samples2 = samples;
    //     break;
    //   }
    // }      
    Mat samples2 = samples; // disable normalisation

    int bestK = totalClusters ? totalClusters : findBestK2(samples2);
    if (bestK == 0) {
      cerr << "WARNING: Apparently, tried to select the best number K for skin "
        "detector training automatically. However, 0 clusters were "
        "chosen. Since this won't work, an arbitrary value K = 7 will be "
        "used. This can be due to bad luck or poor choice of configuration "
        "parameters. In any case, we apologise for this inconvenience." << endl;
      bestK = 7;
    }

    if (bestK < k) {
      cerr << "WARNING: Tried to obtain " << k << " largest clusters out "
           << "of " << bestK << " clusters. Getting as many as can." 
           << endl;
      k = bestK;
    }

    if (debug > 1)
      cerr << "Getting clusters using K = " << bestK << endl;

    // OpenCV 2.2:
    // kmeans(samples2, bestK, labels,
    //        TermCriteria(TermCriteria::MAX_ITER + TermCriteria::EPS, 10, 1.0), 
    //        10, cv::KMEANS_RANDOM_CENTERS, NULL);
    kmeans(samples2, bestK, labels,
           TermCriteria(TermCriteria::MAX_ITER + TermCriteria::EPS, 10, 1.0), 
           10, cv::KMEANS_RANDOM_CENTERS, cv::noArray());


    // label counts
    vector< std::pair<int, int> > labelCounts(bestK);
    for (size_t i = 0; i < labelCounts.size(); ++i)
      labelCounts[i] = std::make_pair(0, i);

    for (int i = 0; i < labels.rows; ++i)
      labelCounts[labels.at<int>(i,0)].first++;

    std::sort(labelCounts.begin(), labelCounts.end());

    // go through the most populated clusters, create corresponding data
    // matrices, and populate them with samples (this approach may be a 
    // little inelegant wrt. processing time)
    clusters = vector<Mat>(k);
    means = Mat(k, samples.cols, samples.type());
    for (int l = 0; l < k; ++l) {
      clusters[l] = Mat(labelCounts[bestK-l-1].first,samples.cols,CV_32FC1);
    
      cv::MatIterator_<float> it = clusters[l].begin<float>();
      for (int i = 0; i < samples.rows; ++i)
        if (labels.at<int>(i,0) == labelCounts[bestK - l - 1].second)
          for (int j = 0; j < samples.cols; ++j)
            *it++ = samples.at<float>(i,j);

      // copy the associated mean-row
      Mat mean = math::meanVector<float>(clusters[l]);
      Mat meanRow = means.row(l);
      mean.copyTo(meanRow);

      //      for (int j = 0; j < meansTemp.cols; ++j) 
      //        means.at<float>(l, j) =
      //          meansTemp.at<float>(labelCounts[bestK-l-1].second, j);
    }
  }


#if 0
  // double SkinDetector::validate(const cv::Mat& input,
  //                               const cv::Mat& validResult,
  //                               double fpCost, double fnCost) {
  double SkinDetector::validate(const cv::Mat&,
                                const cv::Mat&,
                                double, double) {
    assert(false && "This function is currently broken (see process). Come back again later.");
    assert(validResult.type() == CV_8UC3);
    assert(input.type() == CV_8UC3);
    Mat mask;
    process(input, mask);
    assert(mask.size() == validResult.size());
    assert(mask.type() == CV_8UC1);

    double cost = 0;
    MatConstIterator_<Vec3b> vrit = validResult.begin<Vec3b>();
    MatConstIterator_<uchar> mit = mask.begin<uchar>();
    while (mit != mask.end<uchar>()) {
      if (*mit && !toBool(*vrit))
        cost += fpCost;
      else if (!(*mit) && toBool(*vrit))
        cost += fnCost;
      ++mit;
      ++vrit;
    }

    cv::imshow("output", mask);
    cv::waitKey(5);
    return cost;
    return 0;
  }
#endif



  SkinDetector::SkinDetector(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) : Component(blackBoard, frameSource) {
    if (configuration.count("SkinDetector.colourspace")) {
      string s = configuration["SkinDetector.colourspace"].as<string>();
      if (s == "hsv")
        colourSpace.reset(ColourSpace::HSV->clone());
      else if (s == "ycrcb")
        colourSpace.reset(ColourSpace::YCRCB->clone());
      else if (s == "bgr")
        colourSpace.reset(ColourSpace::BGR->clone());
      else if (s == "rgb")
        colourSpace.reset(ColourSpace::RGB->clone());
      else if (s == "xyz")
        colourSpace.reset(ColourSpace::XYZ->clone());
      // else if (s == "cgz")
      //   colourSpace.reset(ColourSpace::CGZ->clone());
      else if (s == "ihls")
        colourSpace.reset(ColourSpace::IHLS->clone());
      else if (s.find("custom:") != string::npos)
        colourSpace.reset(createCustomColourSpace(s).clone());
      else
        throw ConfigurationFileException("'" + s + "' is not a valid colour space!");
    }
    else
      colourSpace.reset(ColourSpace::IHLS->clone());

    if (configuration.count("SkinDetector.postprocess")) {
      auto filters = parseSkinPostprocess(configuration["SkinDetector.postprocess"].as<string>());
      for (auto it = filters.cbegin(); it != filters.cend(); ++it)
        addPostProcessFilter(**it);
    }
    else {
      addPostProcessFilter(PostProcessMorphologicalTransformationFilter(PostProcessMorphologicalTransformationFilter::Type::ERODE, 1));
    addPostProcessFilter(PostProcessWatershedFilter(50));
    addPostProcessFilter(PostProcessMorphologicalTransformationFilter(PostProcessMorphologicalTransformationFilter::Type::CLOSE, 2));
    }
  }

  static boost::program_options::options_description returnSkinDetectorConfigurationFileOptionsDescription() {
    boost::program_options::options_description conffileOpts;
    conffileOpts.add_options()
      ("SkinDetector.colourspace", value<string>()->default_value("ihls"),
       "Possible values are hsv | ycrcb | bgr | rgb | xyz | cgz | ihls | "
       "custom:<string>[,<string>,...]\n"
       "Sets the colour space for skin detection. This is not applicable "
       "for the OpenCV Adaptive Skin Detector.\n"
       "CGZ = Hamilton Chong et al. 2008\n"
       "IHLS = Improved HLS. Description can be found in Allan Hanbury, "
       "Jean Serra. A 3D-polar Coordinate Colour Representation Suitable "
       "for Image Analysis.\n"
       "Specifying custom: and a comma-separated string will allow "
       "arbitrary components. Valid components are r, g, b, h, s, v, y, "
       "cb, cr, x, xyzy (= the Y component from XYZ), z.")
      ("SkinDetector.postprocess", value<string>()->default_value("morph(erode,1);watershed(50);morph(close,2)"),
       "List of post-processing options to apply. These are applied in the "
       "order that they are specified.\n\n"
       "Possible options are:\n\n"
       "- crop(<x>,<y>,<width>,<height>\n"
       "Crops the mask to match the given rectangle\n"
       "- fillHoles()\n"
       "Fills holes in the mask\n"
       "- morph(<type>,<n>)\n"
       "Performs morphological transformations where\n"
       "<type> = open | close | dilate | erode\n"
       "<n> is the number of iterations\n"
       "- watershed(<int>)\n"
       "Dilates the mask, and then applies Watershed segmentation, using "
       "originally detected mask as basis. Marks are set thus that the "
       "dilated region will be zero (i.e. left for the algorithm to fill "
       "in), background is marked region #1. Then, contour finding is "
       "employed and any areas within external contours are recorder with "
       "numbers beginning from 2, and each contour is assigned a unique "
       "identity.\n"
       "- removeSmallComponents(<int>)\n"
       "Removes connected components with less than the specified number of pixels.");
      return conffileOpts;
  }

  static boost::program_options::options_description returnEmptyConfigurationFileOptionsDescription() {
    return boost::program_options::options_description();
  }



  static bool first = true;
  SkinDetector::SkinDetector(bool) : 
    Component(true), 
    conffileOptionsDescriptionFunction(first ? 
                                       &returnSkinDetectorConfigurationFileOptionsDescription : 
                                       &returnEmptyConfigurationFileOptionsDescription) {
    first = false;
  }


  boost::program_options::options_description SkinDetector::getConfigurationFileOptionsDescription() const {
    // this is a hack to make the descriptions unambiguous:
    // only the first class will return these options, others shall not

    return conffileOptionsDescriptionFunction();
  }
}
