#ifndef SLMOTION_TEMPFILE
#define SLMOTION_TEMPFILE

#include <boost/filesystem.hpp>
#include <fstream>

// #include <string>
// #include <memory>

namespace slmotion {
  /**
   * This class represents a temporary file that can be written as a stream. The
   * file is created using standard functions, and erased upon destruction.
   */
  class TempFile {
  public:
    /**
     * Creates a valid temporary file
     */
    TempFile() :
      name(boost::filesystem::unique_path().native()),
      ofs(new std::ofstream(name)) {
    }

    /**
     * The destructor: Erases the file
     */
    ~TempFile() {
      ofs->close();
      if (remove(name.c_str()) != 0)
        perror("Error deleting file");
    }

    TempFile(const TempFile& other) = delete;

    TempFile& operator=(const TempFile& other) = delete;

    TempFile(TempFile&& other) {
      std::swap(this->name, other.name);
      std::swap(this->ofs, other.ofs);
    }

    TempFile& operator=(TempFile&& other) {
      if (this != &other) {
        std::swap(this->name, other.name);
        std::swap(this->ofs, other.ofs);
      }
      return *this;
    }

    /**
     * Returns the name of the file
     */
    inline std::string getName() {
      return name;
    }

    /**
     * Closes the stream
     */
    inline void close() {
      ofs->close();
    }

    /**
     * The outstream function: writes data to the file
     */
    template<typename T>
    TempFile& operator<<(const T& var) {
      if (!ofs->is_open())
        ofs->open (name, std::ofstream::out | std::ofstream::app);
      *ofs << var;
      return *this;
    }

    // for manipulators
    TempFile& operator<<(std::ostream&(*manip)(std::ostream&)) {
      if (!ofs->is_open())
        ofs->open (name, std::ofstream::out | std::ofstream::app);
      manip(*ofs);
      return *this;
    }

  private:
    std::string name;
    std::unique_ptr<std::ofstream> ofs;
  };
}

#endif
