#ifndef SLMOTION_TEMPORAL_SKIN_PROPAGATOR
#define SLMOTION_TEMPORAL_SKIN_PROPAGATOR

#include "Component.hpp"
#include "SkinDetector.hpp"

namespace slmotion {
  class TemporalSkinPropagator : public Component {
  public:
    /**
     * Returns the component name
     */
    inline std::string getComponentName() {
      return "Temporal Skin Propagator";
    }

    /**
     * A very basic constructor.
     *
     * @param blackBoard Black board to associate with
     * @param frameSource Frame source to associate with
     * @param skinDetector Base skin detector
     */
    TemporalSkinPropagator(BlackBoard* blackBoard, 
                           FrameSource* frameSource,
                           SkinDetector* skinDetector) :
      Component(blackBoard, frameSource), skinDetector(skinDetector) {}

    
    /**
     * Default copy constructor
     */
    TemporalSkinPropagator(const TemporalSkinPropagator&) = default;

    /**
     * Default copy-assignment
     */
    TemporalSkinPropagator& operator=(const TemporalSkinPropagator&) = default;


    /**
     * Performs temporal skin propagation
     */
    void process(frame_number_t frameNumber);



    /**
     * Returns a set of property keys which must be found on the blackboard 
     * for each frame that shall be processed by the component (if 
     * available)
     */
    inline virtual property_set_t getRequirements() const {
      return property_set_t { SKINDETECTOR_BLACKBOARD_MASK_ENTRY };
    }



  private:
    /**
     * Performs temporal skin propagation
     *
     * Implementation for the processRange. ProcessRange is guaranteed to 
     * call this function with a valid range with
     *   first in [0,frameSource->size())
     *   last in (first,frameSource->size()]
     *
     * Performs some processing for frames in [first,last), stores results 
     * in the black board.
     *
     * Default action: call process for each frame, separately. May be 
     * overridden in any way.
     *
     * @param first First frame to process (inclusive)
     * @param last Last frame to process (exclusive
     * @param uiCallback Possible user interface callback. The function 
     * should call this function every now and then to notify the parent 
     * program of its status (current frame) if applicable. This may also 
     * be ignored, and the function should be able to handle a NULL pointer.
     * If the callback function returns false, execution should be 
     * terminated prematurely.
     *
     * @return True on successful termination, false otherwise (including 
     * false from callback)
     */
    virtual bool processRangeImplementation(frame_number_t first, 
                                            frame_number_t last, 
                                            UiCallback* uiCallback);



    SkinDetector* skinDetector;
  };
}

#endif
