#ifndef SLMOTION_VIDEOFILESOURCE
#define SLMOTION_VIDEOFILESOURCE

#include <memory>
#include "FrameSource.hpp"
#include "SLIO.hpp"


namespace slmotion {
  /**
   * This is an abstract frontend for accessing footage from video files.
   * Basically, takes in a filename, and then accesses frames.
   *
   * This class takes in a video filename, and then retrieves frames from
   * the file in the given order. An internal frame number is stored, since
   * videos are usually accessed sequentially, and, because of this, the 
   * fastest way to read the video would be through sequential access.
   *
   * However, OpenCV also provides a cumbersome interface to access any 
   * given frame, which will be used whenever the requested frame number 
   * does not match the current internal frame number.
   */
  class VideoFileSource : public FrameSource {
  public:
    /**
     * The default destructor, implemented because of necessity
     */
    virtual ~VideoFileSource() = 0;



    virtual frame_number_type size() const = 0;



    /**
     * Same as above but act as an initialisation function instead
     * (since the former seems to cause trouble with compilation in
     * some cases)
     */
    VideoFileSource(const std::string& filename,
                    size_t cacheSize = SIZE_MAX) :
      FrameSource(cacheSize),
      filename(filename),
      currentFrameNumber(0)
    {
    }



    /**
     * Returns the given frame, if it is in cache. If the requested frame is
     * the next frame to be read, read it in and cache it. Otherwise, wind
     * to the requested frame, while caching any needed frames.
     */
    const cv::Mat& operator[](frame_number_type frameNumber);


    /**
     * Returns information about the file name
     */
    virtual const std::string& getFileName() const {
      return filename;
    }

  private:
    /**
     * Returns a deep copy of the current frame, and advances by one frame
     *
     * @return The frame
     */
    virtual cv::Mat getFrame() = 0;



    /**
     * Resets the source and rewinds the video to the beginning (frame 0)
     */
    virtual void reset(const std::string& filename) = 0;



    std::string filename;
    size_t currentFrameNumber;
#ifdef SLMOTION_THREADING
    mutable std::mutex mutex; ///< If threaded access is used, this object will be used for mutual exclusion
#endif
  };
}
#endif
