#include "ViolaJonesFaceDetector.hpp"
#include "util.hpp"

using cv::Rect;
using std::vector;

namespace slmotion {
  extern int debug;



  const cv::Rect ViolaJonesFaceDetector::INVALID_FACE(~0,~0,0,0);



  std::string getDefaultHaarCascadeFile() {
    // const std::string HAAR_DEFAULT_FACE_CASCADE_FILE = SLMOTION_INSTALL_PREFIX"/share/slmotion/haarcascade_frontalface_alt.xml"; ///< default cascade file
    return getInstallPrefix() + "/share/slmotion/haarcascade_frontalface_alt.xml";
  }



  ViolaJonesFaceDetector::~ViolaJonesFaceDetector() {
  }



  void ViolaJonesFaceDetector::setAndVerifyCascadeFilename(const std::string& filename) {
    cascadeFilename = filename;
    classifier = cv::CascadeClassifier(cascadeFilename);
    if (classifier.empty())
      throw std::invalid_argument("A face detector classifier could not be created using the cascade filename supplied: \"" + filename + "\". Are you sure the file exists?");
  }



  void ViolaJonesFaceDetector::setCascadeFilename(const std::string& filename) {
    setAndVerifyCascadeFilename(filename);
    this->reset();
  }



  std::vector<cv::Rect> ViolaJonesFaceDetector::getRawFaces(frame_number_t frameNumber) {
    return detectFaces(getFrameSource()[frameNumber]);
  }



  cv::Rect ViolaJonesFaceDetector::getRawFace(frame_number_t frameNumber) {
    if (getBlackBoard().has(frameNumber, FACEDETECTOR_RAW_FACE_BLACKBOARD_ENTRY)) 
      return *getBlackBoard().get<cv::Rect>(frameNumber, FACEDETECTOR_RAW_FACE_BLACKBOARD_ENTRY);

    cv::Rect face = detectFace(getFrameSource()[frameNumber]);
    getBlackBoard().set(frameNumber, FACEDETECTOR_RAW_FACE_BLACKBOARD_ENTRY, face);
    return face;
  }




  std::vector<cv::Rect> ViolaJonesFaceDetector::detectFaces(const cv::Mat& frame) {
    cv::Mat gs;
    cvtColor(frame, gs, CV_BGR2GRAY);
    equalizeHist(gs, gs);

    vector<Rect> facelist;
    classifier.detectMultiScale(gs, facelist, scaleFactor, minNeighbours, 
                                CV_HAAR_DO_CANNY_PRUNING, 
                                cv::Size(gs.size().width/16, 
                                         gs.size().height/16));
    return facelist;
  }



  cv::Rect ViolaJonesFaceDetector::detectFace(const cv::Mat& frame) {
    vector<Rect> facelist = detectFaces(frame);
    if (facelist.size() == 1) 
      return facelist[0];
    return INVALID_FACE;
  }



  std::vector<Rect*> ViolaJonesFaceDetector::getDetectionsForRange(frame_number_t first, frame_number_t last) {
    assert(last > first);
    vector<Rect*> vec(last - first, 0);
    for (frame_number_t i = first; i < last; ++i) {
      if (getBlackBoard().has(i, FACEDETECTOR_BLACKBOARD_ENTRY))
        vec[i - first] = &*getBlackBoard().get<Rect>(i, FACEDETECTOR_BLACKBOARD_ENTRY);
      else
        throw std::out_of_range("No face detection results for frame number " + boost::lexical_cast<std::string>(i) + " on the blackboard. Where's me washboard?");
    }
    return vec;
  }
}
