#ifndef SLMOTION_VIOLA_JONES_FACE_DETECTOR
#define SLMOTION_VIOLA_JONES_FACE_DETECTOR

#include "Component.hpp"



namespace slmotion {
  std::string getDefaultHaarCascadeFile();
  const std::string FACEDETECTOR_BLACKBOARD_ENTRY = "facedetection"; // cv::Rect
  const std::string FACEDETECTOR_RAW_FACE_BLACKBOARD_ENTRY = "rawface"; // cv::Rect



  /**
   * A Viola-Jones-style face detector. This is an abstract base class that provides auxiliary
   *  functions that can be used for creating actual face detector components.
   */
  class ViolaJonesFaceDetector : public Component {
  public:
    static const cv::Rect INVALID_FACE; ///< A special value that can be used to signal that a certain face is definitely invalid



    virtual ~ViolaJonesFaceDetector() = 0;



    /**
     * Constructs a new face detector object
     *
     * @param faceDetectorOpts Options struct as defined above
     * @param blackBoard A pointer to a black board object. The board must 
     * outlive the face detector.
     * @param frameSource A pointer to a frame source object. The source 
     * must outlive the face detector.
     */
    explicit ViolaJonesFaceDetector(BlackBoard* blackBoard, FrameSource* frameSource,
                                    const std::string& cascadeFilenameArgument = getDefaultHaarCascadeFile())
      : 
      Component(blackBoard, frameSource),
      scaleFactor(1.2),
      minNeighbours(2)
    {
      setAndVerifyCascadeFilename(cascadeFilenameArgument); // no need to reset
    }



    virtual void process(frame_number_t framenumber) = 0;



    /**
     * Attempts to find a face in frame. If multiple face candidates are 
     * detected, the detection is considered invalid.
     *
     * Begins by checking the cache, and if an entry is found at the given 
     * frame number index, returns the corresponding raw detection result 
     * from cache. Otherwise, calls detectFace.
     *
     * @param frameNumber Current frame number
     *
     * @return The detection, or INVALID_FACE
     */
    cv::Rect getRawFace(frame_number_t frameNumber);



    /**
     * Just like getRawFace, but instead of discarding frames with multiple faces, all faces
     * are returned.
     */
    std::vector<cv::Rect> getRawFaces(frame_number_t frameNumber);



    /**
     * Sets the cascade filename in use, and resets the object.
     */
    void setCascadeFilename(const std::string& filename); 



    inline void setMinNeighbours(int n) {
      minNeighbours = n;
    }



    inline void setScaleFactor(double d) {
      scaleFactor = d;
    }



    virtual property_set_t getProvided() const = 0;
    virtual std::string getShortDescription() const = 0;
    virtual std::string getLongDescription() const = 0;
    virtual property_set_t getRequirements() const = 0;
    virtual std::string getShortName() const = 0;
    virtual std::string getComponentName() const = 0;
    virtual void reset() = 0;
    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const = 0;
    virtual boost::program_options::options_description getCommandLineOptionsDescription() const = 0;



    ViolaJonesFaceDetector(bool) : Component(true) {}



  protected:
    /**
     * Returns a vector of pointers for the range of detection results
     */
    std::vector<cv::Rect*> getDetectionsForRange(frame_number_t first,
                                                 frame_number_t last);



  private:
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const = 0;



    /**
     * Sets the cascade filename in use, resets the cascade object, and checks 
     * that it was opened successfully, or throws an exception.
     */
    void setAndVerifyCascadeFilename(const std::string& filename);
    


    /**
     * Performs a raw face detection and stores the result in the
     * cache, and returns the result. No interpolation is applied, and no 
     * other frames are considered; these are left for other functions. If 
     * multiple face candidates are detected, an INVALID_FACE is returned.
     *
     * @param img Input image. Assumed to be in BGR.
     *
     * @return The raw detection result, or INVALID_FACE in case of failure.
     */
    cv::Rect detectFace(const cv::Mat& img);



    /**
     * Just like above but all results are returned
     */
    std::vector<cv::Rect> detectFaces(const cv::Mat& img);



    std::pair<double,double> scale; ///< first = scaling wrt x axis (1.0 == no scaling), second = scaling wrt y axis
    std::pair<int,int> translation; ///< first = translation wrt x axis, second = translation wrt y axis
    double maxMove; ///< The maximum number of pixels that the detected face can move between frames 
    double maxAreaChange; ///< How great an amount the area of the detected face can change between frames 
    std::string cascadeFilename; ///< Haar cascade filename
    cv::CascadeClassifier classifier; ///< The Haar classifier
    double scaleFactor; ///< OpenCV Haar classifier scaleFactor, i.e. how big a jump there is between each successive scale
    int minNeighbours; ///< OpenCV parameter; Minimum number of overlapping neighbours required for a successful detection
  };
}
#endif
