#include "VisualSilenceDetector.hpp"

namespace slmotion {
  static VisualSilenceDetector DUMMY(true);

  static double getIfSilent(const cv::Mat& prev, const cv::Mat& cur, 
                          const cv::Mat& next, double threshold) {
    cv::Mat diff1 = cv::abs(cur-prev);
    cv::Mat diff2 = cv::abs(next-cur);
    cv::Mat diff = cv::max(diff1,diff2);
    cv::Mat diff_t(diff.size(), diff.type());
    cv::threshold(diff, diff_t, threshold, 255, cv::THRESH_BINARY);
    cv::erode(diff_t, diff, cv::Mat());
    
    return cv::countNonZero(diff)/((double)diff.total());
  }

  bool VisualSilenceDetector::processRangeImplementation(frame_number_t first,
                                                         frame_number_t last,
                                                         UiCallback* uiCallback) {
    cv::Mat cur, prev, next;
    for (frame_number_t frnumber = first; frnumber < last; ++frnumber) {
      // this looks funny but the reason is to force the frame source
      // to be reqd sequentially
      // otherwise, things get slow
      if (frnumber == first) {
        if (frnumber == 0 && last > 1) {
          cv::cvtColor(getFrameSource()[frnumber], cur, CV_BGR2GRAY);
          cv::cvtColor(getFrameSource()[frnumber + 1], next, CV_BGR2GRAY);
          prev = next;
        }
        else if (frnumber > 0 && first + 1 == last) {
          cv::cvtColor(getFrameSource()[frnumber - 1], prev, CV_BGR2GRAY);
          cv::cvtColor(getFrameSource()[frnumber], cur, CV_BGR2GRAY);
          next = prev;
        }
        else if (first > 0) {
          cv::cvtColor(getFrameSource()[frnumber - 1], prev, CV_BGR2GRAY);
          cv::cvtColor(getFrameSource()[frnumber], cur, CV_BGR2GRAY);
          cv::cvtColor(getFrameSource()[frnumber + 1], next, CV_BGR2GRAY);
        }
      }
      else if (frnumber + 1 < getFrameSource().size()) {
        prev = cur;
        cur = next;
        next = cv::Mat(next.size(), next.type());
        cv::cvtColor(getFrameSource()[frnumber + 1], next, CV_BGR2GRAY);
      }
      else if (frnumber + 1  == getFrameSource().size()) 
        prev = next;
      

      getBlackBoard().set(frnumber, VISUAL_SILENCE_BLACKBOARD_ENTRY, 
                          getIfSilent(prev, cur, next, threshold));

      if (uiCallback != nullptr && 
          (*uiCallback)((frnumber+1)*100./(last-first)) == false)
        return false;
    }

    return true;
  }

  void VisualSilenceDetector::process(frame_number_t) {
    assert (false && "single frame processing not supported");
  }

  Component* VisualSilenceDetector::createComponentImpl(const boost::program_options::variables_map& vm, 
                                                        BlackBoard* blackBoard, FrameSource* frameSource) const {
    VisualSilenceDetector vsd(blackBoard, frameSource);
    if (vm.count("VisualSilenceDetector.threshold"))
      vsd.threshold = vm["VisualSilenceDetector.threshold"].as<double>();
    // if (vm.count("VisualSilenceDetector.mincoverage"))
    //   vsd.mincoverage = vm["VisualSilenceDetector.mincoverage"].as<double>();

    return new VisualSilenceDetector(vsd);
  }

  boost::program_options::options_description VisualSilenceDetector::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()
      ("VisualSilenceDetector.threshold", boost::program_options::value<double>()->default_value(10),
       "Threshold used for silencing noise in the difference images");
      // ("VisualSilenceDetector.mincoverage", boost::program_options::value<double>()->default_value(0.1),
      //  "Minimum coverage, ie. the minimal portion of changed pixels. Expected to be in the range [0,1.0].");
    return opts;
  }
}
