#ifndef SLMOTION_XML_NODE
#define SLMOTION_XML_NODE

#include "XmlException.hpp"
#include "XmlNamespace.hpp"
#include "xmlutil.hpp"
#include <libxml/parser.h>
#include <cassert>

namespace slmotion {
  namespace xml {
    // forward declaration
    class XmlDocument;
    class XmlSiblingIterator;

    /**
     * This class represents a node in the XML tree (providing an nicer 
     * interface to libxml2 functions)
     */
    class XmlNode {
    public:
      friend class XmlDocument;
      friend class XmlSiblingIterator;
      friend void dumpElement(FILE* stream, const XmlDocument& doc, 
                              const XmlNode& element);

      /**
       * Creates an empty NULL node
       */
      XmlNode() : internalNode(new xmlNodePtr),
                  shouldDelete(new bool(true)),
                  count (new int(1)) {
        *internalNode = NULL;
      }


      /**
       * Creates an element node with the designated name
       */
      XmlNode(const std::string& name) :
        internalNode(new xmlNodePtr),
        shouldDelete(new bool(true)),
        count (new int(1))
      {
        *internalNode = xmlNewNode(NULL, toXmlChar(name.c_str()));
      }



      /**
       * The destructor. Releases the node if needed (i.e. not assigned to a
       * tree or anything)
       */
      ~XmlNode() {
        // debug code
        assert(shouldDelete.use_count() == count.use_count());
        assert(count.use_count() == *count);

        --*count;
        if (*count == 0 && *shouldDelete) {
          xmlUnlinkNode(*internalNode);
          xmlFreeNode(*internalNode);
        }
      }


      
      /**
       * Unlinks the node from its context (removes it from the tree), and allows it to be 
       * deleted, along with any of its children.
       */
      void unlink() {
        xmlUnlinkNode(*internalNode);
        *shouldDelete = true;
      }



      /**
       * A nice copy constructor. Increments the internal counter of objects
       * (needed for releasing resources)
       */
      XmlNode(const XmlNode& other) :
        internalNode(other.internalNode),
        shouldDelete(other.shouldDelete),
        count(other.count)
      {
        ++*count;
      }



      /**
       * The nice copy assignment. Decrements current counter and, if 
       * needed, releases the resources before performing regular copy.
       */
      XmlNode& operator=(const XmlNode& other) {
        if (&other != this && other.internalNode != this->internalNode) {
          --*count;
          if (*count == 0 && *shouldDelete) {
            xmlUnlinkNode(*internalNode);
            xmlFreeNode(*internalNode);
          }

          internalNode = other.internalNode;
          shouldDelete = other.shouldDelete;
          count = other.count;
          ++*count;
        }
        return *this;
      }



      /**
       * Creates a new namespace
       *
       * @param href The associated URI
       * @param prefix The desired prefix
       * 
       * @return A new namespace object
       */
      XmlNamespace newNamespace(const std::string& href,
                                const std::string& prefix) {
        return XmlNamespace(*internalNode, href, prefix);
      }



      /**
       * Returns the name of the element
       */
      std::string getName() const {
        return toString((*internalNode)->name);
      }



      /**
       * Creates a new text child with the given name and content.
       */
      void newTextChild(const std::string& name, 
                        const std::string& content) {
        xmlNewTextChild(*internalNode, NULL, toXmlChar(name.c_str()),
                        toXmlChar(content.c_str()));
      }



      /**
       * Adds a child
       */
      void addChild(const XmlNode& node) {
        xmlAddChild(*internalNode, *node.internalNode);
        *node.shouldDelete = false;
      }



      /**
       * Adds the node as the next sibling
       */
      void addNextSibling(const XmlNode& node) {
        xmlAddNextSibling(*internalNode, *node.internalNode);
        *node.shouldDelete = false;
      }


      /**
       * Returns the child by name
       */
      XmlNode getChild(const std::string& name) const;



      /**
       * Sets XML text content
       */
      void setContent(const std::string& content) {
        xmlNodeSetContent(*internalNode, toXmlChar(content.c_str()));
      }



      /**
       * Get the XML content
       */
      std::string getContent() const;



      // /**
      //  * FOR DEBUGGERY ONLY
      //  */
      // xmlNodePtr getInternalNode() {
      //   return *internalNode;
      // }

      // /**
      //  * FOR DEBUGGERY ONLY
      //  */
      // xmlNodePtr getInternalNode() const {
      //   return *internalNode;
      // }

      /**
       * Adds a new property (key-value pair) to the node, i.e.
       * <element key="value" />
       */
      void newProperty(const std::string& key, const std::string& value) {
        xmlNewProp(*internalNode, toXmlChar(key), toXmlChar(value));
      }



      /**
       * Returns the given property as string
       */
      std::string getProperty(const std::string& key) const {
        return getXmlProperty(*internalNode, key.c_str());
      }



      /**
       * Returns the line number (a positive integer), or -1 on failure
       */
      long getLineNumber() const {
        return xmlGetLineNo(*internalNode);
      }



      /**
       * Returns a sibling iterator to the first child of the node
       */
      XmlSiblingIterator begin() const;



      /**
       * Returns an iterator that is just past the last immediate child of
       * the node
       */
      XmlSiblingIterator end() const;



      /**
       * Checks if the node has any children
       */
      bool hasChildren() const {
        return xmlFirstElementChild(*internalNode) != NULL;
      }



      /**
       * Returns a boolean specifying if the node has a child with the given name
       */
      bool hasChild(const std::string& name) const;



      /**
       * Adds a new sibling to the node that takes place immediately 
       * preceeding it
       */
      void addPreviousSibling(const XmlNode& node) {
        xmlAddPrevSibling(*internalNode, *node.internalNode);
        *node.shouldDelete = false;
      }



      /**
       * Returns the first child if there is one, or throws an exception
       */
      XmlNode getFirstChild() const; 



      /**
       * Returns the last child with the given name
       */
      XmlNode getLastChild(const std::string& name) const;
     


      /**
       * Returns a Sibling Iterator, pointing at the node itself (somewhere in range between the
       * begin() and the end() of the parent node)
       */
      XmlSiblingIterator getSiblingIterator() const;



    private:
      /**
       * For constructing temporary nodes, e.g. children and such
       */
      explicit XmlNode(xmlNodePtr nodePtr) :
        internalNode(new xmlNodePtr),
        shouldDelete(new bool),
        count(new int) {
        *internalNode = nodePtr;
        *shouldDelete = false;
        *count = 1;
      }

      std::shared_ptr<xmlNodePtr> internalNode; ///< libxml2 node
      std::shared_ptr<bool> shouldDelete; ///< Whether or not to delete the pointer upon destruction
      std::shared_ptr<int> count; ///< copy count, resources are released when this reaches 0
    };
  }
}

#include "XmlSiblingIterator.hpp"

#endif
