#!/bin/sh

# this is a script that will compile slmotion and all of its dependencies for mac

if [ `id -u` -eq 0 ]; then
    echo Do not run this script as root.
    exit 1
fi

DO_INSTALL=true
DO_GUI=true

for arg in "$@"
do
    if [ $arg = --help ] || [ $arg = -h ]; then
             echo "$0: a script to compile slmotion on MacOS X"
             echo "Optional arguments:"
             echo "  --help, -h:          Print this text."
             echo "  --for-mountain-lion: Compile with MacOS X 10.8 SDK"
             echo "  --no-install:        Don't install the CLI version"
             echo "  --no-gui:            Disable GUI"
             echo "  --clean:             Clean the current directory ie. "
             echo "                       deletes the build/ and all "
             echo "                       dependencies"
#             echo   --all:   Build all dependencies. By default, OpenCV and FFMPEG are not built.
#             echo   --clean: Should restore the directories to their natural state.
             exit
    elif [ $arg = --for-mountain-lion ]; then
        export MACOSX_DEPLOYMENT_TARGET=10.8
        export CMAKE_OSX_SYSROOT=/Developer/SDKs/MacOSX10.8.sdk
    elif [ $arg = --no-install ]; then
        unset DO_INSTALL
    elif [ $arg = --no-gui ]; then
        unset DO_GUI
    elif [ $arg = --clean ]; then
        echo Deleting build/ and all dependencies!
        echo If you do not want to proceed, hit Ctrl-C *NOW*
        echo Process will begin in \\c
        for i in `seq 5 1`
        do
            echo $i... \\c
            sleep 1s
        done
        echo
        rm -rfv build opencv opencv-2.4.8 opencv-2.4.8.zip \
            boost_1_55_0.tar.bz2 boost boost_1_55_0 yasm-1.2.0 \
            yasm-1.2.0.tar.gz ffmpeg-git ffmpeg
        echo Done.
        exit
    else
        echo $0: \"$arg\" is an unrecognised argument!
        exit
    fi
done

INSTALL_PREFIX=$HOME/slmotion
# ensure that dependencies get built with the right standard library 
export CXXFLAGS="-stdlib=libc++"
export LDFLAGS="-stdlib=libc++"

echo This script will attempt to build and install slmotion to $INSTALL_PREFIX
echo If you do not want to proceed, hit Ctrl-C *NOW*
echo Process will begin in \\c
for i in `seq 5 1`
do
    echo $i... \\c
    sleep 1s
done
echo

echo Checking operating system version... \\c
OSX_VERSION=$(system_profiler SPSoftwareDataType|grep "System Version"|sed 's/.*OS X \(10\.[0-9]\.[0-9]\).*/\1/')
echo $OSX_VERSION... \\c
if [ $(echo $OSX_VERSION | cut -d '.' -f 2) -lt 8 ]; then
    echo FAIL!
    echo "OS X 10.8 (Mountain Lion) or newer is required!"
    exit 1
else
    echo OK
fi

echo Checking for the existence of XCode... \\c
if [ ! -e /usr/bin/xcodebuild ]; then
    echo FAIL!
    echo Please install XCode from the App Store.
    exit 1
else
    echo OK
fi

echo Checking XCode version... \\c
XCODEVERSION=$(xcodebuild -version | head -n 1 | cut -d ' ' -f 2)
echo $XCODEVERSION... \\c

if [ $(echo $XCODEVERSION | cut -f 1 -d '.') -lt 5 ]; then
    echo FAIL!
    echo XCode 5 or newer is required.
else
    echo OK
fi

echo Checking for Qt... \\c
if [ $DO_GUI ]; then
    if ! command -v qmake --version > /dev/null; then
        echo Fail!
        echo Please go to http://qt-project.org/downloads and install Qt 4.8.
        exit 1
    fi
    echo Found version \\c
    QT_VER=$(qmake --version|tail -n 1|cut -d ' ' -f 4)
    echo $QT_VER... \\c
    QT_VER_MAJOR=$(echo $QT_VER | cut -d '.' -f 1)
    QT_VER_MINOR=$(echo $QT_VER | cut -d '.' -f 2)
    if [ $QT_VER_MAJOR == 4 ] && [ $QT_VER_MINOR == 8 ]; then
        echo OK
    else
        echo Fail!
        echo Qt 4.8 is required.
        exit 1
    fi
fi

echo Checking for CMake... \\c

if ! command -v cmake > /dev/null; then
    echo Fail!
    echo Please go to http://www.cmake.org/cmake/resources/software.html and install CMake.
    exit 1
fi
CMAKE_VERSION=$(cmake --version | cut -d ' ' -f 3)
echo $CMAKE_VERSION... \\c
if [ $(echo $CMAKE_VERSION | cut -d '.' -f 1) -lt 2 ]; then
    echo Error: CMake 2.8 or newer is required
    exit 1
elif [ $(echo $CMAKE_VERSION | cut -d '.' -f 2) -lt 8 ]; then
    echo Error: CMake 2.8 or newer is required
    exit 1    
else
    echo OK
fi

if [ ! -d build ]; then
    mkdir build
fi

YASM_DIR=`pwd`/yasm
YASM_SRC_DIR=`pwd`/yasm-1.2.0
YASM_ARCHIVE=`pwd`/yasm-1.2.0.tar.gz
YASM_SRC_URL=http://www.tortall.net/projects/yasm/releases/yasm-1.2.0.tar.gz
YASM_ARCHIVE_MD5SUM=4cfc0686cf5350dd1305c4d905eb55a6
echo Checking for yasm/... \\c
if [ ! -d yasm/ ]
then
    echo Not found!
    Checking for $YASM_SRC_DIR... \\c
    if [ ! -d $YASM_SRC_DIR ]
    then
        echo Not found!
        echo Checking for $YASM_ARCHIVE... \\c
        if [ ! -e $YASM_ARCHIVE ]
            then
            echo Not found!
            echo "Downloading $YASM_SRC_URL -> $YASM_ARCHIVE"
            curl -L $YASM_SRC_URL -o $YASM_ARCHIVE 
        else
            echo OK
        fi
        echo Verifying checksum... \\c
        if [ `md5 -q $YASM_ARCHIVE` != $YASM_ARCHIVE_MD5SUM ]
        then
            echo Checksum mismatch!
            exit 1
        else
            echo OK
        fi
        echo Extracting... \\c
        tar xzf $YASM_ARCHIVE
        echo OK
    else
        echo OK
    fi
    echo Entering $YASM_SRC_DIR/...
    cd $YASM_SRC_DIR
    ./configure --prefix=$YASM_DIR
    make
    make install
    echo Leaving $YASM_SRC_DIR/...
    cd ..
else
    echo OK
fi

export PATH=$PATH:/$YASM_DIR/bin

echo Checking for ffmpeg/... \\c
FFMPEG_DIR=`pwd`/ffmpeg
if [ ! -d ffmpeg ]; then
    echo Not found!
    echo Checking for ffmpeg-git/... \\c
    if [ ! -d ffmpeg-git ]; then
        echo Not found!
        echo Cloning repository...
        git clone git://source.ffmpeg.org/ffmpeg.git ffmpeg-git
        echo Done.
    fi
    
    echo Entering ffmpeg-git/
    cd ffmpeg-git
    echo Updating...
    git pull
    
    echo Configuring...
    ./configure --prefix=$FFMPEG_DIR --enable-shared --enable-pic --disable-iconv
    
    echo Running make...
    make
    
    echo Installing...
    make install
    
    echo Leaving ffmpeg-git/
    cd ..
    
else
    echo OK
fi

# these are necessary for cmake to locate our custom ffmpeg
export DYLD_LIBRARY_PATH=$DYLD_LIBRARY_PATH:$FFMPEG_DIR/lib/
export PKG_CONFIG_PATH=$PKG_CONFIG_PATH:$FFMPEG_DIR/lib/pkgconfig
export PKG_CONFIG_LIBDIR=$PKG_CONFIG_LIBDIR:$FFMPEG_DIR/lib

OPENCV_DIR=`pwd`/opencv
echo Checking for opencv/... \\c
if [ ! -d opencv/ ]; then
    echo Not found!
    OPENCV_VERSION=2.4.8
    OPENCV_SRC_DIR=opencv-$OPENCV_VERSION
    OPENCV_ARCHIVE=opencv-$OPENCV_VERSION.zip
    OPENCV_ARCHIVE_URL=http://sourceforge.net/projects/opencvlibrary/files/opencv-unix/2.4.8/opencv-2.4.8.zip/download
    OPENCV_ARCHIVE_MD5SUM=50cc1433b3654074206f5b3dbfd49848
    
    echo Checking for $OPENCV_SRC_DIR... \\c
    if [ ! -d $OPENCV_SRC_DIR ]; then
        echo Not found!
        echo Checkig for $OPENCV_ARCHIVE... \\c
        if [ ! -e $OPENCV_ARCHIVE ]; then
            echo Not found!
            echo Downloading...
            curl -L $OPENCV_ARCHIVE_URL -o $OPENCV_ARCHIVE
        else
            echo OK
        fi
        
        echo Verifying checksum... \\c
        if [ `md5 -q $OPENCV_ARCHIVE` != $OPENCV_ARCHIVE_MD5SUM ]; then
            echo Checksum mismatch!
            exit 1
        else
            echo OK
        fi
        
        echo Extracting... \\c
        tar xzf $OPENCV_ARCHIVE
        echo OK
    else
        echo OK
    fi
    echo Creating opencv/
    mkdir opencv
    echo Entering $OPENCV_SRC_DIR
    cd $OPENCV_SRC_DIR
    echo Creating $OPENCV_SRC_DIR/build
    mkdir build/
    echo Entering $OPENCV_SRC_DIR/build
    cd build/
    
        # echo Checking Apple GCC... \\c
        # if [ ! `/usr/bin/g++ --version|grep 'i686-apple-darwin11-llvm-g++-4.2'|wc -l` -eq 1 ];
        #     then
        #     echo Not found!
        #     echo OpenCV must be compiled with Apple GCC because of some Objective-C dependencies
        #     exit 1
        # else
        #     echo OK
        # fi
        
    echo Running CMake...
    cmake -G "Unix Makefiles" -D CMAKE_BUILD_TYPE=RELEASE \
        -D CMAKE_INSTALL_PREFIX=$OPENCV_DIR \
        -D BUILD_PYTHON_SUPPORT=ON -D WITH_FFMPEG=ON \
        -D FFMPEG_INCLUDE_DIR=$FFMPEG_DIR/include  ..

    echo Running make...
    make
        
    echo Installing OpenCV...
    make install
        
    echo Leaving $OPENCV_ARCHIVE/build
    cd ..
    echo Leaving $OPENCV_ARCHIVE/
    cd ..
else
    echo OK
fi

BOOST_DIR=`pwd`/boost
echo Checking for boost/... \\c
if [ ! -d boost/ ]; then
    echo Not found!
    BOOST_SRC_DIR=boost_1_55_0
    BOOST_ARCHIVE=boost_1_55_0.tar.bz2
    BOOST_ARCHIVE_URL=https://sourceforge.net/projects/boost/files/boost/1.55.0/boost_1_55_0.tar.bz2/download
    BOOST_ARCHIVE_MD5SUM=d6eef4b4cacb2183f2bf265a5a03a354
    echo Checking $BOOST_SRC_DIR... \\c
    if [ ! -d $BOOST_SRC_DIR ]; then
        echo Not found!
        echo Checking for $BOOST_ARCHIVE... \\c
        if [ ! -e $BOOST_ARCHIVE ]; then
            echo Not found!
            echo Downloading boost_1_54_0.tar.bz2...
            curl -L $BOOST_ARCHIVE_URL -o $BOOST_ARCHIVE
        fi
        echo Verifying checksum... \\c
        if [ `md5 -q $BOOST_ARCHIVE` != $BOOST_ARCHIVE_MD5SUM ]; then
            echo ERROR!
            echo Checksum mismatch.
            exit 1
        fi
        echo OK
        echo Extracting... \\c
        tar xjf $BOOST_ARCHIVE
        echo OK
    else
        echo OK
    fi
    echo Creating boost/
    mkdir boost
    echo Entering $BOOST_SRC_DIR
    cd $BOOST_SRC_DIR
    echo Bootstrapping Boost...
#     # ensure correct GCC gets selected
#     OLDPATH=$PATH
#     export PATH=`dirname $CXX`:$PATH
#    echo "using darwin : : $CXX ;" >> tools/build/v2/user-config.jam
    ./bootstrap.sh --prefix="$BOOST_DIR"
    echo Building Boost...
    ./b2 --prefix="$BOOST_DIR" toolset=clang \
        cxxflags="-stdlib=libc++ -std=c++11" \
        linkflags="-stdlib=libc++" install
#     export PATH=$OLDPATH
    echo Leaving boost_1_54_0/
    cd ..
else
    echo OK
fi

LIBFLANDMARK_DIR=`pwd`/libflandmark
echo Checking for libflandmark/... \\c
if [ ! -d libflandmark ]; then
    echo Not found!
    echo Checking for flandmark_v107/... \\c
    if [ ! -d flandmark-master ]; then
        echo Not found!
        echo Checking for flandmark_v107.zip... \\c
        if [ ! -e flandmark_v107.zip ]; then
            echo Not found!
            echo Downloading...
            curl -L "http://cmp.felk.cvut.cz/~uricamic/flandmark/ccount/click.php?id=7" \
                -o flandmark_v107.zip
        else 
            echo OK
        fi
        echo Verifying checksum... \\c
        # old checksum 95acd7d73100d3460e6727ce3ecfe17d
        if [ `md5 -q flandmark_v107.zip` != b2c97f50ac8c532ea356e285967e63f1 ]; then
            echo ERROR!
            echo Checksum mismatch!
            exit 1
        else
            echo OK
        fi
        echo Extracting... \\c
        unzip flandmark_v107.zip > /dev/null
        echo OK
    else
        echo OK
    fi
    echo Creating libflandmark/
    mkdir libflandmark
    mkdir libflandmark/lib
    echo Entering flandmark-master/
    cd flandmark-master
    echo Patching libflandmark/CMakeLists.txt... \\c
    head -n 2 libflandmark/CMakeLists.txt > libflandmark/CMakeLists.txt.new
    echo "link_directories($OPENCV_DIR/lib)" >> libflandmark/CMakeLists.txt.new
    tail -n +2 libflandmark/CMakeLists.txt >> libflandmark/CMakeLists.txt.new
    mv libflandmark/CMakeLists.txt.new libflandmark/CMakeLists.txt
    echo Creating bin/
    mkdir bin
    echo Entering bin/
    cd bin
    echo Running CMake...
    cmake -D CMAKE_PREFIX_PATH=$OPENCV_DIR/share/OpenCV \
        -D CMAKE_INSTALL_PREFIX=$LIBFLANDMARK_DIR -D BUILD_MATLAB_BINDINGS=OFF \
        -D BUILD_CPP_EXAMPLES=OFF ..
    make
    echo Copying library files... \\c
    cp libflandmark/libflandmark_shared.dylib ../../libflandmark/lib
    cp libflandmark/libflandmark_shared.dylib ../../libflandmark/lib/libflandmark.dylib
    cp libflandmark/libflandmark_shared.dylib ../../libflandmark/
    cp libflandmark/libflandmark_shared.dylib ../../libflandmark/libflandmark.dylib
    cp ../libflandmark/*.h ../../libflandmark/
    cp ../data/flandmark_model.dat ../../libflandmark/
    echo OK
    cd ..
    cd ..
else
    echo OK
fi

echo Entering build/
cd build

echo Running CMake...
# export CMAKE_PREFIX_PATH=$OPENCV_DIR/share/OpenCV:$WXWIDGETS_DIR 
# export LDFLAGS="-stdlib=libc++"

if [ $DO_GUI ]; then
    GUIARG="-D ENABLE_QT=ON -D MAKE_SLMOTION_GUI=ON"
else
    GUIARG="-D MAKE_SLMOTION_GUI=OFF"
fi

cmake -G Xcode -D OpenCV_DIR=$OPENCV_DIR/share/OpenCV \
    -D USE_CUSTOM_BOOST=ON -D CUSTOM_BOOST_DIR=$BOOST_DIR -D ENABLE_PICSOM=OFF \
    -D TEST_DATA_DIRECTORY=/Users/masa/slmotion_tests \
    -D CMAKE_INSTALL_PREFIX=$INSTALL_PREFIX -D VERBOSE_MAKEFILE=ON \
    -D ENABLE_LIBFLANDMARK=ON -D LIBFLANDMARK_DIR=$LIBFLANDMARK_DIR \
    $GUIARG ..


echo Building with XCode / CLang...

xcodebuild
#xcodebuild -IDEBuildOperationMaxNumberOfConcurrentCompileTasks=1

if [ $DO_INSTALL ]
then
    echo Installing...
    xcodebuild -target install
fi

#     -D LIBUSB_DIR=$LIBUSB_DIR \
#     -D OPENNI_DIR=$OPENNI_SRC_DIR \
#     -D OPENNI_INSTALL_DIR=$OPENNI_DIR -D OPENNI_MODULES_DIR=. -D ENABLE_OPENNI=ON \
#     -D ENABLE_LIBFLANDMARK=ON -D LIBFLANDMARK_DIR=$LIBFLANDMARK_DIR \
#     -D TEST_DATA_DIRECTORY=/Users/masa/slmotion_tests ..


# # parse arguments
# BUILD_ALL=false
# while true; do
#     if [ $1 ]; then
#         if [ $1 = --help ]; then
#             echo $0: a script to compile slmotion on MacOS X
#             echo Optional arguments:
#             echo   --help:  Print this text.
#             echo   --all:   Build all dependencies. By default, OpenCV and FFMPEG are not built.
#             echo   --clean: Should restore the directories to their natural state.
#             exit
#         elif [ $1 = --all ]; then
#             BUILD_ALL=true
#         elif [ $1 = --clean ]; then
#             echo Cleaning... \\c
#             for dir in build opencv opencv-2.4.5 libusb libusb-1.0.9 ffmpeg \
#                 ffmpeg-1.2 boost boost_1_54_0 openni OpenNI-Bin-Dev-MacOSX-v1.5.4.0; do
#                 if [ -d $dir ]; then
#                     rm -r $dir
#                 fi
#             done
#             for file in opencv-2.4.5.tar.gz libusb-1.0.9.tar.bz2 \
#                 ffmpeg-1.2.tar.bz2 boost_1_54_0.tar.bz2 \
#                 OpenNI-Bin-Dev-MacOSX-v1.5.4.0.tar.zip; do
#                 if [ -e $file ]; then
#                     rm $file
#                 fi
#             done

#             echo Done.
#             exit
#         else
#             echo $0: $1: invalid command-line argument!
#             exit 1
#         fi
#         shift
#     else
#         break
#     fi
# done

# # By default, use CXX. If undefined, use g++

# if [ -z "$CXX" ]; then
#     CXX=g++
# fi

# echo Checking G++ version... \\c
# GCC_VERSION_STRING=`$CXX --version|grep '[0-9]\.[0-9]*\.[0-9]*'|sed 's/.* \([0-9]\.[0-9]*\.[0-9]\).*/\1/'`
# echo $GCC_VERSION_STRING... \\c
# GCC_MAJOR_VERSION=`echo $GCC_VERSION_STRING|cut -d '.' -f 1`
# GCC_MINOR_VERSION=`echo $GCC_VERSION_STRING|cut -d '.' -f 2`
# if [ $GCC_MAJOR_VERSION = 4 -a $GCC_MINOR_VERSION -ge 6 -o $GCC_MAJOR_VERSION -gt 4 ]; then
#     echo OK
# else
#     echo FAIL!
#     echo GCC 4.6 or greater is required!
#     exit 1
# fi

# echo Checking for CMake... \\c
# if [ `cmake --version|wc -l` -ge 1 ]; then
#     echo OK
# else
#     echo FAIL!
#     exit 1
# fi

# # OpenNI requires this

# OLD_CC=$CC
# export CC=/usr/bin/gcc
# LIBUSB_DIR=`pwd`/libusb
# echo Checking for libusb/... \\c
# if [ ! -d libusb ]; then
#     echo Not found!
#     echo Checking for libusb-1.0.9.tar.bz2... \\c
#     if [ ! -e libusb-1.0.9.tar.bz2 ]; then
#         echo Not found!
#         echo Downloading libusb-1.0.9.tar.bz2...
#         wget http://sourceforge.net/projects/libusb/files/libusb-1.0/libusb-1.0.9/libusb-1.0.9.tar.bz2
#     fi
#     echo Verifying checksum... \\c
#     if [ `md5 -q libusb-1.0.9.tar.bz2` != 7f5a02375ad960d4e33a6dae7d63cfcb ]; then
#         echo ERROR!
#         echo Checksum mismatch!
#         exit 1
#     else
#         echo OK
#     fi
#     echo Extracting... \\c
#     tar xjf libusb-1.0.9.tar.bz2
#     echo OK
#     echo Creating libusb/
#     mkdir libusb
#     echo Entering libusb-1.0.9/...
#     cd libusb-1.0.9
#     echo Running configure...
#     ./configure --prefix=$LIBUSB_DIR
#     echo Running make...
#     make
#     echo Installing...
#     make install
#     echo Leaving libusb-1.0.9/...
#     cd ..
# else
#     echo OK
# fi

# export CC=$OLD_CC

# # this is for future openni2 support

# # OPENNI_DIR=`pwd`/openni
# # echo Checking for OpenNI/... \\c
# # if [ ! -d openni/ ]; then
# #     echo Not found!
# #     echo Checking for OpenNI-MacOSX-x64-2.2.0/... \\c
# #     if [ ! -d OpenNI-MacOSX-x64-2.2.0 ]; then
# #         echo Not found!
# #         echo Checking for OpenNI-MacOSX-x64-2.2.0.tar.bz2... \\c
# #         if [ ! -e OpenNI-MacOSX-x64-2.2.0.tar.bz2 ]; then
# #             echo Not found!
# #             echo Checking for MacOSX.zip... \\c
# #             if [ ! -e MacOSX.zip ]; then
# #                 echo Not found!
# #                 echo Downloading MacOSX.zip...
# #                 wget http://www.openni.org/wp-content/uploads/2013/05/MacOSX.zip
# #             fi
# #             echo Verifying checksum... \\c
# #             if [ `md5 -q MacOSX.zip` != 464ff59d7c30ecaf859becfc0c857241 ]; then
# #                 echo ERROR!
# #                 echo Checksum mismatch.
# #                 exit 1
# #             fi
# #             echo OK
# #             echo Extracting... \\c
# #             unzip -j MacOSX.zip
# #             echo OK
# #         else
# #             echo OK
# #         fi
# #         echo Verifying checksum... \\c
# #         if [ `md5 -q OpenNI-MacOSX-x64-2.2.0.tar.bz2` != 7cda6be6d5a94b190b436897bf34e78a ]; then
# #             echo ERROR!
# #             echo Checksum mismatch.
# #             exit 1
# #         fi
# #         echo OK
# #         echo Extracting... \\c
# #         tar xjf OpenNI-MacOSX-x64-2.2.0.tar.bz2
# #         echo OK        
# #     else
# #         echo OK
# #     fi
# #     echo Creating openni/
# #     mkdir openni
# #     echo Entering OpenNI-MacOSX-x64-2.2.0
# #     cd OpenNI-MacOSX-x64-2.2.0
# #     echo Leaving OpenNI-MacOSX-x64-2.2.0
# #     cd ..
# # else
# #     echo OK
# # fi

# OPENNI_DIR=`pwd`/openni
# OPENNI_SRC_DIR=`pwd`/OpenNI-Bin-Dev-MacOSX-v1.5.4.0
# echo Checking for openni/... \\c
# if [ ! -d openni/ ]; then
#     echo Not found!
#     echo Checking for OpenNI-Bin-Dev-MacOSX-v1.5.4.0/... \\c
#     if [ ! -d OpenNI-Bin-Dev-MacOSX-v1.5.4.0 ]; then
#         echo Not found!
#         echo Checking for OpenNI-Bin-Dev-MacOSX-v1.5.4.0.tar.zip... \\c
#         if [ ! -e OpenNI-Bin-Dev-MacOSX-v1.5.4.0.tar.zip ]; then
#             echo Not found!
#             echo OpenNI-Bin-Dev-MacOSX-v1.5.4.0.tar.zip...
#             wget http://www.openni.org/wp-content/uploads/2012/12/OpenNI-Bin-Dev-MacOSX-v1.5.4.0.tar.zip
#         fi
#         echo Verifying checksum... \\c
#         if [ `md5 -q OpenNI-Bin-Dev-MacOSX-v1.5.4.0.tar.zip` != eecd5f235af13bfe49dc06dbf28c725d ];
#         then
#             echo ERROR!
#             echo Checksum mismatch.
#             exit 1
#         fi
#         echo OK
#         echo Extracting... \\c
#         unzip -p OpenNI-Bin-Dev-MacOSX-v1.5.4.0.tar.zip|tar xj
#         echo OK
#     else
#         echo OK
#     fi
#     echo Creating openni/
#     mkdir openni
#     echo Entering OpenNI-Bin-Dev-MacOSX-v1.5.4.0
#     cd OpenNI-Bin-Dev-MacOSX-v1.5.4.0
#     # sh install.sh -i -c $OPENNI_DIR
#     mkdir $OPENNI_DIR/usr
#     mkdir $OPENNI_DIR/usr/lib
#     cp Lib/*.dylib $OPENNI_DIR/usr/lib
#     install_name_tool -id libOpenNI.dylib $OPENNI_DIR/usr/lib/libOpenNI.dylib
#     for module in libnimMockNodes.dylib libnimCodecs.dylib libnimRecorder.dylib; do
#         install_name_tool -id $module $OPENNI_DIR/usr/lib/$module
#         install_name_tool -change ../../Bin/x64-Release/libOpenNI.dylib \
#             libOpenNI.dylib $OPENNI_DIR/usr/lib/$module
#     done
#     install_name_tool -change /opt/local/lib/libusb-1.0.0.dylib \
#         libusb-1.0.0.dylib $OPENNI_DIR/usr/lib/libOpenNI.dylib 
#     mkdir $OPENNI_DIR/usr/include
#     mkdir $OPENNI_DIR/usr/include/ni
#     cp Include/*.h $OPENNI_DIR/usr/include/ni
#     mkdir $OPENNI_DIR/var
#     mkdir $OPENNI_DIR/var/lib
#     mkdir $OPENNI_DIR/var/lib/ni
#     cat > $OPENNI_DIR/var/lib/ni/modules.xml << EOF
# <Modules>
#     <Module path="$OPENNI_DIR/usr/lib/libnimMockNodes.dylib" />
#     <Module path="$OPENNI_DIR/usr/lib/libnimCodecs.dylib" />
#     <Module path="$OPENNI_DIR/usr/lib/libnimRecorder.dylib" />
# </Modules>
# EOF
#     echo Leaving OpenNI-Bin-Dev-MacOSX-v1.5.4.0
#     cd ..
# else
#     echo OK
# fi

# if [ $BUILD_ALL = true ]; then
#     WXWIDGETS_DIR=`pwd`/wxWidgets
#     OLD_CC=$CC
#     OLD_CXX=$CXX
#     export CC=/usr/bin/gcc
#     export CXX=/usr/bin/g++
#     echo Checking for wxwidgets/... \\c
#     if [ ! -d wxwidgets ]; then
#         echo Not found!
#         echo Checking for wxWidgets-2.9.4/... \\c
#         if [ ! -d wxWidgets-2.9.4 ]; then
#             echo Not found!
#             echo Checking for wxWidgets-2.9.4.tar.bz2... \\c
#             if [ ! -e wxWidgets-2.9.4.tar.bz2 ]; then
#                 echo Not found!
#                 echo Downloading...
#                 wget https://sourceforge.net/projects/wxwindows/files/2.9.4/wxWidgets-2.9.4.tar.bz2/download -O wxWidgets-2.9.4.tar.bz2
#             else
#                 echo OK
#             fi
#             echo Verifying checksum... \\c
#             if [ `md5 -q wxWidgets-2.9.4.tar.bz2` = f1348f740a569ce2945512d5b0e70e87 ]; then
#                 echo OK
#             else
#                 echo ERROR!
#                 echo Checksum mismatch.
#                 exit 1
#             fi
#             echo Extracting... \\c
#             tar xjf wxWidgets-2.9.4.tar.bz2
#             echo OK
#         else
#             echo OK
#         fi
#         echo Creating wxWidgets
#         mkdir wxWidgets
#         echo Entering wxWidgets-2.9.4
#         cd wxWidgets-2.9.4
#         echo Creating build
#         mkdir build
#         echo Entering build
#         cd build
#         ../configure --with-cocoa --enable-unicode --prefix=$WXWIDGETS_DIR 
#         make install
#         cd ..
#         cd ..
#     else
#         echo OK
#     fi
#     CC=$OLD_CC
#     CXX=$OLD_CXX

# if [ ! -d build/ ]; then
#     echo Creating build/
#     mkdir build
# fi
# echo Entering build/
# cd build

# echo Running CMake...
# export CMAKE_PREFIX_PATH=$OPENCV_DIR/share/OpenCV:$WXWIDGETS_DIR 
# cmake -D OpenCV_DIR=$OPENCV_DIR/share/OpenCV \
#     -D USE_CUSTOM_BOOST=ON -D CUSTOM_BOOST_DIR=$BOOST_DIR -D ENABLE_PICSOM=OFF \
#     -D LIBUSB_DIR=$LIBUSB_DIR \
#     -D OPENNI_DIR=$OPENNI_SRC_DIR \
#     -D OPENNI_INSTALL_DIR=$OPENNI_DIR -D OPENNI_MODULES_DIR=. -D ENABLE_OPENNI=ON \
#     -D ENABLE_LIBFLANDMARK=ON -D LIBFLANDMARK_DIR=$LIBFLANDMARK_DIR \
#     -D TEST_DATA_DIRECTORY=/Users/masa/slmotion_tests ..

# echo Running make...
# make

# cp ../libflandmark/libflandmark.dylib .
# OLDNAME=`otool -D libflandmark.dylib|grep -v .dylib:`
# install_name_tool -change $OLDNAME libflandmark.dylib slmotion
# install_name_tool -change $OLDNAME libflandmark.dylib slmotiongui
# install_name_tool -change $OLDNAME libflandmark.dylib tests
# install_name_tool -change $OLDNAME libflandmark.dylib libslmotion.dylib
# install_name_tool -id libflandmark.dylib libflandmark.dylib

# if [ $BUILD_ALL = true ]; then
#     echo Copying OpenCV dynamic link library binaries... \\c
#     for lib in $OPENCV_DIR/lib/*.dylib; do
#         cp $lib .
#         install_name_tool -id `basename $lib` `basename $lib`
#     done
#     echo OK
#     echo Correcting OpenCV library dependencies \(library names\)... \\c
#     for lib in libopencv_*; do
#         for lib2 in libopencv_*; do
#             install_name_tool -change lib/$lib $lib $lib2
#         done
#         install_name_tool -change lib/$lib $lib libslmotion.dylib
#         install_name_tool -change lib/$lib $lib slmotion
#         install_name_tool -change lib/$lib $lib tests
#         install_name_tool -change lib/$lib $lib libflandmark.dylib
#         install_name_tool -change lib/$lib $lib slmotiongui
#     done
#     echo OK

#     echo Copying FFMPEG dependencies.. \\c
#     cp $FFMPEG_DIR/lib/*.dylib .
#     echo OK
# fi

# echo Creating custom modules.xml for OpenNI... \\c
# mkdir var
# mkdir var/lib
# mkdir var/lib/ni
#     cat > var/lib/ni/modules.xml << EOF
# <Modules>
#     <Module path="libnimMockNodes.dylib" />
#     <Module path="libnimCodecs.dylib" />
#     <Module path="libnimRecorder.dylib" />
# </Modules>
# EOF

# echo OK

# echo Running make install...
# make install

# echo Done.