#include "configuration.hpp"
#include "DummyFrameSource.hpp"
#include "debug.hpp"
#include "slmotion.hpp"
#include "util.hpp"
#include "RuleSkinDetector.hpp"
#include "PropagationSkinDetector.hpp"
#include "TemporalSkinPropagator.hpp"
#include "KLTTracker.hpp"
#include "ImageSequenceSource.hpp"
#include "OpenCVVideoCaptureVideoFileSource.hpp"
#include "MultiFrameSource.hpp"
#include "Analyser.hpp"
#ifdef SLMOTION_ENABLE_OPENNI
#include "OpenNiOniFileSource.hpp"
#endif
#ifdef SLMOTION_ENABLE_VIDFILE_SUPPORT
#include "VidFileSource.hpp"
#endif

#ifdef SLMOTION_ENABLE_PICSOM
#include <DataBase.h>
#endif // SLMOTION_ENABLE_PICSOM

#include <fstream>
#include <boost/algorithm/string.hpp>
#include <boost/python.hpp>

using boost::program_options::options_description;
using boost::program_options::option_description;
using boost::program_options::value;
using boost::is_any_of;
using std::string;
using std::vector;
using std::shared_ptr;

namespace slmotion {
  namespace configuration {
    // some default constants
    std::string getDefaultConfigurationFile() {
      return getInstallPrefix() + "/share/slmotion/default.conf";
    }
    
    // static const vector<string> ALL_COMPONENTS { "ColourSpaceConverter", 
    //     "FaceDetector", "GaussianSkinDetector", "TemporalSkinPropagator", 
    //     "BlobExtractor", "BodyPartCollector", "KLTTracker", "AsmTracker", 
    //     "FeatureCreator" };

    // static const vector<string> DEFAULT_COMPONENTS { "ColourSpaceConverter",
    //     "FaceDetector", "GaussianSkinDetector", "BlobExtractor", 
    //     "BodyPartCollector", "KLTTracker", "AsmTracker", "FeatureCreator" };

  // the program uses only one 'named  window', as they are called in high gui
  // this constant can be used whenever referring to it
  const string WINDOWNAME  = "output";



    static options_description getGlobalConffileOpts(SLMotionConfiguration& config) {
      options_description conffileOpts;
      conffileOpts.add_options()
        ("global.components", value<string>()->default_value("none"),
         "A comma-separated list of components to apply in the analysis phase."
         " Components will be processed in the exact order specified here. "
         "Special options include \"all\" which is synonymous to listing all "
         "known components one-by-one. Prefixing the component name with a "
         "minus sign will cause it to be removed from the list. Omitting only "
         "specified components can be achieved by specifying "
         "\"all,-somecomponent\". \"default\" works just like \"all\" but only "
         "adds the default set of components. A list of supported components "
         "may be obtained by running the program with the parameter "
         "--components help. The same settings can also be set on the command "
         "line using the aforementioned --components option, which will "
         "override any setting set here.")
        ("global.cacheframes", value<bool>()->default_value(false), "If "
         "enabled, all processed frames will be cached in RAM during video "
         "analysis. This may consume a very large amount of memory in case of "
         "long videos.")
        ("global.annformat", value<string>(&config.annotationFormat)->default_value("Frame: %f"),
         "Sets the annotation format. In this case, an annotation is a line of "
         "text that is attached to each frame, formatted according to this "
         "string. The formatting strings resemble those of the printf function "
         "in the standard C library. The following substrings are special and "
         "will be replaced by another string as explained below:\n"
         "- %f : the frame number\n"
         "- %t : the matching line of text from the annotation file (see "
         "below)\n"
         "- %% : a % sign\n"
         "I.e. If the the formatting string were \"%%Frame%% %f %t\", the "
         "result could be: \"%Frame% 5 ann\", assuming \"ann\" were the "
         "matching annotation line for frame number 5.\n\n"
         "The annotation file must be supplied from the command line using the "
         "--ann-file parameter. The file format is very simple; each line has "
         "contains two whitespace separated integers which mark the first and "
         "the last frames for the given annotation line, respectively. "
         "Anything following a whitespace after the latter integer will be "
         "treated as the annotation line, terminated by a newline. Every line "
         "must end in a newline. For example, the file could look like this:\n"
         "1 10 Lines one through ten\n"
         "15 25 Lines 15-25")
        ("global.aspect", value<string>(),
         "The option takes one of the following forms:\n"
         "<double> | <int>:<int> | <int>/<int>\n"
         "Scales the video to given aspect ratio. E.g. to perform a 16:9 "
         "scaling, one would use aspect=1.78 and for 4:3 aspect=1.33. "
         "Optionally, the parameter can be entered as a ratio, using either a "
         "colon or a slash as divisor sign. Thus, aspect=4/3, aspect=4:3, and "
         "aspect=1.33 would have the same effect.")
        ("global.scale", value<string>(),
         "The option teakes the following form: <int>x<int>\n"
         "Scales the video to given absolute resolution before processing it at"
         "all. This is done *before* aspect ratio correction is applied.")
        ("global.videofourcc", value<string>(&config.videoFourCC)->default_value("MJPG"),
         "Sets the fourcc code for the video encoder. The default is MJPG. "
         "Only first four letters are considered.")
        ("global.videofps", value<double>(&config.videoFps)->default_value(25.0),
         "Sets the video frame rate. The default is 25.0. This is used for "
         "both video output and computing ELAN timecodes.")
        ("global.annotationtemplates", value<string>(&config.annotationTemplateFilename)->default_value(/* "default.ann" */ ""), "Annotation templates for generating ELAN annotations as per the schema file")
        ("global.maxthreads", value<size_t>(&config.maxThreads)->default_value(5),
         "Maximum number of concurrent threads. This option is significant "
         "only if the programme has been compiled with -DSLMOTION_THREADING.")
        ("global.videobackend", value<string>()->default_value("opencv"),
         "Sets the backend for video input. Currently, only OpenCV video "
         "capture class is supported.")
        ("global.componentalias", value<std::vector<string>>()->default_value(std::vector<string> { "example=FaceDetector2,FacialLandmarkDetector" }, "example=FaceDetector2,FacialLandmarkDetector"),
         "Sets an alias for components which will be treated as a meta component that is expanded upon specifying it in the component chain. This option can be specified a number of times.");
      return conffileOpts;
    }



    static options_description getConffileOpts(SLMotionConfiguration& config) {
      options_description conffileOpts = Component::getCompleteConfigurationFileOptionsDescription();
      conffileOpts.add(getGlobalConffileOpts(config));
      return conffileOpts;
    }



    /**
     * Prints the text into ostream with the given prefix, cutting lines at
     * whitespaces where appropriate
     */
    static void tokenisePrintLineWithPrefix(std::ostream& os,
                                            std::string firstPrefix,
                                            std::string prefix,
                                            size_t maxLineWidth, 
                                            std::string text) {
      vector<string> tokens;
      split(tokens, text, is_any_of(" "));
      string line;
      bool first = true;
      size_t unprefixedLineWidth = maxLineWidth - firstPrefix.length();
      while (tokens.size() > 0) {
        do {
          if (line.length() > 0)
            line += " ";
          line += tokens.front();
          tokens.erase(tokens.begin());
          size_t s;
          if ((s = line.find('\n')) != std::string::npos) {
            tokens.insert(tokens.begin(), line.substr(s+1));
            line = line.substr(0, s);
            break;
          }
        } while (tokens.size() > 0 && 
                 line.length() + tokens.front().length() < 
                 unprefixedLineWidth);

        if (first) {
          os << firstPrefix << line << std::endl;
          first = false;
          unprefixedLineWidth = maxLineWidth - prefix.length();
        }
        else 
          os << prefix << line << std::endl;
       
        line = "";
      }
    }



    static std::string typeToString(const std::type_info& type) {
      if (type == typeid(string))
        return "string";
      if (type == typeid(double))
        return "double";
      if (type == typeid(unsigned long int))
        return "unsigned int";
      if (type == typeid(int))
        return "int";
      if (type == typeid(bool))
        return "boolean";
      if (type == typeid(unsigned int))
        return "unsigned int";
      return "unknown";
    }
    


    static void printSection(std::ostream& os, size_t /* maxLineWidth */,
                             const std::string& sectionName,
                             const std::vector<std::string>& unPrefixedOptions,
                             const boost::program_options::variables_map& vm) {
      os << "[" << sectionName << "]" << std::endl;
      for (auto it = unPrefixedOptions.cbegin(); 
           it != unPrefixedOptions.cend(); ++it) 
        os << *it + "=" + anyToString(vm[sectionName + "." + *it].value()) << std::endl;
    }



    static void printSection(std::ostream& os, size_t maxLineWidth,
                             const std::string& sectionDescription,
                             const std::vector<option_description>& opts,
                             const std::string& sectionName) {
      // option signatures in the form name=<type>
      std::vector<string> optionSignatures;
      // more detailed descriptions
      std::vector<string> optionDescriptions;
      // default values for options
      std::vector<string> optionDefaults;
    
      for (auto it = opts.cbegin(); it != opts.cend(); ++it) {
        vector<string> tokens;
        split(tokens, it->long_name(), is_any_of("."));
      
        if (tokens.size() == 2) {
          boost::shared_ptr<const boost::program_options::value_semantic> semantic = it->semantic();
          boost::any defaultValue;
          std::string s = tokens[1] + "=";
          if (!semantic->apply_default(defaultValue)) {
            std::cerr << "WARNING: No default value for option \""
                      << it->long_name() << "\"" << std::endl;
            s += "<unspecified>";
            optionDefaults.push_back("# " + tokens[1] + "=");
          }
          else {
            s += "<" + typeToString(defaultValue.type()) + ">";
            optionDefaults.push_back(tokens[1] + "=" + anyToString(defaultValue));
          }

          optionSignatures.push_back(s);
          optionDescriptions.push_back(it->description());
          if (it->description() == "")
            std::cerr << "WARNING: No description for option \""
                      << it->long_name() << "\"" << std::endl;          
        }
        else
          std::cerr << "WARNING: Option \"" << it->long_name() 
                    << "\" ignored because it has no prefix." << std::endl;
      }

      tokenisePrintLineWithPrefix(os, "# ", "# ", maxLineWidth,
                                  sectionDescription);
      os << "# " << std::endl;

      for (size_t i = 0; i < optionSignatures.size(); ++i) {
        os << "# " << optionSignatures[i] << std::endl;
        tokenisePrintLineWithPrefix(os, "#     ", "#   ", maxLineWidth,
                                    optionDescriptions[i]);
        os << "# " << std::endl;
      }

      os << "[" << sectionName << "]" << std::endl;

      for (auto it = optionDefaults.cbegin(); it != optionDefaults.cend();
           ++it)
        os << *it << std::endl;
    }



    void outputConfig(std::ostream& os, size_t maxLineWidth, 
                      const boost::program_options::variables_map& config) {
            // group options by their prefix
      std::map<std::string, std::vector<std::string> > optionGroups;

      for (auto it = config.cbegin(); it != config.cend(); ++it) {
        vector<string> tokens;
        split(tokens, it->first, is_any_of("."));
      
        if (tokens.size() == 2) 
          optionGroups[tokens[0]].push_back(tokens[1]);
        else
          std::cerr << "WARNING: Option \"" << it->first
                    << "\" ignored because it has no prefix." << std::endl;
      }

      for (auto it = optionGroups.cbegin(); it != optionGroups.cend(); ++it) 
        printSection(os, maxLineWidth, it->first, it->second, config);
    }



    void generateDefaultConfig(std::ostream& os, size_t maxLineWidth) {
      configuration::SLMotionConfiguration config;
      boost::program_options::options_description optsd = configuration::getConffileOpts(config);
      const std::vector<boost::shared_ptr<option_description>>& opts = optsd.options();

      // group options by their prefix
      std::map<std::string, std::vector<option_description>> optionGroups;

      for (auto it = opts.cbegin(); it != opts.cend(); ++it) {
        vector<string> tokens;
        split(tokens, (*it)->long_name(), is_any_of("."));
      
        if (tokens.size() == 2) 
          optionGroups[tokens[0]].push_back(**it);
        else
          std::cerr << "WARNING: Option \"" << (*it)->long_name() 
                    << "\" ignored because it has no prefix." << std::endl;
      }
    
      // global options are treated differently
      std::vector<option_description> globals;
      if (optionGroups.count("global")) {
        globals = optionGroups["global"];
        optionGroups.erase("global");
      }
      else
        std::cerr << "WARNING: No global option descriptions available."
                  << std::endl;
    

      os << "# SLMotion default configuration" << std::endl;
      os << "# " << SLMOTION_VCID << std::endl;
      os << "# " << std::endl;

      string temp;
      temp = "This is the default configuration file for slmotion.";
      tokenisePrintLineWithPrefix(os, "# ", "# ", maxLineWidth, temp);
      os << "# " << std::endl;
                                
      temp = "This configuration file is formatted like a typical UNIX "
        "utility configuration file. The format is as follows:";
      tokenisePrintLineWithPrefix(os, "# ", "# ", maxLineWidth, temp);
      os << "# " << std::endl;

      temp = "Configuration consists of key and value pairs. The key is "
        "separated from the value by an equality sign (=), e.g. key=value. "
        "Most of these keys are unique, and only one such key is allowed per "
        "section. However, there are some exceptions which will be explained "
        "later.";
      tokenisePrintLineWithPrefix(os, "# ", "# ", maxLineWidth, temp);
      os << "# " << std::endl;

      temp = "The key-value pairs are divided into sections. Sections are "
        "identified by enclosing the section identifier inside square "
        "brackets, e.g. [section]. Each section is unique, and controls some "
        "particular function of the program.";
      tokenisePrintLineWithPrefix(os, "# ", "# ", maxLineWidth, temp);
      os << "# " << std::endl;

      temp = "Comments begin with a number sign (#). Anything following such "
        "a sign will be ignored until a newline, which terminates comments.";
      tokenisePrintLineWithPrefix(os, "# ", "# ", maxLineWidth, temp);
      os << "# " << std::endl;

      printSection(os, maxLineWidth, "The global section controls some rather"
                   " generic behaviour of the programme that mostly affect "
                   "which operations the programme should perform.",
                   globals, "global");

      // add individual component sections

      for (auto it = optionGroups.cbegin(); it != optionGroups.cend(); 
           ++it) {
        os << std::endl;
        printSection(os, maxLineWidth, 
                     Component::getLongDescriptionForComponent(it->first), 
                     it->second, it->first);
      }
    }



    std::vector<std::string> parseComponents(const std::string& componentString,
                                             const std::map<std::string, 
                                                            std::vector<std::string>>& aliases) {
      if (componentString == "" || 
          boost::algorithm::to_lower_copy(componentString) == "none")
        return vector<string>();

      vector<string> returnStrings;
      vector<string> tokens;
      split(tokens, componentString, is_any_of(","));

      for (auto it = tokens.begin(); it != tokens.end(); ++it) {
        // std::cout << *it << std::endl;
        if (aliases.count(*it)) {
          const std::vector<std::string>& replacement = aliases.find(*it)->second; 
          it = tokens.erase(it);
          tokens.insert(it, replacement.begin(), replacement.end());
          it = tokens.begin();
        } 
      }

      for (auto it = tokens.cbegin(); it != tokens.cend(); ++it) {
        // if (*it == "all")
        //   returnStrings.insert(returnStrings.end(), ALL_COMPONENTS.begin(), ALL_COMPONENTS.end());
        // else if (*it == "default")
        //   returnStrings.insert(returnStrings.end(), DEFAULT_COMPONENTS.begin(), DEFAULT_COMPONENTS.end());
        // else 
        if (it->length() > 1 && (*it)[0] == '-')
          returnStrings.erase(std::remove_if(returnStrings.begin(),
                                             returnStrings.end(),
                                             [&](const std::string& s) {
                                               return s == it->substr(1);
                                             }), returnStrings.end());
        else
          returnStrings.push_back(*it);
      }
      return returnStrings;
    }



    SLMotionConfiguration parseConfigurationFile(const std::string& conffile) {
      std::ifstream ifs(conffile.c_str());
      if (ifs.good()) {
        return parseConfigurationFile(ifs);
      }
      else {
        throw ConfigurationFileException("slmotion: Bad configuration file supplied: \"" + conffile + "\": No configuration will be read.");
      }
    }



    SLMotionConfiguration parseConfigurationFile(std::istream& ifs) {
      SLMotionConfiguration config;

      options_description conffileOpts = getConffileOpts(config);

      // declare section-aware configuration file options
      boost::program_options::variables_map vm;

      boost::program_options::store(boost::program_options::parse_config_file(ifs, conffileOpts), vm);
      boost::program_options::notify(vm);

      *config.configurationVariables = vm;
  
      // Parse configuration
      // if (vm.count("skin.basemethod")) 
      //   config.skinDetectionBaseMethod = vm["skin.basemethod"].as<string>();
      // else
      //   config.skinDetectionBaseMethod = "gaussnd";

      // if (vm.count("skin.postprocess"))
      //   config.skinDetectorPostProcessFilters = parseSkinPostprocess(vm["skin.postprocess"].as<string>());

      // components
      std::map<std::string, std::vector<std::string> > aliases;
       if (vm.count("global.componentalias")) {
        std::vector<string> aliasVector = vm["global.componentalias"].as<std::vector<string>>();
        for (auto it = aliasVector.cbegin(); it != aliasVector.cend(); ++it) {
          std::vector<string> tokens;
          boost::split(tokens, *it, is_any_of("="));
          if (tokens.size() != 2)
            throw ConfigurationFileException("Invalid alias declaration \"" + *it + "\" "
                                             "encountered! An entry of the form key=value was "
                                             "expected.");
          std::vector<string> tokens2;
          boost::split(tokens2, tokens[1], is_any_of(","));
          aliases[tokens[0]] = tokens2;
        }
       }
       
       config.aliases = aliases;
       if (config.script.length() == 0)
         config.analyserComponents = parseComponents(vm.count("global.components") ?
                                                     vm["global.components"].as<string>() :
                                                     "none",
                                                     aliases);

      if (vm.count("global.cacheframes")) 
        config.frameCacheSize = vm["global.cacheframes"].as<bool>() ?
          SIZE_MAX : 0;

      if (vm.count("global.aspect")) {
        string newAspect = vm["global.aspect"].as<string>();
        double d;
        int x, y;
        if (sscanf(newAspect.c_str(), "%i:%i", &x, &y) == 2)  
          config.aspect = static_cast<double>(x) / static_cast<double>(y);
        else if (sscanf(newAspect.c_str(), "%i/%i", &x, &y) == 2)
          config.aspect = static_cast<double>(x) / static_cast<double>(y);
        else if (sscanf(newAspect.c_str(), "%lf", &d) < 1) 
          throw ConfigurationFileException("'aspect' expects an aspect ratio as its argument");
        else 
          config.aspect = d;
      }

      if (vm.count("global.videobackend")) {
        string s = vm["global.videobackend"].as<string>();
        if (s == "opencv")
          config.videoBackend = SLMotionConfiguration::OPENCV;
        else
          throw ConfigurationFileException("'" + s + "' is not a valid video back end");
      }

      if (vm.count("global.scale")) {
        string newResolution = vm["global.scale"].as<string>();
        int x, y;
        if (sscanf(newResolution.c_str(), "%ix%i", &x, &y) == 2) 
          config.scale = cv::Size(x,y);
        else
          throw ConfigurationFileException("'scale' expects two integers separated by an 'x' as its argument.");
      }

      // if (vm.count("skin.colourspace")) {
      //   string s = vm["skin.colourspace"].as<string>();
      //   if (s == "hsv")
      //     config.colourSpace.reset(ColourSpace::HSV->clone());
      //   else if (s == "ycrcb")
      //     config.colourSpace.reset(ColourSpace::YCRCB->clone());
      //   else if (s == "bgr")
      //     config.colourSpace.reset(ColourSpace::BGR->clone());
      //   else if (s == "rgb")
      //     config.colourSpace.reset(ColourSpace::RGB->clone());
      //   else if (s == "xyz")
      //     config.colourSpace.reset(ColourSpace::XYZ->clone());
      //   else if (s == "cgz")
      //     config.colourSpace.reset(ColourSpace::CGZ->clone());
      //   else if (s == "ihls")
      //     config.colourSpace.reset(ColourSpace::IHLS->clone());
      //   else if (s.find("custom:") != string::npos)
      //     config.colourSpace.reset(createCustomColourSpace(s).clone());
      //   else
      //     throw ConfigurationFileException("'" + s + "' is not a valid colour space!");
      // }

      return config;
    }



    boost::program_options::options_description getVisibleOptionsDescription() {
      options_description visible("Global options");
      visible.add_options()
        ("help", "prints this help message")
        ("version", "prints version information and quits")
        ("verbose,v", "enables more verbose output (same as --debug 1)")
        ("debug", value<int>(&slmotion::debug), "sets the debug message verbosity level (disabled at 0)")
        ("paranoid", "enables the paranoia mode; the program shall not trust video file headers at the expense of (a lot of) extra processing time. Try this if you have inexplicable crashes etc. with your substandard video files")
        ("config,c", value<vector<string>>(), "Sets the configuration file. This option may be omitted, or specified once or a number of times matching the number of jobs (i.e. one global configuration, or one configuration per job). If this option is omitted, 'default.conf' is used. The first configuration file specified will be used for non-job-specific global configuration.")
        ("kinect", "Enables the Kinect mode. This is a special short-hand for specifying two tracks where the primary track is an RGB video feed and the secondary track is the depth data. Exactly two video files are expected as input, respectively.")
        // ("face-cascade", value<string>(),
        //  "sets the cascade file for the Haar classifier used by the face detector")
        // ("train-image", value<string>(),
        //  "sets an image to be used for training the skin detector; if this option is omitted, face detection results will be used to initialise the model.")
        ("out,o", value<string>(), "sets the output filename for visual data")
        ("script", value<string>(), "A Python script file. The components "
         "parameter is ignored when a script is specified.")
        ("components", value<string>(), "A comma-separated list of "
         "components to apply in the analysis phase. Components will be "
         "processed in the exact order specified here. Special options "
         "include \"all\" which is synonymous to listing all known "
         "components one-by-one. Prefixing the component name with a minus "
         "sign will cause it to be removed from the list. \"--components "
         "help\" or \"--components ?\" will cause a list of all known "
         "components to be printed. Setting components to \"none\" or not "
         "specifying anything, i.e. \"\", will cause the entire component "
         "chain to be ignored.")
        ("generate-default-config", "Generates the default configuration "
         "and outputs it to standard output")
#ifdef SLMOTION_ENABLE_PICSOM
        ("picsom-features", value<string>(), "If set, prints out a list of "
         "recognised picsom features")
        ("picsom-segmentations", value<string>(), "If set, prints out a "
         "list of recognised picsom segmentations")
        ("db", value<string>(),
         "sets the name of the used PicSOM database")
        ("picsom-args", value<string>(),
         "sets the arguments passed to PicSOM upon invocation")
#endif // SLMOTION_ENABLE_PICSOM
        ("feature-out", value<string>(), "sets the output filename for extracted feature data, stored either in CSV or SOM_PAK format, depending on the file extension given")
        ("elan-out", value<string>(), "sets the output filename for ELAN XML data. If the file exists, new tiers are added to it. Otherwise, a new file is created.")
        ("frames", value<size_t>(),
         "only process arg frames")
        ("skip", value<size_t>(),
         "skip arg first frames")
        ("aspect", value<string>(),
         "scales the video to the desired aspect ratio")
        ("scale", value<string>(),
         "scales the video to the desired resolution")
        ("ann-format", value<string>(),
         "sets a formatting string for the annotation bar")
        ("ann-file", value<string>(),
         "sets the file from which annotations are read")
        ("ann-template-file", value<string>(),
         "sets a file from which annotation templates are read")
        // ("no-blob-detection", "disables blob detection")
        ("complex-visualisation", value<string>(),
         "sets a complex visualisation string")
        ("disable-annotation-bar",
         "when visualising the results, disables the annotation bar")
        ("show-mask", "shows the binary skin detector mask")
        ("show-blob-cent", "draws white crosses at blob centroids")
        ("show-blobs", "draws the blobs on the image in fancy colours")
        ("show-detected-body-parts",
         "colours detected body parts as follows: red for left hand, blue for right hand, green for the head, and combinations of body parts will be coloured by combining the colours, e.g. yellow for a left hand / head combo")
        ("show-corners", "draws white circles around found features")
        ("show-tracked-points", "draws coloured circles around tracked features in the post-processing phase")
        ("show-face", "draws a box where face is detected")
        ("show-motion-vectors", "shows motion vectors. Each frame will be drawn "\
         "one frame late because future information will be needed.")
        ("show-acc-vectors", "shows acceleration vectors like motion vectors")
        ("show-estimated-anchors", "shows estimated anchor points used for body part estimation using ASMs")
        ("show-asms", "draws lines around detected ASMs")
        ("show-gradient-map", "draws a gradient map of the image")
        ("show-2dgradient-map", "draws a gradient map of the image where hue represents the direction and value the magnitude")
        ("show-canny-map", "draws a map of edges as detected by the Canny edge detector")
        ("pause,p", "pauses between each image until a key is pressed")
        ("combination-style", value<string>(), "sets if input images should be combined 'left-to-right' or 'right-to-left'")
        ("frame-cache-size", value<int>(), "sets the maximum number of frames to store in the frame cache")
        ("work-dir", value<string>(), "sets the working directory for locating auxiliary files etc.");
      visible.add(Component::getCompleteCommandLineOptionsDescription());
      return visible;
    }


  
    boost::program_options::variables_map parseCommandLine(int argc, char* argv[]) {
      options_description visible = getVisibleOptionsDescription();

      // input video filenames are passed as 'positional parameters', i.e. 
      // parameters that do not depend on a preceeding parameter name
      options_description hidden("Hidden options");
      hidden.add_options()
        ("inputNames", value<vector<string>>(), "Input filenames. This may be a number of video files or a number of still images. Still images are combined into a single job. A separate job is created for each video file. All jobs are independent of one another.");
      options_description cmdOpts;
      cmdOpts.add(visible).add(hidden);

      boost::program_options::positional_options_description p;
      p.add("inputNames", -1);

      boost::program_options::variables_map vm;
      boost::program_options::store(boost::program_options::command_line_parser(argc, argv).options(cmdOpts).positional(p).
                run(), vm);
      boost::program_options::notify(vm);

      return vm;
    }



    void applyConfiguration(const SLMotionConfiguration& config,
                            Visualiser& visualiser) {
      visualiser.setAnnFormat(config.annotationFormat);
    
      visualiser.setShowBlobCentroids(config.showBlobCentroids);
      visualiser.setShowEstimatedAnchors(config.showEstimatedAnchors);
      visualiser.setShowCorners(config.showCorners);
      visualiser.setShowSkinMask(config.showSkinMask);
      visualiser.setComplexVisualisation(config.complexVisualisation);
      visualiser.setShowBlobs(config.showBlobs);
      visualiser.setShowBodyParts(config.showBodyParts);
      visualiser.setShowBodyParts(config.showBodyParts);
      visualiser.setShowGradientMap(config.showGradientMap);
      visualiser.setShowGradientMap2d(config.showGradientMap2d);
      visualiser.setShowCannyMap(config.showCannyMap);
      visualiser.setShowFace(config.showFace);
      visualiser.setShowAsms(config.showAsms);
      visualiser.setShowTrackedPoints(config.showTrackedPoints);
      visualiser.setShowMotionVectors(config.showMotionVectors);
      visualiser.setShowAccelerationVectors(config.showAccelerationVectors);
      // visualiser.setCombinationStyle(config.visualiserCombinationStyle);
      // visualiser.setAsmFittingContext(config.asmFittingContext);
    }

    void applyConfiguration(const SLMotionConfiguration& config,
                            SLIO& slio) {
      slio.setOutFilename(config.outputFilename);
      slio.setVideoOut(config.videoOutput);

      if (config.annotationFilename.length() > 0) 
        slio.loadAnnFile(config.annotationFilename);

      if (config.featureOutputFilename.length() > 0) 
        slio.setFeatureOutFile(config.featureOutputFilename);

      slio.setOutFps(config.videoFps);

      slio.setVideoFourcc(config.videoFourCC);
    }



    void applyConfiguration(const SLMotionConfiguration& config,
                            FrameSource& frameSource) {
      frameSource.setAspect(config.aspect);
      frameSource.setScale(config.scale);
    }

    std::vector< std::shared_ptr<Component> > createComponents(const SLMotionConfiguration& config, FrameSource* frameSource, BlackBoard* blackBoard) {

      std::vector< std::shared_ptr<Component> > comps;

      std::set<BlackBoard::property_key_t> providedProperties;
      for (auto it = config.analyserComponents.cbegin();
           it != config.analyserComponents.cend(); ++it) {
        shared_ptr<Component> newComponent(COMPONENT_LIST->createComponent(*it, *config.configurationVariables, blackBoard, frameSource));
        if (newComponent.get() == NULL)
          throw ConfigurationFileException("'" + *it + "' is not a valid component.");
      
        
        // ensure that all necessary data will be available in the black 
        // board, and then update the list
        Component::property_set_t requirements = newComponent->getRequirements();
        for (auto it = requirements.cbegin(); it != requirements.end(); 
             ++it)
          if (!providedProperties.count(*it)) {
            vector<string> providers = Component::findProviders(*it);
            string s = "Component '" +newComponent->getComponentName() + "' requires a property called '" + *it + "' to be provided by some other component onto the black board before the component is called. However, no preceeding component in the component chain says it provides this property.";
            if (providers.size() > 0) {
              s += " The following components say they do provide the required property: ";
              int count = 0;
              for (auto jt = providers.begin(); jt != providers.end(); ++jt) {
                if (count > 0)
                  s += ", ";
                s += *jt;
                ++count;
              } 
            }
            throw ConfigurationFileException(s);
          }

        auto provided = newComponent->getProvided();
        providedProperties.insert(provided.cbegin(), provided.cend());

        comps.push_back(newComponent);
      }

      return comps;
    }



    /**
     * Applies relevant command line options to the configuration
     *
     * @param vm Command line options variables map. Should have been parsed 
     * before calling this function.
     */
    void applyCommandLineOptions(int argc, char* argv[],
                                 slmotion::configuration::SLMotionConfiguration& config) {
      boost::program_options::variables_map vm = parseCommandLine(argc, argv);
      config.configurationVariables->insert(vm.begin(), vm.end());

#ifdef SLMOTION_ENABLE_PICSOM
      if (vm.count("db"))
        config.picsomDataBaseName = vm["db"].as<string>();
      if (vm.count("picsom-args"))
        config.picsomArgs = vm["picsom-args"].as<string>();
#endif // SLMOTION_ENABLE_PICSOM

      if (vm.count("elan-out"))
        config.elanOutputFilename = vm["elan-out"].as<string>();

      if (vm.count("components"))
        config.analyserComponents = parseComponents(vm["components"].as<string>(),
                                                    config.aliases);

      if (vm.count("script")) {
        config.script = vm["script"].as<string>();
        config.analyserComponents = std::vector<string>();
      }

      if (vm.count("ann-format"))
        config.annotationFormat = vm["ann-format"].as<string>();

      if (vm.count("ann-file"))
        config.annotationFilename = vm["ann-file"].as<string>();

      if (vm.count("verbose") && slmotion::debug < 1) 
        slmotion::debug = 1;

      if (vm.count("paranoid"))
        slmotion::paranoid = true;

      if (vm.count("out")) {
        config.outputFilename = vm["out"].as<string>();

        string extension = getExtension(vm["out"].as<string>());
        boost::to_lower(extension);

        if(STILL_IMAGE_EXTENSIONS.count(extension))
          config.videoOutput = false;
        else if (VIDEO_EXTENSIONS.count(extension)) 
          config.videoOutput = true;
        else 
          throw CommandLineException(string("Unrecognised output file "
                                            "format: '" +
                                            vm["out"].as<string>() + 
                                            "' given. The extension "
                                            "could not be "
                                            "recognised.").c_str());
      }

      if (vm.count("aspect")) {
        string newAspect = vm["aspect"].as<string>();
        double d;
        int x, y;
        if (sscanf(newAspect.c_str(), "%i:%i", &x, &y) == 2)  
          config.aspect = static_cast<double>(x) / static_cast<double>(y);
        else if (sscanf(newAspect.c_str(), "%i/%i", &x, &y) == 2)
          config.aspect = static_cast<double>(x) / static_cast<double>(y);
        else if (sscanf(newAspect.c_str(), "%lf", &d) == 1) 
          config.aspect = d;
        else 
          throw CommandLineException("'aspect' expects an aspect ratio as its argument");
      }

      if (vm.count("scale")) {
        string scale = vm["scale"].as<string>();
        int x, y;
        if (sscanf(scale.c_str(), "%ix%i", &x, &y) == 2)  
          config.scale = cv::Size(x,y);
        else 
          throw CommandLineException("'scale' expects a resolution denoted by two 'x'-separated integers as its argument");
      }


      if (vm.count("ann-template-file"))
        config.annotationTemplateFilename = vm["ann-template-file"].as<string>();

      // if (vm.count("no-blob-detection")) 
      //   config.doBlobDetection = false;

      if (vm.count("show-tracked-points"))
        config.showTrackedPoints = true;

      if (vm.count("show-motion-vectors"))
        config.showMotionVectors = true;

      if (vm.count("show-acc-vectors"))
        config.showAccelerationVectors = true;

      if (vm.count("feature-out"))
        config.featureOutputFilename = vm["feature-out"].as<string>();
    
      if (vm.count("show-blob-cent"))
        config.showBlobCentroids = true;
    
      if (vm.count("show-estimated-anchors"))
        config.showEstimatedAnchors = true;
    
      if (vm.count("show-corners"))
        config.showCorners = true;
  
      if (vm.count("show-mask"))
        config.showSkinMask = true;

      if (vm.count("disable-annotation-bar"))
        config.disableAnnotationBar = true;

      if (vm.count("complex-visualisation"))
        config.complexVisualisation = vm["complex-visualisation"].as<string>();
    
      if (vm.count("show-blobs"))
        config.showBlobs = true;
 
      if (vm.count("show-detected-body-parts"))
        config.showBodyParts = true;
 
      if (vm.count("show-gradient-map"))
        config.showGradientMap = true;
 
      if (vm.count("show-2dgradient-map")) {
        config.showGradientMap = true;
        config.showGradientMap2d = true;
      }
  
      if (vm.count("show-canny-map"))
        config.showCannyMap = true;
    
      if (vm.count("show-face"))
        config.showFace = true;

      if (vm.count("show-asms"))
        config.showAsms = true;

#if 0
      if (vm.count("combination-style")) {
        if (vm["combination-style"].as<string>() ==
            "left-to-right")
          config.visualiserCombinationStyle = Visualiser::LEFT_TO_RIGHT;
        else if (vm["combination-style"].as<string>() ==
                 "top-to-bottom")
          config.visualiserCombinationStyle = Visualiser::TOP_TO_BOTTOM;
        else if (vm["combination-style"].as<string>() ==
                 "auto")
          config.visualiserCombinationStyle = Visualiser::AUTOMAGIC;
        else
          throw CommandLineException("combination-style must be either 'left-to-right' or 'top-to-bottom'!");
      }
#endif

      if (vm.count("frame-cache-size")) 
        config.frameCacheSize = vm["frame-cache-size"].as<int>();
            
      if (vm.count("pause"))
        config.pauseMode = true;
      if (vm.count("frames"))
        config.framesMax = vm["frames"].as<size_t>();
      if (vm.count("out"))
        config.storeImages = vm["out"].as<string>().length() > 0;
      if (vm.count("skip"))
        config.skipFrames = vm["skip"].as<size_t>();
    }



    void createAnalysers(const std::vector<Job>& jobs, 
                         const std::vector<std::string>& conffiles,
                         std::vector<Analyser>& analysers,
                         SLMotionConfiguration& globalConfiguration,
                         int argc, char* argv[]) {
      auto configs = parseConfigurationFiles(conffiles, jobs.size());
      assert(jobs.size() > 0 && configs.size() == jobs.size());

      analysers.clear();

      for (size_t inputIdx = 0; inputIdx < jobs.size(); ++inputIdx) {
        SLMotionConfiguration& config = configs[inputIdx];
        string inputName = jobs[inputIdx].filenames.size() > 0 ? 
          jobs[inputIdx].filenames.front() : "";

        config.inputFilename = inputName;
        applyCommandLineOptions(argc, argv, config);

        shared_ptr<SLIO> slio(new SLIO(inputName));
        shared_ptr<Visualiser> visualiser(new Visualiser(slio));

        shared_ptr<FrameSource> frameSource = createFrameSourceForJob(jobs[inputIdx],
                                                                      config.frameCacheSize,
                                                                      config.videoBackend);

#ifdef SLMOTION_THREADING
        Thread::setMaxThreads(config.maxThreads);
#endif
        applyConfiguration(config, *visualiser);
        applyConfiguration(config, *slio);
        applyConfiguration(config, *frameSource);

#ifdef SLMOTION_ENABLE_PICSOM
        shared_ptr<BlackBoard> blackboard(new BlackBoard(jobs[inputIdx].picsom,
							 jobs[inputIdx].picsom_db));
#else
	shared_ptr<BlackBoard> blackboard(new BlackBoard());
#endif // SLMOTION_ENABLE_PICSOM

	frameSource->setLabel(jobs[inputIdx].label);

        vector<shared_ptr<slmotion::Component>> comps = createComponents(config, frameSource.get(), blackboard.get());

        analysers.push_back(Analyser(slio, frameSource, visualiser, 
                                     blackboard));
        analysers.back().addPreprocessComponents(std::move(comps));
      }

      globalConfiguration = configs[0];
    }



    std::shared_ptr<FrameSource> createFrameSourceForJob(const Job& job,
                                                         size_t frameCacheSize,
                                                         SLMotionConfiguration::VideoBackEnd videoBackEnd) {
      std::shared_ptr<FrameSource> frameSource;
      std::string firstFilename = job.filenames.size() > 0 ? 
        job.filenames.front() : "";
      switch(job.type) {
      case Job::JobType::DUMMY:
        frameSource.reset(dynamic_cast<FrameSource*>(new DummyFrameSource()));
        break;
      case Job::JobType::IMAGE_SEQUENCE:
        // the frame rate of 25 here is arbitrary
        frameSource.reset(dynamic_cast<FrameSource*>(new ImageSequenceSource(job.filenames, 25., 
                                                                             frameCacheSize)));
        break;

      case Job::JobType::VIDEO_FILE:
        switch (videoBackEnd) {
        case SLMotionConfiguration::OPENCV:
          frameSource.reset(dynamic_cast<FrameSource*>(new OpenCVVideoCaptureVideoFileSource(firstFilename, frameCacheSize)));
          break;
        }
        break;

#ifdef SLMOTION_ENABLE_VIDFILE_SUPPORT
      case Job::JobType::VID_FILE:
        frameSource.reset(dynamic_cast<FrameSource*>(new VidFileSource(firstFilename, 
                                                                       frameCacheSize)));
        break;
#endif
#ifdef SLMOTION_ENABLE_OPENNI
      case Job::JobType::ONI_FILE:
        frameSource.reset(dynamic_cast<FrameSource*>(new OpenNiOniFileSource(firstFilename, frameCacheSize)));
        break;
#endif
      case Job::JobType::KINECT:
        assert(job.filenames.size() == 2);
        frameSource.reset(dynamic_cast<FrameSource*>(new MultiFrameSource(frameCacheSize)));
        dynamic_cast<MultiFrameSource*>(frameSource.get())->addTrack(std::shared_ptr<VideoFileSource>(new OpenCVVideoCaptureVideoFileSource(job.filenames.front(), frameCacheSize)));
        dynamic_cast<MultiFrameSource*>(frameSource.get())->addTrack(std::shared_ptr<VideoFileSource>(new OpenCVVideoCaptureVideoFileSource(job.filenames.back(), frameCacheSize)));
        frameSource->getTrack(0).setTrackInfo(TrackInfo { TrackInfo::Type::BGR_IMAGE, "KinectVideo" });
        frameSource->getTrack(1).setTrackInfo(TrackInfo { TrackInfo::Type::DEPTH_DATA, "KinectDepth" });
        dynamic_cast<MultiFrameSource*>(frameSource.get())->setDefaultTrackNumber(0);
        break;
      }
      return frameSource;
    }




    std::vector<SLMotionConfiguration> parseConfigurationFiles(const std::vector<string>& conffiles, size_t numberOfConfigurations) {
      vector<SLMotionConfiguration> configs;
      for (size_t inputIdx = 0; inputIdx < numberOfConfigurations; 
           ++inputIdx) {
        string conffile = 
          conffiles.size() == 0 ? getDefaultConfigurationFile() :
          conffiles.size() == 1 ? conffiles.front() :
          conffiles[inputIdx];
        configs.push_back(parseConfigurationFile(conffile));
      }
      return configs;
    }



    static string getLowercaseExtension(const std::string& filename) {
      string extension = slmotion::getExtension(filename);
      boost::to_lower(extension);
      return extension;
    }



    /**
     * The Kinect short-hand.
     */
    vector<Job> assignFilenamesToKinectJobs(const vector<string>& inputFilenames) {
      if (inputFilenames.size() != 2 ||
          !VIDEO_EXTENSIONS.count(getLowercaseExtension(inputFilenames[0])) ||
          !VIDEO_EXTENSIONS.count(getLowercaseExtension(inputFilenames[1])))
        throw CommandLineException("Kinect jobs must have exactly two video input files!");

      return vector<Job> { Job { Job::JobType::KINECT, inputFilenames } };
    }



    vector<Job> assignFilenamesToJobs(const vector<string>& inputFilenames,
				      const string& picsom_db_name,
                                      const string& picsomArgs) {
      vector<Job> jobs;
      vector<string> imageSequenceFilenames;

      for(auto it = inputFilenames.cbegin(); it != inputFilenames.cend();
          ++it) {
        string extension = getLowercaseExtension(*it);
        if (*it == "") 
          jobs.push_back(Job(Job::JobType::DUMMY));
        else if (STILL_IMAGE_EXTENSIONS.count(extension))
          imageSequenceFilenames.push_back(*it);
        else if (VIDEO_EXTENSIONS.count(extension)) {
          jobs.push_back(Job(Job::JobType::VIDEO_FILE));
          jobs.back().filenames.push_back(*it);
        }
#ifdef SLMOTION_ENABLE_VIDFILE_SUPPORT
        else if (VID_EXTENSIONS.count(extension)) {
          jobs.push_back(Job(Job::JobType::VID_FILE));
          jobs.back().filenames.push_back(*it);
        }
#endif
#ifdef SLMOTION_ENABLE_OPENNI
        else if (ONI_EXTENSIONS.count(extension)) {
          jobs.push_back(Job(Job::JobType::ONI_FILE));
          jobs.back().filenames.push_back(*it);
        }
#endif
#ifdef SLMOTION_ENABLE_PICSOM
        else if (picsom_db_name!="") {
	  vector<string> picsom_arg { "*picsom*" };
          vector<string> tokens;
          split(tokens, picsomArgs, is_any_of(" "));
	  if (tokens.size()==1 && tokens[0]=="")
	    tokens.clear();
	  if (picsomArgs=="")
	    tokens.push_back("-q");
          picsom_arg.insert(picsom_arg.end(), tokens.cbegin(), 
                            tokens.cend());

	  std::shared_ptr<picsom::PicSOM> picsom(new
						 picsom::PicSOM(picsom_arg));
	  picsom::DataBase *db = picsom->FindDataBaseEvenNew(picsom_db_name,
							     false);
	  if (!db)
	    throw CommandLineException(string("PicSOM database <"+
					      picsom_db_name
					      +"> not found").c_str());
	  picsom::ground_truth gt = db->GroundTruthExpression(*it);
	  vector<size_t> idx = gt.indices(1);
	  if (idx.size()==0)
	    throw CommandLineException(string("PicSOM class expression <"+*it
					      +"> evalutes to an empty set")
				       .c_str());
	  if (slmotion::debug)
	    cerr << "ground truth <" << *it << "> evaluates to " << idx.size()
		 << " objects" << endl;

	  for (size_t q=0; q<idx.size(); q++) {
	    if (!db->ObjectsTargetTypeContains(idx[q], picsom::target_video))
	      continue;

	    std::string label = db->Label(idx[q]);
	    std::string fname = db->SolveObjectPath(label);
	    Job::JobType type = Job::JobType::IMAGE_SEQUENCE;
	    if (db->ObjectsTargetTypeContains(idx[q], picsom::target_video))
	      type = Job::JobType::VIDEO_FILE;
	      
	    jobs.push_back(Job(type, picsom, db));
	    jobs.back().filenames.push_back(fname);
	    jobs.back().label = label;

	    cout << jobs.size() << " " << " " << label << " " << fname << endl;
	  }
        }
#else
	else if (picsom_db_name.size()==picsomArgs.size() && false)
	  ;
#endif // SLMOTION_ENABLE_PICSOM
        else 
          throw CommandLineException(string("Unrecognised input file "
                                            "format: '" +
                                            *it + "' given. The extension "
                                            "could not be "
                                            "recognised.").c_str());
      }
    
      if (imageSequenceFilenames.size() > 0) {
        jobs.push_back(Job(Job::JobType::IMAGE_SEQUENCE));
        jobs.back().filenames = std::move(imageSequenceFilenames);
      }

      return jobs;
    }


    static std::string workingDirectory = ".";
    void setWorkingDirectory(const std::string& dir) {
      if (chdir(dir.c_str()) != 0)
        throw IOException("Invalid working directory: \"" + dir + 
                          "\"; chdir returned error: " + 
                          strerror(errno));
      workingDirectory = getCanonicalPath(dir);
    }

    std::string getWorkingDirectory() {
      return workingDirectory;
    }
  }
}
