#include "math.hpp"
#include "util.hpp"
#include <boost/math/constants/constants.hpp>
#include <stdexcept>
#include <numeric>

namespace slmotion {
  namespace math {
    void expandMat(cv::Mat& matrix, const cv::Size& newSize) {
      assert(newSize.width >= matrix.cols && newSize.height >= matrix.rows);
      cv::Mat newMatrix(newSize, matrix.type(), cv::Scalar::all(0));
      cv::Mat m(newMatrix(cv::Rect(0, 0, matrix.size().width, 
                                   matrix.size().height)));
      matrix.copyTo(m);
      matrix = newMatrix;
    }



    void diagonalise(const cv::Mat& X, cv::Mat& P, cv::Mat& D, cv::Mat& P1) {
      assert(X.type() == CV_32FC1 || X.type() == CV_64FC1);
      cv::Mat eigval, eigvec;
      eigen(X, eigval, eigvec);
      D = cv::Mat::diag(eigval);
      P = eigvec.t();
      P1 = P.inv(cv::DECOMP_SVD);
    }



    template<>
    cv::Mat_<float> meanVector(const cv::Mat_<float>& data) {
      cv::Mat_<float> meanVec(1, data.cols);
      for (int i = 0; i < data.cols; ++i)
        meanVec(0, i) = cv::mean(data.col(i))[0];
      
      return meanVec;
    }



    template<typename T, typename U>
    void copyElement(T& dst, const U& src);

    template<>
    void copyElement(cv::Vec3b& dst, const uchar& src) {
      dst[0] = dst[1] = dst[2] = src;
    }

    template<>
    void copyElement(cv::Vec3b& dst, const float& src) {
      dst[0] = dst[1] = dst[2] = src;
    }

    template<>
    void copyElement(cv::Vec3b& dst, const double& src) {
      dst[0] = dst[1] = dst[2] = src;
    }

    template<typename T, typename U>
    void copyMatrix(cv::Mat& dst, const cv::Mat& src) {
      for (int i = 0; i < src.rows; ++i) 
        for (int j = 0; j < src.cols; ++j) 
          copyElement<T,U>(dst.at<T>(i,j), src.at<U>(i,j));
    }




    void copyMatrix(cv::Mat& dst, const cv::Mat& src) {
      if (dst.size() != src.size())
        throw std::invalid_argument("Matrix size mismatch!");

      if (dst.type() == src.type())
        src.copyTo(dst);
      else if (dst.type() == CV_8UC3 && src.type() == CV_8UC1)
        copyMatrix<cv::Vec3b,uchar>(dst, src);
      else if (dst.type() == CV_8UC3 && src.type() == CV_64FC1)
        copyMatrix<cv::Vec3b,double>(dst, src);
      else if (dst.type() == CV_8UC3 && src.type() == CV_32FC1)
        copyMatrix<cv::Vec3b,float>(dst, src);
      else
        throw std::invalid_argument("Unsupported matrix type!");
    }



#if OPENCV_VERSION_231
    template<>
    double norm(const cv::Vec<unsigned char, 3>& M, int normType) {
      cv::Vec3i temp = M;
      return cv::norm(temp, normType);
    }
#endif



    void zScoreByColumn(const cv::Mat& in, cv::Mat& out, cv::Mat& means, cv::Mat& stddevs) {
      assert(in.type() == CV_32FC1);
      out = cv::Mat(in.size(), in.type(), cv::Scalar::all(0));
      means = cv::Mat(1, in.cols, CV_32FC1, cv::Scalar::all(0));
      stddevs = cv::Mat(1, in.cols, CV_32FC1, cv::Scalar::all(0));
      for (int i = 0; i < in.rows; ++i) {
        for (int j = 0; j < in.cols; ++j) 
          means.at<float>(0,j) += in.at<float>(i,j) / in.rows;
      }

      for (int i = 0; i < in.rows; ++i) {
        for (int j = 0; j < in.cols; ++j) {
          float v = in.at<float>(i,j) - means.at<float>(0,j);
          stddevs.at<float>(0,j) += v*v/in.rows;
        }
      }
      for (int j = 0; j < in.cols; ++j)
        stddevs.at<float>(0,j) = sqrt(stddevs.at<float>(0,j));

      for (int i = 0; i < in.rows; ++i)
        for (int j = 0; j < in.cols; ++j)
          out.at<float>(i,j) = (in.at<float>(i,j)-means.at<float>(0,j))/stddevs.at<float>(0,j);
    }



    const double PI = boost::math::constants::pi<double>();
    const double INFTY = std::numeric_limits<double>::infinity();



    double toRange(double x, double a, double b) {
      if (b < a)
        std::swap(a,b);
      double p = b-a;
      int displacement = std::max<double>(std::abs(x-a), std::abs(x-b))/p;
      if (x > b)
        return x - displacement * p;
      else
        return x + displacement * p;
    }



    double max(const cv::Mat& m) {
      double maxVal = 0;
      cv::minMaxLoc(m, nullptr, &maxVal);
      
      return maxVal;
    }

    double min(const cv::Mat& m) {
      double minVal = 0;
      cv::minMaxLoc(m, &minVal);
      
      return minVal;
    }



    cv::Mat generateDescriptorDmL2(const cv::Mat& X, const cv::Mat& Y) {
      assert(X.type() == CV_32FC1);
      assert(Y.type() == CV_32FC1);
      assert(X.cols == Y.cols);
      cv::Mat XYt = X * Y.t();
      cv::Mat dm;
      cv::sqrt(XYt, dm);
      return dm;
    }



    double chiSqDist(const cv::Mat& P, const cv::Mat& Q) {
      assert(P.type() == CV_32FC1);
      assert(Q.type() == CV_32FC1);
      assert(P.cols == Q.cols);
      assert(P.cols == 1 || P.rows == 1);
      assert(P.rows == Q.rows);

      cv::Mat temp1 = (P-Q).mul(P-Q);
      cv::Mat temp2 = (P+Q);
      cv::Mat temp3;
      cv::divide(temp1, temp2, temp3);
      double result = sum(temp3)[0] * 0.5;
      if (result < 0) {
        std::cerr << "P:" << std::endl
                  << P << std::endl;
        std::cerr << "Q:" << std::endl
                  << Q << std::endl;
        std::cerr << "temp1:" << std::endl
                  << temp1 << std::endl;
        std::cerr << "temp2:" << std::endl
                  << temp2 << std::endl;
        std::cerr << "temp3:" << std::endl
                  << temp3 << std::endl;
      }
        
      return result;
    }



    cv::Mat generateDescriptorDmChiSq(const cv::Mat& X, const cv::Mat& Y) {
      assert(X.type() == CV_32FC1);
      assert(Y.type() == CV_32FC1);
      assert(X.cols == Y.cols);

      cv::Mat dm(X.rows, Y.rows, CV_32FC1);
      float* p = dm.ptr<float>();
      for (int i = 0; i < X.rows; ++i) {
        for (int j = 0; j < Y.rows; ++j) {
          float distance = chiSqDist(X.row(i), Y.row(j));
          assert(distance >= 0);
          *p++ = distance;
        }
      }

      return dm;
    }



    /**
     * Computes directed Chamfer distance X -> Y defined as
     * 1/|X| sum_(x in X) min_(y in Y) ||x-y||
     *
     * @param xContour Binary contour image, type CV_32FC1, where values are 1 or 0
     * @param xSize Number of non-zero pixels
     * @param yDistanceTransform Distance transform of Y (each pixel location 
     *                           contains shortest distance to non-zero pixel), 
     *                           of type CV_32FC1
     */
    double directedChamferDistance(const cv::Mat& xContour,
                                   int xSize,
                                   const cv::Mat& yDistanceTransform) {
      assert(xContour.size() == yDistanceTransform.size());
      assert(yDistanceTransform.type() == CV_32FC1);
      assert(xContour.type() == CV_32FC1);
      if (xSize > 0)
        return cv::sum(xContour.mul(yDistanceTransform))[0]/xSize;
      else
        return DBL_MAX;
    }
  }
}
