#ifndef SLMOTION_MATH
#define SLMOTION_MATH

#include <opencv2/opencv.hpp>
#include <memory>
#include <stdexcept>
#include <numeric>

#define OPENCV_VERSION_242 (CV_MAJOR_VERSION == 2 && CV_MINOR_VERSION == 4 && CV_SUBMINOR_VERSION == 2)
#define OPENCV_VERSION_231 (CV_MAJOR_VERSION == 2 && CV_MINOR_VERSION == 3 && CV_SUBMINOR_VERSION == 1) 

static_assert((OPENCV_VERSION_242) != (OPENCV_VERSION_231) || 
              ((OPENCV_VERSION_242) == 0 &&
               (OPENCV_VERSION_231) == 0), "Invalid opencv versions!");


// this file contains mathematical helper functions

namespace slmotion {
  namespace math {
    /**
     * Returns an integer r in [0,k) such that n = m*k + r for some m in Z
     */
    inline int mod(int n, int k) {
      return ((n % k) + k) % k;
    }



    /**
     * Expands the given matrix to the new target size by filling it
     * with zeros to the right and down.
     */
    void expandMat(cv::Mat& matrix, const cv::Size& newSize);



    /**
     * Diagonalises the given matrix, such that
     * X = P D P^-1
     *   where D is diagonal
     *
     * @param X input matrix
     * @param P output P
     * @param D output diagonal
     * @param P1 output P^-1 
     */
    void diagonalise(const cv::Mat& X, cv::Mat& P, cv::Mat& D, cv::Mat& P1);



    /**
     * Inputs a data matrix where items are stored as row vectors, and
     * computes the corresponding mean row vector
     *
     * @param data the data matrix
     *
     * @return The corresponding mean row vector
     */
    template<typename T>
    cv::Mat_<T> meanVector(const cv::Mat_<T>& data);



    /**
     * Copies the content of a matrix of arbitrary type to another matrix, of the 
     * same size but possibly of different type
     */
    void copyMatrix(cv::Mat& dst, const cv::Mat& src);



    extern const double PI;
    extern const double INFTY;

    

    template<typename _Tp, int n>
    double norm(const cv::Vec<_Tp, n>& M, int normType) {
#if OPENCV_VERSION_242
      using namespace cv;
      return normType == NORM_INF ? (double)normInf<_Tp, _Tp>(M.val, n) :
        normType == NORM_L1 ? (double)normL1<_Tp, _Tp>(M.val, n) :
        std::sqrt((double)normL2Sqr<_Tp, _Tp>(M.val, n));
#else // OPENCV_VERSION_242
      return cv::norm(M, normType);
#endif // CV_MAJOR_VERSION == 2 && CV_MINOR_VERSION == 4 && CV_SUBMINOR_VERSION == 2
    }

#if OPENCV_VERSION_231
    template<>
    double norm(const cv::Vec<unsigned char, 3>& M, int normType);
#endif // OPENCV_VERSION_231



    /**
     * Computes z score by the column. Means and standard deviations
     * are stored as row vectors (number of columns equal to that of in).
     */
    void zScoreByColumn(const cv::Mat& in, cv::Mat& out, cv::Mat& means, cv::Mat& stddevs);



    const double CUBE_ROOT_DBL_EPSILON = std::cbrt(DBL_EPSILON);



    /**
     * Constructs the partial derivative of a function f : R^n -> R
     * f must accept double precision floating point column vectors and output a
     * single double
     *
     * Returns a function that numerically differentiates f wrt. variable i 
     * (zero-based)
     */
    template<typename T>
    class PartialDerivative {
    public:
      PartialDerivative (T f, int i, double minStep = CUBE_ROOT_DBL_EPSILON) : 
        f(f), i(i), minStep(minStep) {
      }      

      double operator()(const cv::Mat& x) {
        assert(x.rows > 0 && x.cols == 1 && x.type() == CV_64FC1);
        assert(i >= 0 && i < x.rows);
        cv::Mat deltaXi(x.rows, 1, CV_64FC1);
        double step = std::max(CUBE_ROOT_DBL_EPSILON*abs(x.at<double>(i)),minStep);
        for (int j = 0; j < deltaXi.rows; ++j)
          deltaXi.at<double>(j) = 0;
        deltaXi.at<double>(i) = step;
        double diffLh = f(x + deltaXi);
        double diffRh = f(x - deltaXi);
        double diff = (diffLh - diffRh);
        double val = diff / (2.0*step);
        return val;
      }



      PartialDerivative(const PartialDerivative<T>& that) : f(that.f), i(that.i), 
                                                            minStep(that.minStep) {}

      PartialDerivative& operator=(const PartialDerivative<T>& that) = delete;

    private:
      T f;
      int i;
      double minStep;
    };



    /**
     * A factory function (to avoid having to type template parameters
     * which may in some cases be rather obscure)
     */
    template<typename T>
    PartialDerivative<T> partialDerivative(T f, int i) {
      return PartialDerivative<T>(f,i);
    }



    /**
     * Constructs the Hessian matrix for the given function f : R^n -> R
     */
    template<typename T>
    class Hessian {
    public:
      typedef PartialDerivative<PartialDerivative<T> > second_derivative_t;

      /**
       * f = function R^n -> R (should be callable with cv::Mat argument and 
       *                        return a double)
       * n = dimensionality of input vectors
       * symmetric should be true if the function is expected to have continuous
       *           second derivatives everywhere where the matrix is evaluated
       */
      Hessian(T f, size_t n, bool symmetric = true) :
        symmetric(symmetric) {
        for (size_t i = 0; i < n; ++i) 
          for (size_t j = 0; j < n; ++j) 
            H.push_back(std::unique_ptr<second_derivative_t>(new second_derivative_t(partialDerivative(f, j), i)));
        assert(H.size() == n*n);
      }

      cv::Mat eval(const cv::Mat& x) {
        assert(x.type() == CV_64FC1);
        assert(x.cols == 1);
        assert(x.rows*x.rows == static_cast<int>(H.size()));

        int n = x.rows;
        cv::Mat evalH(n, n, CV_64FC1);
        if (symmetric) {
          for (int i = 0; i < n; ++i) {
            for (int j = i; j < n; ++j) {
              double val = (*H[n*i+j])(x);
              if (i == j)
                evalH.at<double>(i,j) = val;
              else
                evalH.at<double>(j,i) = evalH.at<double>(i,j) = val;
            }
          }
        }
        else {
          for (int i = 0; i < n; ++i) {
            for (int j = 0; j < n; ++j) {
              evalH.at<double>(i,j) = (*H[n*i+j])(x);
            }
          }
        }

        return evalH;
      }

    private:
      bool symmetric;
      std::vector<std::unique_ptr<PartialDerivative<PartialDerivative<T> > > > H;
    };



    /**
     * A factory function
     */
    template<typename T>
    Hessian<T> hessian(T f, size_t n, bool symmetric = true) {
      return Hessian<T>(f,n,symmetric);
    }



    /**
     * Constructs the Gradient vector for the given function f : R^n -> R
     */
    template <typename T>
    class Gradient {
    public:
      /**
       * f = function R^n -> R (should be callable with cv::Mat argument and 
       *                        return a double)
       * n = dimensionality of input vectors
       */
      Gradient(T f, size_t n, double minStep = CUBE_ROOT_DBL_EPSILON) {
        for (size_t i = 0; i < n; ++i) 
          G.push_back(std::unique_ptr<PartialDerivative<T> >(new PartialDerivative<T>(f, i, minStep)));
      }

      cv::Mat eval(const cv::Mat& x) {
        assert(x.type() == CV_64FC1);
        assert(x.cols == 1);
        assert(x.rows == static_cast<int>(G.size()));
        int n = x.rows;
        cv::Mat evalG(n, 1, CV_64FC1);
        for (int i = 0; i < n; ++i) 
          evalG.at<double>(i) = (*G[i])(x);

        return evalG;
      }

    private:
      std::vector<std::unique_ptr<PartialDerivative<T> > > G;
    };



    /**
     * A factory function
     */
    template<typename T>
    Gradient<T> gradient(T f, size_t n, double minStep = CUBE_ROOT_DBL_EPSILON) {
      return Gradient<T>(f,n, minStep);
    }



    /**
     * Returns true if x in [a,b]
     */
    inline bool inRange(double x, double a, double b) {
      return x >= a && x <= b;
    }



    /**
     * Forces the periodic value x into range [a,b]
     */
    double toRange(double x, double a, double b);

    /**
     * signum
     */
    inline double sgn(double x) {
      return x > 0 ? 1.0 : 
        x < 0 ? -1.0 :
        0.0;
    }



    /**
     * Given an N×M matrix of doubles, sorts the matrix in an ascending
     * order and returns an NM-vector of points that correspond to the
     * matrix elements.
     *
     * The template parameter T corresponds to element type
     */
    template<typename T>
    std::vector<cv::Point> sortMat(const cv::Mat& m) {
      assert(m.cols > 0 && m.rows > 0 && m.isContinuous());
      static_assert(std::is_same<T,float>::value ||
                    std::is_same<T,double>::value,
                    "Only float and double are supported");
      if (std::is_same<T,double>::value)
        assert(m.type() == CV_64FC1);
      if (std::is_same<T,float>::value)
        assert(m.type() == CV_32FC1);
      std::vector<const T*> ptrs(m.cols*m.rows);
      for (int i = 0; i < m.rows; ++i) {
        for (int j = 0; j < m.cols; ++j) {
          const T* e = &m.at<T>(i,j);
          assert(e == reinterpret_cast<const T*>(m.data) + i*m.cols + j);
          ptrs[i*m.cols + j] = e;
        }
      }
      std::sort(ptrs.begin(), ptrs.end(), [](const T* l, const T* r) {
          return *l < *r;
        });
      std::vector<cv::Point> pts(ptrs.size());
      for (size_t i = 0; i < ptrs.size(); ++i) {
        ptrdiff_t diff = ptrs[i] - reinterpret_cast<const T*>(m.data);
        pts[i] = cv::Point(diff % m.cols, diff / m.cols);
      }
      return pts;
    }



    /**
     * Computes the sum of all elements of the matrix
     */
    template<typename T>
    T sum(const cv::Mat& m) {
      static_assert(std::is_same<T,float>::value ||
                    std::is_same<T,double>::value,
                    "Only float and double are supported");
      int mType = 0;
      if (std::is_same<T,float>::value)
        mType = CV_32FC1;
      else if (std::is_same<T,double>::value)
        mType = CV_64FC1;

      if (m.type() != mType)
        throw std::invalid_argument("The matrix has invalid element type!");
      
      cv::Mat temp1;
      cv::Mat temp2;
      cv::reduce(m, temp1, 0, CV_REDUCE_SUM);
      cv::reduce(temp1, temp2, 1, CV_REDUCE_SUM);
      
      return temp2.at<T>(0);
    }
    


    /**
     * Returns the maximal element
     */
    double max(const cv::Mat& m);



    /**
     * Returns the minimum element
     */
    double min(const cv::Mat& m);



    /**
     * Computes directed Chamfer distance X -> Y defined as
     * 1/|X| sum_(x in X) min_(y in Y) ||x-y||
     *
     * @param xContour Binary image, type CV_32FC1, where values are 1 or 0
     * @param xSize Number of non-zero pixels
     * @param yDistanceTransform Distance transform of Y (each pixel location 
     *                           contains shortest distance to non-zero pixel), 
     *                           of type CV_32FC1
     */
    double directedChamferDistance(const cv::Mat& X,
                                   int XSize,
                                   const cv::Mat& yDistanceTransform);



    /**
     * Computes undirected chamfer distance, defined as the sum of
     * directed chamfer distances in both directions.
     *
     * Both X and Y are expected to be CV_32FC1 with 0 and 1 entries,
     * sizes corresponding to the count of such entries,
     * and distance transform defined as the distance to the closest non-zero entry.
     */
    inline double chamferDistance(const cv::Mat& X,
                                  int xSize,
                                  const cv::Mat& XDistanceTransform,
                                  const cv::Mat& Y,
                                  int ySize,
                                  const cv::Mat& YDistanceTransform) {
      return directedChamferDistance(X, xSize, YDistanceTransform) +
        directedChamferDistance(Y, ySize, XDistanceTransform);
    }



    template<typename T>
    T emdHat(const cv::Mat& P, const cv::Mat& Q, const cv::Mat& D,
             double alpha = 1.0) {
      static_assert(std::is_same<T,float>::value ||
                    std::is_same<T,double>::value,
                    "Only float and double are supported");
      int N = std::max(P.rows, P.cols);
      if (std::min(P.rows, P.cols) != 1)
        throw std::invalid_argument("Only vectors can be processed, but P is "
                                    "not a vector!");

      int M = std::max(Q.rows, Q.cols);
      if (std::min(Q.rows, Q.cols) != 1)
        throw std::invalid_argument("Only vectors can be processed, but Q is "
                                    "not a vector!");

      if (D.rows != N || D.cols != M) {
        std::ostringstream oss;
        oss << "Distance matrix is of invalid size! An " << N << "x" << M 
            << " matrix was expected, but got a " << D.rows << "x" 
            << D.cols << " matrix.";

        throw std::invalid_argument(oss.str());
      }
                                    
      int mType = 0;
      if (std::is_same<T,float>::value)
        mType = CV_32FC1;
      else if (std::is_same<T,double>::value)
        mType = CV_64FC1;

      if (P.type() != mType)
        throw std::invalid_argument("P has invalid element type!");
      if (Q.type() != mType)
        throw std::invalid_argument("Q has invalid element type!");
      if (D.type() != mType)
        throw std::invalid_argument("The distance matrix has invalid element "
                                    "type!");

      T sumP = sum<T>(P);
      T sumQ = sum<T>(Q);
      T maxD = max(D);

      cv::Mat flowMat(D.size(), mType, cv::Scalar::all(0));

      float emd = cv::EMD(P, Q, CV_DIST_USER, D, nullptr, flowMat);

      // T sumF = math::sum<T>(flowMat);
      T result = emd * std::min(sumP, sumQ) + alpha * std::abs(sumP - sumQ) * maxD;
      return result;
    }



    /**
     * Given two descriptor matrices where rows correspond to
     * descriptor vectors, computes a distance matrix using the
     * Euclidean norm as ground distance.
     *
     * I.e. the entry (i,j) corresponds to the euclidean distance
     * between vectors X[i] and Y[j]
     */
    cv::Mat generateDescriptorDmL2(const cv::Mat& X, const cv::Mat& Y);



    /**
     * Given two histograms of equal length, computes the square chi
     * distance between them.
     */
    double chiSqDist(const cv::Mat& P, const cv::Mat& Q);



    /**
     * Given two descriptor matrices where rows correspond to
     * descriptor vectors, computes a distance matrix using the
     * suqare Chi as ground distance.
     *
     * I.e. the entry (i,j) corresponds to the square chi distance
     * between vectors X[i] and Y[j]
     */
    cv::Mat generateDescriptorDmChiSq(const cv::Mat& X, const cv::Mat& Y);

  }
}

#endif // SLMOTION_MATH
