#include "CsvToEafConverter.hpp"
#include "slmotion.hpp"
#include <iostream>
#include <fstream>
#include <boost/lexical_cast.hpp>

/*
 * This little program performs conversions on the data as generated by
 * slmotion
 */


namespace slmotion {
  extern int debug;
}



void printHelp() {
  std::cerr << "usage: slconvert [options] --csv-to-eaf <infile.csv> <infile.ann> <outfile.eaf>"
            << std::endl << std::endl
            << "The program creates an ELAN annotation file from the source CSV file (which"
            << std::endl
            << "should be created by running the slmotion program). Annotation templates are"
            << std::endl
            << "read from the file set by the second input parameter. If a file with the output"
            << std::endl
            << "filename already exists, it is considered an EAF file and its validity will be"
            << std::endl
            << "checked. If it is found a valid EAF file, the resulting annotations will be"
            << std::endl
            << "added to it as separate tiers. Otherwise, a new file is created." 
            << std::endl << std::endl
            << "Options:" << std::endl
            << "--help, -h" << std::endl
            << "  Prints this text" << std::endl
            << "--debug n" << std::endl
            << "  Enables debug output at debug level n" << std::endl
            << "--media-name <string>" << std::endl
            << "  Sets the media filename for creating media descriptor entries" << std::endl
            << std::endl << std::endl
            << "Version information:" << std::endl
            << slmotion::getVersions() << std::endl;
}



/**
 * Creates annotations into an EAF file from input CSV file, using input
 * annotation templates
 * 
 * @param inCsv Input CSV data file
 * @param inAnnTemp Input annotation template file
 * @param outEaf Output EAF filename
 * @param mediaFilename Output media filename for creating media descriptor
 * entries
 */
void csvToEaf(const std::string& inCsv, const std::string& inAnnTemp,
	      const std::string& outEaf, const std::string& mediaFilename) {
  slmotion::CsvToEafConverter csvToEafConverter;
  csvToEafConverter.convert(inCsv, inAnnTemp, outEaf, mediaFilename);
}



/**
 * Reads in an Eaf file, and converts annotations into CSV format, marking
 * annotated frames with a non-zero number.
 */
void eafToCsv(const std::string& inEaf, const std::string& outCsv) {
  slmotion::EafDocument ed(inEaf);
  slmotion::tier_deque_t td = ed.getTiers();

  std::ofstream ofs(outCsv);

  int numberOfFrames = ed.getLastTimecode() * 25.0 / 1000.0;

  // Store the values in a "matrix", number of frames times number of tiers
  std::vector<int> matrix(numberOfFrames * td.size(), 0);

  for (size_t col = 0; col < td.size(); ++col) {
    for (size_t annNum = 1; annNum <= td[col].second.size(); ++annNum)
      for (size_t row =
             td[col].second[annNum-1].timerefs.first * 25.0 / 1000.0;
           row < td[col].second[annNum-1].timerefs.second* 25.0 / 1000.0;
           ++row)
        matrix[row*td.size() + col] = annNum;
  }

  for (size_t col = 0; col < td.size(); ++col) {
    ofs << td[col].first;
    if (col + 1 < td.size())
      ofs << ",";
    else
      ofs << "\n";
  }

  for (int row = 0; row < numberOfFrames; ++row)
    for (size_t col = 0; col < td.size(); ++col) {
      ofs << matrix[row*td.size() + col];
      if (col + 1 < td.size())
        ofs << ",";
      else
        ofs << "\n";
    }
}



int main(int argc, char** argv) {
  std::vector<std::string> filenames;
  bool optAllowed = true; // set false after a --
  std::string mediaName; // Filename for creating media descriptor entries

  enum Mode {
    CSV2EAF,
    EAF2CSV,
    HELP
  } mode = HELP;

  if (argc < 2) {
    printHelp();
    return 0;
  }

  // parse parameters
  for (int i = 1; i < argc; i++) {
    std::string s(argv[i]);
    if (optAllowed && s.at(0) == '-') {
      if (s == "--csv-to-eaf")
	mode = CSV2EAF;
      else if (s == "--eaf-to-csv")
        mode = EAF2CSV;
      else if (s == "--debug") {
	if (i + 1 < argc) {
	  i++;
	  slmotion::debug = boost::lexical_cast<int>(argv[i]);
	}
	else
	  slmotion::debug = 1;
      }
      else if (s == "--media-name") {
	if (i + 1 < argc) {
	  i++;
	  mediaName = argv[i];
	}
	else {
          std::cerr << "A filename is expected after --media-name" << std::endl;
	  return -1;
	}
      }
      else if (s == "--help" ||
	       s == "-h") {
	printHelp();
	return 0;
      }
      else if (s == "--version") {
        std::cout << slmotion::getVersions() << std::endl;
        return 0;
      }
      else if (s == "--")
	optAllowed = false;
      else {
        std::cerr << "slconvert: Unknown option: " << s << std::endl;
	return -1;
      }
    }
    else 
      filenames.push_back(s);
  }

  switch(mode) {
  case HELP:
    printHelp();
    return 0;
  
  case CSV2EAF:
    if (filenames.size() == 3) {

      try {
	csvToEaf(filenames[0], filenames[1], filenames[2], mediaName);
      }
      catch(slmotion::ConvertException& e) {
        std::cerr << "slconvert: " << e.what() << std::endl;
	return -1;
      }
      return 0;
    }
    else {
      std::cerr << "slconvert: --csv-to-eaf specified, but a wrong number of filenames was provided." << std::endl;
      return -1;
    }
  
  case EAF2CSV:
    if (filenames.size() == 2) {
      try {
        eafToCsv(filenames[0], filenames[1]);
      }
      catch (slmotion::ConvertException& e) {
        std::cerr << "slconvert: " << e.what() << std::endl;
        return -1;
      }
      return 0;
    }
    else {
      std::cerr << "slconvert: --eaf-to-csv specified, but a wrong number of filenames was provided." << std::endl;
    }
  }
}
