#include "slmotion.hpp"
#include "CliCallback.hpp"

#ifdef SLMOTION_ENABLE_PICSOM
#include <PicSOM.h>
#include <Feature.h>
#endif // SLMOTION_ENABLE_PICSOM

#include <sstream>
#include <opencv2/opencv.hpp>
#include <libxml/parser.h>
#include <boost/python.hpp>
#include "util.hpp"
#include "SLIO.hpp"
#include "Analyser.hpp"
#include "configuration.hpp"
#include "pythonapi.hpp"
#include "AnalysisController.hpp"

#ifdef __APPLE__
#include <Availability.h>
#endif


using std::cerr;
using std::vector;
using std::endl;
using std::string;
using std::cout;
using namespace slmotion::configuration;
namespace po = boost::program_options;



namespace slmotion {
  int debug = 0;

  bool paranoid = false;



  static std::string getCompiler() {
    std::ostringstream ss;
#ifdef __clang__
    ss << "CLang " << __clang_major__ << "." << __clang_minor__ << "." << __clang_patchlevel__;
#elif __GNUC__
    ss << "GCC " << __GNUC__ << "." << __GNUC_MINOR__ << "." << __GNUC_PATCHLEVEL__;
#elif _MSC_VER == 1700 
    ss << "Visual Studio 2012";
#elif _MSC_VER == 1600 
    ss << "Visual Studio 2010";
#elif _MSC_VER > 0
    ss << "Visual Studio (unknown version)";
#else 
    ss << "an unknown compiler";
#endif
    return ss.str();
  }



  static std::string getOs() {
    std::ostringstream ss;
#ifdef __ANDROID__
    ss << "Android";
#elif __FreeBSD__
    ss << "FreeBSD";
#elif __NetBSD__
    ss << "NetBSD";
#elif __OpenBSD__
    ss << "OpenBSD";
#elif __DragonFly__
    ss << "DragonFly BSD";
#elif __CYGWIN__
    ss << "Cygwin";
#elif __linux__
    ss << "Linux";
#elif __APPLE__ && __MAC_10_8
    ss << "MacOS X 10.8";
#elif __APPLE__ && __MAC_10_7
    ss << "MacOS X 10.7";
#elif __APPLE__ && __MAC_10_6
    ss << "MacOS X 10.6";
#elif __APPLE__
    ss << "MacOS X (unknown version)";
#elif __sun
    ss << "Sun Solaris";
#elif __unix__
    ss << "Unknown UNIX-compliant OS";
#elif _WIN64
    ss << "Microsoft Windows (64-bit)";
#elif _WIN32
    ss << "Microsoft Windows (32-bit)";     
#else
    ss << "Completely unknown obscure OS";
#endif
    return ss.str();
  }



  static std::string getArch() {
#ifdef __x86_64__
    return "amd64";
#elif __arm__
    return "arm";
#elif __i686__
    return "i686";
#elif __i386__
    return "i386";
#elif __ppc__
    return "ppc";
#else
    return "Unknown architecture";
#endif
  }



  std::string getVersions() {
    std::ostringstream ss;
    ss << SLMOTION_VCID << std::endl;
    ss << "Compiled with " << getCompiler() << " on " << getOs() << " " 
       << getArch() << std::endl
       << "Libraries: " << std::endl;
#ifdef SLMOTION_ENABLE_PICSOM
    ss << "PicSOM.C " << picsom::PicSOM::PicSOM_C_Version() << endl;
#endif // SLMOTION_ENABLE_PICSOM
    ss << "OpenCV " << CV_VERSION << std::endl
       << "libxml2 " << LIBXML_VERSION_STRING[0] << ".";
    if (LIBXML_VERSION_STRING[1] != '0')
      ss << LIBXML_VERSION_STRING[1];
    ss << LIBXML_VERSION_STRING[2] << ".";
    if (LIBXML_VERSION_STRING[3] != '0')
      ss << LIBXML_VERSION_STRING[3];
    ss << LIBXML_VERSION_STRING[4];
    ss << std::endl << "Python " << Py_GetVersion() << std::endl;
    return ss.str();
  }


  int main(int argc, char* argv[], std::shared_ptr<BlackBoard> globalBlackBoard, 
           UiCallback* uiCallback) {
    storeExecutableName(argv[0]);

#ifdef SLMOTION_ENABLE_PICSOM
    if (argc>1 && strlen(argv[1])==2 && *argv[1]=='-' &&
        std::string(argv[1]+1).find_first_of("fsrp")==0)
      return picsom::PicSOM::Main(argc, (const char**)argv);
#endif // SLMOTION_ENABLE_PICSOM

    slmotion::setArgV(argc, argv);
    // initialise the default random number generator (e.g. for K-means)
    cv::theRNG().state = time(NULL);

    // these display formats are not widely used, they are inteded for possibly
    // altering textual output to an ELAN compatible format in the future
    SLIO::setDebugFormat("", cerr);
    SLIO::setWarningFormat("slmotion: WARNING: ", cerr);
    SLIO::setInfoFormat("", cerr);

    vector<Analyser> analysers;
    SLMotionConfiguration opts;
    vector<Job> jobs;
    po::variables_map vm; 

    // Parse command line arguments using libbboost program options
    try {  
      vm = slmotion::configuration::parseCommandLine(argc, argv);

      // start by setting current directory
      if (vm.count("work-dir"))
        setWorkingDirectory(vm["work-dir"].as<string>());

      // input video filenames
      if (vm.count("inputNames")) {
        const vector<string> fnames = vm["inputNames"].as<vector<string>>();
        string dbname;
        string picsomArgs;
#ifdef SLMOTION_ENABLE_PICSOM
        dbname = vm.count("db") ? vm["db"].as<string>() : "";
        picsomArgs = vm.count("picsom-args") ? vm["picsom-args"].as<string>() 
          : "";
#endif // SLMOTION_ENABLE_PICSOM
        if (vm.count("kinect"))
          jobs = assignFilenamesToKinectJobs(fnames);
        else
          jobs = assignFilenamesToJobs(fnames, dbname, picsomArgs);
      }
      else 
        jobs = vector<Job> { Job { Job::JobType::DUMMY, vector<string>() }  };

      // configuration file names
      vector<string> conffiles;
      if (vm.count("config"))
        conffiles = move(vm["config"].as<vector<string>>());

      if (vm.count("version")) {
        cout << slmotion::getVersions() << endl;
        return 0;
      }

      if (vm.count("components") && 
          ((vm["components"].as<string>() == "?") ||
           (vm["components"].as<string>() == "help"))) {
        cout << "Known components: " << endl;
        cout << Component::getRegisteredComponents(80) << endl;
        return 0;
      }

      if (vm.count("generate-default-config")) {
        generateDefaultConfig(std::cout, 80);
        return 0;
      }

#ifdef SLMOTION_ENABLE_PICSOM
      if (vm.count("picsom-features")) {
        std::cout << "Known PicSOM features:" << std::endl;
        picsom::Feature::PrintFeatureList(cout, true);
        return 0;
      }
      if (vm.count("picsom-segmentations")) {
        std::cout << "Known PicSOM segmentations:" << std::endl;
        picsom::Segmentation::ListMethods(cout, false);
        return 0;
      }
#endif // SLMOTION_ENABLE_PICSOM


      if (argc == 1 || vm.count("help") || jobs.size() == 0) {
        cout << "usage: " << argv[0] << " [options] <file1> [file2] ..." << endl;
        cout << "Input files may be any ordinary image or video files. All still "
             << "image files are" << endl << "treated as if coming from the same image "
             << "sequence." << endl << endl;
        cout << getVisibleOptionsDescription() << endl;
        cout << endl << "Versions:" << endl 
             << slmotion::getVersions() << endl;

        return 0;
      }
    
      if (jobs.size() != conffiles.size() && conffiles.size() > 1)
        throw CommandLineException("There must be an equal amount of "
                                   "configuration files and input jobs. "
                                   "Otherwise, there can be only one "
                                   "(global) configuration file.");

      assert(conffiles.size() == jobs.size() || conffiles.size() <= 1);

      createAnalysers(jobs, conffiles, analysers, opts, argc, argv);
    }
    catch (ConfigurationFileException& e) {
      cerr << "slmotion: Invalid configuration file supplied: " << e.what()
           << endl;
      return -1;
    }
    catch (CommandLineException& e) {
      cerr << "slmotion: Invalid command line parameters specified: " <<
        e.what() << "\n";
      return -1;
    }
    catch (std::exception& e) {
      cerr << "slmotion: An unhandled exception was thrown while processing configuration: " << e.what() << "\n";
      return -1;
    }
    catch (...) {
      cerr << "slmotion: An unhandled exception, not derived from std::exception, was thrown while processing configuration.\n";
      return -1;
    }

    std::unique_ptr<UiCallback> cliCallback;
    if (uiCallback == nullptr) {
      cliCallback.reset(new CliCallback(std::cerr));
      uiCallback = cliCallback.get();
    }

    try {
      if (opts.script.length() > 0) {
        slmotion::python::PythonEnvironment pyEnv(&jobs, &opts, &vm, 
                                                  analysers.front().getFrameSourcePtr(),
                                                  globalBlackBoard,
                                                  uiCallback);
        pyEnv.processScript(opts.script);
      }
      else {
        AnalysisController controller(analysers.front().getVisualiser(), 
                                      analysers.front().getSlio(), 
                                      opts.pauseMode);
        assert(analysers.size() == 1);
        controller.analyseVideofiles(analysers.front(), opts.skipFrames, 
                                     opts.framesMax,
                                     opts.storeImages,
                                     // opts.disableAnnotationBar,
                                     opts.elanOutputFilename, 
                                     opts.annotationTemplateFilename,
                                     opts.inputFilename,
                                     // opts.visualiserCombinationStyle,
                                     nullptr,
                                     uiCallback);
      }
    }
    catch (slmotion::SLIOException& e) {
      cerr << "slmotion: ERROR: an exception occurred while performing an I/O " \
        "operation: " << e.what() << endl;
      return -1;
    }
    catch (std::invalid_argument& e) {
      cerr << "slmotion: ERROR:  " << e.what() << endl;
    
      return -1;
    }
    catch (std::exception& e) {
      cerr << "slmotion: ERROR: " << e.what() << std::endl;
      return -1;
    }
    catch (boost::python::error_already_set&) {
      cerr << "slmotion: ERROR: Python script processing was terminated because"
           << " of an error on the Python side." << std::endl;
      return -1;
    }
    catch(...) {
      cerr << "slmotion: ERROR: An unhandled exception of unknown type occurred." << std::endl;
      return -1;
    }

    // if (cliCallback.get())
    //   std::cerr << std::endl;

    return 0;
  }
}

