// this is a part of tests.cpp (the unit tests file) which has grown huge

BOOST_AUTO_TEST_CASE(testTrackedPoint1) {
  typedef TrackedPoint tp;
  unsigned long long int id0 = tp::uniqueIdCounter;
  BOOST_REQUIRE(id0 == 1);

  tp::point_t p1(1,2);
  tp::point_t p2(3,4);
  tp::point_t p3(5,6);

  tp tp2(p1);
  tp tp3(p2);
  tp tp4(p3);
  BOOST_CHECK(tp::uniqueIdCounter == id0 + 3);
  BOOST_CHECK(tp2.id == id0 + 0);
  BOOST_CHECK(tp3.id == id0 + 1);
  BOOST_CHECK(tp4.id == id0 + 2);
  BOOST_CHECK(tp2 != tp3);
  BOOST_CHECK(tp3 != tp4);

  tp tp1 = tp4;
  BOOST_CHECK(tp1 == tp4);
  BOOST_CHECK(tp1 != tp2);
  BOOST_CHECK(tp::uniqueIdCounter == id0 + 3);
  
  tp2 = tp(0, p3, BodyPart::UNKNOWN);
  BOOST_CHECK(tp::uniqueIdCounter == id0 + 3);
  BOOST_CHECK(tp1 != tp2);
  BOOST_CHECK(tp2.id == 0);

  BOOST_CHECK(tp1.location == p3);
  BOOST_CHECK(tp2.location == p3);
  BOOST_CHECK(tp3.location == p2);
  BOOST_CHECK(tp4.location == p3);

  BOOST_CHECK(tp2 < tp1);
  BOOST_CHECK(tp2 < tp4);
  BOOST_CHECK(!(tp1 < tp4));
  BOOST_CHECK(tp3 < tp1);

  BOOST_CHECK(static_cast<tp::point_t>(tp1) == p3);
  BOOST_CHECK(static_cast<tp::point_t>(tp2) == p3);
  BOOST_CHECK(static_cast<tp::point_t>(tp3) == p2);
  BOOST_CHECK(static_cast<tp::point_t>(tp4) == p3);
}

BOOST_AUTO_TEST_CASE( testMath ) {
  using namespace slmotion::math;
  using namespace slmotion::random;
  for (double d : unifrnd<double>(-PI, PI, 100)) {
    BOOST_REQUIRE(toRange(d, -PI, PI) == d);
  }

  for (double d : unifrnd<double>(-3.0*PI, -2.0*PI, 100)) {
    BOOST_REQUIRE(toRange(d, -PI, PI) == d + 2.0*PI);
  }

  for (double d : unifrnd<double>(5.0*PI, 6.0*PI, 100)) {
    BOOST_REQUIRE(toRange(d, -PI, PI) == d - 6.0*PI);
  }
}

BOOST_AUTO_TEST_CASE( testRandom ) {
  uint64_t seed = time(nullptr);
  std::mt19937_64 eng(seed);
  slmotion::random::seed(seed);
  BOOST_REQUIRE(eng.min() == slmotion::random::min());
  BOOST_REQUIRE(0 == slmotion::random::min());
  BOOST_REQUIRE(eng.max() == slmotion::random::max());
  BOOST_REQUIRE(~((uint64_t)0) == slmotion::random::max());
  for (int i = 0; i < 19937; ++i)
    BOOST_REQUIRE(eng() == slmotion::random::rand());

  int a = rand() % 100 - 50;
  int b = rand() % 100 - 50;
  std::uniform_int_distribution<int> d1(a,b);
  for (int i = 0; i < 19937; ++i)
    BOOST_REQUIRE(d1(eng) == slmotion::random::unifrnd<int>(a,b));

  std::vector<int> v1(19937);
  for (int i = 0; i < 19937; ++i)
    v1[i] = d1(eng);

  std::vector<int> v2 = slmotion::random::unifrnd<int>(a,b,19937);
  BOOST_REQUIRE(v1 == v2);

  cv::Mat m1(512, 512, CV_32SC1);
  for (int i = 0; i < 512; ++i)
    for (int j = 0; j < 512; ++j)
      m1.at<int>(i,j) = d1(eng);

  cv::Mat m2 = slmotion::random::unifrnd<int>(a,b,512,512);
  BOOST_REQUIRE(m1.type() == m2.type());
  BOOST_REQUIRE(m1.cols == m2.cols);
  BOOST_REQUIRE(m1.rows == m2.rows);
  for (int i = 0; i < 512; ++i)
    for (int j = 0; j < 512; ++j)
      BOOST_REQUIRE(m1.at<int>(i,j) == m2.at<int>(i,j));

  std::uniform_real_distribution<float> d2(a,b);
  for (int i = 0; i < 512; ++i)
    BOOST_REQUIRE(d2(eng) == slmotion::random::unifrnd<float>(a,b));

  std::vector<float> v3(19937);
  for (int i = 0; i < 19937; ++i)
    v3[i] = d2(eng);

  std::vector<float> v4 = slmotion::random::unifrnd<float>(a,b,19937);
  BOOST_REQUIRE(v3 == v4);

  cv::Mat m3(512, 512, CV_32FC1);
  for (int i = 0; i < 512; ++i)
    for (int j = 0; j < 512; ++j)
      m3.at<float>(i,j) = d2(eng);

  cv::Mat m4 = slmotion::random::unifrnd<float>(a,b,512,512);
  BOOST_REQUIRE(m3.type() == m4.type());
  BOOST_REQUIRE(m3.cols == m4.cols);
  BOOST_REQUIRE(m3.rows == m4.rows);
  for (int i = 0; i < 512; ++i)
    for (int j = 0; j < 512; ++j)
      BOOST_REQUIRE(m3.at<float>(i,j) == m4.at<float>(i,j));

  std::uniform_real_distribution<double> d3(a,b);
  for (int i = 0; i < 19937; ++i)
    BOOST_REQUIRE(d3(eng) == slmotion::random::unifrnd<double>(a,b));

  std::vector<double> v5(19937);
  for (int i = 0; i < 19937; ++i)
    v5[i] = d3(eng);

  std::vector<double> v6 = slmotion::random::unifrnd<double>(a,b,19937);
  BOOST_REQUIRE(v5 == v6);

  cv::Mat m5(512, 512, CV_64FC1);
  for (int i = 0; i < 512; ++i)
    for (int j = 0; j < 512; ++j)
      m5.at<double>(i,j) = d3(eng);

  cv::Mat m6 = slmotion::random::unifrnd<double>(a,b,512,512);
  BOOST_REQUIRE(m5.type() == m6.type());
  BOOST_REQUIRE(m5.cols == m6.cols);
  BOOST_REQUIRE(m5.rows == m6.rows);
  for (int i = 0; i < 512; ++i)
    for (int j = 0; j < 512; ++j)
      BOOST_REQUIRE(m5.at<double>(i,j) == m6.at<double>(i,j));
}



BOOST_AUTO_TEST_CASE( testAdvancedString ) {
  srand(time(nullptr));
  AdvancedString as;
  BOOST_REQUIRE(as == "");
  BOOST_REQUIRE(as == std::string());
  std::string s = as;
  BOOST_REQUIRE(s == "");
  BOOST_REQUIRE(s == as);
  BOOST_REQUIRE("" == as);
  BOOST_REQUIRE(std::string() == as);
  BOOST_REQUIRE(as == s);
  BOOST_REQUIRE(as.c_str()[0] == '\0');
  BOOST_REQUIRE(as.length() == 0);
  BOOST_REQUIRE(as.size() == 0);
  BOOST_REQUIRE(as.begin() == as.end());
  BOOST_REQUIRE(as.cbegin() == as.cend());
  BOOST_REQUIRE(as.rbegin() == as.rend());
  BOOST_REQUIRE(as.crbegin() == as.crend());

  std::string rs = getRandomString(20);
  BOOST_REQUIRE(AdvancedString(rs) == rs);
  size_t pos = rand() % 10 + 1;
  size_t len = rand() % 7 + 1;
  BOOST_REQUIRE(AdvancedString(rs, pos, len) == std::string(rs, pos, len));
  BOOST_REQUIRE(AdvancedString(rs.c_str()) == rs);
  BOOST_REQUIRE(AdvancedString(rs.c_str(), len) == std::string(rs.c_str(), 
                                                               len));

  std::vector<char> randChars(11);
  for (int i = 0; i < 10; ++i)
    randChars[i] = 'a' + rand() % ('z'-'a'+1);
  randChars[10] = '\0';
  BOOST_REQUIRE(AdvancedString(randChars.cbegin(), randChars.cend()) == 
                std::string(randChars.cbegin(), randChars.cend()));

  as = AdvancedString();
  BOOST_REQUIRE(as != rs);
  BOOST_REQUIRE(rs != as);
  as = AdvancedString(rs);
  BOOST_REQUIRE(as != "");
  BOOST_REQUIRE("" != as);
  BOOST_REQUIRE(as == rs);
  BOOST_REQUIRE(rs == as);
  BOOST_REQUIRE(as == AdvancedString(rs));
  BOOST_REQUIRE(AdvancedString(rs) == as);

  as = AdvancedString();
  BOOST_REQUIRE(as != rs);
  BOOST_REQUIRE(rs != as);
  as = rs;
  BOOST_REQUIRE(as != "");
  BOOST_REQUIRE("" != as);
  BOOST_REQUIRE(as == rs);
  BOOST_REQUIRE(rs == as);
  BOOST_REQUIRE(as == AdvancedString(rs));
  BOOST_REQUIRE(AdvancedString(rs) == as);

  as = AdvancedString();
  BOOST_REQUIRE(as != rs);
  BOOST_REQUIRE(rs != as);
  as = rs.c_str();
  BOOST_REQUIRE(as != "");
  BOOST_REQUIRE("" != as);
  BOOST_REQUIRE(as == rs);
  BOOST_REQUIRE(rs == as);
  BOOST_REQUIRE(as == AdvancedString(rs));
  BOOST_REQUIRE(AdvancedString(rs) == as);

  std::string s2;
  BOOST_REQUIRE(s2 != rs);
  s2 = as;
  BOOST_REQUIRE(s2 == rs);

  for (auto it = as.cbegin(), jt = rs.cbegin(); it != as.cend(); ++it, ++jt) 
    BOOST_REQUIRE(*it == *jt);
  for (auto it = as.crbegin(), jt = rs.crbegin(); it != as.crend(); ++it, ++jt) 
    BOOST_REQUIRE(*it == *jt);

  std::string rs2 = getRandomString(20);
  AdvancedString as2(rs2);

  BOOST_REQUIRE(as + as2 == AdvancedString(rs + rs2));
  BOOST_REQUIRE(as + as2 == rs + rs2);
  BOOST_REQUIRE(as + rs2 == rs + rs2);
  BOOST_REQUIRE(rs + as2 == rs + rs2);
  BOOST_REQUIRE(as + rs2.c_str() == rs + rs2);
  BOOST_REQUIRE(rs.c_str() + as2 == rs + rs2);

  s = as2;
  BOOST_REQUIRE(s == as2);
  BOOST_REQUIRE(strcmp(s.c_str(), as2.c_str()) == 0);

  int i1 = 123456;
  BOOST_REQUIRE(AdvancedString(i1) == "123456");
  int i2 = -654321;
  BOOST_REQUIRE(AdvancedString(i2) == "-654321");
  double d1 = 3.141;
  BOOST_REQUIRE(AdvancedString(d1) == "3.141");
  double d2 = -3.141;
  BOOST_REQUIRE(AdvancedString(d2) == "-3.141");
  for (int i = 0; i < 25; ++i) {
    double d = slmotion::random::unifrnd<double>(-DBL_MAX,DBL_MAX);
    BOOST_REQUIRE(AdvancedString(d) == boost::lexical_cast<std::string>(d));
    d = slmotion::random::unifrnd<double>(DBL_MIN,DBL_MAX);
    BOOST_REQUIRE(AdvancedString(d) == boost::lexical_cast<std::string>(d));
    d = slmotion::random::unifrnd<double>(0,1);
    BOOST_REQUIRE(AdvancedString(d) == boost::lexical_cast<std::string>(d));

    float f = slmotion::random::unifrnd<float>(-FLT_MAX, FLT_MAX);
    BOOST_REQUIRE(AdvancedString(f) == boost::lexical_cast<std::string>(f));
    f = slmotion::random::unifrnd<float>(FLT_MIN,FLT_MAX);
    BOOST_REQUIRE(AdvancedString(f) == boost::lexical_cast<std::string>(f));
    f = slmotion::random::unifrnd<float>(0,1);
    BOOST_REQUIRE(AdvancedString(f) == boost::lexical_cast<std::string>(f));
  }
  unsigned long long u1 = 987654321;
  BOOST_REQUIRE(AdvancedString(u1) == "987654321");
  BOOST_REQUIRE(AdvancedString(true) == "1");
  BOOST_REQUIRE(AdvancedString(false) == "0");
  char c = 'a' + rand() % ('z' - 'a' + 1);
  BOOST_REQUIRE(AdvancedString(c).length() == 1);
  BOOST_REQUIRE(AdvancedString(c)[0] == c);

  as = "314159";
  BOOST_REQUIRE(static_cast<unsigned int>(as) == 314159);
  as = "-8264";
  BOOST_REQUIRE(static_cast<short>(as) == -8264);
  as = "1.234";
  BOOST_REQUIRE(static_cast<double>(as) == 1.234);  
  as = "-2.2e16";
  BOOST_REQUIRE(abs((static_cast<float>(as) - -2.2e16) / -2.2e16) < 1e5);

  as = "";
  BOOST_REQUIRE(as.empty());
  as = rs;
  BOOST_REQUIRE(!as.empty());
  BOOST_REQUIRE(as.length() == rs.length());
  BOOST_REQUIRE(as.size() == rs.size());
  for (size_t i = 0; i < as.length(); ++i)
    BOOST_REQUIRE(as[i] == rs[i]);
  BOOST_REQUIRE(as.front() == rs[0]);
  BOOST_REQUIRE(as.back() == rs[as.length()-1]);
  as += as;
  BOOST_REQUIRE(as.length() == 2*rs.length());
  BOOST_REQUIRE(as == rs + rs);
  as += rs2;
  BOOST_REQUIRE(as.length() == 2*rs.length() + rs2.length());
  BOOST_REQUIRE(as == rs + rs + rs2);
  as += "foofaafuu";
  BOOST_REQUIRE(as.length() == 2*rs.length() + rs2.length() + 
                strlen("foofaafuu"));
  BOOST_REQUIRE(as == rs + rs + rs2 + "foofaafuu");

  for (int i = 0; i < 25; ++i) {
    as = getRandomString(20);
    s = getRandomString(20);
    BOOST_REQUIRE(as.length() > 4);
    size_t pos = rand() % 10 + 1;
    as2 = as.substr(0, pos);
    BOOST_REQUIRE(as2.length() == pos);
    for (size_t j = 0; j < pos; ++j)
      BOOST_REQUIRE(as[j] == as2[j]);
    s2 = as.insert(pos, s);
    BOOST_REQUIRE(s2 == as.substr(0,pos) + s + as.substr(pos));

    size_t len = rand() % 7 + 1;
    as2 = as.substr(pos+len);
    BOOST_REQUIRE(as2.length() == as.length() - pos - len);
    for (size_t j = 0; j < as.length() - pos - len; ++j)
      BOOST_REQUIRE(as[j+pos+len] == as2[j]);

    BOOST_REQUIRE(as.erase(pos, len) == as.substr(0,pos) + as.substr(pos+len));

    as2 = getRandomString(20);

    std::ostringstream oss;
    oss << as;
    BOOST_REQUIRE(as.length() == oss.str().length());
    BOOST_REQUIRE(oss.str() == as);
    BOOST_REQUIRE(oss.str() == as.c_str());

    oss.str("");
    oss << as2;
    BOOST_REQUIRE(oss.str() == as2);
    BOOST_REQUIRE(oss.str() == as2.c_str());

    BOOST_REQUIRE(as.replace(pos, len, as2) == as.substr(0,pos) + 
                  as2 + as.substr(pos+len));

    AdvancedString text = 
      "1:1 in principio erat Verbum et Verbum erat apud Deum et Deus erat "
      "Verbum\n"
      "1:2 hoc erat in principio apud Deum\n"
      "1:3 omnia per ipsum facta sunt et sine ipso factum est nihil quod "
      "factum est\n"
      "1:4 in ipso vita erat et vita erat lux hominum\n"
      "1:5 et lux in tenebris lucet et tenebrae eam non conprehenderunt\n"
      "1:6 fuit homo missus a Deo cui nomen erat Iohannes\n"
      "1:7 hic venit in testimonium ut testimonium perhiberet de lumine ut "
      "omnes crederent per illum\n"
      "1:8 non erat ille lux sed ut testimonium perhiberet de lumine\n"
      "1:9 erat lux vera quae inluminat omnem hominem venientem in mundum\n"
      "1:10 in mundo erat et mundus per ipsum factus est et mundus eum non "
      "cognovit\n"
      "1:11 in propria venit et sui eum non receperunt\n"
      "1:12 quotquot autem receperunt eum dedit eis potestatem filios Dei "
      "fieri his qui credunt in nomine eius\n"
      "1:13 qui non ex sanguinibus neque ex voluntate carnis neque ex "
      "voluntate viri sed ex Deo nati sunt\n"
      "1:14 et Verbum caro factum est et habitavit in nobis et vidimus gloriam "
      "eius gloriam quasi unigeniti a Patre plenum gratiae et veritatis\n";

    size_t first = text.find("Verbum");
    len = strlen("Verbum");
    BOOST_REQUIRE(text.substr(first, len) == "Verbum");

    pos = text.find("Verbum", first);
    BOOST_REQUIRE(pos == first);

    pos = text.find("Verbum", first+1);
    BOOST_REQUIRE(text.substr(pos, len) == "Verbum");

    size_t last = text.rfind("Verbum");
    BOOST_REQUIRE(text.substr(last, len) == "Verbum");
    BOOST_REQUIRE(text.find("Verbum", last+1) == AdvancedString::npos);
    pos = text.rfind("Verbum", last - 1);
    BOOST_REQUIRE(pos > first && pos < last);
    BOOST_REQUIRE(text.substr(pos, len) == "Verbum");

    std::vector<size_t> poss = text.findAll("Verbum");
    BOOST_REQUIRE(poss.front() == first);
    BOOST_REQUIRE(poss.back() == last);
    for (size_t pos : poss)
      BOOST_REQUIRE(text.substr(pos, len) == "Verbum");
    size_t count = poss.size();


    poss = text.findAll("Verbum", first + 1);
    BOOST_REQUIRE(poss.size() == count-1);
    BOOST_REQUIRE(poss.front() != first);
    BOOST_REQUIRE(poss.front() == text.find("Verbum", first+1));
    BOOST_REQUIRE(poss.back() == last);
    for (size_t pos : poss)
      BOOST_REQUIRE(text.substr(pos, len) == "Verbum");


    AdvancedString text2 = 
      "1:1 in principio erat Verbum AND Verbum erat apud Deum et Deus erat "
      "Verbum\n"
      "1:2 hoc erat in principio apud Deum\n"
      "1:3 omnia per ipsum facta sunt et sine ipso factum est nihil quod "
      "factum est\n"
      "1:4 in ipso vita erat et vita erat lux hominum\n"
      "1:5 et lux in tenebris lucet et tenebrae eam non conprehenderunt\n"
      "1:6 fuit homo missus a Deo cui nomen erat Iohannes\n"
      "1:7 hic venit in testimonium ut testimonium perhiberet de lumine ut "
      "omnes crederent per illum\n"
      "1:8 non erat ille lux sed ut testimonium perhiberet de lumine\n"
      "1:9 erat lux vera quae inluminat omnem hominem venientem in mundum\n"
      "1:10 in mundo erat et mundus per ipsum factus est et mundus eum non "
      "cognovit\n"
      "1:11 in propria venit et sui eum non receperunt\n"
      "1:12 quotquot autem receperunt eum dedit eis potestatem filios Dei "
      "fieri his qui credunt in nomine eius\n"
      "1:13 qui non ex sanguinibus neque ex voluntate carnis neque ex "
      "voluntate viri sed ex Deo nati sunt\n"
      "1:14 et Verbum caro factum est et habitavit in nobis et vidimus gloriam "
      "eius gloriam quasi unigeniti a Patre plenum gratiae et veritatis\n";

    BOOST_REQUIRE(text.replace("et", "AND") == text2);

    AdvancedString text2b = 
      "1:1 in principio erat Verbum et Verbum erat apud Deum AND Deus erat "
      "Verbum\n"
      "1:2 hoc erat in principio apud Deum\n"
      "1:3 omnia per ipsum facta sunt et sine ipso factum est nihil quod "
      "factum est\n"
      "1:4 in ipso vita erat et vita erat lux hominum\n"
      "1:5 et lux in tenebris lucet et tenebrae eam non conprehenderunt\n"
      "1:6 fuit homo missus a Deo cui nomen erat Iohannes\n"
      "1:7 hic venit in testimonium ut testimonium perhiberet de lumine ut "
      "omnes crederent per illum\n"
      "1:8 non erat ille lux sed ut testimonium perhiberet de lumine\n"
      "1:9 erat lux vera quae inluminat omnem hominem venientem in mundum\n"
      "1:10 in mundo erat et mundus per ipsum factus est et mundus eum non "
      "cognovit\n"
      "1:11 in propria venit et sui eum non receperunt\n"
      "1:12 quotquot autem receperunt eum dedit eis potestatem filios Dei "
      "fieri his qui credunt in nomine eius\n"
      "1:13 qui non ex sanguinibus neque ex voluntate carnis neque ex "
      "voluntate viri sed ex Deo nati sunt\n"
      "1:14 et Verbum caro factum est et habitavit in nobis et vidimus gloriam "
      "eius gloriam quasi unigeniti a Patre plenum gratiae et veritatis\n";

    BOOST_REQUIRE(text.replace("et", "AND", text.find("et") + 1) == text2b);


    AdvancedString text3 = 
      "1:1 in principio erat Verbum AND Verbum erat apud Deum AND Deus erat "
      "Verbum\n"
      "1:2 hoc erat in principio apud Deum\n"
      "1:3 omnia per ipsum facta sunt AND sine ipso factum est nihil quod "
      "factum est\n"
      "1:4 in ipso vita erat AND vita erat lux hominum\n"
      "1:5 AND lux in tenebris lucAND AND tenebrae eam non conprehenderunt\n"
      "1:6 fuit homo missus a Deo cui nomen erat Iohannes\n"
      "1:7 hic venit in testimonium ut testimonium perhiberAND de lumine ut "
      "omnes crederent per illum\n"
      "1:8 non erat ille lux sed ut testimonium perhiberAND de lumine\n"
      "1:9 erat lux vera quae inluminat omnem hominem venientem in mundum\n"
      "1:10 in mundo erat AND mundus per ipsum factus est AND mundus eum non "
      "cognovit\n"
      "1:11 in propria venit AND sui eum non receperunt\n"
      "1:12 quotquot autem receperunt eum dedit eis potestatem filios Dei "
      "fieri his qui credunt in nomine eius\n"
      "1:13 qui non ex sanguinibus neque ex voluntate carnis neque ex "
      "voluntate viri sed ex Deo nati sunt\n"
      "1:14 AND Verbum caro factum est AND habitavit in nobis AND vidimus gloriam "
      "eius gloriam quasi unigeniti a Patre plenum gratiae AND veritatis\n";

    BOOST_REQUIRE(text.replaceAll("et", "AND") == text3);

    AdvancedString text4 = 
      "1:1 in principio erat Verbum et WORD erat apud Deum et Deus erat "
      "Verbum\n"
      "1:2 hoc erat in principio apud Deum\n"
      "1:3 omnia per ipsum facta sunt et sine ipso factum est nihil quod "
      "factum est\n"
      "1:4 in ipso vita erat et vita erat lux hominum\n"
      "1:5 et lux in tenebris lucet et tenebrae eam non conprehenderunt\n"
      "1:6 fuit homo missus a Deo cui nomen erat Iohannes\n"
      "1:7 hic venit in testimonium ut testimonium perhiberet de lumine ut "
      "omnes crederent per illum\n"
      "1:8 non erat ille lux sed ut testimonium perhiberet de lumine\n"
      "1:9 erat lux vera quae inluminat omnem hominem venientem in mundum\n"
      "1:10 in mundo erat et mundus per ipsum factus est et mundus eum non "
      "cognovit\n"
      "1:11 in propria venit et sui eum non receperunt\n"
      "1:12 quotquot autem receperunt eum dedit eis potestatem filios Dei "
      "fieri his qui credunt in nomine eius\n"
      "1:13 qui non ex sanguinibus neque ex voluntate carnis neque ex "
      "voluntate viri sed ex Deo nati sunt\n"
      "1:14 et Verbum caro factum est et habitavit in nobis et vidimus gloriam "
      "eius gloriam quasi unigeniti a Patre plenum gratiae et veritatis\n";
    BOOST_REQUIRE(text.replaceRe("Ver[a-z]*", "WORD", 
                                 text.findRe("Ver[a-z]*")+1) == 
                  text4);
    AdvancedString text5 = 
      "1:1 in principio erat Verbum et Verbum erat apud Deum et Deus erat "
      "WORD\n"
      "1:2 hoc erat in principio apud Deum\n"
      "1:3 omnia per ipsum facta sunt et sine ipso factum est nihil quod "
      "factum est\n"
      "1:4 in ipso vita erat et vita erat lux hominum\n"
      "1:5 et lux in tenebris lucet et tenebrae eam non conprehenderunt\n"
      "1:6 fuit homo missus a Deo cui nomen erat Iohannes\n"
      "1:7 hic venit in testimonium ut testimonium perhiberet de lumine ut "
      "omnes crederent per illum\n"
      "1:8 non erat ille lux sed ut testimonium perhiberet de lumine\n"
      "1:9 erat lux vera quae inluminat omnem hominem venientem in mundum\n"
      "1:10 in mundo erat et mundus per ipsum factus est et mundus eum non "
      "cognovit\n"
      "1:11 in propria venit et sui eum non receperunt\n"
      "1:12 quotquot autem receperunt eum dedit eis potestatem filios Dei "
      "fieri his qui credunt in nomine eius\n"
      "1:13 qui non ex sanguinibus neque ex voluntate carnis neque ex "
      "voluntate viri sed ex Deo nati sunt\n"
      "1:14 et WORD caro factum est et habitavit in nobis et vidimus gloriam "
      "eius gloriam quasi unigeniti a Patre plenum gratiae et veritatis\n";
    BOOST_REQUIRE(text.replaceAllRe("Ver[a-z]*", "WORD", 
                                    text.findRe("Ver[a-z]*", 
                                                text.find("Verbum") + 1)+1) == 
                                    text5);

    as2 = as*i;
    BOOST_REQUIRE(as2.length() == i*as.length());
    for (int j = 0; j < i; ++j)
      BOOST_REQUIRE(as2.substr(j*as.length(), as.length()) == as);

    as2 = i*as;
    BOOST_REQUIRE(as2.length() == i*as.length());
    for (int j = 0; j < i; ++j)
      BOOST_REQUIRE(as2.substr(j*as.length(), as.length()) == as);

  }

  BOOST_REQUIRE(AdvancedString(getRandomString()).match("[a-z][a-z]*"));
  BOOST_REQUIRE(!AdvancedString(getRandomString()).match("[0-9][a-z]*"));

  std::vector<AdvancedString> strings {
    getRandomString(), getRandomString(), getRandomString(), getRandomString(), getRandomString()
  };
  AdvancedString merged = AdvancedString(",").join(strings);
  BOOST_REQUIRE(merged == strings[0] + "," + strings[1] + "," + strings[2] + 
                "," + strings[3] + "," + strings[4]);
  BOOST_REQUIRE(merged == AdvancedString::join(",", strings));

  BOOST_REQUIRE(strings == merged.split(","));

  for (int i = 0; i < 25; ++i) {
    std::string rs1 = getRandomString();
    std::string rs2 = getRandomString();
    BOOST_REQUIRE(rs1.length() == rs2.length() && rs2.length() == 8);
    BOOST_REQUIRE((rs1.compare(rs2) == 0 && strcmp(rs1.c_str(), rs2.c_str()) == 0) ||
                  (rs1.compare(rs2) < 0 && strcmp(rs1.c_str(), rs2.c_str()) < 0) ||
                  (rs1.compare(rs2) > 0 && strcmp(rs1.c_str(), rs2.c_str()) > 0));
    as = rs1;
    as2 = rs2;
    BOOST_REQUIRE((as.compare(as2) == 0 && strcmp(as.c_str(), as2.c_str()) == 0) ||
                  (as.compare(as2) < 0 && strcmp(as.c_str(), as2.c_str()) < 0) ||
                  (as.compare(as2) > 0 && strcmp(as.c_str(), as2.c_str()) > 0));
    BOOST_REQUIRE(as.compare(as2) == std::string(as).compare(as2));
    BOOST_REQUIRE(as.compare(as2) == std::string(as).compare(std::string(as2)));

    BOOST_REQUIRE((as.compare(std::string(as2)) == 0 && strcmp(as.c_str(), as2.c_str()) == 0) ||
                  (as.compare(std::string(as2)) < 0 && strcmp(as.c_str(), as2.c_str()) < 0) ||
                  (as.compare(std::string(as2)) > 0 && strcmp(as.c_str(), as2.c_str()) > 0));
    BOOST_REQUIRE(signEqual(std::string(as).compare(as2), strcmp(as.c_str(), as2.c_str())));
    BOOST_REQUIRE(signEqual(as.compare(as2.c_str()), strcmp(as.c_str(), as2.c_str())));
    BOOST_REQUIRE(signEqual(as.compare(as2), strcmp(as.c_str(), as2.c_str())));
    BOOST_REQUIRE(as.compare(as) == 0);
    BOOST_REQUIRE(as2.compare(as2) == 0);

    BOOST_REQUIRE((as != as2 && strcmp(as.c_str(), as2.c_str()) != 0) ||
                  (as == as2 && strcmp(as.c_str(), as2.c_str()) == 0));
    BOOST_REQUIRE(as == as);
    BOOST_REQUIRE(!(as != as));
  }

  BOOST_REQUIRE(AdvancedString::npos == std::string::npos);

  BOOST_REQUIRE(AdvancedString("0.557681 0.28743 0.145816").split(" ") ==
                (std::vector<AdvancedString> { "0.557681", "0.28743", "0.145816" }));
}

BOOST_AUTO_TEST_CASE( testVisualiser ) {
  cv::Mat m(0,0,CV_8UC3);
  cv::Point2i cursor(0,0);
  cv::Point2i origin(0,0);
  expandIfNecessary(m, cursor, origin, cv::Size(0,0));
  BOOST_REQUIRE(m.empty());
  BOOST_REQUIRE(cursor == cv::Point2i(0,0));
  BOOST_REQUIRE(origin == cv::Point2i(0,0));

  expandIfNecessary(m, cursor, origin, cv::Size(5,5));
  BOOST_REQUIRE(!m.empty());
  BOOST_REQUIRE(m.size() == cv::Size(5,5));
  BOOST_REQUIRE(countNonZero<cv::Vec3b>(m) == 0);
  BOOST_REQUIRE(cursor == cv::Point2i(0,0));
  BOOST_REQUIRE(origin == cv::Point2i(0,0));
  m.at<cv::Vec3b>(4,4) = cv::Vec3b(255,255,255);
  BOOST_REQUIRE(countNonZero<cv::Vec3b>(m) == 1);

  expandIfNecessary(m, cursor, origin, cv::Size(3,3));
  BOOST_REQUIRE(m.size() == cv::Size(5,5));
  BOOST_REQUIRE(countNonZero<cv::Vec3b>(m) == 1);
  BOOST_REQUIRE(cursor == cv::Point2i(0,0));
  BOOST_REQUIRE(origin == cv::Point2i(0,0));
  m.at<cv::Vec3b>(2,2) = cv::Vec3b(255,255,255);
  BOOST_REQUIRE(countNonZero<cv::Vec3b>(m) == 2);

  expandIfNecessary(m, cursor, origin, cv::Size(8,8));
  BOOST_REQUIRE(m.size() == cv::Size(8,8));
  BOOST_REQUIRE(countNonZero<cv::Vec3b>(m) == 2);
  BOOST_REQUIRE(cursor == cv::Point2i(0,0));
  BOOST_REQUIRE(origin == cv::Point2i(0,0));
  m.at<cv::Vec3b>(7,7) = cv::Vec3b(255,255,255);
  BOOST_REQUIRE(countNonZero<cv::Vec3b>(m) == 3);

  cursor = cv::Point2i(-5, -5);
  BOOST_REQUIRE(origin == cv::Point2i(0,0));
  expandIfNecessary(m, cursor, origin, cv::Size(8,8));
  BOOST_REQUIRE(m.size() == cv::Size(13,13));
  BOOST_REQUIRE(countNonZero<cv::Vec3b>(m) == 3);
  BOOST_REQUIRE(cursor == cv::Point2i(-5,-5));
  BOOST_REQUIRE(origin == cv::Point2i(5,5));
  m.at<cv::Vec3b>(0,0) = cv::Vec3b(255,255,255);
  BOOST_REQUIRE(countNonZero<cv::Vec3b>(m) == 4);

  for (int i = 0; i < m.rows; ++i) {
    for (int j = 0; j < m.cols; ++j) {
      const cv::Vec3b& v = m.at<cv::Vec3b>(i,j);
      const cv::Vec3b ONE(255,255,255);
      const cv::Vec3b ZERO(0,0,0);
      if ((i == 0 && j == 0) ||
          (i == 7 && j == 7) ||
          (i == 9 && j == 9) ||
          (i == 12 && j == 12))
        BOOST_REQUIRE(v == ONE);
      else
        BOOST_REQUIRE(v == ZERO);
    }
  }

  std::unique_ptr<OpenCVVideoCaptureVideoFileSource> vfs(new OpenCVVideoCaptureVideoFileSource(TESTVIDEOFILE_FFV1));
  std::vector<cv::Mat> frames(vfs->begin(), vfs->end());
  BlackBoard bb;
  std::shared_ptr<SLIO> slio(new SLIO(""));
  Visualiser v(slio);
  v.setComplexVisualisation("showFrame");
  BOOST_REQUIRE(vfs->size() == frames.size());
  for (size_t i = 0; i < vfs->size(); ++i)
    BOOST_REQUIRE(equal(v.visualise(*vfs, bb, i), frames[i]));

  v.setComplexVisualisation("locate 10 20; showFrame");
  for (size_t f = 0; f < vfs->size(); ++f) {
    cv::Mat visualised = v.visualise(*vfs, bb, f);
    cv::Mat inFrame = (*vfs)[f];
    BOOST_REQUIRE(visualised.cols == inFrame.cols + 10);
    BOOST_REQUIRE(visualised.rows == inFrame.rows + 20);
    for (int i = 0; i < visualised.rows; ++i) {
      for (int j = 0; j < visualised.cols; ++j) {
        if (i >= 20 && j >= 10) {
          // std::cerr << visualised.at<cv::Vec3b>(i,j) << " " 
          //           << inFrame.at<cv::Vec3b>(i-20,j-10) << std::endl;
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i,j) == 
                        inFrame.at<cv::Vec3b>(i-20,j-10));
        }
        else
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i,j) == cv::Vec3b(0,0,0));
      }
    }
  }

  v.setComplexVisualisation("locate -10 -20; showFrame; locate 33 57; showFrame");
  for (size_t f = 0; f < vfs->size(); ++f) {
    cv::Mat visualised = v.visualise(*vfs, bb, f);
    cv::Mat inFrame = (*vfs)[f];
    BOOST_REQUIRE(visualised.cols == inFrame.cols + 10 + 33);
    BOOST_REQUIRE(visualised.rows == inFrame.rows + 20 + 57);
    for (int i = 0; i < visualised.rows; ++i) {
      for (int j = 0; j < visualised.cols; ++j) {
        if ((i < 57+20 && j < 720) || (j < 33+10 && i < 596))
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i,j) == 
                        inFrame.at<cv::Vec3b>(i,j));
        else if (i >= 57+20 && j >= 33+10) 
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i,j) == 
                        inFrame.at<cv::Vec3b>(i-57-20,j-33-10));
        else 
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i,j) == cv::Vec3b(0,0,0));
      }
    }
  }

  v.setComplexVisualisation("showFrame; locate frame-width 0; showFrame;"
                            "locate 0 frame-height; showFrame;"
                            "locate frame-width frame-height; showFrame");
  for (size_t f = 0; f < vfs->size(); ++f) {
    cv::Mat visualised = v.visualise(*vfs, bb, f);
    cv::Mat inFrame = (*vfs)[f];
    BOOST_REQUIRE(visualised.cols == inFrame.cols*2);
    BOOST_REQUIRE(visualised.rows == inFrame.rows*2);
    for (int i = 0; i < visualised.rows; ++i) 
      for (int j = 0; j < visualised.cols; ++j) 
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i, j) == 
                        inFrame.at<cv::Vec3b>(i % inFrame.rows, 
                                              j % inFrame.cols));
  }

#ifdef SLMOTION_ENABLE_OPENNI
  OpenNiOniFileSource ofs(MARKUS_ONI);

  v.setComplexVisualisation("showFrame; locate frame-width 0; showFrame depth");
  for (size_t f = 0; f < ofs.size(); ++f) {
    cv::Mat visualised = v.visualise(ofs, bb, f);
    cv::Mat inFrame = ofs[f];
    cv::Mat inDepth = convertToMarkusFormat(ofs.getTrack(1)[f]);
    BOOST_REQUIRE(visualised.cols == inFrame.cols + inDepth.cols);
    BOOST_REQUIRE(visualised.rows == std::max(inFrame.rows, inDepth.rows));
    for (int i = 0; i < visualised.rows; ++i) {
      for (int j = 0; j < visualised.cols; ++j) { 
        if (j < inFrame.cols && i < inFrame.rows)
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i, j) == 
                        inFrame.at<cv::Vec3b>(i, j));
        else if (j - inFrame.cols < inDepth.cols && i < inDepth.rows)
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i, j) == 
                        inDepth.at<cv::Vec3b>(i, j-inFrame.cols));
        else
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i,j) == cv::Vec3b(0,0,0));
      }
    }
  }

  OpenNiOniFileSource ofs2(CAPTURE_ONI_SKEL, 0);
  auto oniFrames = getOniTestVideoFrameFilenames();
  std::vector<string> rgb = oniFrames["rgb-skel"];
  ImageSequenceSource rgbiss(std::move(rgb), 30, 0);
  std::vector<string> depthskel = oniFrames["depth-skel"];
  ImageSequenceSource depthissskel(std::move(depthskel), 30, 0);
  v.setComplexVisualisation("showFrame 0; locate frame-width 0; showFrame 1; "
                            "showKinectSkeleton");
  for (size_t f = 0; f < ofs2.size(); ++f) {
    cv::Mat visualised = v.visualise(ofs2, bb, f);
    cv::Mat rgbFrame = rgbiss[f];
    cv::Mat depthFrame = depthissskel[f];
    if (visualised.rows != rgbFrame.rows) {
      cv::imshow("", visualised);
      cv::waitKey(0);
    }
      
    BOOST_REQUIRE(visualised.cols == rgbFrame.cols + depthFrame.cols);
    BOOST_REQUIRE(visualised.rows == rgbFrame.rows);
    BOOST_REQUIRE(visualised.rows == depthFrame.rows);
    cv::Mat faultMatrix(visualised.size(), CV_8UC3, cv::Scalar::all(0));
    for (int i = 0; i < visualised.rows; ++i) {
      for (int j = 0; j < visualised.cols; ++j) { 
        if (j < rgbFrame.cols)
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i, j) == 
                        rgbFrame.at<cv::Vec3b>(i, j));
        else {
          BOOST_REQUIRE(visualised.at<cv::Vec3b>(i,j) == 
                        depthFrame.at<cv::Vec3b>(i,j-depthFrame.cols));
        }
      }
    }
  }
#endif
}

BOOST_AUTO_TEST_CASE( testDump ) {
  // force the same random number seed
  size_t randomSeed = reinterpret_cast<size_t>(&slmotion::debug);

  slmotion::debug = 0;
  // slmotion::debug = 1;
  std::shared_ptr<BlackBoard> bb1, bb2;

  {
    // vector<std::string> argsS = { "--dump-file", "foo.dump", "--components", "FaceDetector,GaussianSkinDetector,BlobExtractor,BodyPartCollector,ColourSpaceConverter,KLTTracker,AsmTracker,BlackBoardDumpWriter" };
    vector<std::string> argsS = { "./tests", "--out-dump-file", "foo.dump", 
                                  "--components", 
                                  "FaceDetector,GaussianSkinDetector,"
                                  "BlobExtractor,BodyPartCollector,"
                                  "ColourSpaceConverter,KLTTracker,"
                                  "AsmTracker,"
#ifdef SLMOTION_ENABLE_LIBFLANDMARK                                  
                                  "FacialLandmarkDetector,"
#endif
                                  "BlackBoardDumpWriter"};
    vector<char*> args;
    for (auto it = argsS.cbegin(); it != argsS.cend(); ++it) {
      args.push_back(new char[it->length() + 1]);
      strcpy(args.back(), it->c_str());
    }

    srand(randomSeed);
    cv::theRNG().state = randomSeed;
    vector<Analyser> anals;
    SLMotionConfiguration opts;

    createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                    vector<string>(), anals, opts, args.size(), &args[0]);
    
#ifdef SLMOTION_THREADING   
    slmotion::Thread::setMaxThreads(0);
#endif
    // reset TrackedPoint id numbers
    TrackedPoint::uniqueIdCounter = 1;

#ifdef SLMOTION_ENABLE_LIBFLANDMARK                                  
    assert(anals.size() == 1);
    FacialLandmarkDetector* flt = getComponent<FacialLandmarkDetector>(anals[0]);
    flt->enableConfidence = true;
#endif 
    
    assert(anals.size() == 1);
    AnalysisController analCon(anals[0].visualiser, anals[0].slio, false);
    analCon.analyseVideofiles(anals[0], 0, SIZE_MAX, false, // false, 
                              "", "", ""
                              // opts.visualiserCombinationStyle
                              );
    bb1 = anals[0].blackboard;

    for (auto it = args.cbegin(); it != args.cend(); ++it) 
      delete *it;
  }

  {
    vector<std::string> argsS = { "./tests", "--in-dump-file", "foo.dump", 
                                  "--components", "BlackBoardDumpReader" };
    vector<char*> args;
    for (auto it = argsS.cbegin(); it != argsS.cend(); ++it) {
      args.push_back(new char[it->length() + 1]);
      strcpy(args.back(), it->c_str());
    }

    srand(randomSeed);
    cv::theRNG().state = randomSeed;
    vector<Analyser> anals;
    SLMotionConfiguration opts;
    createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                    vector<string>(), anals, opts, args.size(), &args[0]); 
    AnalysisController analCon(anals[0].visualiser, anals[0].slio, false);
    assert(anals.size() == 1);
    analCon.analyseVideofiles(anals[0], 0, SIZE_MAX, false, // false, 
                              "", "", ""
                              // opts.visualiserCombinationStyle
                              );
    bb2 = anals[0].blackboard;
    for (auto it = args.cbegin(); it != args.cend(); ++it) 
      delete[] *it;
  }

  BOOST_REQUIRE(bb1->size() == bb2->size());
  BOOST_REQUIRE(bb1->frameBoard.size() == bb2->frameBoard.size());
  for (auto it = bb1->frameBoard.cbegin(), jt = bb2->frameBoard.cbegin(); 
       it != bb1->frameBoard.cend(); ++it, ++jt) {
    BOOST_CHECK(it->first == jt->first);
    BOOST_CHECK(bbEqual(it->second->referenceAny(it->second), 
                        jt->second->referenceAny(jt->second)));
  }
  BOOST_REQUIRE(bb1->globalBoard.size() == bb2->globalBoard.size());
  for (auto it = bb1->globalBoard.cbegin(), jt = bb2->globalBoard.cbegin(); 
       it != bb1->globalBoard.cend(); ++it, ++jt) {
    BOOST_CHECK(it->first == jt->first);
    BOOST_CHECK(bbEqual(it->second->referenceAny(it->second), 
                        jt->second->referenceAny(jt->second)));
  }

  const Pdm* head = bb2->get<Asm>(ASM_TRACKER_HEAD_ASM)->pdm.get();
  const Pdm* leftHand = bb2->get<Asm>(ASM_TRACKER_LEFT_HAND_ASM)->pdm.get();
  const Pdm* rightHand = bb2->get<Asm>(ASM_TRACKER_RIGHT_HAND_ASM)->pdm.get();
  std::unique_ptr<OpenCVVideoCaptureVideoFileSource> vfs_p(new OpenCVVideoCaptureVideoFileSource(TESTVIDEOFILE7));
    OpenCVVideoCaptureVideoFileSource& vfs = *vfs_p;
  size_t nFrames = vfs.size();
  for (size_t i = 0; i < nFrames; ++i) {
    BOOST_CHECK(bb2->get<Asm::Instance>(i, ASM_TRACKER_HEAD_ASM_INSTANCE)->pdm.get() == head);
    BOOST_CHECK(bb2->get<Asm::Instance>(i, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE)->pdm.get() == leftHand);
    BOOST_CHECK(bb2->get<Asm::Instance>(i, ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE)->pdm.get() == rightHand);
  }

  std::shared_ptr<BlackBoard> bb3;
  {
    // a new test will be added here where the first stages of the analysis
    // chain are gone through, then the KLT and ASM tracker stages are done
    // separately and the result is checked

    vector<std::string> argsS = { "./tests", "--out-dump-file", "foo.dump", 
                                  "--components", 
                                  "FaceDetector,GaussianSkinDetector,"
                                  "BlobExtractor,BodyPartCollector,"
                                  "ColourSpaceConverter,"
                                  "BlackBoardDumpWriter" };
    vector<char*> args;
    for (auto it = argsS.cbegin(); it != argsS.cend(); ++it) {
      args.push_back(new char[it->length() + 1]);
      strcpy(args.back(), it->c_str());
    }

    srand(randomSeed);
    cv::theRNG().state = randomSeed;
    vector<Analyser> anals;
    SLMotionConfiguration opts;

    createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                    vector<string>(), anals, opts, args.size(), &args[0]);
    
#ifdef SLMOTION_THREADING   
    slmotion::Thread::setMaxThreads(0);
#endif
    // reset TrackedPoint id numbers
    TrackedPoint::uniqueIdCounter = 1;
    
    AnalysisController analCon(anals[0].visualiser, anals[0].slio, false);
    assert(anals.size() == 1);
    analCon.analyseVideofiles(anals[0], 0, SIZE_MAX, false, // false, 
                              "", "", ""
                              // opts.visualiserCombinationStyle
                              );
    // bb3 = anals[0].blackboard;

    for (auto it = args.cbegin(); it != args.cend(); ++it) 
      delete *it;
  }

  {
    // a new test will be added here where the first stages of the analysis
    // chain are gone through, then the KLT and ASM tracker stages are done
    // separately and the result is checked

    vector<std::string> argsS = { "./tests", "--in-dump-file", "foo.dump", 
                                  "--components", 
                                  "BlackBoardDumpReader,KLTTracker,"
                                  "AsmTracker" 
#ifdef SLMOTION_ENABLE_LIBFLANDMARK                                  
                                  ",FacialLandmarkDetector"
#endif
    };
    vector<char*> args;
    for (auto it = argsS.cbegin(); it != argsS.cend(); ++it) {
      args.push_back(new char[it->length() + 1]);
      strcpy(args.back(), it->c_str());
    }

    srand(randomSeed);
    cv::theRNG().state = randomSeed;
    vector<Analyser> anals;
    SLMotionConfiguration opts;

    createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                    vector<string>(), anals, opts, args.size(), &args[0]);
    
#ifdef SLMOTION_THREADING   
    slmotion::Thread::setMaxThreads(0);
#endif
    // reset TrackedPoint id numbers
    TrackedPoint::uniqueIdCounter = 1;

#ifdef SLMOTION_ENABLE_LIBFLANDMARK                                  
    FacialLandmarkDetector* flt = getComponent<FacialLandmarkDetector>(anals[0]);
    flt->enableConfidence = true;
#endif 
    
    AnalysisController analCon(anals[0].visualiser, anals[0].slio, false);
    assert(anals.size() == 1);
    analCon.analyseVideofiles(anals[0], 0, SIZE_MAX, false, // false, 
                              "", "", ""
                              // opts.visualiserCombinationStyle
                              );
    bb3 = anals[0].blackboard;

    for (auto it = args.cbegin(); it != args.cend(); ++it) 
      delete[] *it;
  }

  BOOST_REQUIRE(bb1->size() == bb3->size());
  BOOST_REQUIRE(bb1->frameBoard.size() == bb3->frameBoard.size());
  for (auto it = bb1->frameBoard.cbegin(), jt = bb3->frameBoard.cbegin(); 
       it != bb1->frameBoard.cend(); ++it, ++jt) {
    BOOST_CHECK(it->first == jt->first);
    BOOST_CHECK(bbEqual(it->second->referenceAny(it->second), 
                        jt->second->referenceAny(jt->second)));
  }
  BOOST_REQUIRE(bb1->globalBoard.size() == bb3->globalBoard.size());
  for (auto it = bb1->globalBoard.cbegin(), jt = bb3->globalBoard.cbegin(); 
       it != bb1->globalBoard.cend(); ++it, ++jt) {
    BOOST_CHECK(it->first == jt->first);
    BOOST_CHECK(bbEqual(it->second->referenceAny(it->second), 
                        jt->second->referenceAny(jt->second)));
  }
}

BOOST_AUTO_TEST_CASE( testDump2 ) {
  TempFile script;
  script << "from slmotion import *" << endl
         << "setComponents([Component('ColourSpaceConverter', {})," << endl
         << "Component('FaceDetector', {'scale': '0.6:0.9'})," << endl
         << "Component('GaussianSkinDetector', {'logdensitythreshold': -13}), " << endl
         << "Component('BlobExtractor', {})," << endl
         << "Component('BodyPartCollector', {})," << endl
         << "Component('KLTTracker', {})," << endl
         << "Component('AsmTracker', {'maxshapeparameterdeviation': 1.0," << endl
         << "          'pcacomponentcount': 3})," << endl
         << "Component('FeatureCreator', {})," << endl
         << "])" << endl
         << "process()" << endl;
  script.close();

  vector<string> args {
    "./tests", "--script", script.getName(), SUVI_TEST_VIDEO
  };
  vector<char*> argv(args.size());
  vector<vector<char> > argvs(args.size());
  for (size_t i = 0; i < args.size(); ++i) {
    argvs[i] = vector<char>(args[i].size() + 1);
    strcpy(&argvs[i][0], args[i].c_str());  
    argv[i] = &argvs[i][0];
  }
  std::shared_ptr<BlackBoard> bb1(new BlackBoard);
  slmotion::main(argv.size(), &argv[0], bb1, nullptr);
  BOOST_REQUIRE(bb1->size() == 1351);

  DummyFrameSource dfs;
  TempFile dumpFile;
  dumpFile.close();
  BlackBoardDumpWriter bbdw(bb1.get(), &dfs);
  bbdw.setFilename(dumpFile.getName());
  bbdw.processRange(0, SIZE_MAX);

  BlackBoard bb2;
  BlackBoardDumpReader bbdr(&bb2, &dfs);
  bbdr.setFilenames(vector<string> { dumpFile.getName() } );
  bbdr.processRange(0, SIZE_MAX);
  BOOST_REQUIRE(bbEqual(*bb1, bb2));
}

BOOST_AUTO_TEST_CASE( testXml ) {
  setArgV(0, NULL);

  xml::XmlDocument doc;
  std::ostringstream oss;
  oss << doc;
  BOOST_REQUIRE(oss.str() == "<?xml version=\"1.0\"?>\n");

  // check for memory leaks
  {
    XmlNode n;
  }

  {
    XmlNode n("foo");
  }

  {
    XmlNode n("foo");
    XmlDocument m;
    m.setRootElement(n);
  }

  {
    xmlDocPtr doc = xmlNewDoc((const xmlChar*)"1.0");
    xmlNodePtr node = xmlNewNode(NULL, (const xmlChar*)"foo");
    xmlDocSetRootElement(doc, node);
    xmlNodePtr node2 = xmlNewNode(NULL, (const xmlChar*)"bar");
    xmlDocSetRootElement(doc, node2);
    xmlUnlinkNode(node);
    xmlFreeNode(node);
    xmlFreeDoc(doc);
  }

  {
    XmlNode n("foo");
    XmlDocument m;
    m.setRootElement(n);
    XmlNode q("bar");
    m.setRootElement(q);
  }

  {
    XmlNode n("foo");
    n.newTextChild("muu", "moo");
    XmlDocument m;
    m.setRootElement(n);
    n.newTextChild("buu", "boo");
    std::ostringstream oss;
    oss << m;
    BOOST_CHECK(oss.str() == "<?xml version=\"1.0\"?>\n"
                "<foo>\n"
                "  <muu>moo</muu>\n"
                "  <buu>boo</buu>\n"
                "</foo>\n");
    oss.seekp(0);


    XmlNode q("bar");
    q.newTextChild("fuu", "foo");
    m.setRootElement(q);
    q.newTextChild("guu", "goo");
    oss << m;
    BOOST_CHECK(oss.str() == "<?xml version=\"1.0\"?>\n"
                "<bar>\n"
                "  <fuu>foo</fuu>\n"
                "  <guu>goo</guu>\n"
                "</bar>\n");

  }

  {
    XmlNode n("foo");
    n.newTextChild("muu", "moo");
    XmlNode r("poo");
    r.newTextChild("puu", "paa");
    n.addChild(r);
    XmlDocument m;
    m.setRootElement(n);
    n.newTextChild("buu", "boo");
    std::ostringstream oss;
    oss << m;
    BOOST_CHECK(oss.str() == "<?xml version=\"1.0\"?>\n"
                "<foo>\n"
                "  <muu>moo</muu>\n"
                "  <poo>\n"
                "    <puu>paa</puu>\n"
                "  </poo>\n"
                "  <buu>boo</buu>\n"
                "</foo>\n");
    oss.seekp(0);


    XmlNode q("bar");
    q.newTextChild("fuu", "foo");
    m.setRootElement(q);
    XmlNode r2("poo");
    q.addChild(r2);
    r2.newTextChild("puu", "paa");
    q.newTextChild("guu", "goo");
    oss << m;
    BOOST_CHECK(oss.str() == "<?xml version=\"1.0\"?>\n"
                "<bar>\n"
                "  <fuu>foo</fuu>\n"
                "  <poo>\n"
                "    <puu>paa</puu>\n"
                "  </poo>\n"
                "  <guu>goo</guu>\n"
                "</bar>\n");

  }

 {
   XmlDocument m;
   {
     XmlNode n("foo");
     n.newTextChild("bar", "bir");
     XmlNode r("foofoo");
     r.newTextChild("puu", "paa");
     n.addChild(r);
     n.newTextChild("bur", "ber");
     XmlNode s("foofoofoo");
     s.newTextChild("poi", "pai");
     r.addChild(s);
     XmlNode t("barbar");
     n.addChild(t);
     m.setRootElement(n);
   }
   std::ostringstream oss;
   oss << m;

   BOOST_CHECK(oss.str() ==
               "<?xml version=\"1.0\"?>\n"
               "<foo>\n"
               "  <bar>bir</bar>\n"
               "  <foofoo>\n"
               "    <puu>paa</puu>\n"
               "    <foofoofoo>\n"
               "      <poi>pai</poi>\n"
               "    </foofoofoo>\n"
               "  </foofoo>\n"
               "  <bur>ber</bur>\n"
               "  <barbar/>\n"
               "</foo>\n");
   BOOST_CHECK_THROW(m.getRoot().getChild("barbarbar"), XMLException);
   XmlNode n = m.getRoot().getChild("barbar");
   n.newTextChild("fui", "fai");
   oss.seekp(0);
   oss << m;
   BOOST_CHECK(oss.str() ==
               "<?xml version=\"1.0\"?>\n"
               "<foo>\n"
               "  <bar>bir</bar>\n"
               "  <foofoo>\n"
               "    <puu>paa</puu>\n"
               "    <foofoofoo>\n"
               "      <poi>pai</poi>\n"
               "    </foofoofoo>\n"
               "  </foofoo>\n"
               "  <bur>ber</bur>\n"
               "  <barbar>\n"
               "    <fui>fai</fui>\n"
               "  </barbar>\n"
               "</foo>\n");
 }

 {
   XmlDocument m;
   {
     XmlNode n("foo");
     n.newTextChild("bar", "bir");
     XmlNode r("foofoo");
     r.newTextChild("puu", "paa");
     n.addChild(r);
     n.newTextChild("bur", "ber");
     XmlNode s("foofoofoo");
     s.newTextChild("poi", "pai");
     r.addChild(s);
     XmlNode t("barbar");
     n.addChild(t);
     m.setRootElement(n);
   }
   XmlNode n = m.getRoot().getChild("barbar");
   n.newTextChild("fui", "fai");
   n = m.getRoot().getChild("foofoo");
   XmlNode o("foobar");
   n.addChild(o);
   o.setContent("FoObAr");
   std::ostringstream oss;
   oss << m;
   BOOST_REQUIRE(oss.str() ==
               "<?xml version=\"1.0\"?>\n"
               "<foo>\n"
               "  <bar>bir</bar>\n"
               "  <foofoo>\n"
               "    <puu>paa</puu>\n"
               "    <foofoofoo>\n"
               "      <poi>pai</poi>\n"
               "    </foofoofoo>\n"
               "    <foobar>FoObAr</foobar>\n"
               "  </foofoo>\n"
               "  <bur>ber</bur>\n"
               "  <barbar>\n"
               "    <fui>fai</fui>\n"
               "  </barbar>\n"
               "</foo>\n");
 }

 {
   XmlDocument m;
   {
     XmlNode n("foo");
     n.newTextChild("bar", "bir");
     XmlNode r("foofoo");
     r.newTextChild("puu", "paa");
     n.addChild(r);
     n.newTextChild("bur", "ber");
     XmlNode u("boar");
     u.newProperty("bear", "buar");
     n.addChild(u);     
     XmlNode s("foofoofoo");
     s.newTextChild("poi", "pai");
     r.addChild(s);
     XmlNode t("barbar");
     n.addChild(t);
     m.setRootElement(n);
     BOOST_CHECK(t.getName() == "barbar");
   }
   XmlNode n = m.getRoot().getChild("barbar");
   n.newTextChild("fui", "fai");
   n = m.getRoot().getChild("foofoo");
   XmlNode o("foobar");
   n.addChild(o);
   o.setContent("FoObAr");
   o.newProperty("foo", "bar");
   BOOST_CHECK(o.getName() == "foobar");
   std::ostringstream oss;
   oss << m;
   BOOST_CHECK(oss.str() ==
               "<?xml version=\"1.0\"?>\n"
               "<foo>\n"
               "  <bar>bir</bar>\n"
               "  <foofoo>\n"
               "    <puu>paa</puu>\n"
               "    <foofoofoo>\n"
               "      <poi>pai</poi>\n"
               "    </foofoofoo>\n"
               "    <foobar foo=\"bar\">FoObAr</foobar>\n"
               "  </foofoo>\n"
               "  <bur>ber</bur>\n"
               "  <boar bear=\"buar\"/>\n"
               "  <barbar>\n"
               "    <fui>fai</fui>\n"
               "  </barbar>\n"
               "</foo>\n");

   int i = 1;
   for (auto it = m.getRoot().begin(); it != m.getRoot().end(); ++it) {
     BOOST_REQUIRE(i < 6);
     switch(i) {
     case 1:
       BOOST_CHECK(it->getName() == "bar");
       break;
     case 2:
       BOOST_CHECK(it->getName() == "foofoo");
       break;
     case 3:
       BOOST_CHECK(it->getName() == "bur");
       break;
     case 4:
       BOOST_REQUIRE(it->getName() == "boar");
       BOOST_CHECK(it->getProperty("bear") == "buar");
       break;
     case 5:
       BOOST_CHECK(it->getName() == "barbar");
       break;
     }

     ++i;
   }
 }

 /*
  * Testing validator
  */
 {
   Validator validator(getAnnotationSchemaFile());
   BOOST_CHECK(validator.validate(getInstallPrefix() + "/share/slmotion/default.ann"));
   XmlDocument defaultAnn(getInstallPrefix() + "/share/slmotion/default.ann");
   BOOST_REQUIRE(validator.validate(defaultAnn));
   BOOST_REQUIRE(!validator.validate(SLMOTION_TEST_DATA_DIRECTORY"/default.ann.broken"));
   BOOST_REQUIRE(validator.validate(defaultAnn));
   BOOST_REQUIRE(!validator.validate(SLMOTION_TEST_DATA_DIRECTORY"/default.ann.broken"));
   XmlDocument defaultAnnBroken(SLMOTION_TEST_DATA_DIRECTORY"/default.ann.broken");

   BOOST_REQUIRE(validator.validate(defaultAnn));



   xmlDocPtr brokenDoc = xmlReadFile(SLMOTION_TEST_DATA_DIRECTORY"/default.ann.broken",
                                     NULL, 0);



   xmlDocPtr annotationSchemaDocPtr = xmlReadFile(getAnnotationSchemaFile().c_str(),
                                                  NULL, 0);

   xmlSchemaParserCtxtPtr annotationSchemaParserContext = xmlSchemaNewDocParserCtxt(annotationSchemaDocPtr);
   xmlSchemaPtr schemaPtr = xmlSchemaParse(annotationSchemaParserContext);
   xmlSchemaValidCtxtPtr validatorContextPtr = xmlSchemaNewValidCtxt(schemaPtr);
   // there is something fishy going on in here
   // on MacOS, if we call these functions first, the validator works
   // otherwise it fails via a segfault
   // this suggests that there is something major wrong
   // possibly even with libxml itself
   xmlSchemaSetValidErrors(validatorContextPtr, 
                           DUMMYFUNCTION,
                           NULL, NULL);
   BOOST_REQUIRE(xmlSchemaValidateDoc(validatorContextPtr, brokenDoc) != 0);
   // On Ubuntu 14.04, this seems to cause random segfaults without
   // any apparent reason
   // no time to investigate, unfortunately...
#if LIBXML_VERSION != 20901
   BOOST_REQUIRE(xmlSchemaValidateDoc(validator.validatorCtxt.get(), brokenDoc) != 0);
#endif // LIBXML_VERSION != 20901
   BOOST_REQUIRE(xmlSchemaValidateDoc(validatorContextPtr, defaultAnnBroken.internalDoc.get()));

   xmlSchemaFreeValidCtxt(validatorContextPtr);
   xmlSchemaFree(schemaPtr);
   xmlSchemaFreeParserCtxt(annotationSchemaParserContext);
   xmlFreeDoc(annotationSchemaDocPtr);
   xmlFreeDoc(brokenDoc);

#ifndef __APPLE__ 
#if LIBXML_VERSION != 20901
   // this causes an unexpected crash on MacOS for some unknown reason
   // ALSO verified on Ubuntu 14.04
   // no time to investigate, unfortunately...
   BOOST_REQUIRE(!validator.validate(defaultAnnBroken));
#endif // 2BLIBXML_VERSION != 20901
#endif // __APPLE__ 

   {
     XmlDocument annotationTemplateSchema(getAnnotationSchemaFile());

     validator = Validator(annotationTemplateSchema);
   }

   BOOST_CHECK(validator.validate(defaultAnn));

   BOOST_CHECK(!validator.validate(defaultAnnBroken));   

 }

  oss.seekp(0);
  SomPakHeaderDocument doc2("slmotion", "slmotion feature", "all features produced by SLMotion in one long vector", "video" /* or should it be image? */);
  oss << doc2;
  std::istringstream iss(oss.str());
  std::string s;
  getline(iss, s);
  BOOST_CHECK(s == "<?xml version=\"1.0\"?>");
  getline(iss, s);
  BOOST_CHECK(s == "<feature>");
  getline(iss, s);
  BOOST_CHECK(s == "  <name>slmotion</name>");
  getline(iss, s);
  BOOST_CHECK(s == "  <longname>slmotion feature</longname>");
  getline(iss, s);
  BOOST_CHECK(s == "  <shorttext>all features produced by SLMotion in one long vector</shorttext>");
  getline(iss, s);
  BOOST_CHECK(s == "  <target>video</target>");
  getline(iss, s);
  BOOST_CHECK(s == "  <vectorlength>0</vectorlength>");
  getline(iss, s);
  BOOST_CHECK(s == "  <options></options>");
  getline(iss, s);
  BOOST_CHECK(s == "  <hasrawin>yes</hasrawin>");
  getline(iss, s);
  BOOST_CHECK(s == "  <hasrawout>yes</hasrawout>");
  getline(iss, s);
  BOOST_CHECK(s == "  <components/>");
  getline(iss, s);
  BOOST_CHECK(s == "  <version>$Id: </version>");
  getline(iss, s);
  BOOST_CHECK(s.substr(0, 8) == "  <date>");
  BOOST_CHECK(s.substr(32) == "</date>");
  getline(iss, s);
  BOOST_CHECK(s == string("  <user>") + getlogin() + "</user>");
  getline(iss, s);
  char hostname[256];
  gethostname(hostname, 256);
  // BOOST_CHECK(s == "  <host>pc115.ics.hut.fi</host>");
  BOOST_CHECK(s == "  <host>" + string(hostname) + "</host>");
  getline(iss, s);
  char* cwd = getcwd(NULL, 0);
  BOOST_CHECK(s == string("  <cwd>") + cwd + "</cwd>");
  free(cwd);
  getline(iss, s);
  BOOST_CHECK(s == "  <cmdline></cmdline>");
  getline(iss, s);
  BOOST_CHECK(s == "</feature>");

  XmlDocument doc3;
  XmlNode root("juuri");
  XmlNamespace schemaInstanceNamespace(root.newNamespace("http://www.w3.org/2001/XMLSchema-instance", "xsi"));
  schemaInstanceNamespace.newProperty("noNamespaceSchemaLocation",
                                      "file:avatech-tier.xsd");
  doc3.setRootElement(root);

  BOOST_CHECK(!root.hasChildren());

  root.addChild(XmlNode("foo"));
  root.addChild(XmlNode("bar"));

  BOOST_CHECK(root.hasChildren());
  BOOST_CHECK(root.getFirstChild().getName() == "foo");

  root.getFirstChild().addPreviousSibling(XmlNode("fuu"));
  BOOST_CHECK(root.getFirstChild().getName() == "fuu");

  oss.str("");
  oss << doc3;
  std::istringstream iss2(oss.str());
  std::getline(iss2, s);
  BOOST_CHECK(s == "<?xml version=\"1.0\"?>");
  std::getline(iss2, s);
  BOOST_CHECK(s == "<juuri xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"file:avatech-tier.xsd\">");
  std::getline(iss2, s);
  BOOST_CHECK(s == "  <fuu/>");
  std::getline(iss2, s);
  BOOST_CHECK(s == "  <foo/>");
  std::getline(iss2, s);
  BOOST_CHECK(s == "  <bar/>");
  std::getline(iss2, s);
  BOOST_CHECK(s == "</juuri>");

  EafDocument eaf1;
  oss.str("");
  oss << eaf1;
  std::istringstream iss3(oss.str());
  std::getline(iss3, s);
  BOOST_CHECK(s == "<?xml version=\"1.0\"?>");
  std::getline(iss3, s);
#if 0
  BOOST_CHECK(s.substr(0, 125) == "<ANNOTATION_DOCUMENT xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"EAFv2.6.xsd\" DATE=\"");
  BOOST_CHECK(s.substr(150) == "\" AUTHOR=\"\" VERSION=\"2.6\" FORMAT=\"2.6\">");
#endif
  const std::string foo = "<ANNOTATION_DOCUMENT xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"";
  BOOST_CHECK(s.substr(0, foo.length()) == foo);
  std::getline(iss3, s);
  BOOST_CHECK(s == "  <HEADER TIME_UNITS=\"milliseconds\"/>");
  std::getline(iss3, s);
  BOOST_CHECK(s == "  <TIME_ORDER/>");
  std::getline(iss3, s);
  BOOST_CHECK(s == "  <LINGUISTIC_TYPE LINGUISTIC_TYPE_ID=\"default-lt\" TIME_ALIGNABLE=\"true\" GRAPHIC_REFERENCES=\"false\"/>");
  std::getline(iss3, s);
  BOOST_CHECK(s == "</ANNOTATION_DOCUMENT>");

  eaf1.addMediaDescriptor(TESTVIDEOFILE);
  std::deque<Annotation> annotations;
  annotations.push_back(Annotation { "Foo", 0, 10 } );
  annotations.push_back(Annotation { "Bar", 11, 20 } );  
  eaf1.addTier(std::pair<string, decltype(annotations)>("FooBar", annotations));
  std::string documentContent =
    "<?xml version=\"1.0\"?>\n"
    "<ANNOTATION_DOCUMENT xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"EAFv2.6.xsd\" DATE=\"2012-08-13T19:29:40+03:00\" AUTHOR=\"\" VERSION=\"2.6\" FORMAT=\"2.6\">\n"
    "  <HEADER TIME_UNITS=\"milliseconds\">\n"
    "    <MEDIA_DESCRIPTOR MEDIA_URL=\"file:///fs/project/imagedb/slmotion/tests/testivideo4.avi\" RELATIVE_MEDIA_URL=\"/share/imagedb/slmotion/tests/testivideo4.avi\" MIME_TYPE=\"video/*\"/>\n"
    "  </HEADER>\n"
    "  <TIME_ORDER>\n"
    "    <TIME_SLOT TIME_SLOT_ID=\"sltc0\" TIME_VALUE=\"0\"/>\n"
    "    <TIME_SLOT TIME_SLOT_ID=\"sltc1\" TIME_VALUE=\"10\"/>\n"
    "    <TIME_SLOT TIME_SLOT_ID=\"sltc2\" TIME_VALUE=\"11\"/>\n"
    "    <TIME_SLOT TIME_SLOT_ID=\"sltc3\" TIME_VALUE=\"20\"/>\n"
    "  </TIME_ORDER>\n"
    "  <TIER ANNOTATOR=\"SLMotion\" LINGUISTIC_TYPE_REF=\"default-lt\" TIER_ID=\"FooBar\">\n"
    "    <ANNOTATION>\n"
    "      <ALIGNABLE_ANNOTATION ANNOTATION_ID=\"slann0\" TIME_SLOT_REF1=\"sltc0\" TIME_SLOT_REF2=\"sltc1\">\n"
    "        <ANNOTATION_VALUE>Foo</ANNOTATION_VALUE>\n"
    "      </ALIGNABLE_ANNOTATION>\n"
    "    </ANNOTATION>\n"
    "    <ANNOTATION>\n"
    "      <ALIGNABLE_ANNOTATION ANNOTATION_ID=\"slann1\" TIME_SLOT_REF1=\"sltc2\" TIME_SLOT_REF2=\"sltc3\">\n"
    "        <ANNOTATION_VALUE>Bar</ANNOTATION_VALUE>\n"
    "      </ALIGNABLE_ANNOTATION>\n"
    "    </ANNOTATION>\n"
    "  </TIER>\n"
    "  <LINGUISTIC_TYPE LINGUISTIC_TYPE_ID=\"default-lt\" TIME_ALIGNABLE=\"true\" GRAPHIC_REFERENCES=\"false\"/>\n"
    "</ANNOTATION_DOCUMENT>";

  auto replaceDate = [](std::string in)->std::string {
    std::regex rx("DATE=\"[0-9]{4}-[0-9]{2}-[0-9]{2}T[0-9]{2}:[0-9]{2}:[0-9]{2}\\+[0-9]{2}:[0-9]{2}\"");
    in = std::regex_replace(in, rx, "DATE=\"\"");
    rx = std::regex("MEDIA_URL=\".*?\"");
    return std::regex_replace(in, rx, "MEDIA_URL=\"\"");
  };

  auto lineByLineEqual = [](const std::string& lh, const std::string& rh) {
    std::istringstream issl(lh), issr(rh);
    std::string sl, sr;
    while (true) {
      bool b1 = !getline(issl, sl).fail();
      bool b2 = !getline(issr, sr).fail();
      if (!b1 && !b2)
        break;
      if (b1 != b2)
        return false;
      if (sl != sr) {
        std::cerr << "Conflict:" << std::endl << sl << std::endl << "vs." << std::endl
                  << sr << std::endl;
        return false;
      }
    }
    return true;
  };

  documentContent = replaceDate(documentContent);

  oss.str("");
  oss << eaf1;
  // fix the eaf xsd file location
  s = replaceRegex("xsi:noNamespaceSchemaLocation=\"[^\"]*\"", 
                   "xsi:noNamespaceSchemaLocation=\"EAFv2.6.xsd\"", oss.str());
  BOOST_CHECK(lineByLineEqual(replaceDate(s), documentContent));
  eaf1.save("tests_cpp_temp_testfile.eaf");
  EafDocument eaf2("tests_cpp_temp_testfile.eaf");
  oss.str("");
  oss << eaf2;
  s = replaceRegex("xsi:noNamespaceSchemaLocation=\"[^\"]*\"", 
                   "xsi:noNamespaceSchemaLocation=\"EAFv2.6.xsd\"", oss.str());
  BOOST_CHECK(lineByLineEqual(replaceDate(s), documentContent));  
}


BOOST_AUTO_TEST_CASE( testNorm ) {
  std::mt19937 rng;
  std::normal_distribution<double> normal_dist(0,1);
  cv::Vec3b v1;
  cv::Vec3f v2;
  cv::Vec3d v3;
  cv::Vec<float,8> v4;
  for (int i = 0; i < 3; ++i) {
    v1[i] = normal_dist(rng)*255;
    v2[i] = normal_dist(rng);
    v3[i] = normal_dist(rng);
  }
  for (int i = 0; i < 8; ++i) 
    v4[i] = normal_dist(rng);

  cv::Mat norms(4,3,CV_64FC1);
  norms.at<double>(0,0) = std::abs(v1[0])+std::abs(v1[1])+std::abs(v1[2]);
  norms.at<double>(1,0) = std::abs(v2[0])+std::abs(v2[1])+std::abs(v2[2]);
  norms.at<double>(2,0) = std::abs(v3[0])+std::abs(v3[1])+std::abs(v3[2]);
  norms.at<double>(3,0) = 0;
  for (int i = 0; i < 8; ++i)
    norms.at<double>(3,0) += std::abs(v4[i]);

  norms.at<double>(0,1) = norms.at<double>(1,1) = norms.at<double>(2,1) = 0;
#if OPENCV_VERSION_242
  uchar uc1 = 0;
#endif
  for (int i = 0; i < 3; ++i) {
#if OPENCV_VERSION_242
    uc1 += v1[i]*v1[i];
#else // OPENCV_VERSION_242
    norms.at<double>(0,1) += v1[i]*v1[i];
#endif // OPENCV_VERSION_242
    norms.at<double>(1,1) += v2[i]*v2[i];
    norms.at<double>(2,1) += v3[i]*v3[i];
  }
#if OPENCV_VERSION_242
  norms.at<double>(0,1) = uc1;
#endif // OPENCV_VERSION_242
  for (int i = 0; i < 3; ++i) 
    norms.at<double>(i,1) = std::sqrt(norms.at<double>(i,1));
  
  norms.at<double>(3,1) = 0;
  for (int i = 0; i < 8; ++i) 
    norms.at<double>(3,1) += v4[i]*v4[i];
  norms.at<double>(3,1) = std::sqrt(norms.at<double>(3,1));

  norms.at<double>(0,2) = norms.at<double>(1,2) = norms.at<double>(2,2) = norms.at<double>(3,2) = -1;

  for (int i = 0; i < 3; ++i) {
    if (std::abs(v1[i]) > norms.at<double>(0,2))
      norms.at<double>(0,2) = std::abs(v1[i]);

    if (std::abs(v2[i]) > norms.at<double>(1,2))
      norms.at<double>(1,2) = std::abs(v2[i]);

    if (std::abs(v3[i]) > norms.at<double>(2,2))
      norms.at<double>(2,2) = std::abs(v3[i]);
  }

  for (int i = 0; i < 8; ++i) 
    if (std::abs(v4[i]) > norms.at<double>(3,2))
      norms.at<double>(3,2) = std::abs(v4[i]);

  BOOST_CHECK(norms.at<double>(0,0) == norm(v1, cv::NORM_L1));
  BOOST_CHECK(norms.at<double>(1,0) == norm(v2, cv::NORM_L1));
  BOOST_CHECK(norms.at<double>(2,0) == norm(v3, cv::NORM_L1));
  BOOST_CHECK(almostEqual(norms.at<double>(3,0), norm(v4, cv::NORM_L1), 1e-06));
#if !OPENCV_VERSION_242


#if !OPENCV_VERSION_231
  BOOST_CHECK(norms.at<double>(0,0) == cv::norm(v1, cv::NORM_L1));
#endif // !(CV_MAJOR_VERSION == 2 && CV_MINOR_VERSION == 3 && CV_SUBMINOR_VERSION == 1) 


  BOOST_CHECK(norms.at<double>(1,0) == cv::norm(v2, cv::NORM_L1));
  BOOST_CHECK(norms.at<double>(2,0) == cv::norm(v3, cv::NORM_L1));
  BOOST_CHECK(almostEqual(norms.at<double>(3,0), cv::norm(v4, cv::NORM_L1), 1e-06));
#endif // !(CV_MAJOR_VERSION == 2 && CV_MINOR_VERSION == 4 && CV_SUBMINOR_VERSION == 2)

  BOOST_CHECK(almostEqual(norms.at<double>(0,1), norm(v1, cv::NORM_L2), 1e-05));
  BOOST_CHECK(almostEqual(norms.at<double>(1,1), norm(v2, cv::NORM_L2), 1e-05));
  BOOST_CHECK(norms.at<double>(2,1) == norm(v3, cv::NORM_L2));
  BOOST_CHECK(almostEqual(norms.at<double>(3,1), norm(v4, cv::NORM_L2), 1e-05));
#if !OPENCV_VERSION_242

#if !OPENCV_VERSION_231
  BOOST_CHECK(almostEqual(norms.at<double>(0,1), cv::norm(v1, cv::NORM_L2), 1e-05));
#endif // !OPENCV_VERSION_231

  BOOST_CHECK(almostEqual(norms.at<double>(1,1), cv::norm(v2, cv::NORM_L2), 1e-06));
  BOOST_CHECK(norms.at<double>(2,1) == cv::norm(v3, cv::NORM_L2));
  BOOST_CHECK(almostEqual(norms.at<double>(3,1), cv::norm(v4, cv::NORM_L2), 1e-05));

#endif // !(CV_MAJOR_VERSION == 2 && CV_MINOR_VERSION == 4 && CV_SUBMINOR_VERSION == 2)

  BOOST_CHECK(norms.at<double>(0,2) == norm(v1, cv::NORM_INF));
  BOOST_CHECK(norms.at<double>(1,2) == norm(v2, cv::NORM_INF));
  BOOST_CHECK(almostEqual(norms.at<double>(2,2), norm(v3, cv::NORM_INF), 1e-05));
  BOOST_CHECK(almostEqual(norms.at<double>(3,2), norm(v4, cv::NORM_INF), 1e-05));

#if !OPENCV_VERSION_242

#if !OPENCV_VERSION_231
  BOOST_CHECK(norms.at<double>(0,2) == cv::norm(v1, cv::NORM_INF));
#endif

  BOOST_CHECK(norms.at<double>(1,2) == cv::norm(v2, cv::NORM_INF));
  BOOST_CHECK(almostEqual(norms.at<double>(2,2), cv::norm(v3, cv::NORM_INF), 1e-05));
  BOOST_CHECK(almostEqual(norms.at<double>(3,2), cv::norm(v4, cv::NORM_INF), 1e-05));
#endif // !(CV_MAJOR_VERSION == 2 && CV_MINOR_VERSION == 4 && CV_SUBMINOR_VERSION == 2)
}

BOOST_AUTO_TEST_CASE( testLimitedPriorityQueue ) { 
  std::vector<int> numbers { 10222, 2764, 29010, 17726, 10807, 16556, 1222, 5928, 30047, 28327,
      21624, 20584, 3436, 32335, 29120, 18549, 19309, 10588, 368, 6560 };
  std::vector<int> sortedNumbers(numbers.begin(), numbers.end());
  std::sort(sortedNumbers.begin(), sortedNumbers.end());

  LimitedPriorityQueue<int> lpq1(5);
  for (size_t i = 0; i < numbers.size(); ++i) {
    lpq1.insert(numbers[i]);
    BOOST_REQUIRE(lpq1.size() == (i > 4 ? 5 : i + 1));
    if (i > 0) {
      BOOST_REQUIRE(lpq1.top() > lpq1.bottom());
      std::vector<int> temp(lpq1.begin(), lpq1.end());
      BOOST_REQUIRE(temp.size() == lpq1.size());
      auto it = lpq1.begin();
      auto jt = it++;
      while (it != lpq1.end())
        BOOST_REQUIRE(*jt++ < *it++);
    }
    else
      BOOST_REQUIRE(lpq1.top() == lpq1.bottom());
  }

  BOOST_REQUIRE(lpq1.top() == sortedNumbers.back());
  BOOST_REQUIRE(lpq1.bottom() == sortedNumbers[15]);

  std::vector<cv::Point> points;
  for (size_t i = 0; i < numbers.size(); i += 2)
    points.push_back(cv::Point(numbers[i], numbers[i+1]));

  LimitedPriorityQueue<cv::Point, decltype(&L2Compare)> lpq2(3, &L2Compare);
  
  for (size_t i = 0; i < points.size(); ++i) {
    lpq2.insert(points[i]);
    BOOST_REQUIRE(lpq2.size() == (i > 2 ? 3 : i + 1));
    if (i > 0) {
      BOOST_REQUIRE(cv::norm(lpq2.top()) > cv::norm(lpq2.bottom()));
      std::vector<cv::Point> temp(lpq2.begin(), lpq2.end());
      BOOST_REQUIRE(temp.size() == lpq2.size());
      auto it = lpq2.begin();
      auto jt = it++;
      while (it != lpq2.end())
        BOOST_REQUIRE(cv::norm(*jt++) < cv::norm(*it++));
    }
    else
      BOOST_REQUIRE(lpq2.top() == lpq2.bottom());
  }

  struct XCompare {
    bool operator()(const cv::Point& lh, const cv::Point& rh) {
      return lh.x < rh.x;
    }
  };

  LimitedPriorityQueue<cv::Point, XCompare> lpq3(3);

  for (size_t i = 0; i < points.size(); ++i) {
    lpq3.insert(points[i]);
    BOOST_REQUIRE(lpq3.size() == (i > 2 ? 3 : i + 1));
    if (i > 0) {
      BOOST_REQUIRE(lpq3.top().x > lpq3.bottom().x);
      std::vector<cv::Point> temp(lpq3.begin(), lpq3.end());
      BOOST_REQUIRE(temp.size() == lpq3.size());
      auto it = lpq3.begin();
      auto jt = it++;
      while (it != lpq3.end())
        BOOST_REQUIRE((jt++)->x < (it++)->x);
    }
    else
      BOOST_REQUIRE(lpq3.top() == lpq3.bottom());
  }

  auto YCompare = [](const cv::Point& lh, const cv::Point& rh)->bool {
    return lh.y < rh.y;
  };

  LimitedPriorityQueue<cv::Point, decltype(YCompare)> lpq4(3, YCompare);

  for (size_t i = 0; i < points.size(); ++i) {
    lpq4.insert(points[i]);
    BOOST_REQUIRE(lpq4.size() == (i > 2 ? 3 : i + 1));
    if (i > 0) {
      BOOST_REQUIRE(lpq4.top().y > lpq4.bottom().y);
      std::vector<cv::Point> temp(lpq4.begin(), lpq4.end());
      BOOST_REQUIRE(temp.size() == lpq4.size());
      auto it = lpq4.begin();
      auto jt = it++;
      while (it != lpq4.end())
        BOOST_REQUIRE((jt++)->y < (it++)->y);
    }
    else
      BOOST_REQUIRE(lpq4.top() == lpq4.bottom());
  }
}



BOOST_AUTO_TEST_CASE( testAnalysisController1 ) { 
  if (slmotion::debug > 0)
    slmotion::debug = 0;

  vector<Analyser> anals;
  SLMotionConfiguration opts;
  vector<string> argvs { "./tests", "--components", 
      "RuleSkinDetector,BlobExtractor"
      };
  vector<vector<char> > argvc(argvs.size());
  vector<char*> argv(argvc.size());
  for (size_t i = 0; i < argvs.size(); ++i) {
    argvc[i] = vector<char>(argvs[i].c_str(), argvs[i].c_str() + 
                            argvs[i].length() + 1);
    argv[i] = &argvc[i][0];
  }

  createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                  vector<string>(), anals, opts, argvc.size(),
                  &argv[0]);

   BOOST_CHECK(anals[0].getBlackBoard().has(0, BLACKBOARD_BLOBS_ENTRY) == false);

  AnalysisController analCon(anals[0].visualiser, anals[0].slio, false);
  assert(anals.size() == 1);
  analCon.analyseVideofiles(anals.front(), 0, 10, false, // false, 
                            "", "", ""
                            // opts.visualiserCombinationStyle
                            );
  BOOST_REQUIRE(anals[0].getBlackBoard().has(0, BLACKBOARD_BLOBS_ENTRY));
  BOOST_CHECK(anals[0].getBlackBoard().get<std::vector<Blob> >(0, BLACKBOARD_BLOBS_ENTRY)
              ->size() > 0);
}



BOOST_AUTO_TEST_CASE(testAnalysisController2) {
  // vector<Analyser> anals = 
  //   createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE3 }),
  //                   parseConfigurationFiles(vector<string>(), 1));
  // AnalysisController analCon(anals[0].visualiser,
  //                            anals[0].slio,
  //                            false);
  // analCon.analyseVideofiles(anals, 0, SIZE_MAX, false, false);
  vector<Analyser> anals;
  SLMotionConfiguration opts;
  vector<string> argvs { "./tests", "--components", 
      "RuleSkinDetector,BlobExtractor,FaceDetector,"
      "BodyPartCollector,ColourSpaceConverter,KLTTracker,"
      "AsmTracker,FeatureCreator"
      };
  vector<vector<char> > argvc(argvs.size());
  vector<char*> argv(argvc.size());
  for (size_t i = 0; i < argvs.size(); ++i) {
    argvc[i] = vector<char>(argvs[i].c_str(), argvs[i].c_str() + 
                            argvs[i].length() + 1);
    argv[i] = &argvc[i][0];
  }
  createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                  vector<string>(), anals, opts, argv.size(), 
                  &argv[0]);

  AnalysisController analCon(anals[0].visualiser, anals[0].slio, false);
  assert(anals.size() == 1);
  analCon.analyseVideofiles(anals.front(), 0, SIZE_MAX, false, // false,
                            "", "", ""
                            //opts.visualiserCombinationStyle
                            );

  BlackBoard& bb = *anals[0].blackboard;
  FeatureCreator fc(&bb, anals[0].frameSource.get());
  AnalysisResult analRes = fc.extractRawResults(0,10,bb);
  BOOST_REQUIRE(analRes.firstFrame == 0);
  BOOST_REQUIRE(analRes.lastFrame == 10);
  BOOST_REQUIRE(analRes.KLTPoints.size() == 10);
  BOOST_REQUIRE(analRes.headASMs.size() == 10);
  BOOST_REQUIRE(analRes.leftHandASMs.size() == 10);
  BOOST_REQUIRE(analRes.rightHandASMs.size() == 10);
  for (size_t i = analRes.firstFrame; i < analRes.lastFrame; ++i) {
    BOOST_CHECK(analRes.KLTPoints[i] == *bb.get<std::set<TrackedPoint>>(i, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY));
    BOOST_CHECK(analRes.headASMs[i] == *bb.get<Asm::Instance>(i, ASM_TRACKER_HEAD_ASM_INSTANCE));
    // BOOST_CHECK(analRes.headASMs[i].second == bb.get<Point>(i, ASM_TRACKER_HEAD_ANCHOR_BLACKBOARD_ENTRY));
    BOOST_CHECK(analRes.leftHandASMs[i] == *bb.get<Asm::Instance>(i, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE));
    // BOOST_CHECK(analRes.leftHandASMs[i].second == bb.get<Point>(i, ASM_TRACKER_LEFT_HAND_ANCHOR_BLACKBOARD_ENTRY));
    BOOST_CHECK(analRes.rightHandASMs[i] == *bb.get<Asm::Instance>(i, ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE));
    // BOOST_CHECK(analRes.rightHandASMs[i].second == bb.get<Point>(i, ASM_TRACKER_RIGHT_HAND_ANCHOR_BLACKBOARD_ENTRY));
  }

  // czech features
  BlackBoardPointer<FeatureVector> fv = anals[0].getBlackBoard().get<FeatureVector>(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY);
  BOOST_REQUIRE(fv->components == SOM_PAK_COMPONENTS);
  BOOST_REQUIRE(fv->features.size() == fv->components.size());

  // KLT features have already been checked, so concentrate on ASM features
  BOOST_REQUIRE(fv->firstFrame == 0);
  BOOST_REQUIRE(fv->lastFrame == 10);
  for (size_t i = fv->firstFrame; i < fv->lastFrame; ++i) {
#if 0
    PDM hPdm = bb.get<PDM>(i, ASM_TRACKER_HEAD_ASM);
    Point hAnc = bb.get<Point>(i, ASM_TRACKER_HEAD_ANCHOR_BLACKBOARD_ENTRY);
    PDM lhPdm = bb.get<PDM>(i, ASM_TRACKER_LEFT_HAND_ASM);
    Point lhAnc = bb.get<Point>(i, ASM_TRACKER_LEFT_HAND_ANCHOR_BLACKBOARD_ENTRY);
    PDM rhPdm = bb.get<PDM>(i, ASM_TRACKER_RIGHT_HAND_ASM);
    Point rhAnc = bb.get<Point>(i, ASM_TRACKER_RIGHT_HAND_ANCHOR_BLACKBOARD_ENTRY);
    BOOST_CHECK(PDM::intersectionArea(hPdm, lhPdm, hAnc, lhAnc) == fv.get<int>(37, i));
    BOOST_CHECK(PDM::intersectionArea(hPdm, rhPdm, hAnc, rhAnc) == fv.get<int>(38, i));
    BOOST_CHECK(PDM::intersectionArea(lhPdm, rhPdm, lhAnc, rhAnc) == fv.get<int>(39, i));
#endif
    Asm::Instance hInstance = *bb.get<Asm::Instance>(i, ASM_TRACKER_HEAD_ASM_INSTANCE);
    Asm::Instance lhInstance = *bb.get<Asm::Instance>(i, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE);
    Asm::Instance rhInstance = *bb.get<Asm::Instance>(i, ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE);
    BOOST_CHECK(intersectionArea(hInstance, lhInstance) == fv->get<int>(37, i));
    BOOST_CHECK(intersectionArea(hInstance, rhInstance) == fv->get<int>(38, i));
    BOOST_CHECK(intersectionArea(lhInstance, rhInstance) == fv->get<int>(39, i));

    size_t j = i > 0 ? i - 1: 0;
#if 0
    PDM hPdmPast = bb.get<PDM>(j, ASM_TRACKER_HEAD_ASM);
    Point hAncPast = bb.get<Point>(j, ASM_TRACKER_HEAD_ANCHOR_BLACKBOARD_ENTRY);
    PDM lhPdmPast = bb.get<PDM>(j, ASM_TRACKER_LEFT_HAND_ASM);
    Point lhAncPast = bb.get<Point>(j, ASM_TRACKER_LEFT_HAND_ANCHOR_BLACKBOARD_ENTRY);
    PDM rhPdmPast = bb.get<PDM>(j, ASM_TRACKER_RIGHT_HAND_ASM);
    Point rhAncPast = bb.get<Point>(j, ASM_TRACKER_RIGHT_HAND_ANCHOR_BLACKBOARD_ENTRY);
#endif
    Asm::Instance hInstancePast = *bb.get<Asm::Instance>(j, ASM_TRACKER_HEAD_ASM_INSTANCE);
    Asm::Instance lhInstancePast = *bb.get<Asm::Instance>(j, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE);
    Asm::Instance rhInstancePast = *bb.get<Asm::Instance>(j, ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE);

    j = i < 9 ? i + 1 : 9;
#if 0
    PDM hPdmFuture = bb.get<PDM>(j, ASM_TRACKER_HEAD_ASM);
    Point hAncFuture = bb.get<Point>(j, ASM_TRACKER_HEAD_ANCHOR_BLACKBOARD_ENTRY);
    PDM lhPdmFuture = bb.get<PDM>(j, ASM_TRACKER_LEFT_HAND_ASM);
    Point lhAncFuture = bb.get<Point>(j, ASM_TRACKER_LEFT_HAND_ANCHOR_BLACKBOARD_ENTRY);
    PDM rhPdmFuture = bb.get<PDM>(j, ASM_TRACKER_RIGHT_HAND_ASM);
    Point rhAncFuture = bb.get<Point>(j, ASM_TRACKER_RIGHT_HAND_ANCHOR_BLACKBOARD_ENTRY);
#endif
    Asm::Instance hInstanceFuture = *bb.get<Asm::Instance>(j, ASM_TRACKER_HEAD_ASM_INSTANCE);
    Asm::Instance lhInstanceFuture = *bb.get<Asm::Instance>(j, ASM_TRACKER_LEFT_HAND_ASM_INSTANCE);
    Asm::Instance rhInstanceFuture = *bb.get<Asm::Instance>(j, ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE);

    // Point2d futCent = hPdmFuture.computeCentroid(hAncFuture);
    // Point2d pastCent = hPdmPast.computeCentroid(hAncPast);
    Point2d futCent = hInstanceFuture.computeCentroid();
    Point2d pastCent = hInstancePast.computeCentroid();

    Point2d cVel = (futCent - pastCent) * 0.5;
    BOOST_CHECK(cVel.x == fv->get<double>(40, i));
    BOOST_CHECK(cVel.y == fv->get<double>(41, i));
    BOOST_CHECK(cv::norm(cVel) == fv->get<double>(42, i));

    // futCent = lhPdmFuture.computeCentroid(lhAncFuture);
    // pastCent = lhPdmPast.computeCentroid(lhAncPast);
    futCent = lhInstanceFuture.computeCentroid();
    pastCent = lhInstancePast.computeCentroid();
    cVel = (futCent - pastCent) * 0.5;
    BOOST_CHECK(cVel.x == fv->get<double>(43, i));
    BOOST_CHECK(cVel.y == fv->get<double>(44, i));
    BOOST_CHECK(cv::norm(cVel) == fv->get<double>(45, i));

    futCent = rhInstanceFuture.computeCentroid();
    pastCent = rhInstancePast.computeCentroid();
    cVel = (futCent - pastCent) * 0.5;
    BOOST_CHECK(cVel.x == fv->get<double>(46, i));
    BOOST_CHECK(cVel.y == fv->get<double>(47, i));
    BOOST_CHECK(cv::norm(cVel) == fv->get<double>(48, i));

    double theta = computeOrientation(hInstance);
    BOOST_CHECK(theta == fv->get<double>(49, i));
    double omega = (computeOrientation(hInstanceFuture) - 
                    computeOrientation(hInstancePast)) * 0.5;
    BOOST_CHECK(omega == fv->get<double>(50, i));

    theta = computeOrientation(lhInstance);
    BOOST_CHECK(theta == fv->get<double>(51, i));
    omega = (computeOrientation(lhInstanceFuture) - 
             computeOrientation(lhInstancePast)) * 0.5;
    BOOST_CHECK(omega == fv->get<double>(52, i));

    theta = computeOrientation(rhInstance);
    BOOST_CHECK(theta == fv->get<double>(53, i));
    omega = (computeOrientation(rhInstanceFuture) - 
             computeOrientation(rhInstancePast)) * 0.5;
    BOOST_CHECK(omega == fv->get<double>(54, i));
  }
}


BOOST_AUTO_TEST_CASE( testBlackBoardPointers ) {
  int a = rand(), b = rand();
  cv::Vec2i v1 { rand(), rand() }, v2 { rand(), rand() };
  cv::Point2i p1 { rand(), rand() };

  std::vector<std::string> labels(5);
  for (int i = 0; i < 5; ++i)
    labels[i] = getRandomString();

  BlackBoard bb;
  bb.set(a, labels[0], v1);
  bb.set(b, labels[1], v2);
  bb.set(labels[2], v1);
  bb.set(labels[3], v2);
  bb.set(labels[4], p1);

  BlackBoardPointer<Vec2i> v1p1 = bb.get<Vec2i>(a, labels[0]);
  BOOST_CHECK(*v1p1 == v1);
  BOOST_CHECK(v1p1.parent->refCount == 1);
  BlackBoardPointer<Vec2i> v1p2(v1p1);
  BOOST_CHECK(*v1p1 == v1);
  BOOST_CHECK(*v1p2 == v1);
  BOOST_CHECK(v1p1.parent == v1p2.parent);
  BOOST_CHECK(v1p1.parent->refCount == 2);
  BlackBoardPointer<Vec2i> v1p3 = v1p2;
  BOOST_CHECK(*v1p1 == v1);
  BOOST_CHECK(*v1p2 == v1);
  BOOST_CHECK(*v1p3 == v1);
  BOOST_CHECK(v1p1.parent == v1p2.parent);
  BOOST_CHECK(v1p2.parent == v1p3.parent);
  BOOST_CHECK(v1p1.parent->refCount == 3);

  BlackBoardPointer<Vec2i>* v1p4 = new BlackBoardPointer<Vec2i>(bb.get<Vec2i>(a, labels[0]));
  BOOST_CHECK(v1p1.parent->refCount == 4);
  delete v1p4;
  BOOST_CHECK(v1p1.parent->refCount == 3);

  BlackBoardPointer<Vec2i>* v2p1 = new BlackBoardPointer<Vec2i>(bb.get<Vec2i>(b, labels[1]));
  BOOST_CHECK(**v2p1 == v2);
  BOOST_CHECK(v2p1->parent->refCount == 1);
  int* rc = &v2p1->parent->refCount;
  BOOST_CHECK(*rc == 1);
  delete v2p1;
  BOOST_CHECK(*rc == 0);
  BlackBoardPointer<Vec2i> v2p2 = bb.get<Vec2i>(b, labels[1]);
  BOOST_CHECK(*rc == 1);
  BOOST_CHECK(*v2p2 == v2);

  BlackBoardPointer<Vec2i> gv1 = bb.get<Vec2i>(labels[2]);
  BOOST_CHECK(*gv1 == v1);
  BlackBoardPointer<Vec2i> gv2 = bb.get<Vec2i>(labels[3]);
  BOOST_CHECK(*gv2 == v2);
  BOOST_CHECK(*rc == 1);

  BlackBoardPointer<Point2i> pp = bb.get<Point2i>(labels[4]);
  BOOST_CHECK(pp->x == p1.x && pp->y == p1.y);

  std::mt19937 engine(time(NULL));
  std::uniform_real_distribution<float> distribution {-100,100};
  auto gen = std::bind(distribution, engine);
  std::vector<double> randomValues(200);
  for (int i = 0; i < 200; ++i)
    randomValues[i] = gen();

  Mat values(20, 10, CV_64FC1);
  for (int i = 0; i < 20; ++i)
    for (int j = 0; j < 10; ++j)
      values.at<double>(i,j) = randomValues[i*10+j];

  std::string matLabel = getRandomString();
  bb.set(matLabel, values, BlackBoardEntry::NONE);
  BOOST_REQUIRE(bb.globalBoard[matLabel]->attribs == BlackBoardEntry::NONE);
  BOOST_REQUIRE(bb.globalBoard[matLabel]->status == BlackBoardEntry::UNCOMPRESSED);
  BOOST_REQUIRE(bb.globalBoard[matLabel]->refCount == 0);
  BOOST_REQUIRE(values.data == boost::any_cast<cv::Mat>(bb.globalBoard[matLabel]->value.get())->data);
  BOOST_CHECK(memcmp(values.data, &randomValues[0], 200*sizeof(double)) == 0);
  BlackBoardPointer<cv::Mat> mp1 = bb.get<cv::Mat>(matLabel);
  BOOST_REQUIRE(bb.globalBoard[matLabel]->refCount == 1);
  BOOST_CHECK(values.data == mp1->data);

  std::string matLabel2 = getRandomString();
  bb.set(matLabel2, values);
  BOOST_REQUIRE(bb.globalBoard[matLabel2]->attribs == BlackBoardEntry::COMPRESS_WITHOUT_REFERENCES);
  BOOST_REQUIRE(bb.globalBoard[matLabel2]->status == BlackBoardEntry::COMPRESSED);
  BOOST_REQUIRE(bb.globalBoard[matLabel2]->refCount == 0);
  BOOST_REQUIRE(values.data != boost::any_cast<cv::Mat>(bb.globalBoard[matLabel2]->value.get())->data);
  BOOST_REQUIRE(boost::any_cast<cv::Mat>(bb.globalBoard[matLabel2]->value.get())->data == nullptr);
  size_t compressedSize = bb.globalBoard[matLabel2]->compressedData.size();
  BOOST_REQUIRE(compressedSize < sizeof(double)*200);

  for (int i = 0; i < 20; ++i)
    for (int j = 0; j < 10; ++j)
      BOOST_CHECK(values.at<double>(i,j) == randomValues[i*10+j]);

  {
    BlackBoardPointer<cv::Mat> bbp2(bb.get<cv::Mat>(matLabel2));
    BOOST_REQUIRE(bb.globalBoard[matLabel2]->attribs == BlackBoardEntry::COMPRESS_WITHOUT_REFERENCES);
    BOOST_REQUIRE(bb.globalBoard[matLabel2]->status == BlackBoardEntry::UNCOMPRESSED);
    BOOST_REQUIRE(bb.globalBoard[matLabel2]->refCount == 1);
    BOOST_REQUIRE(values.data != bbp2->data);
    BOOST_REQUIRE(bbp2->data != nullptr);
    BOOST_REQUIRE(bb.globalBoard[matLabel2]->compressedData.size() == 0);
    
    for (int i = 0; i < 20; ++i)
      for (int j = 0; j < 10; ++j)
        BOOST_CHECK(bbp2->at<double>(i,j) == randomValues[i*10+j]);
  }

  BOOST_REQUIRE(bb.globalBoard[matLabel2]->attribs == BlackBoardEntry::COMPRESS_WITHOUT_REFERENCES);
  BOOST_REQUIRE(bb.globalBoard[matLabel2]->status == BlackBoardEntry::COMPRESSED);
  BOOST_REQUIRE(bb.globalBoard[matLabel2]->refCount == 0);
  BOOST_REQUIRE(boost::any_cast<cv::Mat>(bb.globalBoard[matLabel2]->value.get())->data == nullptr);
  BOOST_REQUIRE(bb.globalBoard[matLabel2]->compressedData.size() == compressedSize);
}



BOOST_AUTO_TEST_CASE( testPruneTrackedAndVelocityAndAccelerationVectors ) {
  Point2f p(1,1);
  TrackedPoint p1(p), p2(p), p3(p), p4(p), p5(p), p6(p);
  BOOST_REQUIRE(p1 != p2);
  BOOST_REQUIRE(p2 != p3);
  BOOST_REQUIRE(p3 != p4);
  BOOST_REQUIRE(p4 != p5);
  vector<set<TrackedPoint>> points;
  points.push_back(set<TrackedPoint> {
      p1, p2, p3
    });

  p1 = p1.relocate(Point2f(2,5));
  p3 = p3.relocate(Point2f(5,2));
  points.push_back(set<TrackedPoint> {
      p1, p3, p6
  });

  p3 = p3.relocate(Point2f(56,2));
  points.push_back(set<TrackedPoint> {
      p3, p4, p6
  });

  p3 = p3.relocate(Point2f(56,20));
  p4 = p4.relocate(Point2f(83,234));
  points.push_back(set<TrackedPoint> {
      p3, p4, p5, p6
  });

  p3 = p3.relocate(Point2f(56,20));
  p5 = p5.relocate(Point2f(83,2));
  points.push_back(set<TrackedPoint> {
      p3, p5
  });

  FeatureCreator::tracked_point_map_vector_t maps = FeatureCreator::pruneTracked(points, 3);
  BOOST_REQUIRE(maps.size() == 5);

  FeatureCreator::velocity_vector_map_vector_t vmaps = FeatureCreator::createVelocityVectors(maps); 
  BOOST_REQUIRE(vmaps.size() == 5);

  FeatureCreator::acceleration_vector_map_vector_t amaps = FeatureCreator::createAccelerationVectors(maps); 
  BOOST_REQUIRE(amaps.size() == 5);

  BOOST_REQUIRE(maps[0].size() == 1);
  BOOST_CHECK(maps[0].count(p3.getId()));
  BOOST_CHECK(maps[0].find(p3.getId())->second == p3);
  BOOST_REQUIRE(vmaps[0].size() == 1);
  BOOST_CHECK(vmaps[0].count(p3.getId()));
  BOOST_CHECK(vmaps[0].find(p3.getId())->second == FeatureCreator::velocity_vector_t(0,0));
  BOOST_REQUIRE(amaps[0].size() == 1);
  BOOST_CHECK(vmaps[0].count(p3.getId()));
  BOOST_CHECK(vmaps[0].find(p3.getId())->second == FeatureCreator::acceleration_vector_t(0,0));

  BOOST_REQUIRE(maps[1].size() == 2);
  BOOST_CHECK(maps[1].count(p3.getId()));
  BOOST_CHECK(maps[1].find(p3.getId())->second == p3);
  BOOST_CHECK(maps[1].count(p6.getId()));
  BOOST_CHECK(maps[1].find(p6.getId())->second == p6);
  BOOST_REQUIRE(vmaps[1].size() == 2);
  BOOST_CHECK(vmaps[1].count(p3.getId()));
  BOOST_CHECK(vmaps[1].find(p3.getId())->second == 0.5 * (Point2f(56,2)-Point2f(1,1)));
  BOOST_CHECK(vmaps[1].count(p6.getId()));
  BOOST_CHECK(vmaps[1].find(p6.getId())->second == FeatureCreator::velocity_vector_t(0,0));
  BOOST_REQUIRE(amaps[1].size() == 2);
  BOOST_CHECK(amaps[1].count(p3.getId()));
  BOOST_CHECK(amaps[1].find(p3.getId())->second == Point2f(56,2)+Point2f(1,1) - 2.0*Point2f(5,2));
  BOOST_CHECK(amaps[1].count(p6.getId()));
  BOOST_CHECK(amaps[1].find(p6.getId())->second == FeatureCreator::acceleration_vector_t(0,0));

  BOOST_REQUIRE(maps[2].size() == 2);
  BOOST_CHECK(maps[2].count(p3.getId()));
  BOOST_CHECK(maps[2].find(p3.getId())->second == p3);
  BOOST_CHECK(maps[2].count(p6.getId()));
  BOOST_CHECK(maps[2].find(p6.getId())->second == p6);
  BOOST_REQUIRE(vmaps[2].size() == 2);
  BOOST_CHECK(vmaps[2].count(p3.getId()));
  BOOST_CHECK(vmaps[2].find(p3.getId())->second == 0.5 * (Point2f(56,20) - Point2f(5,2)));
  BOOST_CHECK(vmaps[2].count(p6.getId()));
  BOOST_CHECK(vmaps[2].find(p6.getId())->second == FeatureCreator::velocity_vector_t(0,0));
  BOOST_REQUIRE(amaps[2].size() == 2);
  BOOST_CHECK(amaps[2].count(p3.getId()));
  BOOST_CHECK(amaps[2].find(p3.getId())->second == Point2f(56,20)+Point2f(5,2) - 2.0*Point2f(56,2));
  BOOST_CHECK(amaps[2].count(p6.getId()));
  BOOST_CHECK(amaps[2].find(p6.getId())->second == FeatureCreator::acceleration_vector_t(0,0));



  BOOST_REQUIRE(maps[3].size() == 2);
  BOOST_CHECK(maps[3].count(p3.getId()));
  BOOST_CHECK(maps[3].find(p3.getId())->second == p3);
  BOOST_CHECK(maps[3].count(p6.getId()));
  BOOST_CHECK(maps[3].find(p6.getId())->second == p6);
  BOOST_REQUIRE(vmaps[3].size() == 2);
  BOOST_CHECK(vmaps[3].count(p3.getId()));
  BOOST_CHECK(vmaps[3].find(p3.getId())->second == 0.5 * (Point2f(56,20)-Point2f(56,2)));
  BOOST_CHECK(vmaps[3].count(p6.getId()));
  BOOST_CHECK(vmaps[3].find(p6.getId())->second == FeatureCreator::velocity_vector_t(0,0));
  BOOST_REQUIRE(amaps[3].size() == 2);
  BOOST_CHECK(amaps[3].count(p3.getId()));
  BOOST_CHECK(amaps[3].find(p3.getId())->second == Point2f(56,20)+Point2f(56,2) - 2.0*Point2f(56,20));
  BOOST_CHECK(amaps[3].count(p6.getId()));
  BOOST_CHECK(amaps[3].find(p6.getId())->second == FeatureCreator::acceleration_vector_t(0,0));


  BOOST_REQUIRE(maps[4].size() == 1);
  BOOST_CHECK(maps[4].count(p3.getId()));
  BOOST_CHECK(maps[4].find(p3.getId())->second == p3);
  BOOST_REQUIRE(vmaps[4].size() == 1);
  BOOST_CHECK(vmaps[4].count(p3.getId()));
  BOOST_CHECK(vmaps[4].find(p3.getId())->second == FeatureCreator::velocity_vector_t(0,0));  
  BOOST_REQUIRE(amaps[4].size() == 1);
  BOOST_CHECK(amaps[4].count(p3.getId()));
  BOOST_CHECK(amaps[4].find(p3.getId())->second == FeatureCreator::acceleration_vector_t(0,0));
}



BOOST_AUTO_TEST_CASE( testCoordinateTransforms) {
  double bodyWidth = 75;
  Point2d throat(150, 50);
  BOOST_CHECK(norm(transformPixelToBodyCoordinates(Point2d(150,50),
                                                   bodyWidth,
                                                   throat) - Point2d(0,0))
              == 0);
  BOOST_CHECK(norm(transformPixelToBodyCoordinates(Point2d(225,50),
                                                   bodyWidth,
                                                   throat) - Point2d(1,0))
              == 0);
  BOOST_CHECK(norm(transformPixelToBodyCoordinates(Point2d(262.5,50),
                                                   bodyWidth,
                                                   throat) - Point2d(1.5,0))
              == 0);
  BOOST_CHECK(almostEqual(norm(transformPixelToBodyCoordinates(Point2d(-50,50),
                                                               bodyWidth,
                                                               throat) 
                               - Point2d(-200./75.,0)),
                          0));
  BOOST_CHECK(almostEqual(norm(transformPixelToBodyCoordinates(Point2d(500,500),
                                                   bodyWidth,
                                                   throat) 
                               - Point2d(350./75.,6))
                          , 0));

  BOOST_CHECK(norm(transformBodyToPixelCoordinates(Point2d(0,0),
                                                   bodyWidth,
                                                   throat) 
                   - Point2d(150,50))
              == 0);
  BOOST_CHECK(norm(transformBodyToPixelCoordinates(Point2d(1,0),
                                                   bodyWidth,
                                                   throat) 
                   - Point2d(225,50))
              == 0);
  BOOST_CHECK(norm(transformBodyToPixelCoordinates(Point2d(1.5,0),
                                                   bodyWidth,
                                                   throat) 
                   - Point2d(262.5,50))
              == 0);
  BOOST_CHECK(almostEqual(norm(transformBodyToPixelCoordinates(Point2d(-200./75.,0),
                                                               bodyWidth,
                                                               throat) 
                               - Point2d(-50,50)),
                          0));
  BOOST_CHECK(almostEqual(norm(transformBodyToPixelCoordinates(Point2d(350./75.,6),
                                                   bodyWidth,
                                                   throat) 
                               - Point2d(500,500))
                          , 0));

  std::uniform_int_distribution<int> distribution(0, 99);
  std::mt19937 engine(time(NULL));
  auto generator = std::bind(distribution, engine);
  PixelToBodyCoordinateTransformer ptbcf(bodyWidth, throat);
  BodyToPixelCoordinateTransformer btpcf(bodyWidth, throat);
  for (int i = 0; i < 10; ++i) {
    Point2d p(generator(), generator());
    
    BOOST_CHECK(ptbcf(p) == transformPixelToBodyCoordinates(p, bodyWidth,
                                                            throat));
    BOOST_CHECK(btpcf(p) == transformBodyToPixelCoordinates(p, bodyWidth,
                                                            throat));
  }
}



BOOST_AUTO_TEST_CASE( testVideoFileSourceOrder ) {
  std::vector<cv::Mat> frames;
  {
    cv::VideoCapture vc(TESTVIDEOFILE6);
    while(vc.grab()) {
      Mat temp;
      vc.retrieve(temp);
      frames.push_back(temp.clone());
    }
  }
  BOOST_REQUIRE(frames.size() == 20);
  std::unique_ptr<vfs_t> vfs_p(new vfs_t(TESTVIDEOFILE6));
  vfs_t& vfs = *vfs_p;
  vector<size_t> frameOrder { 4, 0, 5, 0, 1, 6, 2, 1, 2 };
  for (auto it = frameOrder.cbegin(); it != frameOrder.cend(); ++it) {
    BOOST_CHECK(equal(vfs[*it], frames[*it]));
  }
}



BOOST_AUTO_TEST_CASE( testChains ) {
  cv::Mat m;
  ColourSpace::GREY->convertColour(testVideoFrames[0], m);
  cv::threshold(m, m, 150, 255, cv::THRESH_BINARY);

  vector< vector<Point> > contours_simple;
  cv::findContours(m, contours_simple, CV_RETR_EXTERNAL, 
                   CV_CHAIN_APPROX_SIMPLE);

  vector< vector<Point> > contours_none;
  cv::findContours(m, contours_none, CV_RETR_EXTERNAL,
                   CV_CHAIN_APPROX_NONE);

  BOOST_REQUIRE(contours_simple.size() == contours_none.size());

  for (size_t i = 0; i < contours_simple.size(); ++i) {
    vector<Point> contours_modified = chainApproxSimpleToChainApproxNone(contours_none[i]);
    BOOST_CHECK(contours_none[i] == contours_modified);
  }
}



BOOST_AUTO_TEST_CASE( testMod ) {
  BOOST_CHECK(mod(200, 2) == 0);
  BOOST_CHECK(mod(31, 30) == 1);
  BOOST_CHECK(mod(67, 30) == 7);
  BOOST_CHECK(mod(-15, 5) == 0);
  BOOST_CHECK(mod(-8, 3) == 1);
  BOOST_CHECK(mod(-7, 3) == 2);
}



BOOST_AUTO_TEST_CASE( testLinePointIterator ) {
  Mat testMat(300, 300, CV_32SC1);
  for (int i = 0; i < testMat.rows; ++i)
    for (int j = 0; j < testMat.cols; ++j)
      testMat.at<int>(i,j) = testMat.cols*i + j;

  std::mt19937 rng;
  rng.seed(time(NULL));
  std::uniform_int_distribution<int> rd(0,299);
  auto r = std::bind(rd, rng);

  for (int i = 0; i < 10; ++i) {
    size_t len = 0;

    LinePointIterator it(Point(r(), r()), 
                         Point(r(), r()));
    cv::LineIterator jt(testMat, it.start(), it.end());

    for (size_t j = 0; j < it.length(); ++it, ++jt, ++j) {
      ++len;
      BOOST_CHECK(*reinterpret_cast<int*>(*jt) == testMat.at<int>(*it));
    }
    BOOST_CHECK(it.length() == len);
    BOOST_CHECK(static_cast<int>(len) == jt.count);
  }
}



BOOST_AUTO_TEST_CASE( testCsvImporter ) {
  char filename1[] = "fileXXXXXX.csv";
  char filename2[] = "fileXXXXXX.csv";
  int filed1 = mkstemps(filename1, 4);
  int filed2 = mkstemps(filename2, 4);
  BOOST_REQUIRE(filed1 != -1 && filed2 != -1);
  FILE *file1 = fdopen(filed1, "w");
  FILE *file2 = fdopen(filed2, "w");
  BOOST_REQUIRE(file1 != nullptr && file2 != nullptr);

  fprintf(file1, "1,2,3,4,5,6,7,8,9,10\n");
  fprintf(file1, "11,12.5,13.6,14.7,15.555,16.683,-17.453,18,19,20\n");
  fprintf(file1, "21,22,23,24,25,26,27,28,29,30\n");
  fprintf(file1, "31,32,33,34,35,36,37,38,39,40\n");
  fprintf(file1, "41,42,43,44,45,46,47,38,39,50\n");

  fclose(file1);

  fprintf(file2, "-3245.325, 3543.252768754, -0.5387\n");
  fclose(file2);

  vector<std::string> argsS = { "./tests", "--in-csv-file", 
                                std::string(filename1) + "," + filename2 + "," +
                                filename1, "--csv-fields", "framenr:0,"
                                "head_pos:1-3,lefthand_pos:4:5:6,"
                                "righthand_pos:7-8:9; global_x:0,global_y:1,"
                                "global_z:2; head_pos:1-3,lefthand_pos:4:5:6,"
                                "righthand_pos:7-8:9",
                                "--components", "CsvImporter" };
  vector<char*> args;
  for (auto it = argsS.cbegin(); it != argsS.cend(); ++it) {
    args.push_back(new char[it->length() + 1]);
    strcpy(args.back(), it->c_str());
  }
  
  vector<Analyser> anals;
  SLMotionConfiguration opts;
  createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                    vector<string>(), anals, opts, args.size(), &args[0]); 
  
  for (auto it = args.cbegin(); it != args.cend(); ++it) 
    delete[] *it;

  BOOST_REQUIRE(anals.size() == 1);
  BOOST_REQUIRE(anals[0].components.size() == 1);
  BOOST_REQUIRE(typeid(*anals[0].components[0]) == typeid(CsvImporter));
  CsvImporter& csvi = dynamic_cast<CsvImporter&>(*(anals[0].components[0]));
  BOOST_REQUIRE(csvi.specs.size() == 3);
  BOOST_REQUIRE(csvi.specs[0].size() == 4);
  BOOST_REQUIRE(csvi.specs[0][0].second.size() == 1);
  BOOST_REQUIRE(csvi.specs[0][1].second.size() == 3);
  BOOST_REQUIRE(csvi.specs[0][2].second.size() == 3);
  BOOST_REQUIRE(csvi.specs[0][3].second.size() == 3);

  BOOST_REQUIRE(csvi.specs[0][0].first == "framenr");
  BOOST_REQUIRE(csvi.specs[0][0].second == std::vector<size_t> { 0 } );

  BOOST_REQUIRE(csvi.specs[0][1].first == "head_pos");
  BOOST_REQUIRE((csvi.specs[0][1].second == std::vector<size_t> { 1,2,3 } ));
  BOOST_REQUIRE(csvi.specs[0][2].first == "lefthand_pos");
  BOOST_REQUIRE((csvi.specs[0][2].second == std::vector<size_t> { 4,5,6 } ));
  BOOST_REQUIRE(csvi.specs[0][3].first == "righthand_pos");
  BOOST_REQUIRE((csvi.specs[0][3].second == std::vector<size_t> { 7,8,9 } ));

  BOOST_REQUIRE(csvi.specs[1].size() == 3);
  BOOST_REQUIRE(csvi.specs[1][0].second.size() == 1);
  BOOST_REQUIRE(csvi.specs[1][1].second.size() == 1);
  BOOST_REQUIRE(csvi.specs[1][2].second.size() == 1);
  BOOST_REQUIRE(csvi.specs[1][0].first == "global_x");
  BOOST_REQUIRE(csvi.specs[1][0].second == std::vector<size_t> { 0 } );
  BOOST_REQUIRE(csvi.specs[1][1].first == "global_y");
  BOOST_REQUIRE(csvi.specs[1][1].second == std::vector<size_t> { 1 } );
  BOOST_REQUIRE(csvi.specs[1][2].first == "global_z");
  BOOST_REQUIRE(csvi.specs[1][2].second == std::vector<size_t> { 2 } );

  BOOST_REQUIRE(csvi.specs[2].size() == 3);
  BOOST_REQUIRE(csvi.specs[2][0].second.size() == 3);
  BOOST_REQUIRE(csvi.specs[2][1].second.size() == 3);
  BOOST_REQUIRE(csvi.specs[2][2].second.size() == 3);
  BOOST_REQUIRE(csvi.specs[2][0].first == "head_pos");
  BOOST_REQUIRE((csvi.specs[2][0].second == std::vector<size_t> { 1,2,3 } ));
  BOOST_REQUIRE(csvi.specs[2][1].first == "lefthand_pos");
  BOOST_REQUIRE((csvi.specs[2][1].second == std::vector<size_t> { 4,5,6 } ));
  BOOST_REQUIRE(csvi.specs[2][2].first == "righthand_pos");
  BOOST_REQUIRE((csvi.specs[2][2].second == std::vector<size_t> { 7,8,9 } ));

  BOOST_REQUIRE((csvi.getProvided() == std::set<std::string> {
        "head_pos", "lefthand_pos", "righthand_pos", "global_x", "global_y", 
          "global_z"
          }));

  csvi.processRange(0,50);
  BlackBoard& bb = csvi.getBlackBoard();
  for (int i : { 0, 1, 2, 3, 4, 11, 21, 31, 41 }) {
    BOOST_REQUIRE(bb.has(i, "head_pos"));
    BOOST_REQUIRE(bb.has(i, "lefthand_pos"));
    BOOST_REQUIRE(bb.has(i, "righthand_pos"));
  }
  BOOST_REQUIRE(bb.has("global_x"));
  BOOST_REQUIRE(bb.has("global_y"));
  BOOST_REQUIRE(bb.has("global_z"));
  BOOST_REQUIRE((*bb.get<std::vector<float> >("global_x") == 
                 std::vector<float> { -3245.325 }
                 ));
  BOOST_REQUIRE((*bb.get<std::vector<float> >("global_y") == 
                 std::vector<float> { 3543.252768754 }
                 ));
  BOOST_REQUIRE((*bb.get<std::vector<float> >("global_z") == 
                 std::vector<float> { -0.5387 }
                 ));

  BOOST_REQUIRE((*bb.get<std::vector<float> >(0, "head_pos") ==
                 std::vector<float> { 2,3,4 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(0, "lefthand_pos") ==
                 std::vector<float> { 5,6,7 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(0, "righthand_pos") ==
                 std::vector<float> { 8,9,10 }));

  BOOST_REQUIRE((*bb.get<std::vector<float> >(1, "head_pos") ==
                 std::vector<float> { 12.5,13.6,14.7 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(1, "lefthand_pos") ==
                 std::vector<float> { 15.555,16.683,-17.453 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(1, "righthand_pos") ==
                 std::vector<float> {18,19,20 }));

  BOOST_REQUIRE((*bb.get<std::vector<float> >(2, "head_pos") ==
                 std::vector<float> { 22,23,24 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(2, "lefthand_pos") ==
                 std::vector<float> { 25,26,27 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(2, "righthand_pos") ==
                 std::vector<float> { 28,29,30 }));

  BOOST_REQUIRE((*bb.get<std::vector<float> >(21, "head_pos") ==
                 std::vector<float> { 22,23,24 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(21, "lefthand_pos") ==
                 std::vector<float> { 25,26,27 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(21, "righthand_pos") ==
                 std::vector<float> { 28,29,30 }));

  BOOST_REQUIRE((*bb.get<std::vector<float> >(31, "head_pos") ==
                 std::vector<float> { 32,33,34 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(31, "lefthand_pos") ==
                 std::vector<float> { 35,36,37 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(31, "righthand_pos") ==
                 std::vector<float> { 38,39,40 }));

  BOOST_REQUIRE((*bb.get<std::vector<float> >(3, "head_pos") ==
                 std::vector<float> { 32,33,34 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(3, "lefthand_pos") ==
                 std::vector<float> { 35,36,37 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(3, "righthand_pos") ==
                 std::vector<float> { 38,39,40 }));

  BOOST_REQUIRE((*bb.get<std::vector<float> >(41, "head_pos") ==
                 std::vector<float> { 42,43,44 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(41, "lefthand_pos") ==
                 std::vector<float> { 45,46,47 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(41, "righthand_pos") ==
                 std::vector<float> { 38,39,50 }));

  BOOST_REQUIRE((*bb.get<std::vector<float> >(4, "head_pos") ==
                 std::vector<float> { 42,43,44 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(4, "lefthand_pos") ==
                 std::vector<float> { 45,46,47 }));
  BOOST_REQUIRE((*bb.get<std::vector<float> >(4, "righthand_pos") ==
                 std::vector<float> { 38,39,50 }));

  if (remove(filename1) != 0)
    perror("Error deleting file");
  if (remove(filename2) != 0)
    perror("Error deleting file");

  // std::string filename3 = tmpnam(nullptr);
  // std::ofstream ofs(filename3);
  TempFile tf;
  std::mt19937 engine(time(nullptr));
  std::uniform_real_distribution<float> distribution(-1e10, 1e10);
  std::vector < std::vector<float> > randomValues;
  
  for (int k = 0; k < 13; ++k) {
    std::vector<float> line(13);
    for (int l = 0; l < 13; ++l) 
      line[l] = distribution(engine);
    randomValues.push_back(line);
  }

  std::vector<std::string> linesWritten;

  for (const auto& line : randomValues) {
    std::vector<std::string> stringValues;
    for (float v : line)
      stringValues.push_back(boost::lexical_cast<std::string>(v));

    std::string s = boost::algorithm::join(stringValues, ",");
    /* ofs */ tf << s << std::endl;
    linesWritten.push_back(s);
  }

  // ofs.close();
  tf.close();

  std::vector<std::string> linesRead = readFileLines(tf.getName());
  BOOST_REQUIRE(linesRead.size() == linesWritten.size());
  for (size_t i = 0; i < linesWritten.size(); ++i)
    BOOST_REQUIRE(linesRead[i] == linesWritten[i]);

  std::vector < std::vector < std::pair < std::string, std::vector<size_t> > > > specs;
  std::vector<size_t> frameNrIndices;

  std::set<std::string> fieldNames;
  for (int i = 0; i < 1000; ++i) {
    size_t len = 1 + rand() % 30;
    char fieldname[len+1];
    for (size_t i = 0; i < len; ++i)
      fieldname[i] = 'a' + rand() % 26;
    fieldname[len] = '\0';
    if (!fieldNames.count(fieldname))
      fieldNames.insert(fieldname);
  }

  for (int k = 0; k < 13; ++k) {
    frameNrIndices.push_back((rand() % 2) ? (rand() % 13) : SIZE_MAX);

    std::vector<size_t> unassignedIndices(13);
    for (size_t i = 0; i < 13; ++i)
      unassignedIndices[i] = i;
    std::random_shuffle(unassignedIndices.begin(), unassignedIndices.end());

    std::vector < std::pair < std::string, std::vector<size_t> > > spec;

    while (unassignedIndices.size() > 0) {
      std::string fieldname = *fieldNames.begin();
      fieldNames.erase(fieldname);
      std::pair < std::string, std::vector<size_t> > entry;
      entry.first = fieldname;

      for (size_t fieldSize = rand() % unassignedIndices.size() + 1;
           fieldSize > 0; --fieldSize) {
        entry.second.push_back(unassignedIndices.back());
        unassignedIndices.pop_back();
      }
      spec.push_back(entry);
    }

    specs.push_back(spec);
  }

  std::vector<std::string> specStrings;
    
  size_t index = 0;
  for (const auto& spec : specs) {
    std::vector<std::string> specEntries;
    for (const auto& entry : spec) {
      std::vector<std::string> strings;
      strings.push_back(entry.first);
      for (size_t j : entry.second)
        strings.push_back(boost::lexical_cast<std::string>(j));

      specEntries.push_back(boost::algorithm::join(strings, ":"));
    }
    if (frameNrIndices[index] < SIZE_MAX)
      specEntries.push_back("framenr:" + 
                            boost::lexical_cast<std::string>(frameNrIndices[index]));
    ++index;

    specStrings.push_back(boost::algorithm::join(specEntries,","));
  }

  std::vector<std::string> filenames;
  for (size_t i = 0; i < specStrings.size(); ++i)
    // filenames.push_back(filename3);
    filenames.push_back(tf.getName());
  argsS = std::vector<std::string> { "./tests", "--in-csv-file",
                                     boost::algorithm::join(filenames, ","),
                                     "--csv-fields",
                                     boost::algorithm::join(specStrings, ";"),
                                     "--components", "CsvImporter" };
  args.clear();
  for (auto it = argsS.cbegin(); it != argsS.cend(); ++it) {
    args.push_back(new char[it->length() + 1]);
    strcpy(args.back(), it->c_str());
  }
  anals.clear();
  createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                  vector<string>(), anals, opts, args.size(), &args[0]); 

  for (auto it = args.cbegin(); it != args.cend(); ++it) 
    delete[] *it;

  BOOST_REQUIRE(anals.size() == 1);
  BOOST_REQUIRE(anals[0].components.size() == 1);
  BOOST_REQUIRE(typeid(*anals[0].components[0]) == typeid(CsvImporter));
  CsvImporter& csvi2 = dynamic_cast<CsvImporter&>(*(anals[0].components[0]));

  BOOST_REQUIRE(csvi2.specs.size() == specs.size());
  for (int i = 0; i < 13; ++i) {
    BOOST_REQUIRE(csvi2.specs[i].size() == specs[i].size() + 
                  (frameNrIndices[i] != SIZE_MAX));
    for (size_t j = 0; j < specs[i].size(); ++j) {
      BOOST_REQUIRE(specs[i][j].first == csvi2.specs[i][j].first);
      BOOST_REQUIRE(specs[i][j].second == csvi2.specs[i][j].second);
    }
    if (frameNrIndices[i] != SIZE_MAX) {
      BOOST_REQUIRE(csvi2.specs[i].back().first == "framenr");
      BOOST_REQUIRE(csvi2.specs[i].back().second.size() == 1);
      BOOST_REQUIRE(frameNrIndices[i] == csvi2.specs[i].back().second[0]);
    }
  }

  csvi2.processRange(0,50);
  BlackBoard& bb2 = csvi2.getBlackBoard();

  BOOST_REQUIRE(csvi2.specs.size() == 13);
  // for (int i = 0 ; i < 13 ; ++i) { 
  //   std::cerr << "i = " << i << ", spec[i]:" << std::endl;
  //   for (const auto& spec : csvi2.specs[i])
  //     std::cerr << spec.first << ", " << spec.second.size() << std::endl;
  // }

  for (int i = 0; i < 13; ++i) {
    for (size_t j = 0; j < randomValues.size(); ++j) {
      size_t f = frameNrIndices[i] < SIZE_MAX ? randomValues[j][frameNrIndices[i]] : j;
      for (const auto& spec : csvi2.specs[i]) {
        if (spec.first == "framenr")
          continue;

        BOOST_REQUIRE(bb2.has(f, spec.first));
        BlackBoardPointer<std::vector<float> > vals = bb2.get<std::vector<float> >(f, spec.first);

        // std::cerr << "spec.first = " << spec.first
        //           << ", vals->size() = " << vals->size() 
        //           << ", spec.second.size() = " << spec.second.size()
        //           << ", f = " << f << ", j = " << j << ", i = " << i
        //           << std::endl;

        if (vals->size() != spec.second.size()) {
          std::cerr << "Error case: vals->size() = " << vals->size() 
                    << ", spec.second.size() = " << spec.second.size()
                    << ", f = " << f << ", j = " << j << ", i = " << i
                    << std::endl;
        }

        BOOST_REQUIRE(vals->size() == spec.second.size());
        for (size_t k = 0; k < spec.second.size(); ++k) {
          BOOST_REQUIRE((*vals)[k] == randomValues[j][spec.second[k]]);
        }
      }
    }
  }
  

  // if (remove(filename3.c_str()) != 0)
  //  perror("Error deleting file");
  // the tempfile will be implicitly deleted
}


BOOST_AUTO_TEST_CASE( testPython ) {
  vector<Job> jobs {
    Job(Job::JobType::DUMMY)
      };
  SLMotionConfiguration opts;
  std::shared_ptr<BlackBoard> bb(new BlackBoard());
  slmotion::python::PythonEnvironment pyEnv(&jobs, &opts, nullptr, nullptr, bb);
  
  bb->set("globalvar", std::string("global"));
  bb->set(256, "framevar", std::string("frame"));

  TempFile t;
  t << "import slmotion\n"
    << "assert slmotion.blackboard.has('globalvar')\n"
    << "assert not slmotion.blackboard.has('framevar')\n"
    << "assert not slmotion.blackboard.has('framevar', 0)\n"
    << "assert slmotion.blackboard.has('framevar', 256)\n"
    << "assert slmotion.blackboard.get('globalvar') == 'global'\n"
    << "assert slmotion.blackboard.get('framevar', 256) == 'frame'\n";

  t.close();

  pyEnv.processScript(t.getName());

  t = TempFile();
  bb->clear();
  t << "import slmotion\n";
  t << "import numpy\n";

  // store random matrices at random locations and verify that they
  // still exist on the Python side
  std::map<std::pair<size_t, std::string>, cv::Mat > randomMatrixEntries;
  for (int idx = 0; idx < 5; ++idx) {
    int type = 
      idx == 0 ? CV_8UC1 :
      idx == 1 ? CV_8UC3 :
      idx == 2 ? CV_16SC1 :
      idx == 3 ? CV_32FC1 :
      idx == 4 ? CV_64FC1 : 0;
    
    cv::Mat m(1 + rand() % 256, 1 + rand() % 256, type);
    auto key = std::make_pair(size_t(rand()), getRandomString());
    for (int i = 0; i < m.rows; ++i) {
      for (int j = 0; j < m.cols; ++j) {
        switch(type) {
        case CV_8UC1:
          m.at<uchar>(i,j) = rand() % 256;
          break;

        case CV_8UC3:
          m.at<cv::Vec3b>(i,j) = cv::Vec3b(rand() % 256,
                                           rand() % 256,
                                           rand() % 256);
          break;

        case CV_16SC1:
          m.at<short>(i,j) = rand() % (2 << 15) - (2 << 14);
          break;

        case CV_32FC1:
          m.at<float>(i,j) = exp(1)*(rand() % (2 << 15) - (2 << 14));
          break;          

        case CV_64FC1:
          m.at<double>(i,j) = exp(1)*(rand() % (2 << 15) - (2 << 14));
          break;
        }
      }
    }
    bb->set(key.first, key.second, m);
    randomMatrixEntries[key] = m;
  }

  for (auto it = randomMatrixEntries.cbegin(); it != randomMatrixEntries.cend();
       ++it) {
    t << "assert slmotion.blackboard.has('" << it->first.second << "', "
      << boost::lexical_cast<std::string>(it->first.first) << ")\n";
    t << "m = slmotion.blackboard.get('" << it->first.second << "', "
      << boost::lexical_cast<std::string>(it->first.first) << ")\n";
    t << "mRef = [";
    for (int i = 0; i < it->second.rows; ++i) {
      if (i > 0)
        t << ",";
      t << "[";
      for (int j = 0; j < it->second.cols; ++j) {
        if (j > 0)
          t << ", ";
        const cv::Mat& m = it->second;
        switch(m.type()) {
        case CV_8UC1:
          t << boost::lexical_cast<std::string>((int)m.at<uchar>(i,j));
          break;

        case CV_8UC3:
          t << "[" 
            << boost::lexical_cast<std::string>((int)m.at<cv::Vec3b>(i,j)[0])
            << ","
            << boost::lexical_cast<std::string>((int)m.at<cv::Vec3b>(i,j)[1])
            << ","
            << boost::lexical_cast<std::string>((int)m.at<cv::Vec3b>(i,j)[2])
            << "]";
          break;

        case CV_16SC1:
          t << boost::lexical_cast<std::string>(m.at<short>(i,j));
          break;

        case CV_32FC1:
          t << boost::lexical_cast<std::string>(m.at<float>(i,j));
          break;          

        case CV_64FC1:
          t << boost::lexical_cast<std::string>(m.at<double>(i,j));
          break;
        }
      }
      t << "]\n";
    }
    t << "]\n";
    t << "assert numpy.mean(numpy.abs(numpy.array(m)-numpy.array(mRef))) < 1e-04\n";
  }

  t.close();
  pyEnv.processScript(t.getName());

  // test the inverse, i.e. create the matrices in Python and then try
  // to read them from the Black Board on the C++ side.

  t = TempFile();
  bb->clear();
  t << "import slmotion\n";
  t << "import numpy\n";
  for (auto it = randomMatrixEntries.cbegin(); it != randomMatrixEntries.cend();
       ++it) {
    t << "m = numpy.array([";
    for (int i = 0; i < it->second.rows; ++i) {
      if (i > 0)
        t << ",";
      t << "[";
      for (int j = 0; j < it->second.cols; ++j) {
        if (j > 0)
          t << ", ";
        const cv::Mat& m = it->second;
        switch(m.type()) {
        case CV_8UC1:
          t << boost::lexical_cast<std::string>((int)m.at<uchar>(i,j));
          break;

        case CV_8UC3:
          t << "[" 
            << boost::lexical_cast<std::string>((int)m.at<cv::Vec3b>(i,j)[0])
            << ","
            << boost::lexical_cast<std::string>((int)m.at<cv::Vec3b>(i,j)[1])
            << ","
            << boost::lexical_cast<std::string>((int)m.at<cv::Vec3b>(i,j)[2])
            << "]";
          break;

        case CV_16SC1:
          t << boost::lexical_cast<std::string>(m.at<short>(i,j));
          break;

        case CV_32FC1:
          t << boost::lexical_cast<std::string>(m.at<float>(i,j));
          break;          

        case CV_64FC1:
          t << boost::lexical_cast<std::string>(m.at<double>(i,j));
          break;
        }
      }
      t << "]\n";
    }
    t << "],";
    // set the dtype
    switch(it->second.type()) {
    case CV_8UC1:
    case CV_8UC3:
      t << "numpy.uint8";
      break;

    case CV_16SC1:
      t << "numpy.int16";
      break;

    case CV_32FC1:
      t << "numpy.float32";
      break;          

    case CV_64FC1:
      t << "numpy.float64";
      break;
    }
    t << ")\n";
    t << "slmotion.blackboard.set(m, '" << it->first.second << "', "
      << boost::lexical_cast<std::string>(it->first.first) << ")\n";
  }

  t.close();
  pyEnv.processScript(t.getName());

  for (auto it = randomMatrixEntries.cbegin(); it != randomMatrixEntries.cend();
       ++it) {
    BOOST_REQUIRE(bb->has(it->first.first, it->first.second));
    BlackBoardPointer<cv::Mat> m(bb->get<cv::Mat>(it->first.first, it->first.second));
    BOOST_CHECK(equal(*m, it->second));
  }

  std::shared_ptr<OpenCVVideoCaptureVideoFileSource> vfs(new OpenCVVideoCaptureVideoFileSource(TESTVIDEOFILE6));

  pyEnv.setFrameSource(vfs);
  bb->clear();

  t = TempFile();
  t << "import slmotion\n"
    << "import cv2\n"
    << "for i in range(len(slmotion.framesource)):\n"
    << "  slmotion.blackboard.set(slmotion.framesource[i], 'frame', i)\n";
  t.close();
  pyEnv.processScript(t.getName());

  for (size_t i = 0; i < vfs->size(); ++i) {
    BOOST_REQUIRE(bb->has(i, "frame"));
    BOOST_CHECK(equal(*bb->get<cv::Mat>(i, "frame"), (*vfs)[i]));
  }

  shared_ptr<VideoFileSource> vfs1(new OpenCVVideoCaptureVideoFileSource(KINECT_VIDEO));
  shared_ptr<VideoFileSource> vfs2(new OpenCVVideoCaptureVideoFileSource(KINECT_DEPTH));
  shared_ptr<MultiFrameSource> mfs(new MultiFrameSource);
  mfs->addTrack(vfs1);
  mfs->addTrack(vfs2);
  pyEnv.setFrameSource(mfs);
  t = TempFile();
  t << "import slmotion\n"
    << "import cv2\n"
    << "assert slmotion.framesource.getNumberOfTracks() == 2\n"
    << "track1 = slmotion.framesource.getTrack(0)\n"
    << "track2 = slmotion.framesource.getTrack(1)\n"
    << "for i in range(len(slmotion.framesource)):\n"
    << "  slmotion.blackboard.set(track1[i], 'frame1', i)\n"
    << "  slmotion.blackboard.set(track2[i], 'frame2', i)\n";
  t.close();
  pyEnv.processScript(t.getName());
  
  for (size_t i = 0; i < mfs->size(); ++i) {
    BOOST_REQUIRE(bb->has(i, "frame1"));
    BOOST_REQUIRE(bb->has(i, "frame2"));
    BOOST_CHECK(equal(*bb->get<cv::Mat>(i, "frame1"), (*vfs1)[i]));
    BOOST_CHECK(equal(*bb->get<cv::Mat>(i, "frame2"), (*vfs2)[i]));
  }

  bb->clear();
  bb->set(0, "vals1", rand() % 50);
  bb->set(1, "vals1", rand() % 50);
  bb->set(2, "vals1", rand() % 50 + 60);
  bb->set(3, "vals1", rand() % 50 + 60);
  bb->set(4, "vals1", rand() % 50 + 60);
  bb->set(5, "vals1", rand() % 50);
  bb->set(6, "vals1", rand() % 50 + 60);
  bb->set(7, "vals1", rand() % 50 + 60);
  bb->set(8, "vals1", rand() % 50 + 60);
  bb->set(9, "vals1", rand() % 50 + 60);

  t = TempFile();
  t << "import slmotion\n"
    << "slmotion.constructFeatureVector([{'property': 'vals1',\n"
    << "                                  'interpretation': 'Value',\n"
    << "                                  'name': 'SomeValue1',\n"
    << "                                  'minimum': 0,\n"
    << "                                  'maximum': 120,\n"
    << "                                  'type': 'integer',\n"
    << "                                  'unit': '1'\n"
    << "                                  }],\n"
    << "                                 0, 9)\n";
  t.close();
  pyEnv.processScript(t.getName());
  BOOST_REQUIRE(bb->has(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY));
  BlackBoardPointer<FeatureVector> fv = bb->get<FeatureVector>(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY);
  for (int i = 0; i < 10; ++i) {
    BOOST_CHECK(fv->get<int>(0,i) == *bb->get<int>(i, "vals1"));
  }  
}



BOOST_AUTO_TEST_CASE( testPython2 ) {
  shared_ptr<VideoFileSource> vfs(new OpenCVVideoCaptureVideoFileSource(TESTVIDEOFILE6));
  std::shared_ptr<BlackBoard> bb(new BlackBoard());
 
  char* currentDir = getcwd(nullptr, 0);
  TempFile t;
  t << "from slmotion import *\n"
    << "import imp\n"
    << "PythonExampleComponent = imp.load_source('PythonExampleComponent', '" 
    << getInstallPrefix()
    << "/share/slmotion/PythonExampleComponent.py" << "')\n"
    << "PythonExampleComponent = PythonExampleComponent.PythonExampleComponent\n"
    << "setComponents([Component('PythonComponent', {'class': 'PythonExampleComponent',\n"
    << "                                             'multiplier': 0.5,\n"
    << "                                             'addend': 0.5})])\n"
    << "process()\n";
  free(currentDir);
  t.close();

  vector<Job> jobs {
    Job(Job::JobType::DUMMY)
      };
  SLMotionConfiguration opts;
  slmotion::python::PythonEnvironment pyEnv(&jobs, &opts, nullptr, vfs, bb);

  pyEnv.processScript(t.getName());

  for (size_t i = 0; i < vfs->size(); ++i) {
    cv::Mat gsimg;
    cv::Mat inFrame = (*vfs)[i];
    cv::cvtColor(inFrame, gsimg, CV_BGR2GRAY);
    cv::Mat result;
    gsimg.convertTo(result, CV_64FC1);
    result = (result/255.)*0.5 + 0.5;
    BOOST_CHECK(almostEqual(*bb->get<cv::Mat>(i, "PythonExampleEntry"), result));
  }
}




BOOST_AUTO_TEST_CASE( testPython3 ) {
  shared_ptr<VideoFileSource> vfs(new OpenCVVideoCaptureVideoFileSource(TESTVIDEOFILE6));
  std::shared_ptr<BlackBoard> bb(new BlackBoard());
 
  TempFile t;
  t << "from slmotion import *\n"
    << "import numpy\n"
    << "blackboard.setAs(numpy.array([123, 456, 789, 101112]), 'Rect', 'foo')";
    
  t.close();

  vector<Job> jobs {
    Job(Job::JobType::DUMMY)
      };
  SLMotionConfiguration opts;
  slmotion::python::PythonEnvironment pyEnv(&jobs, &opts, nullptr, vfs, bb);

  pyEnv.processScript(t.getName());

  BOOST_REQUIRE(bb->has("foo"));
  BlackBoardPointer<cv::Rect> r = bb->get<cv::Rect>("foo");
  BOOST_REQUIRE(r->x == 123);
  BOOST_REQUIRE(r->y == 456);
  BOOST_REQUIRE(r->width == 789);
  BOOST_REQUIRE(r->height == 101112);

  vector<Point> pointVector(10);
  t = TempFile();
  t << "from slmotion import *\n"
    << "import numpy\n"
    << "pv = blackboard.get('foo')\n"
    << "m = [";
  for (int i = 0; i < 10; ++i) {
    int i1 = rand(), i2 = rand();
    pointVector[i] = cv::Point(i1, i2);
    t << "[[" << i1 << "," << i2 << "]]";
    if (i < 9)
      t << ",";
  }
  t << "]\n"
    << "assert numpy.sum(pv) > 0\n"
    << "assert numpy.sum(numpy.array(m) - pv) == 0";

  bb->clear();
  bb->set("foo", pointVector);

  t.close();
  pyEnv.processScript(t.getName());

  t = TempFile();
  t << "from slmotion import *\n"
    << "import numpy\n"
    << "v = numpy.array([\n";
  for (int i = 0; i < 10; ++i) {
    int i1 = rand(), i2 = rand();
    pointVector[i] = cv::Point(i1, i2);
    t << "[[" << i1 << "," << i2 << "]]";
    if (i < 9)
      t << ",";
  }
  t << "], numpy.int32)\n"
    << "blackboard.setAs(v, 'vector<Point>', 'foo')";

  t.close();
  bb->clear();
  BOOST_REQUIRE(!bb->has("foo"));
  pyEnv.processScript(t.getName());
  BOOST_REQUIRE(bb->has("foo"));
  BlackBoardPointer<std::vector<cv::Point> > pointVectorPtr = bb->get<std::vector<cv::Point> >("foo");
  BOOST_REQUIRE(pointVectorPtr->size() == pointVector.size());
  for (size_t i = 0; i < pointVector.size(); ++i) {
    BOOST_CHECK(pointVector[i].x == (*pointVectorPtr)[i].x);
    BOOST_CHECK(pointVector[i].y == (*pointVectorPtr)[i].y);
  }

  vector<Point2f> pointVectorFloat(10);
  t = TempFile();
  t << "from slmotion import *\n"
    << "import numpy\n"
    << "pv = blackboard.get('foo')\n"
    << "m = [";
  for (int i = 0; i < 10; ++i) {
    float f1 = (rand() % 100)*exp(1), f2 = (rand() % 100)*exp(1);
    pointVectorFloat[i] = cv::Point2f(f1, f2);
    t << "[[" << f1 << "," << f2 << "]]";
    if (i < 9)
      t << ",";
  }
  t << "]\n"
    << "assert numpy.sum(pv) > 0\n"
    << "assert abs(numpy.mean(numpy.array(m) - pv)) < 1e-3";

  bb->clear();
  bb->set("foo", pointVectorFloat);

  t.close();
  pyEnv.processScript(t.getName());

  t = TempFile();
  t << "from slmotion import *\n"
    << "import numpy\n"
    << "v = numpy.array([\n";
  for (int i = 0; i < 10; ++i) {
    float f1 = (rand() % 100)*exp(1), f2 = (rand() % 100)*exp(1);
    pointVectorFloat[i] = cv::Point2f(f1, f2);
    t << "[[" << f1 << "," << f2 << "]]";
    if (i < 9)
      t << ",";
  }
  t << "], numpy.float32)\n"
    << "blackboard.setAs(v, 'vector<Point2f>', 'foo')";

  t.close();
  bb->clear();
  BOOST_REQUIRE(!bb->has("foo"));
  pyEnv.processScript(t.getName());
  BOOST_REQUIRE(bb->has("foo"));
  BlackBoardPointer<std::vector<cv::Point2f> > pointVectorFloatPtr = bb->get<std::vector<cv::Point2f> >("foo");
  BOOST_REQUIRE(pointVectorFloatPtr->size() == pointVectorFloat.size());
  for (size_t i = 0; i < pointVectorFloat.size(); ++i) {
    BOOST_CHECK(abs(pointVectorFloat[i].x - (*pointVectorFloatPtr)[i].x) < 1e-3);
    BOOST_CHECK(abs(pointVectorFloat[i].y - (*pointVectorFloatPtr)[i].y) < 1e-3);
  }

  std::vector<cv::Rect> rects(10);
  t = TempFile();
  t << "import slmotion\n"
    << "import numpy\n"
    << "rects = slmotion.blackboard.get('foo')\n";
  for (int i = 0; i < 10; ++i) {
    rects[i] = cv::Rect(rand(), rand(), rand(), rand());
    t << "assert rects[" << i << "][0] == "
      << rects[i].x << "\n";
    t << "assert rects[" << i << "][1] == "
      << rects[i].y << "\n";
    t << "assert rects[" << i << "][2] == "
      << rects[i].width << "\n";    
    t << "assert rects[" << i << "][3] == "
      << rects[i].height << "\n";
  }
  t.close();
  bb->clear();
  bb->set("foo", rects);
  pyEnv.processScript(t.getName());

  t = TempFile();
  t << "import slmotion\n"
    << "import numpy\n"
    << "rects = list()\n";
  for (int i = 0; i < 10; ++i) {
    rects[i] = cv::Rect(rand(), rand(), rand(), rand());
    t << "rects.append(numpy.array([" << rects[i].x << ","
      << rects[i].y << ","
      << rects[i].width << ","    
      << rects[i].height << "]))\n";
  }
  t << "slmotion.blackboard.setAs(rects, 'vector<Rect>', 'foo')";
  t.close();
  bb->clear();
  pyEnv.processScript(t.getName());
  BlackBoardPointer<std::vector<cv::Rect> > rectsPtr = bb->get<std::vector<cv::Rect> >("foo");
  BOOST_REQUIRE(rectsPtr->size() == rects.size());
  for (size_t i = 0; i < rects.size(); ++i) {
    BOOST_CHECK((*rectsPtr)[i].x == rects[i].x);
    BOOST_CHECK((*rectsPtr)[i].y == rects[i].y);
    BOOST_CHECK((*rectsPtr)[i].width == rects[i].width);
    BOOST_CHECK((*rectsPtr)[i].height == rects[i].height);
  }


  TempFile csvfile;
  cv::Mat csvValues(10,6,CV_32FC1);
  for (int i = 0; i < csvValues.rows; ++i) {
    for (int j = 0; j < csvValues.cols; ++j) {
      float f = exp(1)*(rand() % 100);
      csvValues.at<float>(i,j) = f;
      csvfile << f;
      if (j < csvValues.cols - 1)
        csvfile << ",";
    }
    csvfile << endl;
  }
  csvfile.close();

  bb->clear();
  t = TempFile();
  t << "from slmotion import *";
  t << std::endl;
  t << "setComponents([Component('CsvImporter', {'filenames': '"
    << csvfile.getName() << "'," << std::endl
    << "'fields': 'foo1:0:2:4,foo2:1-5'})])" << endl
    << "process(0,10)";
  t.close();
  pyEnv.processScript(t.getName());
  for (int f = 0; f < 10; ++f) {
    BOOST_REQUIRE(bb->has(f, "foo1"));
    BlackBoardPointer<std::vector<float> > v = bb->get<std::vector<float> >(f, "foo1");
    BOOST_CHECK(abs((*v)[0] - csvValues.at<float>(f,0)) < 1e-3);
    BOOST_CHECK(abs((*v)[1] - csvValues.at<float>(f,2)) < 1e-3);
    BOOST_CHECK(abs((*v)[2] - csvValues.at<float>(f,4)) < 1e-3);
    v = bb->get<std::vector<float> >(f, "foo2");
    BOOST_CHECK(abs((*v)[0] - csvValues.at<float>(f,1)) < 1e-3);
    BOOST_CHECK(abs((*v)[1] - csvValues.at<float>(f,2)) < 1e-3); 
    BOOST_CHECK(abs((*v)[2] - csvValues.at<float>(f,3)) < 1e-3);
    BOOST_CHECK(abs((*v)[3] - csvValues.at<float>(f,4)) < 1e-3);
    BOOST_CHECK(abs((*v)[4] - csvValues.at<float>(f,5)) < 1e-3);
  }

  bb->clear();
  std::vector<int> randints(27);
  std::vector<float> randfloats(27);
  for (size_t i = 0; i < randints.size(); ++i)
    randints[i] = rand();
  for (size_t i = 0; i < randfloats.size(); ++i)
    randfloats[i] = (rand() % 100)*exp(1);
  bb->set("foo1", cv::Rect(randints[0], randints[1], randints[2], randints[3]));
  bb->set("foo2", cv::Point(randints[4], randints[5]));
  bb->set("foo3", cv::Point2f(randfloats[0], randfloats[1]));
  cv::Mat m(4,1,CV_32FC1);
  for (int i = 0; i < 4; ++i)
    m.at<float>(i,0) = randfloats[i+2];
  bb->set("foo4", m);
  m = cv::Mat(1,4,CV_32FC1);
  for (int i = 0; i < 4; ++i)
    m.at<float>(0,i) = randfloats[i+6];
  bb->set("foo5", m);
  m = cv::Mat(1,4,CV_32SC1);
  for (int i = 0; i < 4; ++i)
    m.at<int>(0,i) = randints[i+6];
  bb->set("foo6", m);
  int sizes[] = { 3, 3, 3 };
  m = cv::Mat(3, sizes, CV_32SC1);
  for (int i = 0; i < 3; ++i)
    for (int j = 0; j < 3; ++j)
      for (int k = 0; k < 3; ++k)
        m.at<int>(i,j,k) = randints[9*i+3*j+k];
  bb->set("foo7", m);
  m = cv::Mat(3, sizes, CV_32FC1);
  for (int i = 0; i < 3; ++i)
    for (int j = 0; j < 3; ++j)
      for (int k = 0; k < 3; ++k)
        m.at<float>(i,j,k) = randfloats[9*i+3*j+k];
  bb->set("foo8", m);

  bb->set("foo11", randfloats[0]);
  bb->set("foo12", double(randfloats[1]) + double(randfloats[2]));
  bb->set("foo9", randints[0]);
  bb->set("foo10", long(randints[1]) + long(randints[2]));
  
  bb->set("foo13", (unsigned int)randints[3]);
  bb->set("foo14", (unsigned long)randints[4]);
  bb->set("foo15", (short)(randints[5] & 0x0fff));
  bb->set("foo16", (unsigned short)(randints[6] & 0x0fff));
          
  t = TempFile();
  t << "from slmotion import *" << endl
    << "import numpy" << endl
    << "import math" << endl
    << "m = blackboard.get('foo1')" << endl
    << "assert m.dtype == numpy.int32" << endl
    << "assert m.ndim == 1" << endl
    << "assert m.shape == (4,)" << endl
    << "assert numpy.sum(m - numpy.array([" << randints[0] << "," << randints[1]
    << "," << randints[2] << "," << randints[3] << "], numpy.int32)) == 0" 
    << endl
    << "m = blackboard.get('foo2')" << endl
    << "assert m.dtype == numpy.int" << endl
    << "assert m.ndim == 2" << endl
    << "assert m.shape == (1,2)" << endl
    << "assert numpy.sum(m - numpy.array([" << randints[4] << "," << randints[5]
    << "], numpy.int32)) == 0" 
    << endl
    << "m = blackboard.get('foo3')" << endl
    << "assert m.dtype == numpy.float" << endl
    << "assert m.ndim == 2" << endl
    << "assert m.shape == (1,2)" << endl
    << "assert abs(numpy.mean(m - numpy.array([" << randfloats[0] << "," << randfloats[1]
    << "], numpy.float32))) < 1e-3" 
    << endl
    << "m = blackboard.get('foo4')" << endl
    << "assert m.dtype == numpy.float32" << endl
    << "assert m.ndim == 1" << endl
    << "assert m.shape == (4,)" << endl
    << "assert abs(numpy.mean(m - numpy.array([" << randfloats[2] << "," << randfloats[3]
    << "," << randfloats[4] << "," << randfloats[5]
    << "], numpy.float32))) < 1e-3" 
    << endl
    << "m = blackboard.get('foo5')" << endl
    << "assert m.dtype == numpy.float32" << endl
    << "assert m.ndim == 1" << endl
    << "assert m.shape == (4,)" << endl
    << "assert abs(numpy.mean(m - numpy.array([" << randfloats[6] << "," << randfloats[7]
    << "," << randfloats[8] << "," << randfloats[9]
    << "], numpy.float32))) < 1e-3" 
    << endl
    << "m = blackboard.get('foo6')" << endl
    << "assert m.dtype == numpy.int32" << endl
    << "assert m.ndim == 1" << endl
    << "assert m.shape == (4,)" << endl
    << "assert numpy.sum(m - numpy.array([" << randints[6] << "," << randints[7]
    << "," << randints[8] << "," << randints[9]
    << "], numpy.int32)) == 0" 
    << endl
    << "m = blackboard.get('foo7')" << endl
    << "assert m.dtype == numpy.int32" << endl
    << "assert m.ndim == 3" << endl
    << "assert m.shape == (3,3,3)" << endl
    << "assert numpy.sum(m - numpy.array([";
  for (int i = 0; i < 3; ++i) {
    t << "[";
    for (int j = 0; j < 3; ++j) {
      t << "[";
      for (int k = 0; k < 3; ++k) {
        t << randints[9*i+3*j+k];
        if (k < 2)
          t << ",";
      }
      t << "]";
      if (j < 2)
        t << ",";
    }
    t << "]";
    if (i < 2)
      t << ",";
  }
  t << "], numpy.int32)) == 0" 
    << endl
    << "m = blackboard.get('foo8')" << endl
    << "assert m.dtype == numpy.float32" << endl
    << "assert m.ndim == 3" << endl
    << "assert m.shape == (3,3,3)" << endl
    << "assert abs(numpy.mean(m - numpy.array([";
  for (int i = 0; i < 3; ++i) {
    t << "[";
    for (int j = 0; j < 3; ++j) {
      t << "[";
      for (int k = 0; k < 3; ++k) {
        t << randfloats[9*i+3*j+k];
        if (k < 2)
          t << ",";
      }
      t << "]";
      if (j < 2)
        t << ",";
    }
    t << "]";
    if (i < 2)
      t << ",";
  }
  t << "], numpy.float32))) < 1e-3" 
    << endl
    << "m = blackboard.get('foo9')" << endl
    << "assert m.__class__.__name__ == 'int'" << endl
    << "assert m == " << randints[0] << endl
    << "m = blackboard.get('foo10')" << endl
    << "assert m.__class__.__name__ == 'int'" << endl
    << "assert m == " << long(randints[1]) + long(randints[2]) << endl
    << "m = blackboard.get('foo11')" << endl
    << "assert m.__class__.__name__ == 'float'" << endl
    << "assert abs(m - " << randfloats[0] << ") < 1e-3" << endl
    << "m = blackboard.get('foo12')" << endl
    << "assert m.__class__.__name__ == 'float'" << endl
    << "assert abs(m - " << double(randfloats[1]) + double(randfloats[2]) 
    << ") < 1e-3" << endl
    << "m = blackboard.get('foo13')" << endl
    << "assert m.__class__.__name__ == 'int'" << endl
    << "assert m == " << randints[3] << endl
    << "m = blackboard.get('foo14')" << endl
    << "assert m.__class__.__name__ == 'int'" << endl
    << "assert m == " << randints[4] << endl
    << "m = blackboard.get('foo15')" << endl
    << "assert m.__class__.__name__ == 'int'" << endl
    << "assert m == " << (randints[5] & 0x0fff) << endl 
    << "m = blackboard.get('foo16')" << endl
    << "assert m.__class__.__name__ == 'int'" << endl
    << "assert m == " << (randints[6] & 0x0fff) << endl;
  // inverse conversions
  t << "blackboard.set(blackboard.get('foo1'), 'bar1m')" << endl
    << "blackboard.setAs(blackboard.get('foo1'), 'Rect', 'bar1r')" << endl
    << "blackboard.set(blackboard.get('foo2'), 'bar2m')" << endl
    << "blackboard.setAs(blackboard.get('foo2'), 'Point', 'bar2p')" << endl
    << "blackboard.set(blackboard.get('foo3'), 'bar3m')" << endl
    << "blackboard.setAs(blackboard.get('foo3'), 'Point2f', 'bar3p')" << endl
    << "blackboard.set(0x0EADBEEF, 'bar4')" << endl
    << "blackboard.setAs(0xDEADBEEF, 'long', 'bar5')" << endl
    << "blackboard.setAs(0xDEADBEEF, 'unsigned long', 'bar6')" << endl
    << "blackboard.setAs(0xDEADBEEF, 'unsigned int', 'bar7')" << endl
    << "blackboard.setAs(0xBEEF, 'unsigned short', 'bar8')" << endl
    << "blackboard.setAs(0x0EAD, 'short', 'bar9')" << endl
    << "blackboard.set(math.exp(1), 'bar10')" << endl
    << "blackboard.setAs(math.exp(1), 'float', 'bar11')" << endl;        
  t.close();
  pyEnv.processScript(t.getName());  
  BOOST_REQUIRE(bb->typeOf("bar4") == typeid(int));
  BOOST_REQUIRE(*bb->get<int>("bar4") == (int)0x0EADBEEF);
  BOOST_REQUIRE(bb->typeOf("bar5") == typeid(long));
  BOOST_REQUIRE(*bb->get<long>("bar5") == (long)0xDEADBEEF);
  BOOST_REQUIRE(bb->typeOf("bar6") == typeid(unsigned long));
  BOOST_REQUIRE(*bb->get<unsigned long>("bar6") == 0xDEADBEEF);
  BOOST_REQUIRE(bb->typeOf("bar7") == typeid(unsigned int));
  BOOST_REQUIRE(*bb->get<unsigned int>("bar7") == 0xDEADBEEF);
  BOOST_REQUIRE(bb->typeOf("bar8") == typeid(unsigned short));
  BOOST_REQUIRE(*bb->get<unsigned short>("bar8") == 0xBEEF);
  BOOST_REQUIRE(bb->typeOf("bar9") == typeid(short));
  BOOST_REQUIRE(*bb->get<short>("bar9") == 0x0EAD);
  BOOST_REQUIRE(bb->typeOf("bar10") == typeid(double));
  BOOST_REQUIRE(abs(*bb->get<double>("bar10") - exp(1)) < 1e-3);
  BOOST_REQUIRE(bb->typeOf("bar11") == typeid(float));
  BOOST_REQUIRE(abs(*bb->get<float>("bar11") - exp(1)) < 1e-3);

  bb->clear();
  for (int i = 0; i < 10; ++i)
    bb->set(i, "vals", i*i);
  t = TempFile();
  t << "import slmotion\n"
    << "slmotion.constructFeatureVector([{'property': 'framenr',\n"
    << "                                  'interpretation': 'Value',\n"
    << "                                  'name': 'framenumber',\n"
    << "                                  'type': 'integer'},\n"
    << "                                 {'property': 'timecode',\n"
    << "                                  'interpretation': 'Value',\n"
    << "                                  'name': 'time',\n"
    << "                                  'type': 'real'},\n"
    << "                                 {'property': 'vals',\n"
    << "                                  'interpretation': 'Value',\n"
    << "                                  'name': 'SomeValue1',\n"
    << "                                  'minimum': 0,\n"
    << "                                  'maximum': 120,\n"
    << "                                  'type': 'integer',\n"
    << "                                  'unit': '1'\n"
    << "                                  }],\n"
    << "                                 0, 9)\n";
  t.close();
  pyEnv.processScript(t.getName());
  BOOST_REQUIRE(bb->has(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY));
  BlackBoardPointer<FeatureVector> fv = bb->get<FeatureVector>(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY);
  for (int i = 0; i < 10; ++i) {
    BOOST_CHECK(*bb->get<int>(i, "vals") == i*i);
    BOOST_CHECK(fv->get<int>("SomeValue1", i) == i*i);
    BOOST_CHECK(fv->get<int>(0, i) == i);
    BOOST_CHECK(fv->get<double>(1, i) == 40.*i);
    BOOST_CHECK(fv->get<int>(2, i) == i*i);
  }  

  t = TempFile();
  t << "import slmotion\n"
    << "fv = slmotion.blackboard.get('featurevector')\n"
    << "for i in range(10):\n"
    << "  assert fv[i][0] == i\n"
    << "  assert fv[i][1] == 40*i\n"
    << "  assert fv[i][2] == i**2\n";
  t.close();
  pyEnv.processScript(t.getName());

  vector<string> names { getRandomString(), getRandomString(), 
      getRandomString(), getRandomString() };
  for (int i = 0; i < 10; ++i)
    bb->set(i, names[0], i);
  bb->set(names[1], (long)0xDEADBEEF);
  for (int i = 0; i < 10; ++i)
    bb->set(i, names[2], i*i);
  bb->set(names[2], (long)0xDEADBEEF);
  for (int i = 0; i < 10; ++i)
    bb->set(i, names[3], i*i*i);

  BOOST_REQUIRE(bb->has(names[1]));
  BOOST_REQUIRE(*bb->get<long>(names[1]) == (long)0xDEADBEEF);
  BOOST_REQUIRE(bb->has(names[2]));
  BOOST_REQUIRE(*bb->get<long>(names[2]) == (long)0xDEADBEEF);
  for (int i = 0; i < 10; ++i) {
    BOOST_REQUIRE(bb->has(i, names[0]));
    BOOST_REQUIRE(*bb->get<int>(i, names[0]) == i);
    BOOST_REQUIRE(bb->has(i, names[2]));
    BOOST_REQUIRE(*bb->get<int>(i, names[2]) == i*i);
    BOOST_REQUIRE(bb->has(i, names[3]));
    BOOST_REQUIRE(*bb->get<int>(i, names[3]) == i*i*i);
  }
  
  t = TempFile();
  t << "import slmotion" << endl
    << "slmotion.blackboard.delete('" << names[1] << "')" << endl;
  t.close();
  pyEnv.processScript(t.getName());

  BOOST_REQUIRE(!bb->has(names[1]));
  BOOST_REQUIRE(bb->has(names[2]));
  BOOST_REQUIRE(*bb->get<long>(names[2]) == (long)0xDEADBEEF);
  for (int i = 0; i < 10; ++i) {
    BOOST_REQUIRE(bb->has(i, names[0]));
    BOOST_REQUIRE(*bb->get<int>(i, names[0]) == i);
    BOOST_REQUIRE(bb->has(i, names[2]));
    BOOST_REQUIRE(*bb->get<int>(i, names[2]) == i*i);
    BOOST_REQUIRE(bb->has(i, names[3]));
    BOOST_REQUIRE(*bb->get<int>(i, names[3]) == i*i*i);
  }

  t = TempFile();
  t << "import slmotion" << endl
    << "for i in range(0, 10, 3):" << endl
    << "  slmotion.blackboard.delete('" << names[0] << "',i)" << endl;
  t.close();
  pyEnv.processScript(t.getName());

  BOOST_REQUIRE(!bb->has(names[1]));
  BOOST_REQUIRE(bb->has(names[2]));
  BOOST_REQUIRE(*bb->get<long>(names[2]) == (long)0xDEADBEEF);
  for (int i = 0; i < 10; ++i) {
    if (i % 3 == 0) {
      BOOST_REQUIRE(!bb->has(i, names[0]));
    }
    else {
      BOOST_REQUIRE(bb->has(i, names[0]));
      BOOST_REQUIRE(*bb->get<int>(i, names[0]) == i);
    }
    BOOST_REQUIRE(bb->has(i, names[2]));
    BOOST_REQUIRE(*bb->get<int>(i, names[2]) == i*i);
    BOOST_REQUIRE(bb->has(i, names[3]));
    BOOST_REQUIRE(*bb->get<int>(i, names[3]) == i*i*i);
  }

  t = TempFile();
  t << "import slmotion" << endl
    << "slmotion.blackboard.deleteAll('" << names[2] << "')" << endl;
  t.close();
  pyEnv.processScript(t.getName());

  BOOST_REQUIRE(!bb->has(names[1]));
  BOOST_REQUIRE(!bb->has(names[2]));
  for (int i = 0; i < 10; ++i) {
    if (i % 3 == 0) {
      BOOST_REQUIRE(!bb->has(i, names[0]));
    }
    else {
      BOOST_REQUIRE(bb->has(i, names[0]));
      BOOST_REQUIRE(*bb->get<int>(i, names[0]) == i);
    }
    BOOST_REQUIRE(!bb->has(i, names[2]));
    BOOST_REQUIRE(bb->has(i, names[3]));
    BOOST_REQUIRE(*bb->get<int>(i, names[3]) == i*i*i);
  }

  t = TempFile();
  t << "import slmotion" << endl
    << "slmotion.blackboard.clear()" << endl;
  t.close();
  pyEnv.processScript(t.getName());
  BOOST_REQUIRE(bb->size() == 0);
}



BOOST_AUTO_TEST_CASE( testPdm ) {
  std::unique_ptr<OpenCVVideoCaptureVideoFileSource> vfs_p(new OpenCVVideoCaptureVideoFileSource(TESTVIDEOFILE6));
    OpenCVVideoCaptureVideoFileSource& vfs = *vfs_p;

  BlackBoard bb;
  // RuleSkinDetector rsd(&bb, &vfs, RuleSkinDetector::Ruleset::B);
  OpenCvAdaptiveSkinDetector asd(&bb, &vfs);

  asd.addPostProcessFilter(PostProcessCropFilter(Rect(0,0,500,576)));
  asd.addPostProcessFilter(PostProcessMorphologicalTransformationFilter(PostProcessMorphologicalTransformationFilter::Type::ERODE, 2));
  asd.addPostProcessFilter(PostProcessMorphologicalTransformationFilter(PostProcessMorphologicalTransformationFilter::Type::DILATE, 3));
  asd.addPostProcessFilter(PostProcessHoleFillerFilter());

  vector<Blob> blobs;
  for (size_t i = 0; i < vfs.size(); ++i) {
    asd.process(i);
    BlackBoardPointer<cv::Mat> m = bb.get<Mat>(i, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);

    blobs.push_back(Blob::extractBlobsFromMask(*m, 5000, 10000).front());
  }

  Mat m1 = (Mat_<double>(8,1) << 50,50, 0,50, 0,0, 50,0);
  Mat m2 = (Mat_<double>(8,1) << 30,60, 0,30, 30,0, 60,30 );

  double s, t, tx, ty;
  align(m1, m2, &m2, &s, &t, &tx, &ty);
  static const double PI = boost::math::constants::pi<double>();
  BOOST_CHECK(almostEqual(s, 1.1785, 1e-04));
  BOOST_CHECK(almostEqual(t, -PI/4));
  BOOST_CHECK(almostEqual(tx, -25));
  BOOST_CHECK(almostEqual(ty, 25));

  size_t nLandmarks = 30;
  vector<Mat> blobLandmarks;
  Point2d p;
  getBlobLandmarks(blobs, nLandmarks, blobLandmarks, &p);
  for (auto landmarks = blobLandmarks.begin(); 
       landmarks != blobLandmarks.end(); ++landmarks) {
    double e1 = 0;
    Mat m = blobLandmarks.front() - *landmarks;
    cv::pow(m, 2, m);
    e1 = cv::sum(m)[0];

    double e2 = align(blobLandmarks.front(), *landmarks, &(*landmarks),
                      0, 0, 0 ,0);
    BOOST_CHECK(e1 >= e2);
  }

  Pdm pdm(blobs, nLandmarks);

  Mat& eigenValues = pdm.eigenValues;
  BOOST_REQUIRE(eigenValues.rows == 2*static_cast<int>(nLandmarks));
  BOOST_REQUIRE(eigenValues.cols == 1);
  BOOST_REQUIRE(eigenValues.type() == CV_64FC1);

  Mat& eigenVectors = pdm.eigenVectors;
  BOOST_REQUIRE(eigenVectors.cols == 2*static_cast<int>(nLandmarks));
  BOOST_REQUIRE(eigenVectors.rows == 2*static_cast<int>(nLandmarks));
  BOOST_REQUIRE(eigenVectors.type() == CV_64FC1);

  for (size_t i = 1; i < 2*nLandmarks; ++i) 
    BOOST_CHECK(eigenValues.at<double>(i-1,0) > 
                eigenValues.at<double>(i,0));

  Mat b(0,1,CV_64FC1);
  BOOST_CHECK(equal(pdm.generateShape(b), pdm.mean));
  b = (Mat_<double>(3, 1, CV_64FC1) << 1,2,3);
  
  BOOST_CHECK(almostEqual(pdm.generateShape(b),
                          pdm.mean + pdm.eigenVectors.colRange(0,1) +
                          pdm.eigenVectors.colRange(1,2)*2 +
                          pdm.eigenVectors.colRange(2,3)*3));

  Asm asmi(blobs, nLandmarks, 3, 1.0); // 3 = t = principal component count
  BOOST_CHECK(asmi.pdm->nLandmarks == nLandmarks);
  BOOST_CHECK(equal(asmi.pdm->mean, pdm.mean));
  BOOST_CHECK(equal(asmi.pdm->eigenVectors, pdm.eigenVectors));
  BOOST_CHECK(equal(asmi.pdm->eigenValues, pdm.eigenValues));

  cv::Size frameSize = vfs[0].size();

  // test guess masks
  std::list<BodyPart> bodyParts {
    BodyPart {
      BodyPart::LEFT_RIGHT_HAND_HEAD, true,
      Blob {
        vector<Point> {
          Point(0,0), Point(frameSize.width-1, 0),
          Point(frameSize.width-1, frameSize.height-1),
          Point(0, frameSize.height-1)
        }
      }
    }
  };
  Mat lhMask, rhMask, headMask;
  createGuessMasks(bodyParts, lhMask, rhMask, headMask, frameSize);
  BOOST_CHECK(lhMask.at<uchar>(frameSize.height/4, frameSize.width/2) == 0);
  BOOST_CHECK(lhMask.at<uchar>(frameSize.height*3./4., 
                               frameSize.width*3./2.) == 0);
  BOOST_CHECK(lhMask.at<uchar>(frameSize.height*3./4., 
                               frameSize.width/4) == 255);
  BOOST_CHECK(rhMask.at<uchar>(frameSize.height/4, frameSize.width/2) == 0);
  BOOST_CHECK(rhMask.at<uchar>(frameSize.height*3./4., 
                               frameSize.width*3./2.) == 255);
  BOOST_CHECK(rhMask.at<uchar>(frameSize.height*3./4., 
                               frameSize.width/4) == 0);
  BOOST_CHECK(headMask.at<uchar>(frameSize.height/4, 
                                 frameSize.width/2) == 255);
  BOOST_CHECK(headMask.at<uchar>(frameSize.height*3./4., 
                               frameSize.width*3./2.) == 0);
  BOOST_CHECK(headMask.at<uchar>(frameSize.height*3./4., 
                               frameSize.width/4) == 0);

  BodyPartCollector bpc(&bb, &vfs);
  FaceDetector fd(&bb, &vfs);
  BlobExtractor be(&bb, &vfs);


  std::unique_ptr<Asm::Instance> oldInstance;

  std::vector<cv::Mat> bts;
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 34.464, -18.9304, 
                 65.792));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 246.355, -335.72, 
                 189.388));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 32.9904, -18.1004, 
                 39.4331));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 38.8958, -23.6274, 
                 58.4888));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 15.3246, -0.377646, 
                 23.0608));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 8.55494, -0.0481016, 
                 23.8607));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 2.44962, 28.2675, 
                 18.3252));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 3.76757, 2.95271, 
                 17.3059));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 5.12734, -8.54888, 
                 8.41653));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 289.083, -205.125, 
                 119.208));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 71.234, -43.6517, 
                 17.4855));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 92.4351, -32.0096, 
                 21.4691));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 163.711, -150.263, 
                 45.358));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 45.9919, -25.3352, 
                 59.6144));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 2.56632, -4.53313, 
                 13.2328));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 7.81165, 9.78028, 
                 8.06017));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 107.57, -89.5288, 
                 123.083));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 428.948, -402.562, 
                 269.65));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << -0.837285, -4.71993, 
                 31.3988));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 3.74766, 18.7318, 
                 -10.9567));
#if 0
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 196.134, -296.149, 
                 155.356));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 635.784, -436.935, 
                 573.918));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 113.257, -142.746, 
                 148.146));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 587.884, -495.574, 
                 651.237));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 10.7555, -13.6888, 
                 35.1005));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 8.49953, -67.0602, 
                 47.3097));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 4.67089, 25.0601, 
                 16.0964));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 5.51179, 3.07883, 
                 15.9988));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 18.1599, 19.427, 
                 13.8093));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 14.6411, 18.5324, 
                 21.1972));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 210.21, -164.424, 
                 198.299));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 103.411, -26.6725, 
                 49.6524));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 251.847, -146.171, 
                 -162.77));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << -18.8521, -240.761, 
                 178.078));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 7.49376, -18.5208, 
                 17.7066));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 11.7388, 18.1222, 
                 19.3834));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 6.23818, -2.1784, 
                 -17.716));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 5.28144, 20.3624, 
                 -24.7879));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << -9.02398, -30.1715, 
                 30.5073));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 3.74766, 18.7318, 
                 -10.9567));

  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 196.134, -296.149, 
                 155.356));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 622.595, -436.935, 
                 431.7));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 113.257, -142.746, 
                 148.146));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 582.416, -495.574, 
                 540.592));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 10.7555, -13.6888, 
                 35.1005));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 8.49953, -67.0602, 
                 47.3097));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 4.67089, 25.0601, 
                 16.0964));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 5.51179, 3.07883, 
                 15.9988));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 18.1599, 19.427, 
                 13.8093));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 14.6411, 18.5324, 
                 21.1972));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 210.21, -164.424, 
                 198.299));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 103.411, -26.6725, 
                 49.6524));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 251.847, -146.171, 
                 -162.77));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << -18.8521, -240.761, 
                 178.078));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 7.49376, -18.5208, 
                 17.7066));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 11.7388, 18.1222, 
                 19.3834));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 6.23818, -2.1784, 
                 -17.716));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 5.28144, 20.3624, 
                 -24.7879));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << -9.02398, -30.1715, 
                 30.5073));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1)
                 << 3.74766, 18.7318, 
                 -10.9567));
  // #if 0
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 11.81631454404221, -9.384474655871845, 
                 40.60715079190668));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 4.937405885535841, -0.6156903212685014,
                 39.8242985922288));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 4.509418767078389, -12.53086757370325, 
                 40.73356420174073));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 6.314391139051474, -3.853989291739575, 
                 34.33307733909905));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 11.27187728644101, -1.163673577130693, 
                 34.25849558316301));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 10.82877120670388, 6.32879135016084, 
                 14.68700348143664));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 11.81865932947525, 1.822172426929044, 
                 20.04829084843242));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 13.82923214230453, 2.735596662092461, 
                 12.09502154447629));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 9.73852231851715, -0.03978052024811429, 
                 9.619181202255801));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 7.418777727380345, 4.984870908963553, 
                 9.625034861313816));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 1.459506507987342, -0.6928862957738384, 
                 14.47586017232866));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 4.572439253775446, 0.1551644466132602, 
                 17.06294107876817));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 7.044876404632086, 2.867685981359404, 
                 16.03250126471279));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 7.499179149512718, 2.233651539172819, 
                 19.7402755762095));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 8.165040901452631, 1.725550631640152, 
                 20.55773850135289));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 6.997223411986617, 2.642602101030929, 
                 18.60239744587716));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 2.301814076379886, -0.2945234669882524, 
                 26.55330595242599));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << -1.031027913749914, 4.780841097980703, 
                 18.75919309946007));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 5.161090421507653, 13.23259973130065, 
                 -8.404374251474163));
  bts.push_back((Mat_<double>(3, 1, CV_64FC1) 
                 << 10.32982302927738, 11.04348036830603, 
                 -19.12622745630224));
#endif

  vector<cv::Mat> landmarks;
#if 0
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 372.2176100992081, 489.4334663161725, 384.5960003163739, 483.8543622086555, 391.1653717743146, 472.6351788394318, 394.379032306232, 457.353427620259, 390.8504458563053, 441.3363852536501, 386.2668668427096, 426.571118220105, 372.5353116203901, 420.0115700481724, 352.476886614143, 418.7896210899795, 342.3835467936646, 408.4989857073441, 347.4753505675185, 389.521345065656, 345.3396086055281, 371.5442978303578, 341.1092905207758, 358.3207763800925, 346.3392616557345, 340.8784593714267, 339.4251234064397, 325.0857447137259, 329.9024808090076, 326.1050380040973, 321.0790982329339, 324.9922217803204, 314.9970738497657, 330.1847493496968, 320.8334870486934, 327.5942902976582, 328.3789807408323, 340.0378562499773, 322.7322963070961, 357.405059561628, 315.2450135206843, 373.3324633814988, 313.166091400757, 386.166540320387, 311.19265765722, 400.9035481893624, 311.6652359835603, 417.3331960566941, 319.0646183402615, 426.6846948285823, 324.749383967116, 436.2395718328956, 326.7713656198884, 451.2523999897838, 337.3520194518256, 463.6643057276802, 346.7181822282841, 475.8921719409328, 353.7852463399997, 487.9484269717228));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 366.5984809377176, 486.552789773442, 379.5654437936118, 482.4061756826583, 386.5919365417184, 472.3474304072802, 390.4769112631678, 457.9039397955893, 387.7694982322441, 442.1104845920104, 385.0238192708452, 426.933272407312, 372.355288192082, 419.7685302467426, 353.0298213874392, 418.0811861896708, 341.9925277294388, 409.1301416528219, 346.2004994092998, 391.2450704670616, 344.5659322261751, 374.2259166063731, 340.399907294535, 360.9416270207142, 345.7360017303519, 344.1718513180815, 340.2436615395264, 328.2929682384378, 330.0557081222952, 325.4891506227719, 321.6271701759876, 326.280593511188, 317.2017402779941, 334.4762286431239, 322.2517881082665, 328.8520819542002, 327.7016108305006, 338.9862870784248, 321.7569175086156, 354.5575653464998, 315.0500995688565, 370.2700619515317, 311.8122300899589, 383.5044262021856, 309.9788289808339, 397.6156345796438, 309.5260392151908, 414.0904567829107, 316.6816679503755, 423.6888402712329, 323.6057142388213, 432.44489360432, 325.4997021142868, 447.0716682080579, 334.7304267704646, 459.6214967989696, 343.650001204335, 471.6091690431262, 350.1322424162113, 483.4034298228504));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 369.0784398277461, 486.3645034169224, 381.0895179103522, 481.2705414989646, 387.9861160386839, 470.9676517206029, 391.5916702584882, 456.328520515432, 388.6986971646484, 440.6879985614395, 385.1002558898473, 425.9988079987161, 372.3576051102713, 418.989102894024, 352.6632880309767, 417.8385282504471, 341.8832657356365, 408.2170218153303, 346.4172528752478, 390.4385452954791, 345.066200663993, 372.6611141098559, 340.8850860551063, 359.7428607358165, 346.5767899246326, 342.621598432409, 341.8154504143827, 325.9901333904876, 332.6862041685355, 323.949598275176, 322.2822405384128, 320.3732985633517, 315.0495887357499, 326.8730396329838, 321.4646528456918, 326.7275236430426, 328.7731105536247, 339.5626406477832, 322.9311756606256, 356.9192047620855, 314.9788336172287, 372.2022863580678, 313.3416342808528, 384.7030516534078, 310.4554779373947, 399.2771875759469, 310.6544908768727, 415.1484915425582, 318.0105598175705, 423.9906059782874, 323.0056997434301, 434.0435668011016, 324.7529338776051, 448.6661712070763, 335.1261108837029, 460.9545354555886, 344.0676515407482, 472.9040131525684, 350.7460233492509, 484.6038404090825));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 366.9742108586954, 488.647233134113, 380.3281030016225, 484.1985856239425, 387.613564633732, 473.2735532241908, 391.4655780817868, 458.06548237563, 388.9662037023264, 441.7126216952666, 385.4108000283306, 426.3924515704778, 372.3931756272158, 418.8504446247522, 353.3198952493786, 416.2151909642472, 342.4347892579718, 406.3640810938772, 346.1498121943461, 388.1874023129382, 344.6651117269826, 370.5026315519534, 340.7438415854436, 356.6123310652687, 345.5098798440235, 339.429344301088, 339.9216371248586, 323.4673105183778, 330.0746956409498, 321.6717029675274, 321.2213761380393, 321.5044196355558, 315.5027318621961, 327.8329528573063, 319.7448877233165, 325.2402050513839, 324.9146541342391, 336.6677337609794, 318.9111835025506, 352.7838384453879, 311.877427198391, 368.5600608267129, 309.5522698099071, 381.875795233008, 308.0855114901397, 396.5479697348007, 309.0943529554291, 412.6481229887557, 316.8828581692519, 422.2731027384244, 323.1025006332525, 432.0337370185175, 325.3201960209048, 447.0532309184903, 334.5884795469975, 460.01860073526, 343.6637068450119, 472.4646496610516, 350.5017109054951, 484.7750855676607));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 367.1372538720956, 488.1915326072854, 380.9074340957262, 483.6685302710415, 388.1730764429871, 472.2240512173632, 392.0340096116975, 456.6218290967556, 389.3760348896101, 439.9236339901715, 385.4350251797301, 424.4292691475411, 371.8362976499503, 416.9480705553567, 352.5365511682566, 413.8899404767748, 342.2764658332841, 403.4340473926873, 346.6847860284499, 384.4904253435303, 344.9447689817275, 366.6563079572466, 341.2120842488928, 352.4486978835564, 345.9385638529169, 335.1340521735431, 339.0806389438124, 319.6592744466234, 328.8709912922468, 319.7721896499311, 321.2276691703959, 321.4106553067309, 316.3724684109923, 326.836475735396, 320.1882084663593, 322.3359917999998, 325.2110447551667, 333.476349342649, 319.0377755911445, 349.4180653908235, 312.1188192442286, 365.5540896580609, 309.1766922398699, 379.1189586922385, 308.150117658337, 393.8608324486009, 309.0339675045029, 410.406054456653, 316.6893097337847, 420.5616611422503, 323.3403242165145, 430.0120608065332, 325.5036433490786, 445.3514315354909, 334.647903568186, 458.6084053128131, 343.8163731053669, 471.4207327979113, 350.7135280421372, 484.1156149119554));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 363.9283538580223, 492.5764317412236, 379.6528151581632, 489.3559431633571, 387.8300997889707, 476.6772770440833, 392.0818241374128, 460.1719748775872, 391.0847760759398, 442.56741697546, 386.7806968986472, 426.71036344154, 373.6622636660134, 417.8034227145679, 357.4549405237353, 410.7770925306779, 347.3167036222064, 399.5675418435852, 348.5707967273954, 381.4018257664838, 347.4403086038288, 363.8815089231593, 344.8299224111237, 347.9916157056384, 346.8411882860772, 331.1135739894561, 340.2521925495096, 317.0939080292234, 330.2554342755536, 317.0747082323114, 323.9538549500824, 321.4928926079377, 319.1635008176956, 324.1521675445835, 317.7987018446392, 322.2952003526171, 318.0016768073205, 332.515573060834, 311.7963910521661, 346.1198850809801, 306.1765825416582, 361.855469525799, 304.0752649688626, 375.9804345276297, 305.7412635220745, 390.8716175466623, 310.4347161466863, 405.7960267457269, 319.491818379733, 416.4102139712668, 326.325987639189, 426.8118795972075, 329.6392049259275, 442.313133427888, 336.1279172056982, 456.5165711864108, 344.9719646856186, 469.9705099217342, 352.0948991886788, 483.5800253062355));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 366.408591686566, 491.4677191636949, 381.5816409814443, 487.7340875679121, 389.6456386527252, 475.1920330721832, 393.8985518191514, 458.715282449299, 392.5297798871814, 441.1838615385664, 388.1197652243542, 425.3742992617119, 374.7177205024541, 416.7149467399877, 357.4723199159272, 410.5831160802028, 347.5451866775688, 399.0877346583322, 350.0622237848709, 380.492627804775, 348.9584610775306, 362.5152245183535, 346.2089063231854, 346.9704348702601, 349.2103463746079, 329.7735548413012, 342.5834050581021, 315.1047039664343, 332.6761783838497, 315.648471130141, 325.6706844273654, 318.4476648822921, 320.1976588523131, 321.1371756928978, 320.5041670511573, 319.7113559575536, 322.3648649157476, 330.9742696652438, 315.9700901709782, 345.6934952590409, 309.4935117529714, 361.5312851394218, 307.3423356166571, 375.4302734303352, 307.9609592398464, 390.5221296213356, 311.679871290744, 405.9282234726538, 320.3564604281444, 416.4716219578417, 326.6419095949413, 427.0428502000558, 329.4722679849963, 442.6955521502629, 336.8737927136236, 456.8119874090208, 345.7643805446354, 470.3383595265639, 352.8178930038691, 483.9181552898407));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
                      << 364.8013664662368, 491.58187063503, 
                      380.5448879978945, 488.0612408240403, 
                      388.8186205565165, 474.7436918751783, 
                      393.0419814127841, 457.7791486801963, 
                      392.2135993728103, 439.9380395585911, 
                      387.1508323861852, 424.2385824679019, 
                      373.7678755015195, 415.1871781643882, 
                      357.8509434615345, 407.4089229437702, 
                      348.4570485022854, 395.2367876977394, 
                      349.9230353903728, 376.8374462730205, 
                      348.9914934092656, 358.8960348142883, 
                      346.8018673947442, 342.7642655915789,
                      348.5845809645062, 325.763617213999,
                      341.6823931510852, 311.943362068047, 
                      332.1388111696971, 313.6890246298652, 
                      325.7958031459331, 317.2091602801849, 
                      319.9704600305242, 317.6390884051062,
                      318.1717190396249, 318.2135915586007, 
                      318.4440486889381, 329.6564738094469,
                      312.1222224527253, 343.6855559721917,
                      306.0204850800844, 359.2262953126848, 
                      304.5392430168095, 373.1116614045659, 
                      306.2876901132575, 388.2804356579206, 
                      311.8764268358033, 402.8109439990514,
                      321.1152010838821, 413.4218892903282, 
                      327.0332348930408, 424.5227212530315, 
                      330.3305425831171, 440.2127212241294,
                      336.8336075733408, 454.6081929964325, 
                      345.5837728842751, 468.3931459880815, 
                      352.8022327734118, 482.3533700189177));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
                      << 363.1483405880043, 489.4387408323396, 378.6169807131837, 486.1045399493148, 387.0400133849395, 473.1214698076012, 391.3337369642853, 456.429519661674, 390.9271994203662, 438.8512086297686, 386.1145748473675, 423.3636312033188, 373.3793144974829, 414.0675199351431, 358.1130815981387, 406.1431456412545, 348.3390870071202, 394.2702521754204, 348.9209484043857, 376.7092854737272, 348.4271041986765, 358.8963768691497, 346.3153590487113, 342.9110323823749, 347.9001751559982, 326.106932407865, 342.3263811103276, 312.0018436861459, 333.3355980224454, 312.1811002285715, 325.9046606916053, 313.8286626853411, 318.952001652698, 314.3438977114247, 316.9772802725691, 317.6961365538093, 316.9284012828452, 329.6192325411035, 310.7057607471222, 343.6590951665783, 304.4465838140202, 358.6788642922538, 303.7754571057987, 372.2709298904379, 305.2707976981558, 387.4084995046119, 311.4844464838894, 401.2309800411102, 320.9714129372821, 411.3523928786744, 326.1722774671063, 423.0107499089542, 329.5575777499252, 438.4068014546227, 335.8804338320726, 452.6248828774684, 344.4093542696489, 466.1701802008101, 351.5408579503209, 479.8960914470144));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 361.7846573493393, 490.6757216181604, 378.1876765873444, 487.9067486176438, 387.106244831036, 474.9480339934839, 391.8509556531971, 457.9756980554024, 391.6843067837682, 439.8024558073016, 387.5615586582675, 423.4476205508282, 374.7165646313462, 413.6097767539821, 359.0920298071537, 405.3478848416476, 348.4499891090626, 393.6942617979773, 348.7215676443083, 375.7024150194551, 348.2589292314104, 357.6804649530768, 346.0292258207329, 341.0332512808488, 347.7209921914716, 323.7903390561762, 342.2518978366407, 309.1800086099639, 332.4406982073267, 307.7268260496841, 325.1231082028075, 311.0074145715199, 319.1227308731593, 313.243409793049, 316.7174411857872, 314.1934068552258, 315.7588935356441, 325.0812042637154, 309.2369148533944, 338.754649060574, 303.2625406797351, 354.4728284453918, 301.7559908800415, 368.9172764887766, 303.4988993578887, 384.3880716286628, 309.3569567071603, 399.0502835082323, 319.0494621118759, 409.8259684665483, 325.3628134006132, 421.3534777750131, 328.8528472722732, 437.2461371406243, 334.8126023527408, 452.0423035569842, 343.5553178565351, 466.0059541110626, 350.7218726919469, 480.1596484342152));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 362.791094666399, 486.0970292689575, 378.0348205340712, 482.8576598361906, 386.5737663973161, 470.8226141060156, 391.0120669900983, 454.5747980100471, 390.4114800461551, 437.1234214379142, 386.6905089780963, 421.1910492072183, 374.3793485687416, 411.811862755767, 358.1056977082666, 405.3347388711428, 346.5464351288556, 394.7296063034909, 346.6372943904457, 377.5509170241338, 346.187257369538, 359.6336715500676, 343.248812138398, 343.9078871644688, 345.589264865739, 326.7322708075422, 341.6004644099128, 311.1628016275445, 332.2845580051132, 307.4462971015079, 323.0504254338712, 307.4750666960711, 315.7854730575094, 311.5621666923668, 315.4000943748381, 314.5692375134448, 316.1227103329399, 326.234574304774, 309.9481038292612, 340.8931936599913, 303.566798363612, 356.2528531284423, 302.8416463077518, 370.1810295397884, 303.3757682084752, 385.4757858810849, 308.5091462343564, 399.8762402962623, 318.0917155950346, 409.6970248479579, 323.7993885907655, 421.340867589559, 327.181887171101, 436.7131773748898, 334.2133845069588, 450.8051498768202, 343.046292990029, 464.0208196700243, 350.21272707663, 477.3370515899605));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 361.8346795015156, 483.2153228423548, 376.8145714508398, 479.6810788681533, 384.9643762226915, 467.6021776350846, 389.1307477572785, 451.4557916372416, 388.0534986237491, 434.1999692464234, 384.0588265040445, 418.4971758838647, 371.4300590713495, 409.6181726664287, 354.8203726375615, 403.6564326150271, 343.7305887209174, 393.0713249783856, 344.5095963184791, 375.5870037126207, 343.6439679468012, 357.850366042972, 340.5744537076089, 342.4321694538513, 343.0375969687689, 325.3845914853839, 338.1035896221841, 310.2404532230115, 328.6213174778026, 307.8940550332758, 320.1621730769278, 308.8789802546846, 313.6559476680927, 312.7964786931712, 313.7049806802231, 314.0860114402315, 315.0434684819771, 325.4360042607196, 309.0110553581907, 340.0753887640752, 302.8085081820366, 355.5315986658227, 301.6958570524729, 369.3293387330765, 302.3118928423211, 384.3817515106028, 306.9007482410706, 399.0019331349668, 316.1417076421279, 408.8558674969678, 322.1544647822824, 419.9032933997768, 325.4442743955344, 435.1678938583003, 332.7467612823356, 449.0187271664852, 341.6611711654381, 462.0760791031581, 348.8397570573518, 475.2249331183893));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 361.918983475826, 483.9592513099162, 377.4112116721107, 480.2598663950928, 385.5010776256759, 467.6082599077071, 389.5004848545323, 450.9865838399688, 388.1528886295973, 433.3226031988638, 383.720624193031, 417.3609136067386, 370.5442834669852, 408.5360642413306, 353.736876512468, 402.3039916722328, 342.6579031609178, 391.4912182714212, 343.318202332805, 373.4344052051921, 342.0294533899974, 355.5323720662153, 338.8330089505034, 339.7096635492203, 340.8495331196664, 322.4248667615574, 334.9103394655161, 307.6043678319298, 325.0233794545628, 306.2969167556648, 317.3847148608404, 308.90724796879, 311.5813014554245, 312.5106024710607, 311.0106880191526, 312.4565850868103, 312.0392272479737, 323.4594156750076, 306.0773305184534, 337.9761808456702, 300.3052627590197, 353.9034154891344, 298.9822357993387, 368.1018227818309, 300.3160015338975, 383.3048735780533, 305.2706488940556, 398.2135612322461, 314.7617100077424, 408.3951578786133, 321.4273409115079, 419.2280661541532, 325.0481934884439, 434.7794438161759, 332.3452615569661, 448.8878209904009, 341.5988556542694, 462.1678735133987, 349.0944018026956, 475.5982561597224));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 362.6990853446919, 482.2501751994553, 377.8423830774828, 478.4753121967873, 385.8156355157258, 466.1634804123152, 389.8608240712426, 449.8098719728156, 388.3202952849999, 432.3654170134031, 384.1189412741747, 416.4749051908158, 370.9351782155156, 407.8728896133407, 353.6464637293826, 402.2759263753112, 342.7113478104106, 391.5730785999012, 344.1662848274265, 373.3808034392866, 342.8594069016494, 355.5221485720455, 339.5906157750834, 340.0408433480508, 342.2434859542474, 322.7702138414228, 336.2459681030408, 307.764328814531, 326.2819487711032, 306.520755171736, 318.592686618869, 308.8997554380997, 313.0084912717609, 313.0206895716103, 313.3592926755053, 311.9690414442848, 315.1453177930634, 322.9287488592799, 309.0938820214744, 337.6838871853051, 303.0511915772341, 353.6279078782071, 301.3213643395021, 367.7005848061743, 302.093036232026, 382.7966327678453, 306.1102973357268, 398.0701366739258, 315.1792504722661, 408.2698354209798, 321.8383102615682, 418.8152893255661, 325.1009496426055, 434.315723590829, 332.7115600100229, 448.2577547965024, 341.8923289294231, 461.4473622523919, 349.1995985998391, 474.7245678550013));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 363.6312976366526, 482.7472008626133, 378.8683740260284, 478.7626155214825, 386.8444056113455, 466.2410400313639, 390.8493485306868, 449.6517973204883, 389.1300730231404, 432.0047957591676, 384.7376598040714, 415.9599001251426, 371.2636944804335, 407.3790149373306, 353.5792262586126, 401.9051487897669, 342.5763341636377, 391.0234785391624, 344.2251039624422, 372.4766767683009, 342.7928915497291, 354.3409922211479, 339.4128482437538, 338.7317090820177, 342.1561339925858, 321.1947453343319, 335.8805666208255, 306.0088425454599, 325.7937061193387, 305.1057511295032, 318.0388832227184, 307.4585091058453, 312.3768613721235, 311.5390521893069, 312.9449235206309, 310.3811957708099, 315.0607537107326, 321.5978232252772, 308.9747712918077, 336.7086020163977, 302.8177500953469, 352.9057598017656, 301.0795900623845, 367.123571901834, 301.8146386101798, 382.4219370371461, 305.7980624654135, 397.9567755100257, 314.9445693751203, 408.261705634692, 321.671884125429, 418.8926539768864, 324.9665347019091, 434.601850852639, 332.8634636849271, 448.6686214948622, 342.2257739348275, 462.004163685623, 349.6753192385514, 475.4238421581207));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 362.3390236932517, 478.5279507978499, 377.492769992215, 474.8092707979877, 385.4374534225497, 462.5048103624712, 389.4246480579924, 446.1979568218446, 387.9345734706584, 428.8145605337418, 383.7226608284227, 412.9968222671403, 370.6490321470006, 404.3941985402823, 353.6205827646472, 398.6935744588296, 342.6256308545292, 388.1164285922154, 343.738310044816, 370.1214028476256, 342.4188128847506, 352.3955989900206, 339.1449313853096, 336.9220615357971, 341.554061066519, 319.7637571246404, 335.6516474869359, 304.8978093682159, 325.7512024067867, 303.4797776080673, 318.1027989407202, 305.9497985128689, 312.5352017565378, 310.0457834928811, 312.5958509669668, 309.2305944753514, 314.1142524681027, 320.0557121261293, 308.1622218870257, 334.6154114079658, 302.3022525762763, 350.4632423660789, 300.7161258561679, 364.5040755287521, 301.6805298829743, 379.5210117040844, 305.9618850488092, 394.5826188942344, 315.1243594577557, 404.6877243196299, 321.8068745052705, 415.2181225635675, 325.1949811597719, 430.6132695318949, 332.6649339978964, 444.4951340158784, 341.8318125255641, 457.5896398598831, 349.1654411037159, 470.793998630091));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 366.4654296537807, 473.6452364343377, 380.690633141122, 470.0763200516021, 388.5976828515849, 459.0987065322798, 392.890370747099, 443.7454762882645, 391.3476426555119, 427.0346259883374, 388.3074407701619, 411.3436435573368, 375.8673369859104, 402.9226409136423, 357.995920653844, 398.7499873614341, 346.5367750060851, 388.9882347184484, 348.6759745956984, 371.4641909459438, 347.7423132509224, 353.9353621050586, 344.2041054264636, 339.3312583951644, 348.1597183932458, 322.3066474137034, 343.6349815319537, 306.4012694023713, 333.8135963979166, 302.678617026687, 324.8450963814609, 303.0463660995412, 318.966110185095, 309.4995848867029, 321.3430198020443, 308.1157845134519, 324.3056587612688, 318.9796699887175, 318.1340302118611, 334.0828443852973, 311.5173356375515, 349.6711332867566, 309.4271058976215, 363.3456412733584, 308.6828654557988, 378.1029834601235, 310.9029540443375, 393.6314846267701, 319.2817256176886, 403.41980610667, 325.6632746831592, 413.6900353241901, 328.2527439873659, 428.7594841886974, 336.3215746656389, 442.1979114325154, 345.1802916291737, 454.8834253243643, 351.9631404368772, 467.5080561645872));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 366.3055366328809, 470.67730250982, 381.0622267875181, 468.1210088864817, 389.2915291089407, 457.3613558781727, 393.8336149362688, 442.2941455185618, 393.1956312353221, 425.7162065398742, 390.7220530690256, 410.0963576326139, 379.1443780638695, 401.1415763944777, 363.0345799375104, 395.6097865728349, 351.3429350473148, 386.2886523049568, 351.7899798028502, 369.7879370628617, 351.259315658176, 353.028672574123, 348.1649738208691, 338.1200178367321, 351.0528264123163, 321.7459201281344, 347.3900937388033, 306.5663329751686, 337.7614358931677, 301.2500792178716, 329.2755138817009, 302.8721795439669, 323.8524351610583, 309.3698172358022, 324.1005854811423, 308.3425478313073, 324.6657173843169, 317.9152728847558, 318.6062264613867, 331.3789329832208, 312.8439798230825, 346.4492970995173, 310.8740396482313, 360.1821049145423, 311.1151906978729, 374.5608345040024, 314.4821156654041, 389.1866448728974, 323.20118022061, 398.9465865311313, 329.8569799457889, 409.204938106676, 332.8304380494707, 423.8977938805799, 339.398475301456, 437.3734385993012, 347.8497920447027, 449.8983201235501, 354.4223568091178, 462.4620376340143));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
<< 363.009802005542, 479.3667430403203, 381.7213446080034, 477.7659956556556, 391.5147323933562, 463.3576838393843, 396.4298616263131, 445.2071245792067, 397.6534146650487, 425.8608148032191, 393.0624257655719, 408.8349894896059, 380.501419562675, 397.5767328401624, 367.7544815822448, 385.5675991843438, 356.3722464979688, 373.5760896879459, 352.5640113574856, 356.2463205117562, 352.3255933944336, 338.4198788310717, 350.7362055545201, 319.811335942282, 349.1947179464562, 302.6916422478398, 343.926964568061, 289.3481936888942, 334.0228632758006, 286.8960263830158, 327.8689500070173, 293.4821739797672, 322.1552831223345, 293.900456495885, 314.2300400118632, 297.136886061283, 308.2175803082326, 306.7554147910873, 301.9413835540598, 318.1525019878273, 297.7961988698318, 333.9100328580517, 297.3632221516951, 349.3623366929135, 302.3186797947074, 365.1865174562786, 312.4949311243975, 378.3740276880566, 324.1088362735521, 389.5828983740776, 331.2681410777098, 402.0255342311183, 336.4566389511702, 418.2247511303101, 339.9574198103109, 434.0568420523778, 348.8422920532166, 448.6090244321994, 356.6539563133201, 463.6872348022046));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1)
                      << 365.4917022265585, 475.8900167464628, 384.319107308782, 474.305469885894, 394.1167407792007, 458.8067473196484, 398.782271538107, 440.2161005260925, 400.7536019291983, 420.8988360685244, 394.7854865786653, 404.7117060905251, 382.390814923831, 393.2289927260704, 371.9464071540467, 378.9114764765623, 362.2668981071398, 365.6078730038236, 357.465347814365, 348.8583565636295, 357.5936900799399, 331.2777720666959, 357.1070414340236, 312.3304156857643, 353.8758422836786, 295.899176302718, 348.1042534270449, 284.2444094204573, 339.3731667565995, 285.0170127271607, 334.2427722342368, 291.521709687256, 327.3709165596325, 287.1031493466244, 316.6981934531499, 294.8608074835755, 308.9930489135122, 305.4875626223855, 302.9224622380677, 316.0471893656353, 298.9686270028181, 330.7617209686329, 299.9455139005336, 345.4835641179653, 306.2442991944837, 361.1319782736674, 319.2265230187614, 372.4502945481408, 331.3326467324484, 383.3704083574106, 337.0032274048084, 396.6722423229525, 342.6076708570768, 412.559080451614, 344.9161178168262, 428.3900546149269, 353.2753891740022, 443.0795109578888, 361.1448156939525, 458.4490014379522));

 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 372.218, 489.433, 384.596, 483.854, 391.165, 472.635, 394.379, 457.353, 390.85, 441.336, 386.267, 426.571, 372.535, 420.012, 352.477, 418.79, 342.384, 408.499, 347.475, 389.521, 345.34, 371.544, 341.109, 358.321, 346.339, 340.878, 339.425, 325.086, 329.902, 326.105, 321.079, 324.992, 314.997, 330.185, 320.833, 327.594, 328.379, 340.038, 322.732, 357.405, 315.245, 373.332, 313.166, 386.167, 311.193, 400.904, 311.665, 417.333, 319.065, 426.685, 324.749, 436.24, 326.771, 451.252, 337.352, 463.664, 346.718, 475.892, 353.785, 487.948));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 392.546, 474.9, 398.748, 451.831, 395, 442.597, 398.677, 427.291, 368.036, 405, 365.348, 382.947, 321.136, 391.035, 235.284, 423.243, 243.47, 406.625, 332.458, 357.74, 316.527, 326.443, 301.369, 322.458, 354.986, 299.104, 302.635, 266.344, 260.873, 320.792, 271.733, 357.656, 300.99, 393.437, 391.841, 267.734, 462.139, 285.222, 441.604, 326.49, 413.956, 362.593, 365.149, 374.082, 334.975, 385.725, 255.222, 439.581, 231.252, 462.764, 261.284, 447.31, 238.508, 471.092, 281, 489.366, 296.483, 509.861, 298.807, 524.348));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 366.158, 485.049, 378.689, 480.413, 385.662, 470.139, 389.549, 455.54, 386.735, 439.732, 383.518, 424.782, 370.563, 417.718, 350.755, 416.124, 340.451, 406.374, 345.784, 388.161, 344.35, 370.666, 340.411, 357.54, 346.358, 340.624, 340.696, 324.519, 330.922, 323.258, 322.108, 322.392, 316.587, 329.161, 322.582, 325.284, 329.189, 336.867, 322.995, 353.354, 315.371, 368.897, 312.404, 381.722, 309.844, 396.015, 309.132, 412.477, 316.026, 422, 321.958, 431.241, 323.42, 446, 333.208, 458.581, 342.007, 470.789, 348.415, 482.728));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 399.709, 485.218, 400.204, 458.244, 395.297, 453.887, 398.442, 441.766, 366.608, 421.62, 366.637, 401.396, 325.2, 409.886, 232.112, 448.745, 234.619, 435.176, 324.535, 392.396, 311.785, 359.113, 292.822, 358.738, 353.148, 336.276, 314.395, 295.395, 277.633, 334.957, 276.359, 342.662, 290.657, 385.883, 394.734, 285.573, 474.592, 316.092, 452.132, 366.028, 417.944, 398.386, 375.301, 406.642, 334.659, 416.946, 249.805, 467.335, 225.86, 484.613, 248.608, 472.93, 223.726, 494.264, 273.548, 511.875, 289.114, 530.003, 291.365, 541.012));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 370.856, 489.227, 382.632, 484.142, 389.653, 472.557, 392.49, 457.467, 391.157, 442.109, 385.685, 428.97, 374.347, 421.137, 359.188, 416.623, 348.675, 406.359, 348.192, 391.505, 347.728, 374.049, 344.531, 360.641, 346.332, 344.584, 343.905, 329.279, 337.579, 327.098, 325.013, 319.095, 312.892, 319.994, 315.313, 333.98, 319.652, 349.483, 315.022, 366.048, 307.585, 379.178, 310.746, 390.141, 310.023, 404.757, 316.807, 415.889, 326.301, 422.74, 328.214, 435.787, 331.705, 449.35, 340.462, 461.199, 348.877, 472.61, 356.204, 484.07));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 365.832, 486.996, 380.064, 483.272, 387.748, 472.185, 392.1, 456.797, 390.101, 440.016, 386.964, 424.261, 373.845, 416.239, 355.078, 412.404, 344.606, 402.15, 348.691, 383.63, 347.412, 366.071, 343.998, 351.597, 348.764, 334.548, 342.61, 319.033, 332.198, 317.671, 324.704, 320.104, 320.293, 326.276, 323.488, 321.015, 327.259, 331.357, 320.77, 346.444, 313.984, 362.441, 310.523, 376.181, 309.58, 390.777, 310.274, 407.259, 317.851, 417.679, 324.831, 427.094, 326.85, 442.401, 335.135, 455.935, 343.972, 468.888, 350.533, 481.709));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 368.252, 483.436, 381.526, 479.11, 388.96, 468.381, 393.161, 453.289, 390.834, 436.914, 387.573, 421.55, 374.519, 413.883, 355.046, 411.128, 344.698, 400.849, 349.616, 382.467, 348.476, 364.644, 344.868, 350.818, 350.477, 333.695, 344.9, 317.543, 335.015, 316.231, 326.321, 315.831, 320.578, 321.928, 325.527, 318.763, 331.071, 330.503, 324.578, 346.766, 316.909, 362.455, 313.98, 375.631, 311.838, 390.251, 311.868, 406.667, 319.137, 416.561, 325.082, 426.308, 326.688, 441.436, 335.951, 454.592, 344.73, 467.301, 351.206, 479.778));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 360.264, 489.07, 376.626, 487.231, 385.669, 475.484, 390.862, 459.408, 390.967, 441.679, 388.395, 425.149, 376.28, 415.236, 360.624, 407.569, 349.314, 397.027, 349.726, 379.506, 349.448, 362.223, 347.111, 345.931, 349.512, 329.078, 344.876, 314.232, 334.6, 310.068, 327.361, 314.601, 322.786, 319.806, 320.996, 317.15, 319.593, 326.111, 312.938, 338.813, 307.31, 354.549, 304.5, 369.16, 305.794, 384.148, 309.973, 399.346, 319.047, 410.325, 326.381, 420.971, 329.463, 436.569, 334.885, 451.158, 343.319, 464.802, 349.925, 478.56));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 362.593, 488.489, 378.053, 485.239, 386.497, 472.787, 391.318, 456.326, 390.402, 438.618, 386.405, 422.57, 372.984, 413.543, 355.724, 406.761, 346.534, 394.711, 350.219, 375.795, 349.551, 357.745, 347.391, 342.026, 351.075, 324.887, 344.21, 310.207, 333.982, 311.386, 327.529, 315.057, 322.608, 317.337, 322.988, 314.446, 324.532, 325.493, 317.553, 339.889, 310.599, 355.724, 307.502, 369.654, 307.764, 384.759, 310.638, 400.613, 318.771, 411.737, 325.031, 422.236, 327.234, 438.123, 334.136, 452.594, 342.606, 466.575, 349.184, 480.543));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 360.686, 489.291, 376.92, 486.578, 385.8, 474.029, 390.825, 457.311, 390.39, 439.201, 386.765, 422.662, 373.643, 413.034, 356.98, 405.498, 346.736, 393.78, 348.795, 375.146, 348.277, 357.076, 346.052, 340.707, 348.976, 323.372, 342.899, 308.479, 332.533, 307.559, 325.681, 311.718, 320.705, 314.909, 319.754, 312.565, 319.8, 322.993, 312.886, 336.856, 306.471, 352.91, 303.647, 367.409, 304.592, 382.785, 308.508, 398.484, 317.36, 409.73, 324.13, 420.606, 326.895, 436.676, 333.18, 451.503, 341.813, 465.626, 348.605, 479.82));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 362.882, 490.021, 379.084, 486.388, 387.484, 472.832, 391.773, 455.477, 390.696, 437.16, 385.655, 420.925, 371.804, 411.772, 355.103, 404.177, 345.221, 391.95, 346.853, 372.9, 345.68, 354.541, 343.201, 338.029, 345.167, 320.515, 337.864, 306.264, 327.782, 307.806, 321.462, 312.049, 316.031, 313.297, 314.471, 312.477, 314.929, 323.775, 308.477, 338.143, 302.401, 354.334, 300.529, 368.759, 302.331, 384.28, 307.617, 399.531, 317.017, 410.518, 323.581, 421.564, 326.98, 437.699, 333.804, 452.451, 342.931, 466.541, 350.38, 480.802));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 373.853, 479.446, 385.837, 470.677, 389.527, 455.14, 390.437, 437.585, 383.513, 421.165, 373.033, 408.426, 354.414, 405.663, 332.59, 404.131, 329.733, 387.696, 341.468, 361.856, 335.863, 343.572, 332.837, 330.187, 337.389, 312.497, 318.493, 303.722, 310.461, 327.289, 312.5, 336.103, 310.376, 328.712, 314.138, 315.023, 325.311, 327.306, 320.056, 345.116, 313.222, 362.624, 308.355, 375.254, 309.38, 389.468, 309.599, 407.988, 315.325, 419.866, 321.578, 426.449, 323.213, 442.755, 335.074, 454.947, 345.353, 468.451, 353.397, 481.949));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 375.888, 477.98, 386.726, 468.193, 389.459, 451.761, 389.439, 433.987, 382.057, 418.074, 369.769, 406.903, 350.775, 405.147, 329.549, 403.363, 328.315, 385.138, 340.457, 358.716, 334.46, 340.424, 331.938, 327.383, 335.781, 310.033, 315.651, 303.074, 310.028, 331.443, 312.802, 338.678, 308.45, 326.801, 312.077, 316.704, 324.691, 330.582, 320.041, 349.034, 312.812, 365.954, 309.14, 377.46, 310.551, 391.494, 312.178, 408.777, 317.891, 420.17, 322.473, 427.416, 324.219, 443.476, 336.507, 455.018, 346.725, 468.376, 355.08, 481.751));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 383.749, 473.929, 393.66, 467.983, 398.02, 458.023, 398.264, 445.062, 393.769, 432.183, 388.364, 420.245, 377.275, 415.641, 362.183, 416.865, 349.723, 412.475, 346.759, 399.336, 342.735, 384.946, 336.446, 374.399, 336.133, 359.581, 331.225, 345.654, 322.467, 341.169, 311.094, 339.473, 305.409, 347.792, 309.545, 351.243, 315.473, 360.775, 314.243, 375.36, 311.405, 389.021, 314.191, 399.663, 315.734, 412.317, 321.244, 423.45, 330.355, 428.39, 336.725, 436.858, 342.105, 448.127, 352.1, 456.731, 361.858, 464.639, 370.011, 472.873));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 362.579, 483.846, 377.337, 480.2, 385.574, 467.907, 389.567, 451.674, 388.908, 434.523, 384.483, 419.204, 372.322, 410.059, 356.724, 403.455, 345.466, 392.765, 344.871, 376.161, 344.377, 358.353, 341.47, 342.882, 343.113, 325.995, 339.29, 310.816, 330.69, 307.847, 321.051, 306.516, 312.682, 309.163, 312.012, 315.246, 312.889, 327.787, 307.133, 342.616, 300.798, 357.439, 301.243, 370.795, 302.053, 385.966, 308.358, 399.327, 318.237, 408.496, 323.105, 420.579, 326.809, 435.571, 333.95, 449.29, 342.745, 462.174, 350.11, 475.208));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 362.963, 480.367, 377.993, 476.777, 386.287, 464.775, 390.497, 448.543, 389.431, 431.149, 385.588, 415.22, 373.027, 406.165, 356.153, 400.429, 344.395, 390.064, 344.749, 372.673, 343.959, 354.677, 340.647, 339.152, 343.174, 321.824, 338.863, 306.068, 329.359, 302.472, 320.012, 302.476, 312.932, 307.167, 313.291, 309.446, 314.791, 321.135, 308.772, 336.185, 302.429, 351.781, 301.646, 365.736, 302.032, 381.043, 306.751, 395.728, 316.268, 405.453, 322.233, 416.853, 325.654, 432.23, 333.228, 446.168, 342.312, 459.241, 349.631, 472.397));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 364.307, 475.719, 378.89, 472.362, 387.185, 460.41, 391.481, 444.5, 391.021, 427.581, 386.956, 412.375, 374.941, 403.259, 359.369, 396.609, 348.824, 385.682, 349.278, 368.961, 349.053, 351.381, 346.541, 336.149, 348.835, 319.517, 344.865, 304.574, 336.265, 302.239, 327.242, 301.304, 319.402, 303.671, 318.953, 308.38, 319.903, 320.619, 313.812, 335.161, 307.166, 349.77, 306.874, 362.958, 307.225, 377.879, 312.602, 391.527, 321.861, 400.97, 326.571, 412.692, 329.694, 427.624, 336.563, 441.307, 344.954, 454.267, 351.857, 467.319));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 357.856, 485.477, 377.318, 485.296, 386.61, 471.449, 391.271, 454.291, 392.753, 435.666, 389.032, 419.061, 377.246, 408.182, 366.7, 395.17, 355.558, 384.858, 350.059, 368.744, 349.161, 353.272, 347.674, 334.83, 344.324, 319.461, 337.953, 308.781, 327.216, 305.913, 324.977, 319.495, 323.742, 321.262, 312.56, 317.493, 302.908, 322.431, 297.214, 329.909, 295.964, 345.24, 293.956, 360.943, 301.452, 375.055, 311.762, 387.502, 323.087, 398.97, 332.849, 408.89, 338.71, 424.001, 339.832, 439.366, 348.354, 453.035, 355.832, 467.413));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 365.648, 476.244, 383.73, 474.035, 392.506, 460.224, 396.388, 442.809, 396.534, 424.192, 391.997, 407.687, 379.656, 397.392, 366.546, 387.079, 353.868, 377.198, 348.599, 361.144, 347.385, 344.233, 344.399, 326.475, 342.098, 310.048, 337.107, 296.908, 326.905, 292.527, 320.824, 299.444, 315.979, 302.518, 308.731, 304.808, 303.111, 313.502, 297.751, 324.538, 295.014, 340.112, 295.175, 355.369, 300.689, 370.256, 310.714, 382.694, 322.599, 392.783, 330.91, 404.023, 336.924, 419.227, 341.119, 434.236, 350.59, 447.508, 358.903, 461.342));
 landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 364.552, 474.627, 383.796, 474.18, 394.086, 460.236, 399.613, 442.4, 401.628, 423.091, 398.023, 405.768, 386.069, 393.996, 374.273, 381.214, 362.832, 369.459, 358.685, 352.309, 358.85, 335.005, 357.784, 316.148, 356.187, 299.263, 351.307, 286.201, 341.105, 282.396, 335.471, 290.854, 330.951, 292.14, 321.917, 293.094, 314.316, 301.013, 307.844, 311.106, 304.092, 326.841, 302.715, 342.604, 307.976, 358.307, 317.909, 371.75, 329.303, 383.495, 337.069, 395.632, 342.079, 411.849, 344.346, 427.816, 352.788, 442.503, 360.115, 457.745));
#endif
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 367.686, 488.787, 381.013, 484.059, 387.965, 472.31, 391.38, 456.79, 388.865, 440.51, 384.16, 425.813, 370.875, 418.56, 352.857, 414.947, 343.225, 404.293, 346.713, 386.061, 344.987, 368.61, 341.54, 354.643, 345.233, 337.866, 338.426, 323.308, 329.139, 324.611, 321.598, 325.419, 315.837, 328.747, 318.618, 327.313, 323.221, 339.004, 317.525, 354.625, 310.879, 370.006, 309.115, 382.91, 308.709, 397.361, 311.142, 412.585, 319.088, 422.118, 324.825, 431.922, 327.362, 446.754, 336.172, 459.569, 345.131, 472.008, 352.14, 484.415));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 374.874, 478.51, 384.199, 470.175, 388.664, 458.564, 390.425, 443.52, 384.118, 428.81, 377.113, 415.885, 361.429, 412.374, 338.481, 415.229, 331.7, 403.076, 342.417, 381.148, 338.592, 362.722, 334.083, 351.681, 341.594, 333.784, 330.233, 319.188, 321.923, 329.693, 314.485, 327.204, 308.319, 328.779, 319.072, 322.936, 333.713, 337.34, 328.706, 357.767, 319.589, 374.075, 316.75, 385.376, 312.821, 399.953, 310.219, 418.01, 315.507, 427.218, 319.739, 435.515, 320.355, 450.548, 334.267, 461.265, 343.959, 473.04, 351.043, 484.453));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 376.2, 484.994, 385.157, 476.199, 390.311, 462.872, 391.257, 446.609, 386.918, 431.396, 377.802, 419.757, 363.723, 414.418, 344.873, 413.672, 336.116, 400.653, 339.804, 382.397, 337.4, 362.636, 333.396, 350.375, 336.647, 332.469, 330.326, 316.609, 325.86, 323.058, 310.861, 310.382, 294.729, 307.393, 303.873, 324.684, 316.868, 344.873, 313.474, 366.737, 303.509, 381.174, 308.056, 390.523, 305.581, 406.623, 311.483, 418.698, 320.298, 424.852, 320.112, 439.377, 323.147, 453.801, 336.515, 464.401, 346.218, 475.957, 354.714, 487.366));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 373.18, 483.116, 383.7, 475.223, 388.851, 462.241, 390.667, 446.192, 385.749, 430.633, 377.803, 417.765, 362.51, 412.95, 342.023, 412.304, 335.2, 399.1, 342.885, 378.346, 339.828, 359.602, 336.167, 347.151, 341.256, 329.555, 331, 315.601, 323.711, 325.961, 315.789, 322.617, 307.183, 320.725, 314.445, 322.163, 325.887, 337.959, 320.93, 357.517, 312.152, 373.091, 311.497, 384.272, 309.413, 399.228, 310.948, 414.907, 317.851, 423.768, 320.972, 434.092, 322.706, 449.166, 335.029, 460.69, 344.531, 472.995, 352.102, 485.121));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 367.087, 492.037, 381.581, 487.232, 388.77, 474.296, 392.105, 457.723, 389.608, 440.497, 384.129, 425.242, 370.092, 417.62, 352.218, 412.739, 342.487, 401.332, 345.183, 382.292, 343.12, 364.339, 339.74, 349.296, 342.382, 332.024, 334.462, 317.896, 324.809, 320.514, 318.227, 323.322, 312.808, 325.437, 313.81, 323.72, 317.205, 335.317, 311.487, 350.722, 305.381, 366.762, 303.726, 380.405, 304.655, 395.356, 308.538, 410.717, 317.272, 420.824, 323.66, 430.93, 326.877, 446.393, 335.334, 459.927, 344.766, 472.992, 352.342, 486.15));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 368.967, 487.904, 383.101, 483.266, 390.162, 471.105, 393.338, 455.132, 390.762, 438.385, 385.946, 423.205, 372.577, 415.699, 354.873, 411.756, 343.953, 401.732, 345.451, 383.646, 343.316, 366.167, 339.349, 351.575, 341.814, 334.5, 335.133, 319.888, 325.423, 319.593, 317.771, 321.549, 312.368, 325.962, 313.767, 324.553, 317.116, 335.509, 311.835, 350.73, 306.222, 366.593, 304.974, 380.201, 305.756, 394.907, 309.561, 409.98, 318.488, 419.479, 325.211, 429.361, 328.74, 444.343, 337.324, 457.4, 346.857, 469.792, 354.471, 482.292));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 369.567, 484.795, 383.666, 480.148, 390.558, 467.767, 393.644, 451.791, 391.067, 435.144, 385.93, 420.255, 372.436, 412.934, 355.072, 408.59, 344.949, 398.149, 346.844, 379.918, 344.633, 362.659, 340.975, 348.11, 343.264, 331.328, 335.863, 317.476, 326.328, 318.833, 319.69, 321.708, 314.679, 324.878, 315.582, 322.874, 318.722, 333.666, 313.426, 348.497, 307.933, 364.181, 306.473, 377.598, 307.582, 392.058, 311.478, 406.95, 320.189, 416.579, 326.831, 426.2, 330.268, 441.089, 338.558, 454.094, 347.923, 466.519, 355.442, 479.066));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 365.831, 487.011, 380.951, 483.613, 389.018, 471.668, 393.12, 455.637, 391.924, 438.425, 388.14, 422.631, 375.542, 413.843, 358.93, 408.072, 347.531, 397.931, 348.019, 380.507, 346.934, 363.021, 343.657, 347.618, 345.953, 330.649, 340.882, 315.631, 331.095, 312.764, 322.982, 314.767, 317.232, 319.544, 317.1, 319.377, 318.138, 329.952, 312.235, 344.205, 306.496, 359.812, 305.092, 373.802, 306.003, 388.703, 310.425, 403.462, 319.68, 413.367, 326.314, 423.988, 329.763, 439.174, 336.945, 452.964, 345.97, 465.874, 353.19, 478.897));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 366.438, 485.393, 381.322, 481.283, 388.927, 468.812, 392.592, 452.483, 390.689, 435.234, 386.046, 419.663, 372.701, 411.472, 355.461, 406.173, 344.759, 395.534, 346.155, 377.331, 344.49, 359.643, 341.067, 344.386, 343.398, 327.257, 336.853, 312.699, 327.023, 312.424, 319.731, 315.083, 314.309, 318.729, 314.627, 317.579, 316.727, 328.516, 310.981, 343.273, 305.223, 359.157, 303.713, 373.051, 304.785, 387.967, 309.03, 403.026, 318.109, 413.018, 324.799, 423.306, 328.248, 438.62, 336.062, 452.285, 345.368, 465.234, 352.838, 478.301));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 362.762, 492.64, 379.33, 489.61, 388.263, 476.152, 392.875, 458.782, 392.671, 440.352, 388.013, 423.981, 374.897, 414.081, 359.343, 405.441, 348.831, 393.372, 348.872, 375.192, 348.344, 356.879, 346.159, 339.972, 347.48, 322.553, 341.727, 308.032, 332.048, 307.399, 324.722, 310.447, 318.235, 311.679, 315.417, 313.879, 314.398, 325.358, 307.92, 339.292, 301.9, 355.07, 300.851, 369.547, 302.895, 385.239, 309.498, 399.704, 319.508, 410.465, 325.57, 422.362, 329.285, 438.415, 335.337, 453.369, 344.224, 467.506, 351.632, 481.871));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 372.378, 485.757, 385.768, 477.534, 390.04, 460.835, 391.199, 442.418, 385.273, 425.104, 374.227, 412.092, 355.525, 408.309, 335.254, 403.791, 332.877, 386.4, 343.192, 360.97, 338.041, 342.544, 335.801, 327.891, 338.786, 310.512, 319.728, 302.571, 311.697, 327.078, 314.806, 337.562, 312.388, 327.585, 313.208, 316.238, 321.497, 328.762, 315.918, 345.362, 309.561, 362.6, 305.25, 375.467, 307.93, 389.786, 310.488, 407.112, 317.063, 419.328, 323.261, 426.723, 325.498, 443.246, 335.83, 456.382, 345.919, 470.57, 354.166, 484.876));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 374.884, 480.273, 386.462, 470.981, 389.837, 454.684, 390.385, 436.783, 383.457, 420.402, 371.822, 408.424, 352.911, 405.915, 331.581, 403.612, 329.924, 385.846, 341.995, 359.76, 336.405, 341.251, 333.906, 327.827, 338.008, 310.322, 318.428, 302.439, 311.61, 329.052, 313.971, 336.868, 310.327, 326.247, 313.825, 315.516, 325.488, 329.176, 320.363, 347.36, 313.07, 364.448, 309.008, 376.391, 310.295, 390.631, 311.598, 408.261, 317.379, 419.932, 322.434, 427.232, 324.07, 443.505, 335.983, 455.597, 346.112, 469.266, 354.293, 482.937));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 376.671, 477.117, 384.849, 466.222, 388.173, 449.453, 387.948, 431.63, 382.385, 416.311, 368.51, 407.182, 351.338, 404.071, 332.489, 400.702, 331.479, 381.01, 341.465, 358.391, 337.962, 338.302, 336.541, 325.907, 339.751, 308.802, 325.347, 298.96, 323.799, 324.948, 317.095, 316.889, 301.245, 299.744, 308.498, 314.372, 323.743, 336.876, 319.726, 358.934, 308.589, 372.953, 311.663, 380.757, 310.637, 396.091, 316.892, 408.013, 323.862, 415.889, 321.414, 429.568, 323.119, 444.654, 336.454, 455.404, 345.544, 468.448, 353.96, 481.329));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 368.256, 481.496, 381.015, 476.726, 387.904, 465.134, 390.591, 449.922, 388.672, 434.192, 383.675, 420.273, 371.867, 412.627, 356.134, 408.61, 344.548, 399.499, 343.429, 384.118, 342.012, 367.07, 338.037, 353.285, 339.335, 337.036, 335.694, 321.972, 327.793, 318.731, 316.757, 315.154, 307.656, 318.909, 309.375, 326.778, 312.733, 339.67, 308.364, 355.188, 302.477, 369.387, 304.299, 381.459, 304.779, 395.917, 310.933, 408.138, 320.599, 415.606, 325.074, 427.192, 329.169, 440.957, 337.7, 453.004, 346.774, 464.319, 354.446, 475.771));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 364.751, 484.136, 379.514, 480.075, 387.532, 467.537, 391.251, 451.079, 390.089, 433.778, 385.333, 418.36, 372.724, 409.485, 356.504, 403.399, 345.032, 392.773, 344.569, 375.761, 343.633, 357.734, 340.345, 342.247, 341.958, 325.082, 337.53, 309.778, 328.666, 307.285, 319.053, 306.307, 310.888, 309.288, 310.705, 314.642, 312.216, 327.248, 306.634, 342.489, 300.425, 357.71, 300.862, 371.222, 301.796, 386.533, 307.986, 400.204, 317.964, 409.377, 323.22, 421.31, 327.11, 436.447, 334.781, 450.15, 343.95, 463.023, 351.609, 476.052));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 363.002, 480.436, 378.017, 476.83, 386.307, 464.832, 390.515, 448.601, 389.437, 431.208, 385.592, 415.28, 373.022, 406.232, 356.117, 400.526, 344.359, 390.156, 344.744, 372.753, 343.953, 354.743, 340.633, 339.228, 343.187, 321.889, 338.877, 306.112, 329.376, 302.524, 320.004, 302.48, 312.904, 307.178, 313.311, 309.472, 314.86, 321.189, 308.837, 336.273, 302.472, 351.875, 301.69, 365.824, 302.047, 381.138, 306.739, 395.838, 316.248, 405.559, 322.2, 416.965, 325.609, 432.346, 333.212, 446.281, 342.3, 459.354, 349.619, 472.509));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 366.024, 476.668, 380.535, 472.839, 388.662, 461.14, 393.029, 445.16, 391.642, 427.976, 387.931, 412.168, 375.069, 403.475, 357.207, 398.502, 346.281, 387.728, 348.69, 369.715, 347.929, 351.588, 344.745, 336.526, 348.572, 319.17, 343.647, 303.265, 334.034, 301.095, 325.054, 300.916, 318.321, 305.499, 320.265, 305.94, 323.224, 317.929, 316.816, 333.53, 309.708, 349.186, 308.018, 362.883, 307.372, 378.072, 310.353, 393.521, 318.98, 403.515, 324.721, 414.488, 327.309, 429.956, 335.431, 443.799, 344.314, 456.991, 351.25, 470.139));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 389.424, 458.324, 390.691, 443.227, 391.729, 427.019, 389.869, 410.162, 381.741, 397.456, 364.888, 392.102, 346.858, 392.312, 324.4, 395.393, 326.799, 372.425, 343.938, 347.776, 340.153, 325.496, 338.843, 317.025, 346.609, 298.725, 331.645, 287.004, 335.253, 321.2, 321.448, 298.064, 296.357, 275.588, 315.811, 301.844, 344.831, 332.769, 342.07, 362.73, 324.855, 375.792, 330.786, 378.973, 323.374, 395.517, 326.45, 407.286, 330.782, 412.657, 321.324, 429.476, 320.282, 444.239, 339.802, 451.72, 348.526, 463.881, 356.597, 475.305));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 376.088, 471.35, 388.697, 467.445, 395.785, 457.527, 399.48, 443.395, 397.47, 428.074, 394.597, 413.538, 382.958, 406.083, 365.461, 403.611, 354.126, 395.065, 356.188, 378.784, 355.063, 362.225, 351.135, 349.098, 355.249, 332.923, 351.459, 317.317, 342.365, 312.926, 332.736, 311.423, 326.576, 318.631, 330.311, 318.213, 334.582, 328.978, 329.122, 344.078, 322.679, 358.738, 321.217, 371.342, 319.814, 385.243, 321.429, 399.918, 329.266, 408.549, 335.012, 418.204, 337.458, 432.155, 346.03, 444.295, 354.605, 455.758, 361.13, 467.122));
landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << 364.552, 474.627, 383.796, 474.18, 394.086, 460.236, 399.613, 442.4, 401.628, 423.091, 398.023, 405.768, 386.069, 393.996, 374.273, 381.214, 362.832, 369.459, 358.685, 352.309, 358.85, 335.005, 357.784, 316.148, 356.187, 299.263, 351.307, 286.201, 341.105, 282.396, 335.471, 290.854, 330.951, 292.14, 321.917, 293.094, 314.316, 301.013, 307.844, 311.106, 304.092, 326.841, 302.715, 342.604, 307.976, 358.307, 317.909, 371.75, 329.303, 383.495, 337.069, 395.632, 342.079, 411.849, 344.346, 427.816, 352.788, 442.503, 360.115, 457.745));

 vector<Pdm::PoseParameter> poses;
#if 0
 poses.push_back(Pdm::PoseParameter { 0.0330138,1.00795,343.473,404.319 });
 poses.push_back(Pdm::PoseParameter { 0.0619686,0.979025,342.06,403.536 });
 poses.push_back(Pdm::PoseParameter { 0.056353,0.992101,342.651,402.967 });
 poses.push_back(Pdm::PoseParameter { 0.0579414,1.01725,341.631,401.862 });
 poses.push_back(Pdm::PoseParameter { 0.0590554,1.0287,341.722,399.666 });
 poses.push_back(Pdm::PoseParameter { 0.0678192,1.06439,342.126,398.382 });
 poses.push_back(Pdm::PoseParameter { 0.0736434,1.06815,343.824,397.44 });
 poses.push_back(Pdm::PoseParameter { 0.0719175,1.0814,343.057,395.247 });
 poses.push_back(Pdm::PoseParameter { 0.0766527,1.07413,342.241,394.026 });
 poses.push_back(Pdm::PoseParameter { 0.0859378,1.1032,341.735,392.609 });
 poses.push_back(Pdm::PoseParameter { 0.0729432,1.07755,340.841,392.114 });
 poses.push_back(Pdm::PoseParameter { 0.0626798,1.06188,338.854,390.672 });
 poses.push_back(Pdm::PoseParameter { 0.048169,1.08007,337.512,389.624 });
 poses.push_back(Pdm::PoseParameter { 0.054333,1.06811,338.501,389.193 });
 poses.push_back(Pdm::PoseParameter { 0.0503577,1.08148,338.567,388.649 });
 poses.push_back(Pdm::PoseParameter { 0.0506671,1.06407,338.115,385.833 });
 poses.push_back(Pdm::PoseParameter { 0.0733453,1.03299,343.657,384.689 });
 poses.push_back(Pdm::PoseParameter { 0.0836908,1.0168,346.424,382.328 });
 poses.push_back(Pdm::PoseParameter { 0.0716784,1.15598,344.415,374.766 });
 poses.push_back(Pdm::PoseParameter { 0.0716704,1.15015,348.442,369.919 });

 poses.push_back(Pdm::PoseParameter { 0.0330138, 1.00795, 343.473, 404.319 });
 poses.push_back(Pdm::PoseParameter { 0.211286, 0.981219, 327.735, 393.187 });
 poses.push_back(Pdm::PoseParameter { 0.0728624, 0.985415, 341.528, 401.272 });
 poses.push_back(Pdm::PoseParameter { 0.23478, 0.93679, 327.052, 415.102 });
 poses.push_back(Pdm::PoseParameter { 0.0192352, 0.993076, 344.259, 405.002 });
 poses.push_back(Pdm::PoseParameter { 0.0834558, 1.02564, 343.335, 398.081 });
 poses.push_back(Pdm::PoseParameter { 0.0848104, 1.01488, 344.78, 396.29 });
 poses.push_back(Pdm::PoseParameter { 0.107811, 1.07084, 342.952, 394.165 });
 poses.push_back(Pdm::PoseParameter { 0.108391, 1.07081, 343.187, 393.196 });
 poses.push_back(Pdm::PoseParameter { 0.104876, 1.09445, 341.677, 391.944 });
 poses.push_back(Pdm::PoseParameter { 0.0669093, 1.10823, 339.973, 391.566 });
 poses.push_back(Pdm::PoseParameter { -0.0355813, 1.00522, 336.877, 391.035 });
 poses.push_back(Pdm::PoseParameter { -0.0612164, 0.989676, 336.531, 390.777 });
 poses.push_back(Pdm::PoseParameter { -0.190549, 0.866618, 345.742, 408.474 });
 poses.push_back(Pdm::PoseParameter { 0.0550106, 1.06598, 339.332, 391.062 });
 poses.push_back(Pdm::PoseParameter { 0.059484, 1.07623, 339.302, 387.113 });
 poses.push_back(Pdm::PoseParameter { 0.0798504, 1.04792, 343.416, 383.892 });
 poses.push_back(Pdm::PoseParameter { 0.0549327, 1.07481, 341.798, 386.715 });
 poses.push_back(Pdm::PoseParameter { 0.0123289, 1.11045, 342.08, 377.685 });
 poses.push_back(Pdm::PoseParameter { 0.0988353, 1.1467, 349.923, 370.101 });
#endif
 poses.push_back(Pdm::PoseParameter { 0.0428312, 1.0078, 341.848, 401.864 });
 poses.push_back(Pdm::PoseParameter { -0.00569151, 0.955154, 339.613, 399.595 });
 poses.push_back(Pdm::PoseParameter { -0.0537563, 1.03513, 337.508, 400.499 });
 poses.push_back(Pdm::PoseParameter { -0.00563958, 1.00072, 339.199, 398.468 });
 poses.push_back(Pdm::PoseParameter { 0.0280699, 1.05651, 339.795, 400.135 });
 poses.push_back(Pdm::PoseParameter { 0.0112074, 1.03566, 340.821, 399.207 });
 poses.push_back(Pdm::PoseParameter { 0.0129085, 1.0235, 341.986, 396.462 });
 poses.push_back(Pdm::PoseParameter { 0.0549202, 1.05542, 342.503, 395.23 });
 poses.push_back(Pdm::PoseParameter { 0.036378, 1.05693, 340.749, 393.564 });
 poses.push_back(Pdm::PoseParameter { 0.0780047, 1.11972, 341.767, 392.955 });
 poses.push_back(Pdm::PoseParameter { -0.0240032, 1.03849, 337.419, 392.18 });
 poses.push_back(Pdm::PoseParameter { -0.0397476, 1.00643, 337.328, 391.031 });
 poses.push_back(Pdm::PoseParameter { -0.0420535, 1.01089, 337.705, 388.964 });
 poses.push_back(Pdm::PoseParameter { -0.0107887, 1.00029, 339.67, 397.271 });
 poses.push_back(Pdm::PoseParameter { 0.036887, 1.07594, 339.288, 391.017 });
 poses.push_back(Pdm::PoseParameter { 0.0595242, 1.07639, 339.306, 387.188 });
 poses.push_back(Pdm::PoseParameter { 0.0789581, 1.06271, 343.171, 384.446 });
 poses.push_back(Pdm::PoseParameter { -0.0556546, 0.953901, 342.871, 379.716 });
 poses.push_back(Pdm::PoseParameter { 0.0532116, 0.960846, 352.27, 390.067 });
 poses.push_back(Pdm::PoseParameter { 0.0988353, 1.1467, 349.923, 370.101 });


#if 0
 auto poseAlmostEqual = [](const Pdm::PoseParameter& lh,
                           const Pdm::PoseParameter& rh)->bool {
   return std::abs(lh.theta - rh.theta) < 0.001 &&
   std::abs(lh.scale - rh.scale) < 0.001 &&
   std::abs(lh.tx - rh.tx) < 0.001 &&
   std::abs(lh.ty - rh.ty) < 0.001;
 };
#endif
 
 for (size_t i = 0; i < vfs.size(); ++i) {
   fd.process(i);
   be.process(i);
   bpc.process(i);
   Mat lhMask, rhMask, headMask;
   BlackBoardPointer<std::list<BodyPart> > bodyParts = bb.get<std::list<BodyPart>>(i, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY);
   
   createGuessMasks(*bodyParts,
                    lhMask, rhMask, headMask, frameSize);
   if (oldInstance.get())
     *oldInstance = asmi.fit(vfs[i], rhMask, *oldInstance);
   else 
     oldInstance.reset(new Asm::Instance(asmi.fit(vfs[i], rhMask)));    
   
   BOOST_CHECK(oldInstance->pdm.get() == asmi.pdm.get());
   // cout << oldInstance->bt - bts[i] << endl;

   // these exact checks have been disabled because they do not work reliably on differing 
   // platforms

   // BOOST_CHECK(almostEqual(oldInstance->bt, bts[i], 0.001));

   // cout << "bts.push_back((Mat_<double>(3, 1, CV_64FC1)" << endl
   //      << "               <<"
   //      << oldInstance->bt.at<double>(0,0) << ", "
   //      << oldInstance->bt.at<double>(1,0) << ", " << endl
   //      << "               " << oldInstance->bt.at<double>(2,0) << "));"
   //      << endl;

   // BOOST_CHECK(poseAlmostEqual(oldInstance->pose, poses[i]));
   // cout << "poses.push_back(Pdm::PoseParameter { "
   //      << oldInstance->pose.theta << ", " << oldInstance->pose.scale
   //      << ", " << oldInstance->pose.tx << ", " << oldInstance->pose.ty
   //      << " });" << endl;

   // BOOST_CHECK(almostEqual(oldInstance->landmarks, landmarks[i], 0.001));
   // cout << "landmarks.push_back((Mat_<double>(60, 1, CV_64FC1) << ";
   // for (int i = 0; i < oldInstance->landmarks.rows; i += 2) {
   //   if (i > 0)
   //     cout << ", ";
   //   cout << oldInstance->landmarks.at<double>(i,0) << ", ";
   //   cout << oldInstance->landmarks.at<double>(i+1,0);
   // }
   // cout << "));" << endl;
 }

 vector<Blob> lhBlobs, rhBlobs;

 for (size_t i = 0; i < vfs.size(); ++i) {
   BlackBoardPointer<cv::Mat> m = bb.get<Mat>(i, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);
   
   vector<Blob> blobber = Blob::extractBlobsFromMask(*m, 5000, 10000);
   rhBlobs.push_back(blobber[0]);
   if (blobber.size() == 3)
     lhBlobs.push_back(blobber[1]);
 }

 Asm lh(lhBlobs, nLandmarks, 3, 1.0); // 3 = t = principal component count
 Asm rh(rhBlobs, nLandmarks, 3, 1.0); // 3 = t = principal component count
 Asm::Instance lhOldInstance;
 Asm::Instance rhOldInstance;
 bool first = true;

 for (size_t i = 0; i < vfs.size(); ++i) {
   Mat lhMask, rhMask, headMask;
   BlackBoardPointer<std::list<BodyPart> > bodyParts = bb.get<std::list<BodyPart>>(i, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY);
   
   createGuessMasks(*bodyParts,
                    lhMask, rhMask, headMask, frameSize);

   if (first) {
     debug = 1;
     lhOldInstance = lh.fit(vfs[i], lhMask);
     debug = 0;
     rhOldInstance = rh.fit(vfs[i], rhMask);
     first = false;
   }
   else {
     lhOldInstance = lh.fit(vfs[i], lhMask, lhOldInstance);
     rhOldInstance = rh.fit(vfs[i], rhMask, rhOldInstance);
   }

   // cout << intersectionArea(lh, rh)  << endl;
   // cout << computeOrientation(lhOldInstance)/(PI)*180. << endl;

   cv::Size s;
   cv::Point2d p;
   lhOldInstance.getMinSize(s, p);
   Mat temp = vfs[i];
   lhOldInstance.drawLine(temp, 1, cv::Scalar::all(255));
 }


}

BOOST_AUTO_TEST_CASE( testParanoid ) {
  slmotion::paranoid = false;

  {
    std::unique_ptr<OpenCVVideoCaptureVideoFileSource> vfs_p(new OpenCVVideoCaptureVideoFileSource(TESTVIDEOFILE));
    OpenCVVideoCaptureVideoFileSource& vfs = *vfs_p;
    BOOST_CHECK(vfs.size() == 10);
  }
  {
    std::unique_ptr<OpenCVVideoCaptureVideoFileSource> vfs_p(new OpenCVVideoCaptureVideoFileSource(SLMOTION_TEST_DATA_DIRECTORY"/M_1.avi"));
    OpenCVVideoCaptureVideoFileSource& vfs = *vfs_p;
    BOOST_CHECK(vfs.size() == 183 || vfs.size() == 61);
    // 61 is the expected number of frames
  }

  slmotion::paranoid = true;

  {
    std::unique_ptr<OpenCVVideoCaptureVideoFileSource> vfs_p(new OpenCVVideoCaptureVideoFileSource(TESTVIDEOFILE));
    OpenCVVideoCaptureVideoFileSource& vfs = *vfs_p;
    BOOST_CHECK(vfs.size() == 10);
  }

  // it turns out that there is a level of disagreement between the number of 
  // frames in a given video, depending on the library used to decode the video
  // therefore, trust this value, since this is what actualy happens when the 
  // program is run
  cv::VideoCapture vc(SLMOTION_TEST_DATA_DIRECTORY"/M_1.avi");
  size_t nFrames = 0;
  while (vc.grab())
    ++nFrames;

  {
    std::unique_ptr<OpenCVVideoCaptureVideoFileSource> vfs_p(new OpenCVVideoCaptureVideoFileSource(SLMOTION_TEST_DATA_DIRECTORY"/M_1.avi"));
    OpenCVVideoCaptureVideoFileSource& vfs = *vfs_p;
    // BOOST_CHECK(vfs.size() == 61);
    // WHY does this fail???

    BOOST_CHECK(vfs.size() == nFrames);
  }

  slmotion::paranoid = false;
}



#ifdef SLMOTION_ENABLE_VIDFILE_SUPPORT
BOOST_AUTO_TEST_CASE( testVidFileSource ) {
  vector<string> pngFiles;
  for (int i = 0; i < 10; ++i) {
    char s[6];
    sprintf(s, "%05i", i);
    pngFiles.push_back(string(SLMOTION_TEST_DATA_DIRECTORY"/testivideo5.vid.frame.") + s + ".png");
  }

  std::unique_ptr<ImageSequenceSource> issptr(new ImageSequenceSource(pngFiles));
  ImageSequenceSource& iss = *issptr;

  std::unique_ptr<VidFileSource> vfs(new VidFileSource(TESTVIDFILE, 0));

  // for (size_t i = 0; i < vfs->size(); ++i) {
  //   cv::imshow("", (*vfs)[i]);
  //   cv::waitKey(0);
  // }

  BOOST_CHECK(vfs->getNumberOfTracks() == 1);
  BOOST_CHECK(&vfs->getTrack(0) == vfs.get());
  BOOST_CHECK(vfs->getDefaultTrackNumber() == 0);
  BOOST_CHECK_THROW(vfs->getTrack(1), std::out_of_range);

  BOOST_REQUIRE(vfs->size() == iss.size());

  std::uniform_int_distribution<int> distribution(0, iss.size()-1);
  std::mt19937 engine;
  auto generator = std::bind(distribution, engine);
  for (int i = 0; i < 100; ++i) {
    int j = generator();
    Mat diff = (*vfs)[j] - iss[j];
    BOOST_CHECK(allEqual(diff, 0));
  }

  {
    auto jt = iss.begin();
    for (auto it = vfs->begin(); it != vfs->end(); ++it) {
      Mat diff = *it - *jt++;
      BOOST_CHECK(allEqual(diff, 0));
    }
  }

  {
    auto jt = iss.rbegin();
    for (auto it = vfs->rbegin(); it != vfs->rend(); ++it) {
      Mat diff = *it - *jt++;
      BOOST_CHECK(allEqual(diff, 0));
    }
  }

  std::unique_ptr<VidFileSource> vfs2(new VidFileSource(TESTVIDFILE));
  for (int i = 0; i < 100; ++i) {
    int j = generator();
    Mat diff = (*vfs2)[j] - iss[j];
    BOOST_CHECK(allEqual(diff, 0));
  }
}
#endif



#ifdef SLMOTION_ENABLE_OPENNI
BOOST_AUTO_TEST_CASE( testOniFileSource ) {
  
  OpenNiOniFileSource ofs(MARKUS_ONI, 0);
  BOOST_CHECK(ofs.size() == 149);

  vector<Mat> frames;

  for (size_t i = 0; i < ofs.size(); ++i) {
    const Mat& m = ofs[i];
    BOOST_REQUIRE(!m.empty());
    BOOST_REQUIRE(!allEqual(m, 0));
    if (i > 0)
      BOOST_CHECK(!equal(m, frames.back()));
    frames.push_back(m.clone());
  }

  {
    auto jt = frames.rbegin();
    auto it = ofs.rbegin();
    while(it != ofs.rend())
      BOOST_CHECK(equal(*it++, *jt++));
  }

  std::uniform_int_distribution<int> distribution(0, ofs.size()-1);
  std::mt19937 engine;
  auto generator = std::bind(distribution, engine);
  
  for (int i = 0; i < 300; ++i) {
    int j = generator();
    BOOST_CHECK(equal(ofs[j], frames[j]));  
  }

  BOOST_CHECK_THROW(ofs.getTrack(2), std::out_of_range);

  for (int i = 0; i < 300; ++i) {
    int j = generator();
    BOOST_CHECK(equal(ofs.getTrack(0)[j], frames[j]));  
  }

  vector<Mat> depthFrames;
  FrameSource& dt = ofs.getTrack(1);
  for (size_t j = 0; j < dt.size(); ++j) {
    const Mat& m = dt[j];
    BOOST_REQUIRE(!m.empty());
    BOOST_REQUIRE(!allEqual(m, 0));
    if (j > 0)
      BOOST_CHECK(!equal(m, frames.back()));

    depthFrames.push_back(m.clone());
  }

  {
    auto jt = depthFrames.rbegin();
    auto it = dt.rbegin();
    while(it != dt.rend())
      BOOST_CHECK(equal(*it++, *jt++));
  }

  for (int i = 0; i < 300; ++i) {
    int j = generator();
    BOOST_CHECK(equal(dt[j], depthFrames[j]));  
  }  

  OpenNiOniFileSource ofs2(MARKUS_ONI);

  for (int i = 0; i < 300; ++i) {
    int j = generator();
    BOOST_CHECK(equal(ofs2[j], frames[j]));  
  }

  for (int i = 0; i < 300; ++i) {
    int j = generator();
    BOOST_CHECK(equal(ofs2.getTrack(0)[j], frames[j]));  
  }

  FrameSource& dt2 = ofs2.getTrack(1);

  for (int i = 0; i < 300; ++i) {
    int j = generator();
    BOOST_CHECK(equal(dt2[j], depthFrames[j]));  
  }  

}



BOOST_AUTO_TEST_CASE( testOniFileSource2 ) {
  OpenNiOniFileSource ononifs(CAPTURE_ONI_NOSKEL, 0);
  auto oniFrames = getOniTestVideoFrameFilenames();
  std::vector<string> rgb = oniFrames["rgb-noskel"];
  ImageSequenceSource rgbiss(std::move(rgb), 30, 0);
  std::vector<string> depth = oniFrames["depth-noskel"];
  ImageSequenceSource depthiss(std::move(depth), 30, 0);

  BOOST_REQUIRE(ononifs.size() == rgbiss.size());
  BOOST_REQUIRE(ononifs.size() == depthiss.size());
  BOOST_REQUIRE(ononifs.getNumberOfTracks() == 2);

  for (size_t i = 0; i < ononifs.size(); ++i) {
    BOOST_REQUIRE(equal(ononifs[i], rgbiss[i]));
    BOOST_REQUIRE(equal(ononifs.getTrack(0)[i], rgbiss[i]));
    // m = convertToMarkusFormat(ononifs.getTrack(1)[i]);
    BOOST_REQUIRE(equal(convertToMarkusFormat(ononifs.getTrack(1)[i]), depthiss[i]));
  }
}

BOOST_AUTO_TEST_CASE( testOniFileSource3 ) {
  OpenNiOniFileSource ononifs(CAPTURE_ONI_SKEL, 0);
  auto oniFrames = getOniTestVideoFrameFilenames();
  std::vector<string> rgb = oniFrames["rgb-skel"];
  ImageSequenceSource rgbiss(std::move(rgb), 30, 0);
  std::vector<string> depthskel = oniFrames["depth-skel"];
  ImageSequenceSource depthissskel(std::move(depthskel), 30, 0);
  std::vector<string> depth = oniFrames["depth-noskel"];
  ImageSequenceSource depthiss(std::move(depth), 30, 0);

  BOOST_REQUIRE(ononifs.size() == rgbiss.size());
  BOOST_REQUIRE(ononifs.size() == depthiss.size());
  BOOST_REQUIRE(ononifs.getNumberOfTracks() == 3);

  std::vector<cv::Mat> skeletons;

  for (size_t i = 0; i < ononifs.size(); ++i) {
    BOOST_REQUIRE(equal(ononifs[i], rgbiss[i]));
    BOOST_REQUIRE(equal(ononifs.getTrack(0)[i], rgbiss[i]));
    BOOST_REQUIRE(equal(convertToMarkusFormat(ononifs.getTrack(1)[i]), depthiss[i]));
    skeletons.push_back(ononifs.getTrack(2)[i].clone());
  }

  std::uniform_int_distribution<int> distribution(0, ononifs.size()-1);
  std::mt19937 engine;
  auto generator = std::bind(distribution, engine);
  
  for (int i = 0; i < 600; ++i) {
    int j = generator();
    BOOST_REQUIRE(equal(ononifs[j], rgbiss[j]));
    BOOST_REQUIRE(equal(convertToMarkusFormat(ononifs.getTrack(1)[j]), 
                        depthiss[j]));
    BOOST_REQUIRE(equal(skeletons[j], ononifs.getTrack(2)[j]));
  }
}
#endif



BOOST_AUTO_TEST_CASE( testMultiFrameSource ) {
  shared_ptr<ImageSequenceSource> iss(new ImageSequenceSource(getTestVideoFramefilenames()));
  BOOST_CHECK(iss->getNumberOfTracks() == 1);
  BOOST_CHECK(&iss->getTrack(0) == iss.get());
  BOOST_CHECK(iss->getDefaultTrackNumber() == 0);
  BOOST_CHECK_THROW(iss->getTrack(1), std::out_of_range);

  MultiFrameSource mfs;
  BOOST_CHECK(mfs.getNumberOfTracks() == 0);
  BOOST_CHECK(mfs.getDefaultTrackNumber() == 0);
  BOOST_CHECK_THROW(mfs.getTrack(0), std::out_of_range);
  BOOST_CHECK_THROW(mfs.getDefaultTrack(), std::out_of_range);
  
  TrackInfo ti = iss->getTrackInfo();
  BOOST_CHECK(ti.type == TrackInfo::Type::BGR_IMAGE);
  BOOST_CHECK(ti.trackName == "");
  iss->setTrackInfo(TrackInfo { TrackInfo::Type::BGR_IMAGE, 
        "ImageSequence" });
  ti = iss->getTrackInfo();
  BOOST_CHECK(ti.type == TrackInfo::Type::BGR_IMAGE);
  BOOST_CHECK(ti.trackName == "ImageSequence");

  mfs.addTrack(iss);
  BOOST_CHECK(mfs.getNumberOfTracks() == 1);
  BOOST_CHECK(&mfs.getTrack(0) == iss.get());
  BOOST_CHECK(mfs.getDefaultTrackNumber() == 0);
  BOOST_CHECK_THROW(mfs.getTrack(1), std::out_of_range);

  BOOST_REQUIRE(mfs.size() == iss->size());
  {
    auto it = iss->cbegin();
    auto jt = mfs.cbegin();
    for (size_t i = 0; i < iss->size(); ++i)
      BOOST_CHECK(equal(*it++, *jt++));
  }

  paranoid = true;
  shared_ptr<VideoFileSource> vfs1(new OpenCVVideoCaptureVideoFileSource(KINECT_VIDEO));
  vfs1->setTrackInfo(TrackInfo{TrackInfo::Type::BGR_IMAGE, "KinectVideo"});
  BOOST_CHECK(vfs1->getNumberOfTracks() == 1);
  BOOST_CHECK(&vfs1->getTrack(0) == vfs1.get());
  BOOST_CHECK(vfs1->getDefaultTrackNumber() == 0);
  BOOST_CHECK_THROW(vfs1->getTrack(1), std::out_of_range);
  ti = vfs1->getTrackInfo();
  BOOST_CHECK(ti.type == TrackInfo::Type::BGR_IMAGE);
  BOOST_CHECK(ti.trackName == "KinectVideo");

  shared_ptr<VideoFileSource> vfs2(new OpenCVVideoCaptureVideoFileSource(KINECT_DEPTH));
  vfs2->setTrackInfo(TrackInfo{TrackInfo::Type::DEPTH_DATA, "KinectDepth"});
  BOOST_CHECK(vfs2->getNumberOfTracks() == 1);
  BOOST_CHECK(&vfs2->getTrack(0) == vfs2.get());
  BOOST_CHECK(vfs2->getDefaultTrackNumber() == 0);
  BOOST_CHECK_THROW(vfs2->getTrack(1), std::out_of_range);
  ti = vfs2->getTrackInfo();
  BOOST_CHECK(ti.type == TrackInfo::Type::DEPTH_DATA);
  BOOST_CHECK(ti.trackName == "KinectDepth");

  mfs.addTrack(vfs1);
  mfs.addTrack(vfs2);

  BOOST_CHECK(mfs.getNumberOfTracks() == 3);
  BOOST_CHECK(&mfs.getTrack(0) == iss.get());
  BOOST_CHECK(&mfs.getTrack(1) == vfs1.get());
  BOOST_CHECK(&mfs.getTrack(2) == vfs2.get());
  BOOST_CHECK(mfs.getDefaultTrackNumber() == 0);
  BOOST_CHECK_THROW(mfs.getTrack(3), std::out_of_range);

  {
    auto it = iss->cbegin();
    auto jt = mfs.cbegin();
    for (size_t i = 0; i < iss->size(); ++i)
      BOOST_CHECK(equal(*it++, *jt++));
  }


  mfs.setDefaultTrackNumber(2);
  BOOST_CHECK(&mfs.getDefaultTrack() == vfs2.get());

  BOOST_REQUIRE(iss->size() < vfs1->size());
  BOOST_REQUIRE(vfs2->size() == vfs1->size());
  BOOST_CHECK(iss->size() == mfs.size());

  {
    auto it = vfs2->cbegin();
    auto jt = mfs.cbegin();
    for (size_t i = 0; i < mfs.size(); ++i)
      BOOST_CHECK(equal(*it++, *jt++));
  }


  mfs.tracks.pop_front();
  mfs.setDefaultTrackNumber(1);
  BOOST_CHECK(mfs.getNumberOfTracks() == 2);
  BOOST_CHECK(mfs.size() == vfs1->size());

  {
    auto it = vfs2->cbegin();
    auto jt = mfs.cbegin();
    for (size_t i = 0; i < vfs2->size(); ++i)
      BOOST_CHECK(equal(*it++, *jt++));
  }

  {
    auto it = vfs1->cbegin();
    auto jt = mfs.getTrack(0).cbegin();
    for (size_t i = 0; i < vfs2->size(); ++i)
      BOOST_CHECK(equal(*it++, *jt++));
  }

  BOOST_CHECK(mfs.getDefaultTrack().getTrackInfo().trackName == "KinectDepth");
}

BOOST_AUTO_TEST_CASE( testColourSpace1 ) {
  const Mat original = testVideoFrames[0];
  Mat temp = original.clone();
  Mat tempBGR = original.clone();
  Mat tempYCrCb;
  Mat tempHSV;
  Mat tempGREY;

  BOOST_CHECK(equal(temp, original));
  cvtColor(temp, tempYCrCb, CV_BGR2YCrCb);
  BOOST_CHECK(equal(temp, original));
  cvtColor(temp, tempHSV, CV_BGR2HSV);
  BOOST_CHECK(equal(temp, original));
  cvtColor(temp, tempGREY, CV_BGR2GRAY);
  BOOST_CHECK(equal(temp, original));
  BOOST_CHECK(equal(temp, tempBGR));
  BOOST_CHECK(equal(original, tempBGR));

  BOOST_CHECK(dynamic_cast<const YCrCbColourSpace*>(ColourSpace::YCRCB.get()));
  BOOST_CHECK(*ColourSpace::YCRCB == CustomColourSpace({ColourSpace::Y,ColourSpace::CR, ColourSpace::CB}));
  ColourSpace::YCRCB->convertColour(original,temp);
  BOOST_CHECK(equal(original, tempBGR));
  Vec3b mapping1;
  mapping1[0] = 0;
  mapping1[1] = 1;
  mapping1[2] = 2;
  BOOST_CHECK(checkMatMapping<3>(tempYCrCb, temp, mapping1));

  BOOST_CHECK(dynamic_cast<const RGBColourSpace*>(ColourSpace::RGB.get()));
  BOOST_CHECK(*ColourSpace::RGB == CustomColourSpace({ColourSpace::R,ColourSpace::G, ColourSpace::B}));
  ColourSpace::RGB->convertColour(original,temp);
  mapping1[0] = 2;
  mapping1[1] = 1;
  mapping1[2] = 0;
  BOOST_CHECK(checkMatMapping<3>(tempBGR, temp, mapping1));

  BOOST_CHECK(dynamic_cast<const BGRColourSpace*>(ColourSpace::BGR.get()));
  BOOST_CHECK(*ColourSpace::BGR == CustomColourSpace({ColourSpace::B,ColourSpace::G, ColourSpace::R}));
  ColourSpace::BGR->convertColour(original,temp);
  mapping1[0] = 0;
  mapping1[1] = 1;
  mapping1[2] = 2;
  BOOST_CHECK(checkMatMapping<3>(tempBGR, temp, mapping1));

  BOOST_CHECK(dynamic_cast<const HSVColourSpace*>(ColourSpace::HSV.get()));
  BOOST_CHECK(*ColourSpace::HSV == CustomColourSpace({ColourSpace::H, ColourSpace::S, ColourSpace::V}));
  ColourSpace::HSV->convertColour(original,temp);
  mapping1[0] = 0;
  mapping1[1] = 1;
  mapping1[2] = 2;
  BOOST_CHECK(checkMatMapping<3>(tempHSV, temp, mapping1));

  BOOST_CHECK(dynamic_cast<const GreyColourSpace*>(ColourSpace::GREY.get()));
  BOOST_CHECK(*ColourSpace::GREY == CustomColourSpace({ColourSpace::Y}));
  ColourSpace::GREY->convertColour(original,temp);
  Vec<uchar,1> mapping2;
  mapping2[0] = 0;
  BOOST_CHECK(checkMatMapping<1>(tempGREY, temp, mapping2));

  Mat tempHSCrCb(original.size(), CV_8UC4);
  for (int i = 0; i < tempHSCrCb.rows; ++i) {
    for (int j = 0; j < tempHSCrCb.cols; ++j) {
      Vec4b& v = tempHSCrCb.at<Vec4b>(i,j);
      v[0] = tempHSV.at<Vec3b>(i,j)[0];
      v[1] = tempHSV.at<Vec3b>(i,j)[1];
      v[2] = tempYCrCb.at<Vec3b>(i,j)[1];
      v[3] = tempYCrCb.at<Vec3b>(i,j)[2];
    }
  }
  Vec4b mapping3;
  for (int k = 0; k < 4; ++k)
    mapping3[k] = k;
  CustomColourSpace csp { ColourSpace::H, ColourSpace::S, ColourSpace::CR,
      ColourSpace::CB };
  csp.convertColour(original, temp);
  BOOST_CHECK(checkMatMapping<4>(tempHSCrCb, temp, mapping3));

  BOOST_CHECK(dynamic_cast<const XYZColourSpace*>(ColourSpace::XYZ.get()));
  BOOST_CHECK(*ColourSpace::XYZ == CustomColourSpace({ColourSpace::X, ColourSpace::XYZ_Y, ColourSpace::Z}));
  ColourSpace::XYZ->convertColour(original, temp);
  Mat tempXYZ;
  cvtColor(original,tempXYZ,CV_BGR2XYZ);
  BOOST_CHECK(equal(temp, tempXYZ));

  csp = { ColourSpace::Y, ColourSpace::CR, ColourSpace::CB };
  BOOST_CHECK(csp == *ColourSpace::YCRCB);
  csp.convertColour(original, temp);
  BOOST_CHECK(equal(temp, tempYCrCb));
  ColourSpace::YCRCB->convertColour(original, temp);
  BOOST_CHECK(equal(temp, tempYCrCb));

  IHLSColourSpace ihlsCs;
  BOOST_CHECK(ihlsCs == *ColourSpace::IHLS);
  ihlsCs.convertColour(original, temp);
  BOOST_REQUIRE(temp.type() == CV_8UC3);
  BOOST_REQUIRE(temp.size() == original.size());
  for (int i = 0; i < temp.rows; ++i) {
    for (int j = 0; j < temp.cols; ++j) {
      const Vec3b& v = temp.at<const Vec3b>(i, j);
      const Vec3b& e = original.at<const Vec3b>(i, j);
      BOOST_CHECK(v[1] == round(e[2]*0.2126 + 0.7152*e[1] + 0.0722*e[0]));
      BOOST_CHECK(v[2] == std::max(std::max(e[0],e[1]),e[2]) - std::min(std::min(e[0],e[1]),e[2]));
      if (e[0] > e[1])
        BOOST_CHECK(v[0] >= 128);
      else
        BOOST_CHECK(v[0] <= 128);
    }
  }
}

BOOST_AUTO_TEST_CASE( testSkinDetector1 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());
  BlackBoard bb;
  Scalar lowerB = Scalar(10,120,30);
  Scalar upperB = Scalar(140,200,100);
  ThresholdSkinDetector tsd(&bb, &iss, lowerB, upperB);

  BOOST_CHECK(tsd.getComponentName() == "Threshold Skin Detector");
  BOOST_CHECK(*tsd.colourSpace == *ColourSpace::RGB);

  // BOOST_CHECK(tsd.detectBlobs);
  // BOOST_CHECK(tsd.removeSmallBlobs);
  // BOOST_CHECK(tsd.nBlobs == 3);
  // BOOST_CHECK(tsd.minBlobSize == 3300);

  // tsd.setBlobDetection(false);
  // tsd.setBlobRemoval(false);
  // BOOST_CHECK(!tsd.detectBlobs);
  // BOOST_CHECK(!tsd.removeSmallBlobs);

  tsd.process(0);
  BlackBoardPointer<Mat> skinMask = bb.get<Mat>(0, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);
  BOOST_REQUIRE(skinMask->rows > 200 && skinMask->cols > 200);
  BOOST_REQUIRE(cv::countNonZero(*skinMask) > 100);
  Mat original = iss[0];
  for (int i = 0; i < skinMask->rows; ++i) { 
    for (int j = 0; j < skinMask->cols; ++j) {
      Vec3b& v = original.at<Vec3b>(i,j);
      bool b = true;
      for (int k = 0; k < 3; ++k)
        b = b && v[k] >= lowerB[2-k] && v[k] < upperB[2-k];
      BOOST_CHECK((static_cast<bool>(skinMask->at<uchar>(i,j)) && b)
                  || !b);
    }
  }
}



BOOST_AUTO_TEST_CASE( testSkinDetector2 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());

  Scalar lowerB(0,20,40);
  Scalar upperB(40,168,244);

  Scalar mean, stddev;
  Mat original;
  ColourSpace::HSV->convertColour(iss[0], original);
  // cv::meanStdDev(original(Rect(250,150,50,50)), mean, stddev);
  // for (int i = 0; i < 4; ++i)
  //   lowerB[i] = mean[i] - stddev[i], upperB[i] = mean[i] + stddev[i];

  BlackBoard bb;
  ThresholdSkinDetector tsd(&bb, &iss, lowerB, upperB, *ColourSpace::HSV);
  // tsd.setBlobDetection(false);
  // tsd.setBlobRemoval(false);

  tsd.process(0);
  Mat result1 = bb.get<Mat>(0, SKINDETECTOR_BLACKBOARD_MASK_ENTRY)->clone();

  // tsd.setBlobDetection(true);
  // tsd.setBlobRemoval(false);
  BlobExtractor bex(&bb, &iss, false);
  tsd.process(0);
  bex.process(0);
  Mat result2 = bb.get<Mat>(0, SKINDETECTOR_BLACKBOARD_MASK_ENTRY)->clone();
  BlackBoardPointer<vector<Blob> > blobs2 = bb.get< vector<Blob> >(0, BLACKBOARD_BLOBS_ENTRY);
  BOOST_CHECK( equal(result1, result2) );
  BOOST_CHECK( blobs2->size() == 3 );
  for (auto it = blobs2->begin(); it != blobs2->end(); ++it)
    BOOST_CHECK( it->size() > 3300 );

  tsd.addPostProcessFilter(PostProcessHoleFillerFilter());
  tsd.process(0);
  Mat result5 = bb.get<Mat>(0, SKINDETECTOR_BLACKBOARD_MASK_ENTRY)->clone();
  BOOST_CHECK(cv::countNonZero(result5) > cv::countNonZero(result2));

  // tsd.setBlobRemoval(true);
  bex = BlobExtractor(&bb, &iss);
  tsd.process(0);
  bex.process(0);
  Mat result3 = bb.get<Mat>(0, SKINDETECTOR_BLACKBOARD_MASK_ENTRY)->clone();
  BOOST_CHECK( !equal(result3, result2) );
  
  // tsd.setMinBlobSize(20);
  // tsd.setBlobCount(10);
  bex = BlobExtractor(&bb, &iss, true, 10, 20);
  tsd.process(0);
  bex.process(0);
  Mat result4 = bb.get<Mat>(0, SKINDETECTOR_BLACKBOARD_MASK_ENTRY)->clone();
  BOOST_CHECK(cv::countNonZero(result4) > cv::countNonZero(result3));
  BlackBoardPointer<vector<Blob> > blobs4 = bb.get< vector<Blob> >(0, BLACKBOARD_BLOBS_ENTRY);
  BOOST_CHECK(blobs4->size() == 10);

  Mat result4Dilated;
  cv::dilate(result4, result4Dilated, 
             cv::getStructuringElement(cv::MORPH_RECT,Size(3,3)));
  tsd.addPostProcessFilter(PostProcessMorphologicalTransformationFilter(PostProcessMorphologicalTransformationFilter::Type::DILATE, 1));
  tsd.process(0);
  Mat result6 = bb.get<Mat>(0, SKINDETECTOR_BLACKBOARD_MASK_ENTRY)->clone();
  BOOST_CHECK(cv::countNonZero(result6) > cv::countNonZero(result4Dilated));
}

BOOST_AUTO_TEST_CASE( testSkinDetector3 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());
  
  // some lower bounds
  // -13.5229
  // 85.7577
  // 147.308
  // 103.685
  // some upper bounds
  // 37.0685
  // 135.518
  // 162.288
  // 116.974
  
  Scalar lowerB(0,20,100,70);
  Scalar upperB(70,180,200,150);

  Scalar mean, stddev;
  Mat original;
  CustomColourSpace csp { ColourSpace::H, ColourSpace::S, ColourSpace::CR,
      ColourSpace::CB };
  csp.convertColour(iss[0], original);

  BlackBoard bb;
  ThresholdSkinDetector tsd(&bb, &iss, lowerB, upperB, *ColourSpace::HSV);
  // tsd.setBlobDetection(false);
  // tsd.setBlobRemoval(false);

  tsd.process(0);
  BlackBoardPointer<Mat> result = bb.get<Mat>(0, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);

  BOOST_CHECK(cv::countNonZero(*result) > 100);
}



BOOST_AUTO_TEST_CASE( testSkinDetector4 ) {
  cv::theRNG() = cv::RNG(time(nullptr));
  // create three clusters
  Mat samples(100, 2, CV_32FC1);
  Mat classes(100, 1, CV_32SC1);

  int bestK = 0;

  for (int i = 0; i < 1000; ++i) {
    Mat temp = samples.rowRange(0,10);
    randn(temp, Scalar::all(10000), Scalar::all(0.1));

    temp = samples.rowRange(10,40);
    randn(temp, Scalar::all(20000), Scalar::all(0.1));

    temp = samples.rowRange(40,80);
    randn(temp, Scalar::all(30000), Scalar::all(0.1));

    temp = samples.rowRange(80,100);
    randn(temp, Scalar::all(40000), Scalar::all(0.1));

    classes.rowRange(0, 10) = Scalar::all(0);
    classes.rowRange(10, 40) = Scalar::all(1);
    classes.rowRange(40, 80) = Scalar::all(2);
    classes.rowRange(80, 100) = Scalar::all(3);

    bestK = findBestK2(samples);
    if (bestK == 4)
      break;
  }


  BOOST_REQUIRE(bestK == 4);

  std::vector<cv::Mat> clusters;
  cv::Mat clusterMeans;
  getBiggestClusters(samples, clusters, clusterMeans, bestK, bestK);
  BOOST_REQUIRE(clusters.size() == 4);
  BOOST_REQUIRE(clusterMeans.rows == 4);
  BOOST_REQUIRE(clusterMeans.cols == 2);

  for (size_t i = 0; i < clusters.size(); ++i) {
    if (clusters[i].rows == 10)
      std::swap(clusters[0], clusters[i]);
    if (clusters[i].rows == 30)
      std::swap(clusters[1], clusters[i]);
    if (clusters[i].rows == 40)
      std::swap(clusters[2], clusters[i]);
    if (clusters[i].rows == 20)
      std::swap(clusters[3], clusters[i]);
  } 

  BOOST_CHECK(clusters[0].rows == 10);
  BOOST_CHECK(clusters[1].rows == 30);
  BOOST_CHECK(clusters[2].rows == 40);
  BOOST_CHECK(clusters[3].rows == 20);

  for (size_t k = 0; k < clusters.size(); ++k) {
    int j, i;
    for (i = 0; i < clusters[k].rows; ++i) {
      for (j = 0; j < samples.rows; ++j) {
        if (samples.at<float>(j,0) == clusters[k].at<float>(i,0) &&
            samples.at<float>(j,1) == clusters[k].at<float>(i,1)) {
          break;
        }
      }
      BOOST_CHECK(classes.at<int>(j,0) == static_cast<int>(k));
    }
  }
}


BOOST_AUTO_TEST_CASE( testConfiguration1 ) {
  //   SLMotionConfiguration config = parseConfigurationFile("default.conf");
  SLMotionConfiguration config = parseConfigurationFile(slmotion::configuration::getDefaultConfigurationFile());
  BOOST_CHECK(config.annotationFormat == "Frame: %f");
  BOOST_CHECK(config.videoFourCC == "MJPG");
  BOOST_CHECK(config.videoFps == 25.0);
  //  BOOST_CHECK(config.annotationTemplateFilename == "default.ann");
  BOOST_CHECK(config.aspect == -1);
  BOOST_CHECK(config.scale == cv::Size(-1,-1));


  // BOOST_CHECK(config.faceDetectorCascadeFilename == "haarcascade_frontalface_alt.xml");
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.cascade"].as<string>() ==
              getDefaultHaarCascadeFile());
  // BOOST_CHECK(config.faceScale == std::make_pair(0.6,1.0));
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.scale"].as<string>() == "1.0:1.0");
              
  // BOOST_CHECK(config.faceTranslate == std::make_pair(0,0));
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.translate"].as<string>() == "0:0");

  // BOOST_CHECK(config.faceDetectorMaxMove == 25);
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.maxmove"].as<int>() == 25);

  // BOOST_CHECK(config.faceDetectorMaxAreaChange == 0.5);
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.maxareachange"].as<double>() == 0.5);

  // BOOST_CHECK(config.faceDetectorMinNeighbours == 2);
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.minneighbours"].as<int>() == 2);

  // BOOST_CHECK(config.faceDetectorScaleFactor == 1.2);
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.scalefactor"].as<double>() == 1.2);

  // BOOST_CHECK(config.skinDetectionBaseMethod == "gaussnd");
  // BOOST_CHECK(*config.colourSpace == *ColourSpace::IHLS);
  
  vector<shared_ptr<PostProcessFilter>> defaultFilters = getDefaultSkinPostProcessFilters();
  // auto& filters = config.skinDetectorPostProcessFilters;
  // BOOST_REQUIRE(filters.size() == defaultFilters.size());
  // BOOST_CHECK(std::equal(boost::make_indirect_iterator(filters.cbegin()),
  //                        boost::make_indirect_iterator(filters.cend()),
  //                        boost::make_indirect_iterator(defaultFilters.cbegin())));

  // BOOST_CHECK(config.skinDetectorPostProcessFilterString == "morph(erode,1);watershed(50);morph(close,2)");
  // BOOST_CHECK(config.doBlobDetection);
  BOOST_CHECK((*config.configurationVariables)["BlobExtractor.blobremoval"].as<bool>());
  BOOST_CHECK((*config.configurationVariables)["BlobExtractor.blobs"].as<int>() == 3);
  BOOST_CHECK((*config.configurationVariables)["BlobExtractor.minblobsize"].as<int>() == 3300);

  // BOOST_CHECK(config.gaussNDWeights);
  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.weights"].as<bool>());
  // BOOST_CHECK(config.gaussNDLogDensityThreshold == -14);
  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.logdensitythreshold"].as<double>() == -14);
  // BOOST_CHECK(config.gaussNDKColours == 2);
  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.kcolours"].as<int>() == 2);
  // BOOST_CHECK(config.gaussNDRColours == 1);
  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.rcolours"].as<int>() == 1);

  // BOOST_CHECK(config.skinDetectorTrainingImageFilename == "");
  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.trainimage"].as<string>() == "" );

  // BOOST_CHECK(config.KLTRemoveNonMaskedFeatures);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.removenonmaskedfeatures"].as<bool>());
  // BOOST_CHECK(config.KLTMaxFrameError == 1000);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.maxframeerror"].as<double>() == 1000);
  // BOOST_CHECK(config.KLTGFQuality == 0.01);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.gfquality"].as<double>() == 0.01);
  // BOOST_CHECK(config.KLTGFMinDistance == 3);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.gfmindistance"].as<double>() == 3);
  // BOOST_CHECK(config.KLTMaxPoints == 1000);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.maxpoints"].as<int>() == 1000);
  // BOOST_CHECK(config.KLTMaxMove == 30);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.maxmove"].as<int>() == 30);

  // BOOST_CHECK(config.classifyHeadSizeFactor == 3.4);
  // BOOST_CHECK(config.classifyHeadMaxDistance == 50);
  // BOOST_CHECK(config.goodFrameMinHandHDist == 0);
  // BOOST_CHECK(config.goodFrameMinHandHeadVDist == 0);
  // BOOST_CHECK(config.goodFrameMaxHeadDisplacement == 1000);

  //BOOST_CHECK(config.asmNLandmarks == 60);
  //BOOST_CHECK(config.pdmAlignThreshold == 0);
  //BOOST_CHECK(config.pdmAlignMaxIterations == 10);
  //BOOST_CHECK(config.asmSobelApertureSize == "scharr");
  // BOOST_CHECK(config.asmShapeHistorySize == SIZE_MAX);

  BOOST_CHECK((*config.configurationVariables)["AsmTracker.nlandmarks"].as<int>() == 60);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.alignthreshold"].as<double>() == 0);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.alignmaxiterations"].as<unsigned int>() == 10);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.sobelaperturesize"].as<string>() == "scharr");

  // BOOST_CHECK((*config.configurationVariables)["AsmTracker.maxlookdistance"].as<int>() == 20);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.maxlookdistance"].as<int>() == 30);
  // BOOST_CHECK((*config.configurationVariables)["AsmTracker.convergencethreshold"].as<double>() == 0.0);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.convergencethreshold"].as<double>() == 1.0);
  // BOOST_CHECK((*config.configurationVariables)["AsmTracker.intensityfactor"].as<double>() == 1.0);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.intensityfactor"].as<double>() == 5.0);
  // BOOST_CHECK(boost::lexical_cast<double>((*config.configurationVariables)["AsmTracker.minintensity"].as<string>()) == 0);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.minintensity"].as<string>() == "mean");
  // BOOST_CHECK((*config.configurationVariables)["AsmTracker.maxiterations"].as<int>() == 60);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.maxiterations"].as<int>() == 20);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.pcacomponentcount"].as<int>() == 3);
  //BOOST_CHECK(config.asmFittingContext.scaleApproximationFactor == std::make_pair(0.,0.));
  // BOOST_CHECK((*config.configurationVariables)["AsmTracker.scaleapproximationfactor"].as<string>() == "0:0");
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.scaleapproximationfactor"].as<string>() == "1.0:1.0");
  // BOOST_CHECK((*config.configurationVariables)["AsmTracker.maxshapeparameterdeviation"].as<double>() == 1.5);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.maxshapeparameterdeviation"].as<double>() == 3.0);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.blackout"].as<bool>());
  // BOOST_CHECK(config.asmFittingContext.initialPoseEstimationMethod == PDM::AsmFittingContext::POSE_AND_SHAPE_HISTORY);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.initialposeestimationmethod"].as<string>() == "poseandshapehistory");
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.initialposeestimationgoodnessthreshold"].as<double>() == 0.7);
  BOOST_CHECK(!(*config.configurationVariables)["AsmTracker.equalisehistogram"].as<bool>());
  //  BOOST_CHECK(config.asmFittingContext.targetType == PDM::AsmFittingContext::NEGATIVE);
  // BOOST_CHECK((*config.configurationVariables)["AsmTracker.targettype"].as<string>() == "negative");
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.targettype"].as<string>() == "absolute");
  // BOOST_CHECK(config.asmFittingContext.gradientMethod == PDM::AsmFittingContext::SOBEL);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.gradientmethod"].as<string>() == "sobel");
  // BOOST_CHECK((*config.configurationVariables)["AsmTracker.cannythreshold1"].as<double>() == 30000);
  // BOOST_CHECK((*config.configurationVariables)["AsmTracker.cannythreshold2"].as<double>() == 29000);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.cannythreshold1"].as<double>() == 150);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.cannythreshold2"].as<double>() == 100);
}

BOOST_AUTO_TEST_CASE( testConfiguration2 ) {
  std::ifstream ifs(slmotion::configuration::getDefaultConfigurationFile());
  std::string str((std::istreambuf_iterator<char>(ifs)),
                  std::istreambuf_iterator<char>());
  // boost::algorithm::replace_all(str, "# aspect=", "aspect=16:9");
  // boost::algorithm::replace_all(str, "# scale=", "scale=1920x1080");
  // boost::algorithm::replace_all(str, "# trainimage= ", "trainimage=batman and the phantom");
  str = std::regex_replace(str, std::regex("# aspect=$"), 
                           "aspect=16:9");
  str = std::regex_replace(str, std::regex("# scale=$"), 
                           "scale=1920x1080");
  str = std::regex_replace(str, std::regex("trainimage=$"), 
                           "trainimage=batman and the phantom");
  std::istringstream iss(str);
  SLMotionConfiguration config = parseConfigurationFile(iss);
  BOOST_CHECK(config.aspect == 16./9.);
  BOOST_CHECK(config.scale == cv::Size(1920,1080));
  // BOOST_CHECK(config.skinDetectorTrainingImageFilename == "batman and the phantom");
  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.trainimage"].as<string>() == "batman and the phantom");

}

BOOST_AUTO_TEST_CASE( testConfiguration3 ) {
  SLMotionConfiguration config = parseConfigurationFile(slmotion::configuration::getDefaultConfigurationFile());

  vector<const char*> params = { 
    "--help", 
    "--version", 
    "--verbose", 
    //    "--face-cascade", 
    // "fakecascade123", 
    // "--train-image", 
    // "batman rides again", 
    "--out", 
    "invalid output file.avi", 
    "--feature-out", 
    "unimaginably horrible feature file", 
    "--elan-out", 
    "an xml file", 
    "--frames", 
    "42", 
    "--skip", 
    "4242", 
    "--aspect", 
    "200:4", 
    "--scale", 
    "200x400", 
    "--ann-format", 
    "duke nukem", 
    "--ann-file", 
    "homer simpson", 
    "--ann-template-file", 
    "eric cartman", 
    "--components",
    "all,-FaceDetector,-GaussianSkinDetector,-KLTTracker,-BodyPartCollector",
    "--show-mask",
    "--show-blob-cent",
    "--show-blobs",
    "--show-detected-body-parts",
    "--show-corners",
    "--show-tracked-points",
    "--show-face",
    "--show-motion-vectors",
    "--show-acc-vectors",
    "--show-estimated-anchors",
    "--show-asms",
    "--show-gradient-map",
    "--show-2dgradient-map",
    "--show-canny-map",
    "--pause",
    "--combination-style",
    "left-to-right"
  };

  vector<char*> argv;
  for (size_t i = 0; i < params.size(); ++i)
    argv.push_back(const_cast<char*>(params[i]));

  applyCommandLineOptions(argv.size(), &argv[0], config);

  BOOST_CHECK(slmotion::debug == 1);
  // BOOST_CHECK(config.analyserComponents.size() == 5 &&
  //             config.analyserComponents[0] == "ColourSpaceConverter" &&
  //             config.analyserComponents[1] == "TemporalSkinPropagator" &&
  //             config.analyserComponents[2] == "BlobExtractor" &&
  //             config.analyserComponents[3] == "AsmTracker" && 
  //             config.analyserComponents[4] == "FeatureCreator");
  // BOOST_CHECK((*config.configurationVariables)["FaceDetector.cascade"].as<string>() == "fakecascade123");
  // BOOST_CHECK(config.skinDetectorTrainingImageFilename == "batman rides again");
  // BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.trainimage"].as<string>() == "batman rides again" );
  BOOST_CHECK(config.outputFilename == "invalid output file.avi");  
  BOOST_CHECK(config.featureOutputFilename == "unimaginably horrible feature file"); 
  BOOST_CHECK(config.elanOutputFilename == "an xml file");  
  // BOOST_CHECK(config.frameSkip == 4242); 
  BOOST_CHECK(config.aspect == 200./4.);
  BOOST_CHECK(config.scale == cv::Size(200,400));
  BOOST_CHECK(config.annotationFormat == "duke nukem");
  BOOST_CHECK(config.annotationFilename == "homer simpson");
  BOOST_CHECK(config.annotationTemplateFilename == "eric cartman");
  BOOST_CHECK(config.showSkinMask);
  BOOST_CHECK(config.showBlobCentroids);
  BOOST_CHECK(config.showBlobs);
  BOOST_CHECK(config.showBodyParts);
  BOOST_CHECK(config.showCorners);
  BOOST_CHECK(config.showTrackedPoints);
  BOOST_CHECK(config.showFace);
  BOOST_CHECK(config.showMotionVectors);
  BOOST_CHECK(config.showAccelerationVectors);
  BOOST_CHECK(config.showEstimatedAnchors);
  BOOST_CHECK(config.showAsms);
  BOOST_CHECK(config.showGradientMap);
  BOOST_CHECK(config.showGradientMap2d);
  BOOST_CHECK(config.showCannyMap);
  // BOOST_CHECK(config.visualiserCombinationStyle == slmotion::Visualiser::CombinationStyle::LEFT_TO_RIGHT);
}

BOOST_AUTO_TEST_CASE( testConfiguration4 ) {
  vector<Job> jobs;
  vector<string> conffiles;
  vector<string> inFiles;
  jobs = assignFilenamesToJobs(inFiles);
  BOOST_CHECK(jobs.size() == 0);
  inFiles = vector<string> { TESTVIDEOFILE, TESTVIDEOFILE2 };
  jobs = assignFilenamesToJobs(inFiles);

  vector<SLMotionConfiguration> configs = parseConfigurationFiles(conffiles, jobs.size());

  BOOST_REQUIRE(jobs.size() == 2);
  BOOST_CHECK(jobs[0].type == jobs[1].type);
  BOOST_CHECK(jobs[0].type == Job::JobType::VIDEO_FILE);

  vector<Analyser> anals;
  SLMotionConfiguration globalConfig;
  createAnalysers(jobs, conffiles, anals, globalConfig, 0, 0);

  for (auto it = anals.cbegin(); it != anals.cend(); ++it) {
    BOOST_CHECK(it->visualiser->annFormat == "Frame: %f");
    BOOST_CHECK(memcmp(it->slio->fourcc, "MJPG", 4) == 0);
    BOOST_CHECK(it->slio->outFps == 25.0);
    // BOOST_CHECK(globalConfig.annotationTemplateFilename == "default.ann");
    BOOST_CHECK(globalConfig.annotationTemplateFilename == "");
    BOOST_CHECK(it->frameSource->aspect == -1);
    BOOST_CHECK(it->frameSource->targetScale == cv::Size(-1,-1));

#if 0
    const FaceDetector* faceDetector = getPreprocessComponent<FaceDetector>(*it);
    BOOST_REQUIRE(faceDetector != nullptr);
    BOOST_CHECK(faceDetector->cascadeFilename == "haarcascade_frontalface_alt.xml");
    BOOST_CHECK(faceDetector->scale == std::make_pair(1.0,1.0));
    BOOST_CHECK(faceDetector->translation == std::make_pair(0,0));
    BOOST_CHECK(faceDetector->maxMove == 25);
    BOOST_CHECK(faceDetector->maxAreaChange == 0.5);
    BOOST_CHECK(faceDetector->minNeighbours == 2);
    BOOST_CHECK(faceDetector->scaleFactor == 1.2);

    const GaussianSkinDetector* gsd = getPreprocessComponent<GaussianSkinDetector>(*it);
    BOOST_REQUIRE(gsd != nullptr);
    BOOST_CHECK(*gsd->colourSpace == *ColourSpace::IHLS);
 
    vector<shared_ptr<PostProcessFilter>> defaultFilters = getDefaultSkinPostProcessFilters();
    auto& filters = gsd->postProcessFilters;
    BOOST_REQUIRE(filters.size() == defaultFilters.size());
    BOOST_CHECK(std::equal(boost::make_indirect_iterator(filters.cbegin()),
                           boost::make_indirect_iterator(filters.cend()),
                           boost::make_indirect_iterator(defaultFilters.cbegin())));
   
    // BOOST_CHECK(gsd->detectBlobs);
    const BlobExtractor* bex  = getPreprocessComponent<BlobExtractor>(*it);
    BOOST_REQUIRE(bex != nullptr);
    BOOST_CHECK(bex->removeSmallBlobs);
    BOOST_CHECK(bex->nBlobs == 3);
    BOOST_CHECK(bex->minBlobSize == 3300);

    // BOOST_CHECK(gsd->gaussMix.gaussians.size() == 1);
    // BOOST_CHECK(gsd->gaussMix.gaussians[0].nVars == 3);
    // not trained at this point
    BOOST_CHECK(gsd->gaussMix.gaussians.size() == 0);
    // BOOST_CHECK(gsd->gaussMix.gaussians[0].nVars == 0);
    BOOST_CHECK(gsd->weights);
    BOOST_CHECK(gsd->classificationThreshold == -14);
    BOOST_CHECK(gsd->totalColourClasses == 2);
    BOOST_CHECK(gsd->relevantColourClasses == 1);

    const KLTTracker* ft = getPreprocessComponent<KLTTracker>(*it);
    BOOST_REQUIRE(ft != nullptr);

    BOOST_CHECK(ft->removeNonMaskedFeatures);
    BOOST_CHECK(ft->maxFrameError == 1000);
    BOOST_CHECK(ft->qualityLevel == 0.01);
    BOOST_CHECK(ft->minDistance == 3);
    BOOST_CHECK(ft->maxPoints == 1000);
    BOOST_CHECK(ft->maxMove == 30);

    // BOOST_CHECK(it->opts.classifyHeadSizeFactor == 3.4);
    // BOOST_CHECK(it->opts.classifyHeadMaxDistance == 50);
    // BOOST_CHECK(it->opts.goodFrameMinHandHDist == 0);
    // BOOST_CHECK(it->opts.goodFrameMinHandHeadVDist == 0);
    // BOOST_CHECK(it->opts.goodFrameMaxHeadDisplacement == 1000);

    const AsmTracker* at = getPreprocessComponent<AsmTracker>(*it);
    BOOST_REQUIRE(at != nullptr);
    BOOST_CHECK(at->nlandmarks == 60);
    // BOOST_CHECK(at->pdmAlignThreshold == 0);
    BOOST_CHECK(at->pdmAlignMaxIterations == 10);
    //    BOOST_CHECK(it->opts.asmShapeHistorySize == SIZE_MAX);

    // const Visualiser* vis = it->visualiser.get();

    //    BOOST_CHECK(vis->asmFittingContext.sobelApertureSize == CV_SCHARR);
    // BOOST_CHECK(vis->asmFittingContext.sobelApertureSize == 5);
    // BOOST_CHECK(vis->asmFittingContext.maxLookDistance == 30);
    // BOOST_CHECK(vis->asmFittingContext.convergenceThreshold == 1.0);
    // BOOST_CHECK(vis->asmFittingContext.intensityFactor == 5.0);
    // BOOST_CHECK(vis->asmFittingContext.minIntensity == -1);
    // BOOST_CHECK(vis->asmFittingContext.maxIterations == 20);
    // BOOST_CHECK(vis->asmFittingContext.pcaComponentCount == 3);
    // BOOST_CHECK(vis->asmFittingContext.scaleApproximationFactor == std::make_pair(1.,1.));
    // BOOST_CHECK(vis->asmFittingContext.maxShapeDeviation == 3);
    // BOOST_CHECK(vis->asmFittingContext.blackout);
    // BOOST_CHECK(vis->asmFittingContext.initialPoseEstimationMethod == PDM::AsmFittingContext::POSE_AND_SHAPE_HISTORY);
    // BOOST_CHECK(vis->asmFittingContext.initialPoseEstimationGoodnessThreshold == 0.7);
    // BOOST_CHECK(!vis->asmFittingContext.equaliseHistogram);
    // BOOST_CHECK(vis->asmFittingContext.targetType == PDM::AsmFittingContext::ABSOLUTE);
    // BOOST_CHECK(vis->asmFittingContext.gradientMethod == PDM::AsmFittingContext::SOBEL);
    // BOOST_CHECK(vis->asmFittingContext.cannyThreshold1 == 150);
    // BOOST_CHECK(vis->asmFittingContext.cannyThreshold2 == 100);
#endif
  }
}

BOOST_AUTO_TEST_CASE( testConfiguration5 ) {
  // test empty configuration file

  SLMotionConfiguration config = parseConfigurationFile("/dev/null");
  BOOST_CHECK(config.annotationFormat == "Frame: %f");
  BOOST_CHECK(config.videoFourCC == "MJPG");
  BOOST_CHECK(config.videoFps == 25.0);
  //  BOOST_CHECK(config.annotationTemplateFilename == "default.ann");
  BOOST_CHECK(config.aspect == -1);
  BOOST_CHECK(config.scale == cv::Size(-1,-1));


  // BOOST_CHECK(config.faceDetectorCascadeFilename == "haarcascade_frontalface_alt.xml");
  // BOOST_CHECK(config.faceScale == std::make_pair(0.,0.));
  // BOOST_CHECK(config.faceTranslate == std::make_pair(0,0));
  // BOOST_CHECK(config.faceDetectorMaxMove == 25);
  // BOOST_CHECK(config.faceDetectorMaxAreaChange == 0.5);
  // BOOST_CHECK(config.faceDetectorMinNeighbours == 2);
  // BOOST_CHECK(config.faceDetectorScaleFactor == 1.2);
  // BOOST_CHECK(config.faceDetectorCascadeFilename == "haarcascade_frontalface_alt.xml");
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.cascade"].as<string>() == getDefaultHaarCascadeFile());
  //  BOOST_CHECK((*config.configurationVariables)["FaceDetector.scale"].as<string>() == "0:0");
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.scale"].as<string>() == "1.0:1.0");
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.translate"].as<string>() == "0:0");
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.maxmove"].as<int>() == 25);
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.maxareachange"].as<double>() == 0.5);
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.minneighbours"].as<int>() == 2);
  BOOST_CHECK((*config.configurationVariables)["FaceDetector.scalefactor"].as<double>() == 1.2);


  // BOOST_CHECK(config.skinDetectionBaseMethod == "gaussnd");
  // BOOST_CHECK(*config.colourSpace == *ColourSpace::RGB);
  
  // BOOST_REQUIRE(config.skinDetectorPostProcessFilters.size() == 0);

  // BOOST_CHECK(config.skinDetectorPostProcessFilterString == "morph(erode,1);watershed(50);morph(close,2)");
  // BOOST_CHECK(config.doBlobDetection);

  BOOST_CHECK((*config.configurationVariables)["BlobExtractor.blobremoval"].as<bool>());
  BOOST_CHECK((*config.configurationVariables)["BlobExtractor.blobs"].as<int>() == 3);
  BOOST_CHECK((*config.configurationVariables)["BlobExtractor.minblobsize"].as<int>() == 3300);

  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.weights"].as<bool>());
  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.logdensitythreshold"].as<double>() == -14.0);
  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.kcolours"].as<int>() == 2);
  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.rcolours"].as<int>() == 1);
  BOOST_CHECK((*config.configurationVariables)["GaussianSkinDetector.trainimage"].as<string>() == "" );



  // BOOST_CHECK(config.KLTRemoveNonMaskedFeatures);
  // BOOST_CHECK(config.KLTMaxFrameError == 1000);
  // BOOST_CHECK(config.KLTGFQuality == 0.01);
  // BOOST_CHECK(config.KLTGFMinDistance == 3);
  // BOOST_CHECK(config.KLTMaxPoints == 1000);
  // BOOST_CHECK(config.KLTMaxMove == 30);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.removenonmaskedfeatures"].as<bool>());
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.maxframeerror"].as<double>() == 1000);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.gfquality"].as<double>() == 0.01);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.gfmindistance"].as<double>() == 3);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.maxpoints"].as<int>() == 1000);
  BOOST_CHECK((*config.configurationVariables)["KLTTracker.maxmove"].as<int>() == 30);


  // BOOST_CHECK(config.classifyHeadSizeFactor == 3.4);
  // BOOST_CHECK(config.classifyHeadMaxDistance == 50);
  // BOOST_CHECK(config.goodFrameMinHandHDist == 0);
  // BOOST_CHECK(config.goodFrameMinHandHeadVDist == 0);
  // BOOST_CHECK(config.goodFrameMaxHeadDisplacement == 1000);

  BOOST_CHECK((*config.configurationVariables)["AsmTracker.nlandmarks"].as<int>() == 60);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.alignthreshold"].as<double>() == 0);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.alignmaxiterations"].as<unsigned int>() == 10);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.sobelaperturesize"].as<string>() == "scharr");

  // BOOST_CHECK(config.asmNLandmarks == 60);
  // BOOST_CHECK(config.pdmAlignThreshold == 0);
  // BOOST_CHECK(config.pdmAlignMaxIterations == 10);
  // BOOST_CHECK(config.asmSobelApertureSize == "scharr");
  // BOOST_CHECK(config.asmShapeHistorySize == SIZE_MAX);

  BOOST_CHECK((*config.configurationVariables)["AsmTracker.maxlookdistance"].as<int>() == 30);
  //  BOOST_CHECK(config.asmFittingContext.maxLookDistance == 30);

  // BOOST_CHECK(config.asmFittingContext.convergenceThreshold == 1.0);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.convergencethreshold"].as<double>() == 1.0);

  //  BOOST_CHECK(config.asmFittingContext.intensityFactor == 5.0);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.intensityfactor"].as<double>() == 5.0);

  // BOOST_CHECK(config.asmFittingContext.minIntensity == -1);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.minintensity"].as<string>() == "mean");

  // BOOST_CHECK(config.asmFittingContext.maxIterations == 20);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.maxiterations"].as<int>() == 20);

  // BOOST_CHECK(config.asmFittingContext.pcaComponentCount == 3);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.pcacomponentcount"].as<int>() == 3);

  // BOOST_CHECK(config.asmFittingContext.scaleApproximationFactor == std::make_pair(1.0,1.0));
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.scaleapproximationfactor"].as<string>() == "1.0:1.0");

  // BOOST_CHECK(config.asmFittingContext.maxShapeDeviation == 3.0);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.maxshapeparameterdeviation"].as<double>() == 3.0);

  // BOOST_CHECK(config.asmFittingContext.blackout);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.blackout"].as<bool>());

  // BOOST_CHECK(config.asmFittingContext.initialPoseEstimationMethod == PDM::AsmFittingContext::POSE_AND_SHAPE_HISTORY);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.initialposeestimationmethod"].as<string>() == "poseandshapehistory");

  // BOOST_CHECK(config.asmFittingContext.initialPoseEstimationGoodnessThreshold == 0.7);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.initialposeestimationgoodnessthreshold"].as<double>() == 0.7);

  // BOOST_CHECK(!config.asmFittingContext.equaliseHistogram);
  BOOST_CHECK(!(*config.configurationVariables)["AsmTracker.equalisehistogram"].as<bool>());

  // BOOST_CHECK(config.asmFittingContext.targetType == PDM::AsmFittingContext::ABSOLUTE);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.targettype"].as<string>() == "absolute");

  // BOOST_CHECK(config.asmFittingContext.gradientMethod == PDM::AsmFittingContext::SOBEL);
  // BOOST_CHECK(config.asmFittingContext.cannyThreshold1 == 150);
  // BOOST_CHECK(config.asmFittingContext.cannyThreshold2 == 100);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.gradientmethod"].as<string>() == "sobel");
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.cannythreshold1"].as<double>() == 150);
  BOOST_CHECK((*config.configurationVariables)["AsmTracker.cannythreshold2"].as<double>() == 100);
}

#ifdef SLMOTION_ENABLE_LIBFLANDMARK
BOOST_AUTO_TEST_CASE(testConfiguration6) {
  std::ifstream ifs(slmotion::configuration::getDefaultConfigurationFile());
  std::string str((std::istreambuf_iterator<char>(ifs)),
                  std::istreambuf_iterator<char>());
  str = std::regex_replace(str, std::regex("components=none"), 
                           "components=FaceDetector,RuleSkinDetector,"
                           "FacialLandmarkDetector");
  {
    std::istringstream iss(str);
    SLMotionConfiguration config = parseConfigurationFile(iss);
    std::vector < std::shared_ptr<Component> > comps = createComponents(config, NULL, NULL);
    BOOST_REQUIRE(comps.size() == 3);
    FacialLandmarkDetector* fld = dynamic_cast<FacialLandmarkDetector*>(comps[2].get());
    BOOST_CHECK(fld->enableConfidence == false);
  }

  str = std::regex_replace(str, std::regex("enableconfidence=false"), 
                           "enableconfidence=true");
  {
    std::istringstream iss(str);
    SLMotionConfiguration config = parseConfigurationFile(iss);
    std::vector <std::shared_ptr<Component> > comps = createComponents(config, NULL, NULL);
    BOOST_REQUIRE(comps.size() == 3);
    FacialLandmarkDetector* fld = dynamic_cast<FacialLandmarkDetector*>(comps[2].get());
    BOOST_CHECK(fld->enableConfidence == true);
  }
}
#endif // SLMOTION_ENABLE_LIBFLANDMARK

BOOST_AUTO_TEST_CASE(testAlias) {
  SLMotionConfiguration config = parseConfigurationFile(slmotion::configuration::getDefaultConfigurationFile());

  vector<const char*> params = { 
    "./tests",
    "--components",
    "FaceDetector,example,GaussianSkinDetector,example"
  };

  vector<char*> argv;
  for (size_t i = 0; i < params.size(); ++i)
    argv.push_back(const_cast<char*>(params[i]));

  applyCommandLineOptions(argv.size(), &argv[0], config);
  BOOST_CHECK(config.analyserComponents == (std::vector<std::string> {
      "FaceDetector",
        "FaceDetector2",
        "FacialLandmarkDetector",
        "GaussianSkinDetector",
        "FaceDetector2",
        "FacialLandmarkDetector" 
        })
    );


}

BOOST_AUTO_TEST_CASE( testKLTFeatureComputation ) {
  std::mt19937 engine(time(NULL));
  std::uniform_real_distribution<float> distribution {-100,100};
  auto gen = std::bind(distribution, engine);
  auto rp = [&] { return Point2f(gen(), gen()); };

  TrackedPoint p1(rp()), p2(rp()), p3(rp()), p4(rp()), p5(rp()), p6(rp()),
    p7(rp()), p8(rp());
  p1.setPartName(BodyPart::UNKNOWN);
  p2.setPartName(BodyPart::LEFT_HAND);
  p3.setPartName(BodyPart::RIGHT_HAND);
  p4.setPartName(BodyPart::HEAD);
  p5.setPartName(BodyPart::LEFT_HAND_HEAD);
  p6.setPartName(BodyPart::LEFT_RIGHT_HAND);
  p7.setPartName(BodyPart::RIGHT_HAND_HEAD);
  p8.setPartName(BodyPart::LEFT_RIGHT_HAND_HEAD);

  vector<set<TrackedPoint>> points;
  points.push_back(set<TrackedPoint> {
      p1, p2, p3, p4, p5, p6, p7, p8
  });

  points.push_back(set<TrackedPoint> {
      p1.relocate(rp()), p2.relocate(rp()), p3.relocate(rp()), 
        p4.relocate(rp()), p5.relocate(rp()), p6.relocate(rp()),
        p7.relocate(rp()), p8.relocate(rp())
  });
  points.push_back(set<TrackedPoint> {
      p1.relocate(rp()), p2.relocate(rp()), p3.relocate(rp()), 
        p4.relocate(rp()), p5.relocate(rp()), p6.relocate(rp()),
        p7.relocate(rp()), p8.relocate(rp())
  });
  points.push_back(set<TrackedPoint> {
      p1.relocate(rp()), p2.relocate(rp()), p3.relocate(rp()), 
        p4.relocate(rp()), p5.relocate(rp()), p6.relocate(rp()),
        p7.relocate(rp()), p8.relocate(rp())
  });
  points.push_back(set<TrackedPoint> {
      p1.relocate(rp()), p2.relocate(rp()), p3.relocate(rp()), 
        p4.relocate(rp()), p5.relocate(rp()), p6.relocate(rp()),
        p7.relocate(rp()), p8.relocate(rp())
  });

  FeatureCreator::tracked_point_map_vector_t locationVectors = FeatureCreator::pruneTracked(points, 3);
  BOOST_REQUIRE(locationVectors.size() == 5);

  FeatureCreator::velocity_vector_map_vector_t velocityVectors = FeatureCreator::createVelocityVectors(locationVectors); 
  BOOST_REQUIRE(velocityVectors.size() == 5);

  FeatureCreator::acceleration_vector_map_vector_t accelerationVectors = FeatureCreator::createAccelerationVectors(locationVectors); 
  BOOST_REQUIRE(accelerationVectors.size() == 5);

  FeatureCreator::point_id_to_part_name_map_vector_t associations = FeatureCreator::createBodyPartAssociationMaps(locationVectors);
  BOOST_REQUIRE(associations.size() == 5 && 
                associations.size() == locationVectors.size());

  for (size_t i = 0; i < associations.size(); ++i) {
    BOOST_CHECK(associations[i].size() == locationVectors[i].size());
    for (auto it = locationVectors[i].cbegin(); 
         it != locationVectors[i].cend(); ++it) {
      BOOST_REQUIRE(associations[i].count(it->first));
      BOOST_CHECK(associations[i][it->first] == it->second.getPartName());
    }
  }

  vector<vector<boost::any>> features(5);

  FeatureCreator::computeKLTFeatures(features.begin(), features.end(),
                                     velocityVectors, accelerationVectors,
                                     associations, BodyPart::UNKNOWN, 
                                     false);

  auto sumPointVector = [](const FeatureCreator::velocity_vector_map_t& summand) {
    Point2f sum(0,0);
    for (auto it = summand.cbegin(); it != summand.cend(); ++it)
      sum += it->second;
    return sum;
  };

  BOOST_REQUIRE(features.size() == 5);

  for (size_t i = 0; i < features.size(); ++i)   
    BOOST_REQUIRE(features[i].size() == 5);

  for (size_t i = 0; i < features[0].size(); ++i) 
    BOOST_CHECK(boost::any_cast<int>(features[0][i]) == 8);

  for (size_t i = 0; i < features[1].size(); ++i) {
    Point2f sum = sumPointVector(velocityVectors[i]);
    Point2f asum = sumPointVector(accelerationVectors[i]);
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[1][i]),
                            static_cast<double>(sum.x)));
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[2][i]),
                            static_cast<double>(sum.y)));
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[3][i]),
                            cv::norm(sum)));
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[4][i]),
                            cv::norm(asum)));
  }

  typedef vector<TrackedPoint::point_id_type_t> tracked_point_vector_t;
  auto sumWithGivenIds = [](const FeatureCreator::velocity_vector_map_t& summand, const tracked_point_vector_t& ids) {
    Point2f sum(0,0);
    for (auto it = ids.begin(); it != ids.end(); ++it) {
      assert(summand.count(*it));
      sum += summand.find(*it)->second;
    }
    return sum;
  };



  FeatureCreator::computeKLTFeatures(features.begin(), features.end(),
                                     velocityVectors, accelerationVectors,
                                     associations, BodyPart::LEFT_HAND, 
                                     true);
  BOOST_REQUIRE(features.size() == 5);

  for (size_t i = 0; i < features.size(); ++i)   
    BOOST_REQUIRE(features[i].size() == 5);

  for (size_t i = 0; i < features[0].size(); ++i) 
    BOOST_CHECK(boost::any_cast<int>(features[0][i]) == 1);  

  for (size_t i = 0; i < features[1].size(); ++i) {
    Point2f sum = sumWithGivenIds(velocityVectors[i],
                                  tracked_point_vector_t {
                                    p2.getId()
                                      });

    Point2f asum = sumWithGivenIds(accelerationVectors[i],
                                   tracked_point_vector_t {
                                     p2.getId()
                                       });

    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[1][i]),
                            sum.x));
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[2][i]),
                            sum.y));
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[3][i]),
                            cv::norm(sum)));
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[4][i]),
                            cv::norm(asum)));
  }

  FeatureCreator::computeKLTFeatures(features.begin(), features.end(),
                                     velocityVectors, accelerationVectors,
                                     associations, BodyPart::HEAD, 
                                     false);

  for (size_t i = 0; i < features.size(); ++i)   
    BOOST_REQUIRE(features[i].size() == 5);

  for (size_t i = 0; i < features[0].size(); ++i) 
    BOOST_CHECK(boost::any_cast<int>(features[0][i]) == 4);

  for (size_t i = 0; i < features[1].size(); ++i) {
    Point2f sum = sumWithGivenIds(velocityVectors[i],
                                  tracked_point_vector_t {
                                    p4.getId(), p5.getId(),
                                      p7.getId(), p8.getId()
                                      });
    Point2f asum = sumWithGivenIds(accelerationVectors[i],
                                   tracked_point_vector_t {
                                     p4.getId(), p5.getId(),
                                       p7.getId(), p8.getId()
                                       });
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[1][i]),
                            sum.x));
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[2][i]),
                            sum.y));
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[3][i]),
                            cv::norm(sum)));
    BOOST_CHECK(almostEqual(boost::any_cast<double>(features[4][i]),
                            cv::norm(asum)));
  }
}

BOOST_AUTO_TEST_CASE( testColourSpaceConverter ) {
  if (slmotion::debug > 0)
    slmotion::debug = 0;

  ImageSequenceSource iss(getTestVideoFramefilenames());
  BOOST_REQUIRE(iss.size() == 10);
  vector<Mat> gsimg(iss.size());
  vector<Mat> ycrcbimg(iss.size());
  for (size_t i = 0; i < iss.size(); ++i) {
    cv::cvtColor(iss[i], gsimg[i], CV_BGR2GRAY);
    cv::cvtColor(iss[i], ycrcbimg[i], CV_BGR2YCrCb);
  }

  BlackBoard bb;
  ColourSpaceConverter csc(&bb, &iss);
  csc.processRange(0, iss.size());
  ColourSpaceConverter csc2(&bb, &iss, "piupau", *ColourSpace::YCRCB);
  csc2.processRange(0, iss.size());

  for (size_t i = 0; i < iss.size(); ++i)
    BOOST_CHECK(equal(*bb.get<Mat>(i,COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY), gsimg[i]));

  for (size_t i = 0; i < iss.size(); ++i)
    BOOST_CHECK(equal(*bb.get<Mat>(i,"piupau"), ycrcbimg[i]));
}

BOOST_AUTO_TEST_CASE( testKLTTracker1 ) {
  if (slmotion::debug > 0)
    slmotion::debug = 0;

  ImageSequenceSource iss(getTestVideoFramefilenames());
  BlackBoard bb;
  KLTTracker klt(&bb, &iss);
  BOOST_CHECK_THROW(klt.process(0), KLTTrackerException);

  BOOST_CHECK(klt.getComponentName() == "Kanade-Lucas-Tomasi Tracker");

  ColourSpaceConverter csc(&bb, &iss);
  BOOST_CHECK(!bb.has(0, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY));
  csc.process(0);
  BOOST_CHECK(bb.has(0, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY));

  BlackBoardPointer<Mat> gs = bb.get<Mat>(0, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY);
  vector<cv::Point2f> corners = klt.findCorners(*gs, Mat());
  vector<uchar> featuresFound(corners.size(), 1);
  vector<float> errs(corners.size(), 0);
  klt.removeInvalidPoints(corners, featuresFound, errs, Mat());
  vector<cv::Point2f> temp;
  for(size_t i = 0; i < corners.size(); ++i)
    if (featuresFound[i])
      temp.push_back(corners[i]);
  corners = std::move(temp);
  
  // goodFeaturesToTrack(gs, corners, 1000, 0.01, 3);
  BOOST_REQUIRE(corners.size() > 0);

  // prerequisite
  bb.set(0, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY, std::list<BodyPart>());

  BOOST_CHECK(!bb.has(0, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY));
  klt.process(0);
  BOOST_REQUIRE(bb.has(0, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY));
  BlackBoardPointer<set<TrackedPoint> > trackedPoints = bb.get<set<TrackedPoint>>(0, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY);
  BOOST_REQUIRE(trackedPoints->size() == corners.size());
  
  struct ComparePoints {
    bool operator()(const cv::Point2f& lhs, const cv::Point2f rhs) const {
      if (lhs.x == rhs.x)
        return lhs.y < rhs.y;
      return lhs.x < rhs.x;
    }
  };

  set<cv::Point2f, ComparePoints> cornerSet(corners.begin(), corners.end());
  for (auto it = trackedPoints->begin(); it != trackedPoints->end(); ++it)
    BOOST_CHECK(cornerSet.count(static_cast<cv::Point2f>(*it)));

  *trackedPoints = klt.initialise(*gs, Mat());
  BOOST_REQUIRE(trackedPoints->size() == corners.size());
  for (auto it = trackedPoints->begin(); it != trackedPoints->end(); ++it)
    BOOST_CHECK(cornerSet.count(static_cast<cv::Point2f>(*it)));

  csc.process(5);
  gs = bb.get<Mat>(5, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY);
  cornerSet.clear();
  set<TrackedPoint> trackedPoints2 = klt.initialise(*gs, Mat());
  insert_transform(trackedPoints2, cornerSet,
                   [](const TrackedPoint& p) {
                     return static_cast<cv::Point2f>(p);
                   });
    
  // prerequisite
  bb.set(5, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY, std::list<BodyPart>());

  klt.process(5);
  trackedPoints = bb.get<set<TrackedPoint>>(5, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY);

  BOOST_REQUIRE(cornerSet.size() == trackedPoints->size());
  for (auto it = trackedPoints->begin(); it != trackedPoints->end(); ++it)
    BOOST_CHECK(cornerSet.count(static_cast<cv::Point2f>(*it)));
}

BOOST_AUTO_TEST_CASE( testKLTTracker2 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());
  BlackBoard bb;
  KLTTracker klt(&bb, &iss);
  
  struct CompareTrackedPoints {
    bool operator()(const TrackedPoint& a,
                    const TrackedPoint& b) {
      return a.location.x == b.location.x &&
        b.location.y == a.location.y;
    }
  };
    
  ColourSpaceConverter csc(&bb, &iss);
  csc.process(1);
  csc.process(2);
  csc.process(3);
  csc.process(5);

  // prerequisite
  bb.set(1, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY, std::list<BodyPart>());
  bb.set(2, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY, std::list<BodyPart>());
  bb.set(3, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY, std::list<BodyPart>());
  bb.set(5, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY, std::list<BodyPart>());

  klt.process(1);
  BlackBoardPointer<set<TrackedPoint> > set1 = bb.get<std::set<TrackedPoint>>(1, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY);
  set<TrackedPoint> set2 = klt.initialise(*bb.get<cv::Mat>(1, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY), Mat());
  BOOST_REQUIRE(std::equal(set1->begin(), set1->end(), set2.begin(),
                         CompareTrackedPoints()));

  klt.process(2);
  set1 = bb.get<set<TrackedPoint>>(2, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY);
  set2 = klt.track(*bb.get<Mat>(2, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY),
                   *bb.get<Mat>(1, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY),
                   set2, Mat());
  BOOST_REQUIRE(std::equal(set1->begin(), set1->end(), set2.begin(),
                           CompareTrackedPoints()));
  
  klt.process(3);
  set1 = bb.get<set<TrackedPoint>>(3, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY);
  set2 = klt.track(*bb.get<Mat>(3, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY),
                   *bb.get<Mat>(2, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY),
                   set2, Mat());
  BOOST_REQUIRE(std::equal(set1->begin(), set1->end(), set2.begin(),
                           CompareTrackedPoints()));

  klt.process(5);
  set1 = bb.get<std::set<TrackedPoint>>(5, KLTTRACKER_BLACKBOARD_TRACKED_POINTS_ENTRY);
  set2 = klt.initialise(*bb.get<cv::Mat>(5, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY), Mat());
  BOOST_REQUIRE(std::equal(set1->begin(), set1->end(), set2.begin(),
                         CompareTrackedPoints()));
}

BOOST_AUTO_TEST_CASE( testKLTTracker3 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());
  BlackBoard bb;
  KLTTracker klt(&bb, &iss);

  Rect r(200, 200, 200, 200);

  vector<Point2f> tp1 {
    Point2f(0, 0), Point2f(5,5), Point2f(10,10),
      Point2f(200, 200), Point2f(201, 201), Point2f(202, 202), 
      Point2f(205, 205), Point2f(300, 300), Point2f(600,600)
  };
  vector<uchar> ff1(9, 1);
  vector<float> err1(9, 0);
  err1[7] = 2000;

  Mat mask(700, 700, CV_8UC1, Scalar::all(0));
  mask(r) = Scalar::all(255);

  klt.removeInvalidPoints(tp1, ff1, err1, mask);
  BOOST_CHECK(ff1 == (vector<uchar> {0,0,0,1,0,0,1,0,0}));

  Mat gs;
  ColourSpace::GREY->convertColour(iss[0], gs);

  vector<cv::Point2f> corners1 = klt.findCorners(gs, Mat());
  vector<uchar> featuresFound1(corners1.size(), 1);
  vector<float> errors1(corners1.size(), 0);

  klt.removeInvalidPoints(corners1, featuresFound1, errors1, Mat());

  BOOST_REQUIRE(corners1.size() > 0);
  vector<cv::Point2f> corners2(corners1);
  vector<uchar> featuresFound2(featuresFound1);
  vector<float> errors2(errors1);
  klt.purgeAndReplacePoints(gs, corners2, featuresFound2, errors2, 
                            Mat());

  auto c = [](const cv::Point2f& a, const cv::Point2f& b) {
    return a.x == b.x && a.y == b.y;
  };
  BOOST_REQUIRE(featuresFound2.size() == corners2.size() &&
                corners2.size() > corners1.size());
  BOOST_CHECK(std::equal(corners1.begin(), corners1.end(), 
                         corners2.begin(), c));
  BOOST_CHECK(std::count(featuresFound2.begin(), featuresFound2.end(), 1) ==
              std::count(featuresFound1.begin(), featuresFound1.end(), 1));

  corners2 = corners1;
  featuresFound2 = featuresFound1;
  errors2 = errors1;
  size_t i;
  for (i = 0; i < corners1.size(); ++i)
    if (corners1[i].inside(r) && featuresFound1[i])
      break;

  BOOST_REQUIRE(i < corners1.size());
  errors2[i] = 1001;

  mask = Mat(gs.size(), CV_8UC1, Scalar::all(0));
  mask(r) = Scalar::all(255);
  klt.purgeAndReplacePoints(gs, corners2, featuresFound2, errors2, mask);
  BOOST_CHECK(featuresFound2[i] == 0);
  for(i = 0; i < featuresFound2.size(); ++i)
    if (featuresFound2[i])
      BOOST_CHECK(corners2[i].inside(r));
}



BOOST_AUTO_TEST_CASE( testBodyParts1 ) {
  BodyPart p1(BodyPart::HEAD, false, Blob());
  BOOST_CHECK(p1.getIdentity() == BodyPart::HEAD);
  BOOST_CHECK(p1.identity == 1);
  BOOST_CHECK(!p1.isAmbiguous());
  BOOST_CHECK(p1.getBlob().size() == 0);

  BOOST_CHECK(p1.isA(BodyPart::HEAD));
  BOOST_CHECK(!p1.isA(BodyPart::LEFT_HAND));
  BOOST_CHECK(!p1.isA(BodyPart::RIGHT_HAND_HEAD));

  Mat m(576, 720, CV_8UC1, Scalar::all(0));
  rectangle(m, Point(100,100), Point(102,102), Scalar::all(255), CV_FILLED);

  Blob b = Blob::extractBlobsFromMask(m, 0, 1)[0];
  BOOST_CHECK(b.size() == 9);

  BodyPart p2(BodyPart::LEFT_HAND_HEAD, true, b);

  BOOST_CHECK(p2.getIdentity() == BodyPart::LEFT_HAND_HEAD);
  BOOST_CHECK(p2.identity == 3);
  BOOST_CHECK(p2.isAmbiguous());
  BOOST_CHECK(p2.getBlob().size() == 9);

  BOOST_CHECK(p2.isA(BodyPart::HEAD));
  BOOST_CHECK(p2.isA(BodyPart::LEFT_HAND));
  BOOST_CHECK(!p2.isA(BodyPart::RIGHT_HAND));

  p1 = std::move(p2);
  BOOST_CHECK(p2.getBlob().size() == 0);

  BOOST_CHECK(p1.identity == 3);
  BOOST_CHECK(p1.isAmbiguous());
  BOOST_CHECK(p1.getBlob().size() == 9);

  BOOST_CHECK(p1.isA(BodyPart::HEAD));
  BOOST_CHECK(p1.isA(BodyPart::LEFT_HAND));
  BOOST_CHECK(!p1.isA(BodyPart::RIGHT_HAND));
  p1.makeA(BodyPart::RIGHT_HAND);
  BOOST_CHECK(p1.isA(BodyPart::RIGHT_HAND));
  BOOST_CHECK(p1.isA(BodyPart::LEFT_RIGHT_HAND_HEAD));
  BOOST_CHECK(p1.identity == 7);

  BOOST_CHECK(std::round(p1.distanceTo(Point(104, 101))) == 2);
  BOOST_CHECK_THROW(p2.distanceTo(Point(104, 101)), SLMotionException);
  BOOST_CHECK(p1.distanceTo(Point(-100,-100)) == INFINITY);
}



BOOST_AUTO_TEST_CASE( testComponentsOption ) {
#if 0
  BOOST_CHECK(parseComponents("all,default,bogus,-FaceDetector,"
                              "-FaceDetector,all",
                              std::map < std::string, std::vector<std::string> >()) 
              == (vector<string> { 
                  "ColourSpaceConverter", "GaussianSkinDetector", 
                    "TemporalSkinPropagator", 
                    "BlobExtractor", "BodyPartCollector", 
                    "KLTTracker", "AsmTracker", 
                    "FeatureCreator", 
                    "ColourSpaceConverter", "GaussianSkinDetector",
                    "BlobExtractor", "BodyPartCollector", 
                    "KLTTracker", "AsmTracker", 
                    "FeatureCreator", "bogus", 
                    "ColourSpaceConverter", "FaceDetector",
                    "GaussianSkinDetector", 
                    "TemporalSkinPropagator",
                    "BlobExtractor", "BodyPartCollector",
                    "KLTTracker", "AsmTracker", 
                    "FeatureCreator" }));
#endif

  ImageSequenceSource iss(getTestVideoFramefilenames());
  BlackBoard bb;
  SLMotionConfiguration config;
  config.analyserComponents = parseComponents("BodyPartCollector,"
                                              "GaussianSkinDetector",
                                              std::map < std::string, std::vector<std::string> >());

  BOOST_CHECK_THROW(createComponents(config, &iss, &bb), 
                    ConfigurationFileException);
}



BOOST_AUTO_TEST_CASE( testBlackBoardGlobals ) {
  BlackBoard bb;
  BOOST_CHECK(bb.frameBoard.size() == 0);
  BOOST_CHECK(bb.globalBoard.size() == 0);

  bb.set(0, "foo", 11);
  bb.set(1, "bar", 22);
  bb.set("foobar", 33);
  BOOST_CHECK(bb.frameBoard.size() == 2);
  BOOST_CHECK(bb.globalBoard.size() == 1);

  BOOST_CHECK(!bb.has(0, "foobar"));
  BOOST_CHECK(!bb.has(1, "foobar"));
  BOOST_CHECK(bb.has("foobar"));
  BOOST_CHECK(bb.has(0, "foo"));
  BOOST_CHECK(bb.has(1, "bar"));
  BOOST_CHECK(!bb.has("foo"));
  BOOST_CHECK(!bb.has("bar"));

  BOOST_CHECK(*bb.get<int>(0, "foo") == 11);
  BOOST_CHECK(*bb.get<int>(1, "bar") == 22);
  BOOST_CHECK(*bb.get<int>("foobar") == 33);
  *bb.get<int>("foobar") = 44;
  BOOST_CHECK(*bb.get<int>("foobar") == 44);
}

BOOST_AUTO_TEST_CASE( testFrameSourceReferenceSegfault ) {
  std::vector<cv::Mat> frames;
  cv::VideoCapture vc(TESTVIDEOFILE2);
  while(vc.grab()) {
    Mat temp;
    vc.retrieve(temp);
    frames.push_back(temp.clone());
  }

  std::unique_ptr<vfs_t> vfs_p(new vfs_t(TESTVIDEOFILE2));
  vfs_t& vfs = *vfs_p;
  BOOST_REQUIRE(frames.size() > 0);
  BOOST_REQUIRE(vfs.size() == frames.size());
  const Mat& frame0 = vfs[0];
  BOOST_REQUIRE(equal(frame0, frames[0]));
  for (size_t i = 0; i < vfs.size(); ++i) {
    vfs[i];
    BOOST_CHECK(vfs.frameCache.size() == i + 1);
    BOOST_CHECK(&frame0 == &vfs[0]);
    BOOST_CHECK(equal(frame0, frames[0]));
  }
}



BOOST_AUTO_TEST_CASE( testVideoFileSource1 ) {
  std::unique_ptr<vfs_t> vfs_p(new vfs_t(TESTVIDEOFILE));
  vfs_t& vfs = *vfs_p;
  BOOST_CHECK(equal(testVideoFrames[0], vfs[0]));
  BOOST_CHECK(vfs.size() == 10);

  // Iterate over the video file
  auto jt = testVideoFrames.begin();
  for(VideoFileSource::const_iterator it = vfs.begin();
      it != vfs.end(); ++it)
    BOOST_CHECK(equal(*jt++, *it));
}

BOOST_AUTO_TEST_CASE( testVideoFileSource2 ) {
  // Iterate over the video file in reverse direction
  std::unique_ptr<vfs_t> vfs_p(new vfs_t(TESTVIDEOFILE));
  vfs_t& vfs = *vfs_p;
  auto jt = testVideoFrames.rbegin();
  size_t i = 0;
  for(VideoFileSource::const_reverse_iterator it = vfs.rbegin();
      it < vfs.rend(); ++it) {
    if (!equal(*jt, *it))
      cout << i << endl;
    BOOST_CHECK(equal(*jt++, *it));
    ++i;
  }
}

BOOST_AUTO_TEST_CASE( testVideoFileSource3 ) {
  // Iterate over the video file in random order
  std::unique_ptr<vfs_t> vfs_p(new vfs_t(TESTVIDEOFILE));
  vfs_t& vfs = *vfs_p;

  vector<size_t> range(10);
  for(size_t i = 0; i < 10; ++i)
    range[i] = i;
  std::random_shuffle(range.begin(), range.end());

  size_t i;
  for(i = 0; i < range.size(); ++i) {
    BOOST_CHECK(equal(testVideoFrames[range[i]], vfs[range[i]]));
  }
  BOOST_CHECK(i == 10);

  for(i = 0; i < range.size(); ++i) {
    BOOST_CHECK(equal(testVideoFrames[range[i]], vfs[range[i]]));
  }
  BOOST_CHECK(i == 10);
}

BOOST_AUTO_TEST_CASE( testVideoFileSource4 ) {
  // Iterate over the video file in random order, limited cache size
  std::unique_ptr<vfs_t> vfs_p(new vfs_t(TESTVIDEOFILE, 5));
  vfs_t& vfs = *vfs_p;

  vector<size_t> range(10);
  for(size_t i = 0; i < 10; ++i)
    range[i] = i;
  std::random_shuffle(range.begin(), range.end());

  size_t i;
  for(i = 0; i < range.size(); ++i)
    BOOST_CHECK(equal(testVideoFrames[range[i]], vfs[range[i]]));
  BOOST_CHECK(i == 10);

  for(i = 0; i < range.size(); ++i)
    BOOST_CHECK(equal(testVideoFrames[range[i]], vfs[range[i]]));
  BOOST_CHECK(i == 10);
}

BOOST_AUTO_TEST_CASE( testFaceDetector1 ) {
  BlackBoard bb;
  std::unique_ptr<vfs_t> vfs_p(new vfs_t(TESTVIDEOFILE2));
  vfs_t& vfs = *vfs_p;

  // Test constructors
  FaceDetector fd1(&bb, &vfs);
  // BOOST_CHECK(fd1.rawFaceCache.size() == 0);
  BOOST_CHECK_THROW(FaceDetector fd2(&bb, &vfs, "läpälää"),
                    std::invalid_argument);

  Rect face1(Point(185, 104), Point(309,228));
  // BOOST_CHECK(fd1.getRawFace(2) == Rect(186,110,121,121));
  // BOOST_CHECK(fd1.getRawFace(2) == face1);
  // the results are utterly dependent on compiler, OpenCV version etc., and cannot be reliably
  // reproduced under different system configurations; therefore, the exact tests have been
  // disabled
  BOOST_CHECK(fd1.getRawFace(2) != FaceDetector::INVALID_FACE);
  // BOOST_CHECK(fd1.rawFaceCache.size() == 1);

  BOOST_CHECK(fd1.getRawFace(11) == FaceDetector::INVALID_FACE);
  // BOOST_CHECK(fd1.rawFaceCache.size() == 2);
  ;
  //  BOOST_CHECK(fd1.getRawFace(2) == Rect(186,110,121,121));
  // BOOST_CHECK(fd1.getRawFace(2) == face1);
  BOOST_CHECK(fd1.getRawFace(2) != FaceDetector::INVALID_FACE);
  // BOOST_CHECK(fd1.rawFaceCache.size() == 2);
  
  BOOST_CHECK(!bb.has(2, FACEDETECTOR_BLACKBOARD_ENTRY));
  fd1.process(2);
  BOOST_CHECK(bb.has(2, FACEDETECTOR_BLACKBOARD_ENTRY));
  Rect r = *bb.get<Rect>(2, FACEDETECTOR_BLACKBOARD_ENTRY);
  // BOOST_CHECK(r == Rect(186,110,121,121));
  // BOOST_CHECK(r == face1);
  BOOST_CHECK(r != FaceDetector::INVALID_FACE);

  fd1.setScale(0.5, 0.5);
  fd1.process(2);
  r = *bb.get<Rect>(2, FACEDETECTOR_BLACKBOARD_ENTRY);
  Rect face2(Point(216,135),Point(278,197));
  //  BOOST_CHECK(r == Rect(216,140,60,60));
  // BOOST_CHECK(r == face2);
  BOOST_CHECK(r != FaceDetector::INVALID_FACE);


  Rect face3(Point(266,195),Point(328,257));

  fd1.setTranslation(50, 60);
  fd1.process(2);
  r = *bb.get<Rect>(2, FACEDETECTOR_BLACKBOARD_ENTRY);
  //  BOOST_CHECK(r == Rect(266,200,60,60));
  // BOOST_CHECK(r == face3);
  BOOST_CHECK(r != FaceDetector::INVALID_FACE);
}



BOOST_AUTO_TEST_CASE( testFaceDetector2 ) {
  // Create some bogus detections to test interpolation

  vector<Rect> bogusDetections(20, FaceDetector::INVALID_FACE);
  vector<Rect*> bogusDetectionPointers(20, nullptr);
  for(size_t i = 0; i < bogusDetections.size(); ++i)
    bogusDetectionPointers[i] = &bogusDetections[i];

  std::fill(bogusDetections.begin() + 5, bogusDetections.end(),
            Rect(10,10,20,20));

  for(size_t i = 0; i < 5; ++i)
    BOOST_CHECK(bogusDetections[i] == FaceDetector::INVALID_FACE);
  for(size_t i = 5; i < 20; ++i)
    BOOST_CHECK(bogusDetections[i] == Rect(10,10,20,20));

  FaceDetector::interpolate(bogusDetectionPointers);

  BOOST_REQUIRE(bogusDetections.size() == 20);
  for(size_t i = 0; i < 20; ++i)
    BOOST_CHECK(bogusDetections[i] == Rect(10,10,20,20));

  for(size_t i = 0; i < 5; ++i)
    bogusDetections[rand() % bogusDetections.size()] = FaceDetector::INVALID_FACE;

  BOOST_CHECK(std::count(bogusDetections.begin(), bogusDetections.end(),
                         FaceDetector::INVALID_FACE) > 0);

  FaceDetector::interpolate(bogusDetectionPointers);
  BOOST_CHECK(std::count(bogusDetections.begin(), bogusDetections.end(),
                         Rect(10,10,20,20)) == 
              static_cast<int>(bogusDetections.size()));

  std::fill(bogusDetections.begin() + 5, bogusDetections.end(),
            FaceDetector::INVALID_FACE);
  BOOST_CHECK(std::count(bogusDetections.begin(), bogusDetections.end(),
                         FaceDetector::INVALID_FACE) == 15);
  BOOST_CHECK(std::count(bogusDetections.begin(), bogusDetections.end(),
                         Rect(10,10,20,20)) == 5);

  FaceDetector::interpolate(bogusDetectionPointers);
  BOOST_CHECK(std::count(bogusDetections.begin(), bogusDetections.end(),
                         Rect(10,10,20,20)) == 
              static_cast<int>(bogusDetections.size()));

  for(size_t i = 0; i < 20; ++i)
    bogusDetections[i] = Rect(i,2*i,3*i,4*i);
  FaceDetector::interpolate(bogusDetectionPointers);
  for(size_t i = 0; i < 20; ++i)
    BOOST_CHECK(bogusDetections[i] == Rect(i,2*i,3*i,4*i));

  std::fill(bogusDetections.begin(), bogusDetections.begin() + 7,
            FaceDetector::INVALID_FACE);
  std::fill(bogusDetections.begin()+16, bogusDetections.end(),
            FaceDetector::INVALID_FACE);
  FaceDetector::interpolate(bogusDetectionPointers);
  for(size_t i = 0; i < 8; ++i)
    BOOST_CHECK(bogusDetections[i] == Rect(7,2*7,3*7,4*7));
  for(size_t i = 8; i < 16; ++i)
    BOOST_CHECK(bogusDetections[i] == Rect(i,2*i,3*i,4*i));
  for(size_t i = 16; i < 20; ++i)
    BOOST_CHECK(bogusDetections[i] == Rect(15,2*15,3*15,4*15));

  for(size_t i = 0; i < 20; ++i)
    bogusDetections[i] = Rect(i,2*i,3*i,4*i);
  std::fill(bogusDetections.begin() + 2, bogusDetections.begin() + 7,
            FaceDetector::INVALID_FACE);  
  std::fill(bogusDetections.begin() + 15, bogusDetections.begin() + 18,
            FaceDetector::INVALID_FACE);
  FaceDetector::interpolate(bogusDetectionPointers);
  for(size_t i = 0; i < 20; ++i) 
    BOOST_CHECK(bogusDetections[i] == Rect(i,2*i,3*i,4*i));
}

BOOST_AUTO_TEST_CASE( testNonZeroToSamples ) {
  typedef cv::Vec<uchar,5> VT;
  uchar u = 0;
  Mat rangeMatrix(5,5,CV_8UC(5));

  for (int i = 0; i < rangeMatrix.rows; ++i) {
    for (int j = 0; j < rangeMatrix.cols; ++j) {
      VT& v = rangeMatrix.at<VT>(i,j);
      for(int k = 0; k < 5; ++k)
        v[k] = u++;
    }
  }

  Mat mask(5,5,CV_8UC1, cv::Scalar::all(0));
  for (int i = 1; i < mask.rows - 1 ; ++i) {
    for (int j = 1; j < mask.cols - 1; ++j) {
      mask.at<uchar>(i,j) = 1;
    }
  }

  Mat result = slmotion::nonZeroToSamples(rangeMatrix, mask);
  BOOST_REQUIRE( result.rows == 3*3 );
  BOOST_REQUIRE( result.cols == 5 );
  BOOST_REQUIRE( result.type() == CV_32FC1 );
  for (int i = 0; i < result.rows; ++i) {
    for (int j = 0; j < result.cols; ++j) {
      int y = 1 + i/3;
      int x = 1 + (i%3);
      BOOST_CHECK( result.at<float>(i,j) == 
                   rangeMatrix.at<VT>(y,x)[j] );
    }
  }
}

BOOST_AUTO_TEST_CASE( testBlob1 ) {
  Mat bogusMask(576, 720, CV_8UC1, Scalar::all(0));
  bogusMask(Rect(100, 200, 70, 60)) = Scalar::all(255);
  vector<Blob> blobs = Blob::getContourBlobs(bogusMask);
  BOOST_REQUIRE(blobs.size() == 1);
  Blob& blob = blobs[0];
  BOOST_CHECK(blob.points.size() == 4); // check approximation

  // these checks comply with CHAIN_APPROX_NONE
  // BOOST_REQUIRE(blob.points.size() == 256);
  // for (int i = 0; i < 60; ++i)
  //   BOOST_CHECK(blob.points[i] == Point(100,200+i));
  // for (int i = 0; i < 70; ++i) 
  //   BOOST_CHECK(blob.points[59+i] == Point(100+i,259));
  // for (int i = 0; i < 60; ++i) 
  //   BOOST_CHECK(blob.points[59+69+i] == Point(169,259-i));
  // for (int i = 0; i < 69; ++i) 
  //   BOOST_CHECK(blob.points[2*59+69+i] == Point(169-i,200));
  // BOOST_CHECK(blob.size() == 70*60);
}

// Test constructors
BOOST_AUTO_TEST_CASE( testBlobConstructors1 ) {
  Blob orpo;
  BOOST_REQUIRE(Blob::INVALID_POINT == cv::Point(INT_MIN,INT_MIN));
  BOOST_CHECK(orpo.centroid == Blob::INVALID_POINT); 
  BOOST_CHECK(orpo.leftMost == Blob::INVALID_POINT); 
  BOOST_CHECK(orpo.rightMost == Blob::INVALID_POINT); 
  BOOST_CHECK(orpo.topMost == Blob::INVALID_POINT); 
  BOOST_CHECK(orpo.bottomMost == Blob::INVALID_POINT); 
  BOOST_CHECK(orpo.points.size() == 0);
  BOOST_CHECK(orpo.blobSize == 0);
}

BOOST_AUTO_TEST_CASE( testBlobConstructors2 ) {
  const vector<Point> originalPoints = { Point(0,0), Point(1,1), Point(2,2),
                                         Point(3,3), Point(4,4), Point(5,5),
                                         Point(6,6) };
  vector<Point> points = originalPoints;
  Blob orpo1(points);

  BOOST_CHECK(orpo1.centroid != Blob::INVALID_POINT); 
  BOOST_CHECK(orpo1.leftMost != Blob::INVALID_POINT); 
  BOOST_CHECK(orpo1.rightMost != Blob::INVALID_POINT); 
  BOOST_CHECK(orpo1.topMost != Blob::INVALID_POINT); 
  BOOST_CHECK(orpo1.bottomMost != Blob::INVALID_POINT); 

  BOOST_CHECK(orpo1.getCentroid() == Point(3,3)); 
  BOOST_CHECK(orpo1.getLeftMost().x == 0);
  BOOST_CHECK(orpo1.getRightMost().x == 6);
  BOOST_CHECK(orpo1.getTopMost().y == 0);
  BOOST_CHECK(orpo1.getBottomMost().y == 6);

  BOOST_CHECK(orpo1.points == originalPoints);
  BOOST_CHECK(points == originalPoints);

  Blob orpo2(std::move(points));
  BOOST_CHECK(points.size() == 0);
  BOOST_CHECK(orpo1 == orpo2);

  vector<Point> points2 = { Point(6,0), Point(5,1), Point(4,2),
                            Point(3,3), Point(2,4), Point(1,5),
                            Point(0,6) };
  Blob orpo3(points2);
  BOOST_CHECK(points2.size() == 7);
  BOOST_CHECK(orpo3.points == points2);
  
  orpo2 = orpo3;
  BOOST_CHECK(orpo3.points == points2);
  BOOST_CHECK(orpo2.points == points2);
  BOOST_CHECK(orpo2 == orpo3);

  orpo1 = std::move(orpo3);
  BOOST_CHECK(orpo3 == Blob());
  BOOST_CHECK(orpo2 == orpo1);
  BOOST_CHECK(orpo1.points == points2);

  Blob orpo4;
  orpo4 = orpo3;
  BOOST_CHECK(orpo4 == orpo3);
}



BOOST_AUTO_TEST_CASE( testBlob2 ) {
  Mat bogusMask(576, 720, CV_8UC1, Scalar::all(0));
  rectangle(bogusMask, Point(20,20), Point(55,67), Scalar::all(255), 
            CV_FILLED);

  ellipse(bogusMask, Point(500,500), Size(100,50), 0, 0, 360, 
          Scalar::all(255), CV_FILLED);

  rectangle(bogusMask, Point(200,30), Point(600,400), Scalar::all(255), 
            CV_FILLED);

  vector<Blob> blobs = Blob::extractBlobsFromMask(bogusMask, 0, 10);
  BOOST_REQUIRE(blobs.size() == 3);

  BOOST_CHECK(blobs[0].size() == 15933);
  BOOST_CHECK(blobs[1].size() == 148771);
  BOOST_CHECK(blobs[2].size() == 1728);

  Mat mask2(576, 720, CV_8UC1, Scalar::all(0));
  for (auto it = blobs.begin(); it != blobs.end(); ++it)
    it->draw(mask2, Scalar::all(255));

  BOOST_CHECK(equal(bogusMask, mask2));

  mask2 = blobs[0].toMatrix();
  BOOST_CHECK(mask2.cols == 201);
  BOOST_CHECK(mask2.rows == 101);
  BOOST_CHECK(cv::countNonZero(mask2) == 15933);

  blobs = Blob::extractBlobsFromMask(bogusMask, 0, 2);
  BOOST_REQUIRE(blobs.size() == 2);
  BOOST_CHECK(blobs[1].size() == 15933);
  BOOST_CHECK(blobs[0].size() == 148771);

  blobs = Blob::extractBlobsFromMask(bogusMask, 100000, 2);
  BOOST_REQUIRE(blobs.size() == 1);
  BOOST_CHECK(blobs[0].size() == 148771);
}



BOOST_AUTO_TEST_CASE( testBlob3 ) {
  Mat bogusMask(576, 720, CV_8UC1, Scalar::all(0));
  ellipse(bogusMask, Point(500,500), Size(100,50), 0, 0, 360, 
          Scalar::all(255), CV_FILLED);
  vector<Blob> blobs = Blob::extractBlobsFromMask(bogusMask, 0, 10);
  BOOST_REQUIRE(blobs.size() == 1);
  blobs[0].cutPointsOutside(Rect(500,500,220,76));
  Mat bogusMask2 = blobs[0].toMatrix(bogusMask.size());

  BOOST_CHECK(blobs[0].size() == 4057);
  BOOST_CHECK(blobs[0].getCentroid() == Point(542,521));
  BOOST_CHECK(blobs[0].getLeftMost() == Point(500,500));
  BOOST_CHECK(blobs[0].getRightMost() == Point(600,504));
  BOOST_CHECK(blobs[0].getTopMost() == Point(500,500));
  BOOST_CHECK(blobs[0].getBottomMost() == Point(500,550));
}




BOOST_AUTO_TEST_CASE( testGaussian ) {
  Mat samples = (cv::Mat_<float>(7,4) <<
                 -0.1746, 2.0504, 2.0257, -0.3273,
                 3.1768, -0.1099, -0.3298, 1.5547,
                 -0.8941, 4.1406, 1.3399, -1.0282,
                 -0.7377, -2.0230, 1.0702, -0.8270,
                 -0.2190, 1.1955, 2.6554, 1.3863,
                 -4.4886, 0.9171, 3.5865, 4.4653,
                 4.2768, 2.0384, 3.618, -0.1393);

  cv::Mat coVar;
  cv::Mat means;
  cv::calcCovarMatrix(samples, coVar, means, CV_COVAR_NORMAL | 
                      CV_COVAR_SCALE | CV_COVAR_ROWS,
                      CV_32FC1);

  cv::Mat meansML = (cv::Mat_<float>(1,4) << 0.1342,1.1727, 1.9951, 0.7264);
  BOOST_CHECK(almostEqual(meansML, means, 1e-4));

  cv::Mat coVarML = (cv::Mat_<float>(4,4) << 
                     7.1181, 0.0458, -0.9243, -2.1570,
                     0.0458, 3.1788, 0.7190, -0.5601, 
                     -0.9243, 0.7190, 1.7562, 0.8013,
                     -2.1570, -0.5601, 0.8013, 3.2075);
  BOOST_CHECK(almostEqual(coVarML, coVar, 1e-4));

  double determinant { cv::determinant(coVar) };
  BOOST_CHECK(almostEqual(determinant, 72.4227, 1e-4));

  BOOST_REQUIRE(samples.type() == CV_32FC1);
  Gaussian g(samples);

  Mat invCoVarML = (cv::Mat_<float>(4,4) << 
                    0.1796, 0.0081,0.0400, 0.1122,
                    0.0081, 0.3840, -0.2100, 0.1250,
                    0.0400,-0.2100, 0.7685, -0.2018,
                    0.1122, 0.1250, -0.2018, 0.4595);
  Mat invCoVar = coVar.inv(cv::DECOMP_SVD);
  BOOST_CHECK(almostEqual(invCoVar, invCoVarML, 1e-4));
                    
  BOOST_CHECK(equal(g.inverseCovariance, invCoVar));
  BOOST_CHECK(equal(g.means, means));
  BOOST_CHECK(g.covDetSqrt == sqrt(determinant));

  // test samples
  vector< vector<float> > testSamples {
    vector<float> { 9.6813, 3.7463, 1.6848, -0.4204 },
    vector<float> { 7.5230, 3.7469, 0.6572, -4.8911 },
    vector<float> { -3.4672, 2.3467, 3.8132, 1.9343 },
    vector<float> { 8.2313, -0.9279, -0.9384, -2.3245 },
    vector<float> { 2.0696, 2.4639, 0.7392, -0.1781 },
    vector<float> { -0.0340, 4.0781, 1.6982, 0.3084 },
    vector<float> { 2.0411, 2.0566, -1.6244, -2.8451 },
    vector<float> { -0.4126, 3.0139, 4.2313, 1.2967 },
    vector<float> { -0.1970, 2.4665, 2.7281, 0.5340 },
    vector<float> { 4.1087, 0.6573, 0.4401, 0.1389 }
  };

  vector<double> probs {
    6.847551e-07, 3.994177e-06, 6.404013e-04, 7.221559e-06, 9.495488e-04,
    5.397418e-04, 1.845151e-05, 6.062399e-04, 2.122060e-03, 5.787626e-04 
  };

  for (size_t i = 0; i < testSamples.size(); ++i) 
    BOOST_CHECK(almostEqual(probs[i], g.computePdf(testSamples[i]), 1e-4));

  vector<double> logProbs { -14.1942, -12.4307, -7.3534, -11.8384, -6.9595, 
      -7.5244, -10.9004, -7.4082, -6.1554, -7.4546 };

  for (size_t i = 0; i < testSamples.size(); ++i)
    BOOST_CHECK(almostEqual(logProbs[i], 
                            g.computeLogProbability(testSamples[i]), 
                            1e-04));

  // test the GMM
  vector<Mat> clusters {
    (cv::Mat_<float>(15,4) <<
    1.6090,    2.2968,    1.1943,    0.3630,
    2.9065,    1.5172,    0.2856,   -0.2169,
   -2.8476,   -0.6824,    1.7304,   -0.0020,
    3.4976,    2.8866,    1.6119,   -2.1281,
    1.8954,    1.7315,    3.7469,   -0.1800,
   -0.0469,    1.4125,    1.7716,   -0.1419,
   -0.3866,    2.0879,    0.9816,   -0.3132,
   -0.4464,    1.6350,    4.1214,    1.6945,
   -0.6745,   -0.5110,    3.2147,   -2.5515,
    0.1957,    0.8837,    1.6426,   -0.0390,
    0.2710,    0.9132,    0.0913,    1.7606,
    2.3381,    0.2384,    0.9534,   -1.5889,
    4.2081,    4.1978,    1.9610,    0.2710,
    1.3799,   -0.3806,    1.8121,    1.2810,
     -0.4253,    0.3065,    1.5543,    0.8623),

    (cv::Mat_<float>(10,4) <<
      
    0.7038,    1.7537,    4.2566,    1.9635,
   -3.4663,    1.9443,    2.5774,    0.6077,
    0.0667,    5.5711,    5.2640,   -0.5601,
    4.0959,   -1.2312,    1.3562,    1.5545,
    0.5031,    5.9020,    4.8424,    0.8023,
    0.3671,    2.8843,    4.8078,    1.5686,
   -1.4836,    3.9288,    4.9362,    3.0677,
    0.1949,   -0.3088,    4.5689,    3.2313,
    0.8229,    1.4081,    4.7949,    2.3877,
     1.2859,    1.9085,    3.0509,    2.5604),

    (cv::Mat_<float>(5,4) <<
    2.7845,   -2.5441,   -0.0717,   -3.6668,
    0.7678,    0.5431,    0.8016,    0.2469,
   -1.9506,   -3.2327,   -2.9116,   -1.3544,
   -1.8371,    2.5671,    2.4715,    1.6018,
     3.5512,    2.2917,   -1.9837,   -1.1666)
  };

  GaussianMixture gm(clusters);
  BOOST_REQUIRE(gm.gaussians.size() == 3);
  BOOST_REQUIRE(almostEqual(std::accumulate(gm.weights.begin(), 
                                            gm.weights.end(), 0.), 
                            1.0, 1e-15));
  BOOST_CHECK(gm.weights[0] == 15./30.);
  BOOST_CHECK(gm.weights[1] == 10./30.);
  BOOST_CHECK(gm.weights[2] == 5./30.);

  for (size_t i = 0; i < gm.gaussians.size(); ++i)
    BOOST_CHECK(gm.gaussians[i] == Gaussian(clusters[i]));

  Mat x = (cv::Mat_<float>(1,4) << 4.4643, -3.1370, -0.3534, -2.2826);

  BOOST_CHECK(almostEqual(gm.computeMixPdf(x), 2.3141e-11, 1e-4));
  BOOST_CHECK(almostEqual(gm.computeLogMixPdf(x), -24.4894, 1e-4));
}



BOOST_AUTO_TEST_CASE( testRuleskinDetector1 ) {
  // vfs_t vfs { TESTVIDEOFILE2 };
  std::unique_ptr<vfs_t> vfs_p(new vfs_t(TESTVIDEOFILE2));
  vfs_t& vfs = *vfs_p;
  size_t frnumber = rand() % vfs.size();
  Mat frame { vfs[frnumber] };

  BlackBoard bb;
  RuleSkinDetector rsd(&bb, &vfs, RuleSkinDetector::Ruleset::A);

  rsd.process(frnumber);
  BlackBoardPointer<Mat> mask { bb.get<cv::Mat>(frnumber, 
                                                SKINDETECTOR_BLACKBOARD_MASK_ENTRY) };

  // for each pixel:
  //  if it has been classified, the rules must be true
  //  otherwise, the rules must be false
  for (int i = 0; i < frame.rows; ++i) {
    for (int j = 0; j < frame.cols; ++j) {
      const Vec3b& v { frame.at<Vec3b>(i,j) };
      uchar R { v[2] }, G { v[1] }, B { v[0] };
      bool b { mask->at<uchar>(i,j) > 0 };
      BOOST_CHECK((b && R > 95 && G > 40 && B > 20 &&
                   (std::max(std::max(R, G), B) - std::min(std::min(R, G), B)) > 15 &&
                   abs(R-G) > 15 && abs(R > G) && R > B) ||
                  !b);
    }
  }
}



BOOST_AUTO_TEST_CASE( testRuleskinDetector2 ) {
  std::unique_ptr<vfs_t> vfs_p(new vfs_t(TESTVIDEOFILE2));
  vfs_t& vfs = *vfs_p;
  size_t frnumber = rand() % vfs.size();
  Mat frame { vfs[frnumber] };

  BlackBoard bb;
  RuleSkinDetector rsd(&bb, &vfs, RuleSkinDetector::Ruleset::B);

  rsd.process(frnumber);
  BlackBoardPointer<Mat> mask { bb.get<cv::Mat>(frnumber, 
                                                SKINDETECTOR_BLACKBOARD_MASK_ENTRY) };

  // for each pixel:
  //  if it has been classified, the rules must be true
  //  otherwise, the rules must be false
  for (int i = 0; i < frame.rows; ++i) {
    for (int j = 0; j < frame.cols; ++j) {
      const Vec3b& v { frame.at<Vec3b>(i,j) };
      uchar R { v[2] }, G { v[1] }, B { v[0] };
      bool b { mask->at<uchar>(i,j) > 0 };
      BOOST_CHECK((b && R > 220 && G > 210 && B > 170 && 
                   std::abs(R-G) <= 15 && R > B && G > B) ||
                  !b);
    }
  }
}



BOOST_AUTO_TEST_CASE( testFeatureVectorConstruction ) {
  typedef SOM_PAK_Component::ValueType ValueType;

  vector<SOM_PAK_Component> components;
  vector<vector<boost::any>> features;

  // BOOST_CHECK_THROW(FeatureVector(components, features, 0, 0),
  //                   std::invalid_argument);

  components.push_back(SOM_PAK_Component { 0, "abracadabra", -5, 10, 
        ValueType::INTEGER, "" });

  BOOST_CHECK_THROW(FeatureVector(components, features, 0, 0),
                    std::invalid_argument);

  features.push_back(vector<boost::any>());
  features[0].push_back(boost::any(2));
  
  FeatureVector(components, features, 0, 1);
  BOOST_CHECK_THROW(FeatureVector(components, features, 1, 0),
                    std::invalid_argument);

  components[0].index = 123;

  BOOST_CHECK_THROW(FeatureVector(components, features, 0, 1),
                    std::invalid_argument);

  components[0].index = 0;
  components.push_back(SOM_PAK_Component { 1, "hocus pocus", 1000., 1000.5, 
        ValueType::REAL, "" });
  components.push_back(SOM_PAK_Component { 2, "simsalabim", -5, -7, 
        ValueType::BOOLEAN, "" });

  BOOST_CHECK_THROW(FeatureVector(components, features, 0, 1),
                    std::invalid_argument);

  features.push_back(vector<boost::any>());
  features.push_back(vector<boost::any>());
  BOOST_CHECK_THROW(FeatureVector(components, features, 100, 101),
                    std::invalid_argument);
  features[1].push_back(boost::any(1000.3));
  features[2].push_back(boost::any(false));
  FeatureVector(components, features, 1000, 1001);

  features[0].push_back(boost::any(2000));
  features[1].push_back(boost::any(1000.3));
  features[2].push_back(boost::any(false));
  BOOST_CHECK_THROW(FeatureVector(components, features, 1000, 1002),
                    std::invalid_argument);
  features[0].pop_back();
  features[0].push_back(boost::any(3));
  FeatureVector(components, features, 1000, 1002);

  features[0].push_back(boost::any(4));
  features[1].push_back(boost::any(1000.6));
  features[2].push_back(boost::any(false));
  BOOST_CHECK_THROW(FeatureVector(components, features, 1000, 1003),
                    std::invalid_argument);
  features[1].pop_back();
  features[1].push_back(boost::any(1000.4));
  FeatureVector(components, features, 6020, 6023);

  features[0].push_back(boost::any(false));
  features[1].push_back(boost::any(1000.1));
  features[2].push_back(boost::any(true));
  BOOST_CHECK_THROW(FeatureVector(components, features, 0, 4),
                    std::invalid_argument);
  features[0].pop_back();
  features[0].push_back(boost::any(3));
  FeatureVector(components, features, 0, 4);

  features[0].push_back(boost::any(4));
  features[1].push_back(boost::any((int)1000));
  features[2].push_back(boost::any(true));
  BOOST_CHECK_THROW(FeatureVector(components, features, 0, 5),
                    std::invalid_argument);
  features[1].pop_back();
  features[1].push_back(boost::any(1000.4));
  FeatureVector(components, features, 0, 5);

  features[0].push_back(boost::any(4));
  features[1].push_back(boost::any(1000.1));
  features[2].push_back(boost::any(1));
  BOOST_CHECK_THROW(FeatureVector(components, features, 0, 6),
                    std::invalid_argument);
  features[2].pop_back();
  features[2].push_back(boost::any(true));
  FeatureVector f(components, features, 0, 6);

  BOOST_CHECK_THROW(FeatureVector(components, features, 1000, 1101),
                    std::invalid_argument);

  deque<vector<double>> dvd = f.toDoubleVectorDeque();
  BOOST_CHECK((dvd == deque < vector<double> > {
      vector<double> { 2, 1000.3, false },
      vector<double> { 3, 1000.3, false },
      vector<double> { 4, 1000.4, false },
      vector<double> { 3, 1000.1, true },
      vector<double> { 4, 1000.4, true },
      vector<double> { 4, 1000.1, true }
    }));

  // export tests
  // Store the feature vector in a file
  BlackBoard bb;
  SLIO slio("");
  slio.setFeatureOutFile("featureout.temp.csv");
  slio.storeFeatureData(f, 0, "", bb);

  std::ifstream ifs("featureout.temp.csv");
  BOOST_REQUIRE(ifs);

  string s;
  getline(ifs, s);
  BOOST_CHECK(s == "abracadabra,hocus pocus,simsalabim");
  getline(ifs, s);
  BOOST_CHECK(s == "2,1000.3,0");
  getline(ifs, s);
  BOOST_CHECK(s == "3,1000.3,0");
  getline(ifs, s);
  BOOST_CHECK(s == "4,1000.4,0");
  getline(ifs, s);
  BOOST_CHECK(s == "3,1000.1,1");
  getline(ifs, s);
  BOOST_CHECK(s == "4,1000.4,1");
  getline(ifs, s);
  BOOST_CHECK(s == "4,1000.1,1");
  getline(ifs, s);

  // Try to store an empty vector
  slio.storeFeatureData(FeatureVector(), 0, "", bb);
  std::ifstream ifs2("featureout.temp.csv");
  ifs2.seekg(0, std::ios::end);
  BOOST_CHECK(ifs2.tellg() == 0);
}

#if 0
// this test is no longer relevant because "default" does not specify anything
BOOST_AUTO_TEST_CASE( testParseComponents2 ) {
  vector<string> s = parseComponents("default,-FaceDetector,-GaussianSkinDetector,-KLTTracker,-BodyPartCollector", std::map < std::string, std::vector<std::string> >());
  BOOST_CHECK(s.size() == 4);
  BOOST_CHECK(s[0] == "ColourSpaceConverter");
  BOOST_CHECK(s[1] == "BlobExtractor");
  BOOST_CHECK(s[2] == "AsmTracker");
  BOOST_CHECK(s[3] == "FeatureCreator");

  s = parseComponents("all,-FaceDetector,-GaussianSkinDetector,-KLTTracker,-BodyPartCollector",
                      std::map < std::string, std::vector<std::string> >());
  BOOST_CHECK(s.size() == 5);
  BOOST_CHECK(s[0] == "ColourSpaceConverter");
  BOOST_CHECK(s[1] == "TemporalSkinPropagator");
  BOOST_CHECK(s[2] == "BlobExtractor");
  BOOST_CHECK(s[3] == "AsmTracker");
  BOOST_CHECK(s[4] == "FeatureCreator");
}
#endif

BOOST_AUTO_TEST_CASE( testArgs ) {
  vector<const char*> params = { 
    "./tests",
    "--help", 
    "--version", 
    "--verbose", 
    "--face-cascade", 
    "fakecascade123", 
    "--train-image", 
    "batman rides again", 
    "--out", 
    "invalid output file.avi", 
    "--feature-out", 
    "unimaginably horrible feature file", 
    "--elan-out", 
    "an xml file", 
    "--frames", 
    "42", 
    "--skip", 
    "4242", 
    "--aspect", 
    "200:4", 
    "--scale", 
    "200x400", 
    "--ann-format", 
    "duke nukem", 
    "--ann-file", 
    "homer simpson", 
    "--ann-template-file", 
    "eric cartman", 
    "--components",
    "all,-FaceDetector",
    "--show-mask",
    "--show-blob-cent",
    "--show-canny-map",
    "--pause",
    "--combination-style",
    "left-to-right",
    NULL
  };
  vector<char*> chars;
  insert_transform(params, chars, [](const char* c) { 
      return const_cast<char*>(c);
    });
  BOOST_CHECK(getArgV() == "");
  setArgV(chars.size()-1, &chars[0]);
  BOOST_CHECK(getArgV() == "./tests --help --version --verbose --face-cascade fakecascade123 --train-image batman rides again --out invalid output file.avi --feature-out unimaginably horrible feature file --elan-out an xml file --frames 42 --skip 4242 --aspect 200:4 --scale 200x400 --ann-format duke nukem --ann-file homer simpson --ann-template-file eric cartman --components all,-FaceDetector --show-mask --show-blob-cent --show-canny-map --pause --combination-style left-to-right");
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceVectorConstructors ) {
  vector<string> filenames = getTestVideoFramefilenames();
  vector<string> filenamesCopy = filenames;
  BOOST_REQUIRE(filenames == filenamesCopy);
  size_t size = filenames.size();
  std::unique_ptr<ImageSequenceSource> iss_p(new ImageSequenceSource(filenames));
  ImageSequenceSource& iss(*iss_p);
  BOOST_REQUIRE(filenames == filenamesCopy);
  // test the move constructor
  ImageSequenceSource iss2(std::move(filenamesCopy));
  BOOST_REQUIRE(filenames.size() == size);
  BOOST_REQUIRE(filenamesCopy.size() == 0);
  BOOST_REQUIRE(filenames == iss.filenames);
  BOOST_REQUIRE(filenames == iss2.filenames);
  BOOST_CHECK(iss.maxCacheSize == SIZE_MAX);
  BOOST_CHECK(iss2.maxCacheSize == SIZE_MAX);
  std::unique_ptr<ImageSequenceSource> iss_p2(new ImageSequenceSource(filenames,
									  25, 5));
  ImageSequenceSource& iss3(*iss_p2);
  BOOST_CHECK(iss3.maxCacheSize == 5);
  BOOST_CHECK(std::equal(iss.filenames.begin(), iss.filenames.end(),
                         filenames.begin()));
  BOOST_CHECK(std::equal(iss2.filenames.begin(), iss2.filenames.end(),
                         filenames.begin()));
  BOOST_CHECK(std::equal(iss3.filenames.begin(), iss3.filenames.end(),
                         filenames.begin()));
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceBracketOperator1 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());

  BOOST_REQUIRE(!equal(testVideoFrames2[0],
                       testVideoFrames2[9]));

  for (size_t i = 0; i < 10; ++i) {
    Mat j = iss[i];
    BOOST_CHECK(iss.frameCache.size() == i + 1);
    BOOST_CHECK(equal(testVideoFrames2[i], j));
  }
  
  auto it = iss.fifoIndices.begin();
  for (size_t i = 0; i < 10; ++i)
    BOOST_CHECK(*it++ == i);  

  for (size_t i = 0; i < 10; ++i) {
    Mat j = iss[i];
    BOOST_CHECK(iss.frameCache.size() == 10);
    BOOST_CHECK(equal(testVideoFrames2[i], j));
  }
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceBracketOperator2 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames(), 25, 5);

  for (size_t i = 0; i < 5; ++i) {
    Mat j = iss[i];
    BOOST_CHECK(iss.frameCache.size() == i + 1);
    BOOST_CHECK(equal(testVideoFrames2[i], j));
  }
  for (size_t i = 5; i < 10; ++i) {
    Mat j = iss[i];
    BOOST_CHECK(iss.frameCache.size() == 5);
    BOOST_CHECK(equal(testVideoFrames2[i], j));
  }

  
  auto it = iss.fifoIndices.begin();
  for (size_t i = 5; i < 10; ++i)
    BOOST_CHECK(*it++ == i);
  
  for (size_t i = 0; i < 10; ++i) {
    Mat j = iss[i];
    BOOST_CHECK(iss.frameCache.size() == 5);
    BOOST_CHECK(equal(testVideoFrames2[i], j));
  }
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceBracketOperator3 ) {
  // test with a range of numbers, in random order
  ImageSequenceSource iss(getTestVideoFramefilenames(), 25, 3);
  vector<size_t> range(10);
  for(size_t i = 0; i < 10; ++i)
    range[i] = i;
  std::random_shuffle(range.begin(), range.end());
  for (size_t i = 0; i < 10; ++i) {
    Mat j = iss[range[i]];
    switch (i) {
    case 0:
    case 1:
      BOOST_CHECK(iss.frameCache.size() == i+1);
      break;
    default:
      BOOST_CHECK(iss.frameCache.size() == 3);
      break;
    }
    BOOST_CHECK(equal(testVideoFrames2[range[i]], j));
  }

  auto it = iss.fifoIndices.begin();
  for (size_t i = 7; i < 10; ++i) {
    BOOST_CHECK(*it++ == range[i]);
  }

  for (size_t i = 1; i < 5; ++i) {
    Mat j = iss[range[i]];
    BOOST_CHECK(iss.frameCache.size() == 3);
    BOOST_CHECK(equal(testVideoFrames2[range[i]], j));
  }
  it = iss.fifoIndices.begin();
  for (size_t i = 2; i < 5; ++i) {
    BOOST_CHECK(*it++ == range[i]);
  }
  BOOST_CHECK(iss.fifoIndices.size() == iss.frameCache.size());
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceCacheIntegrity ) {
  // try to alter contents of the cache using public functions
  ImageSequenceSource iss(getTestVideoFramefilenames());
  Mat f1 = iss[0];
  // const Mat& f2 = iss[0];
  //  BOOST_CHECK(f1.data != f2.data);
  Mat f3 = f1;
  BOOST_CHECK(f1.data == f3.data);
  f3 = Scalar::all(0);
  BOOST_CHECK(f1.data == f3.data);
  BOOST_REQUIRE(testVideoFrames.size() > 0);
  BOOST_REQUIRE(!testVideoFrames[0].empty());
  BOOST_REQUIRE(!f1.empty());
  BOOST_CHECK(!equal(f1, testVideoFrames[0]));
  // BOOST_CHECK_MESSAGE(equal(f2, testVideoFrames[0]),
  BOOST_WARN_MESSAGE(equal(iss[0], testVideoFrames[0]),
                      "Cache integrity check failed! Making a non-const "
                      "copy of the matrix header allows one to change the "
                      "data of the supposedly-const matrix that resides "
                      "within cache. This is an OpenCV issue and cannot "
                      "really be dealt with.");
  BOOST_CHECK(!equal(f3, testVideoFrames[0]));
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceIterators1 ) {
  // Test default constructor
  ImageSequenceSource iss(getTestVideoFramefilenames());
  {
    ImageSequenceSource::const_iterator it;
    BOOST_CHECK(it.currentFrame == 0);
    BOOST_CHECK(it.frameSource == 0);
  }
  {
    ImageSequenceSource::const_iterator it(&iss);
    BOOST_CHECK(it.currentFrame == 0);
    BOOST_CHECK(it.frameSource == &iss);
  }
  {
    ImageSequenceSource::const_iterator it(&iss, 500);
    BOOST_CHECK(it.currentFrame == 500);
    BOOST_CHECK(it.frameSource == &iss);
  }

  // test copy constructor and copy assignment
  {
    ImageSequenceSource::const_iterator jt(&iss, 500);
    ImageSequenceSource::const_iterator it(jt);
    BOOST_CHECK(it.currentFrame == 500);
    BOOST_CHECK(it.frameSource == &iss);
    jt = ImageSequenceSource::const_iterator();
    it = jt;
    BOOST_CHECK(it.currentFrame == 0);
    BOOST_CHECK(it.frameSource == 0);
  }

  {
    ImageSequenceSource::const_iterator it = iss.begin();
    BOOST_CHECK(it.frameSource == &iss);
    BOOST_CHECK(it.currentFrame == 0);
  }
  {
    ImageSequenceSource::const_iterator it = iss.end();
    BOOST_CHECK(it.frameSource == &iss);
    BOOST_CHECK(it.currentFrame == 10);
  }
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceIterators2) {
  // Test the * operator
  ImageSequenceSource iss(getTestVideoFramefilenames());
  auto it = iss.begin();
  BOOST_CHECK(equal(testVideoFrames[0], *it));
  Mat f1 = *it;
  Mat f2 = *it;
  BOOST_CHECK(f1.data == f2.data);

  // Test the ++ operators
  it++;
  BOOST_CHECK(equal(testVideoFrames2[1], *it));

  ++it;
  BOOST_CHECK(equal(testVideoFrames2[2], *it));

  BOOST_CHECK(equal(testVideoFrames2[2], *it++));
  BOOST_CHECK(equal(testVideoFrames2[4], *++it));

  // Test the -- operators
  --it;
  BOOST_CHECK(equal(testVideoFrames2[3], *it));
  it--;
  BOOST_CHECK(equal(testVideoFrames2[2], *it));
  BOOST_CHECK(equal(testVideoFrames2[2], *it--));
  BOOST_CHECK(equal(testVideoFrames2[0], *--it));
}



BOOST_AUTO_TEST_CASE( testImageSequenceSourceIterators3 ) {
  // test logical comparisons
  ImageSequenceSource iss(getTestVideoFramefilenames());
  auto it = iss.begin();
  size_t i = 0;
  for( ; it != iss.end(); ++it, ++i) 
    BOOST_CHECK(equal(testVideoFrames2[i], *it));  
  BOOST_CHECK(it == iss.end());
  BOOST_CHECK(i == 10);

  it = iss.begin();
  auto jt = it;
  it = it + 3;
  it = 2 + it;
  BOOST_CHECK(jt != it);
  BOOST_CHECK(it > jt);
  jt += 5;
  BOOST_CHECK(jt == it);
  BOOST_CHECK(jt >= it);
  it -= 4;
  BOOST_CHECK(jt != it);
  BOOST_CHECK(it < jt);
  jt = jt - 2;
  jt = 2 - jt;
  BOOST_CHECK(jt == it);
  BOOST_CHECK(it <= jt);
  BOOST_CHECK(it >= jt);
  BOOST_CHECK(equal(testVideoFrames2[1], *jt));
}



BOOST_AUTO_TEST_CASE( testImageSequenceSourceIterators4 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());
  auto it = iss.begin();
  it += 2;
  BOOST_CHECK(it.getCurrentFramenumber() == 2);
  BOOST_CHECK(equal(testVideoFrames2[5], it[3]));
  BOOST_CHECK(it.getCurrentFramenumber() == 2);
  BOOST_CHECK(equal(testVideoFrames2[1], it[-1]));
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceIterators5 ) {
  // Finally, test iterating over the sequence in two ways
  ImageSequenceSource iss(getTestVideoFramefilenames());
  size_t i = 0;
  for (auto it = iss.begin(); it != iss.end(); ++it, ++i) {
    BOOST_CHECK(equal(testVideoFrames2[i], *it));
  }
  BOOST_CHECK(i == 10);

  i = 0;
  auto it = iss.begin();
  while(it < iss.end()) {
    BOOST_CHECK(equal(testVideoFrames2[i++], *it++));
  }
  BOOST_CHECK(i == 10);
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceScale1 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());
  BOOST_CHECK(iss[0].size() == cv::Size(720,596));
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceScale2 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());
  iss.setAspect(16./9.);
  BOOST_CHECK(iss[0].size() == cv::Size(1059,596));
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceScale3 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());
  iss.setScale(cv::Size(1920,1080));
  BOOST_CHECK(iss[0].size() == cv::Size(1920,1080));
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceScale4 ) {
  ImageSequenceSource iss(getTestVideoFramefilenames());
  iss.setScale(cv::Size(1600,1200));
  iss.setAspect(1.6);
  BOOST_CHECK(iss[0].size() == cv::Size(1920,1200));
}

BOOST_AUTO_TEST_CASE( testImageSequenceSourceIterators6 ) {
  // Test reverse iterators just as above
  ImageSequenceSource iss(getTestVideoFramefilenames());
  size_t i = 0;
  size_t size = testVideoFrames2.size();
  ImageSequenceSource::const_reverse_iterator it = iss.rbegin();
  BOOST_CHECK(equal(testVideoFrames2[size-i-1], *it));

  while(it < iss.rend()) {
    BOOST_CHECK(equal(testVideoFrames2[size-i++-1], *it++));
  }
  BOOST_CHECK(i == 10);  

  i = 0;
  for(auto jt = iss.rbegin(); jt != iss.rend(); ++jt, ++i) {
    BOOST_CHECK(equal(testVideoFrames2[size-i-1], *jt));
  }
  BOOST_CHECK(i == 10);  
}



#ifdef SLMOTION_ENABLE_FFMPEG

BOOST_AUTO_TEST_SUITE(ffmpegTests)

BOOST_AUTO_TEST_CASE(testFFmpegVideoFileSource1) {
  FFmpegVideoFileSource fvfs { TESTVIDEOFILE };
  BOOST_CHECK(fvfs.size() == 10);
}

BOOST_AUTO_TEST_CASE( testVideoFileSource1 ) {
  FFmpegVideoFileSource vfs(TESTVIDEOFILE_FFV1);
  BOOST_CHECK(equal(TESTVIDEOFILE_FFV1_FRAMES[0], vfs[0]));
  BOOST_CHECK(vfs.size() == 10);

  // Iterate over the video file
  auto jt = TESTVIDEOFILE_FFV1_FRAMES.begin();
  for(VideoFileSource::const_iterator it = vfs.begin();
      it != vfs.end(); ++it) {
    Mat m = *jt++, n = *it;
    BOOST_CHECK(equal(m, n));
  }
}

BOOST_AUTO_TEST_CASE( testVideoFileSource2 ) {
  // Iterate over the video file in reverse direction
  FFmpegVideoFileSource vfs(TESTVIDEOFILE_FFV1);
  auto jt = TESTVIDEOFILE_FFV1_FRAMES.rbegin();
  size_t i = 0;
  for(VideoFileSource::const_reverse_iterator it = vfs.rbegin();
      it < vfs.rend(); ++it) {
    if (!equal(*jt, *it))
      cout << i << endl;
    Mat m = *jt++, n = *it;
    BOOST_CHECK(equal(m, n));
    ++i;
  }
}


BOOST_AUTO_TEST_CASE( testFrameSourceReferenceSegfaultFfmpeg ) {
  std::vector<cv::Mat> frames;
  cv::VideoCapture vc(TESTVIDEOFILE2);
  while(vc.grab()) {
    Mat temp;
    vc.retrieve(temp);
    frames.push_back(temp.clone());
  }

  FFmpegVideoFileSource vfs { TESTVIDEOFILE2 };
  BOOST_REQUIRE(frames.size() > 0);
  BOOST_REQUIRE(vfs.size() == frames.size());
  const Mat& frame0 = vfs[0];
  BOOST_REQUIRE(equal(frame0, frames[0]));
  for (size_t i = 0; i < vfs.size(); ++i) {
    vfs[i];
    BOOST_CHECK(vfs.frameCache.size() == i + 1);
    BOOST_CHECK(&frame0 == &vfs[0]);
    BOOST_CHECK(equal(frame0, frames[0]));
  }
}

BOOST_AUTO_TEST_SUITE_END()

#endif // SLMOTION_ENABLE_FFMPEG
