#ifndef SLMOTION_SLCONVERT
#define SLMOTION_SLCONVERT

#include <deque>
#include <string>
#include <set>
#include <sstream>
#include <memory>
#include <map>
#include <vector>
#include <libxml/parser.h>



namespace slmotion {
  /**
   * Schema file for annotation specifiers
   */
  std::string getAnnotationSchemaFile();



  // forward declarations
  class SOM_PAK_Component;



  /**
   * Some helper functions for XML processing
   */
  namespace xml {
    class XmlDocument;
    class XmlNode;



    /**
     * Converts something to an xmlChar*. The xmlChar* array is statically
     * allocated, and may be overwritten by a subsequent call to this
     * function.
     */
    template<typename T>
    inline const xmlChar* toXmlChar(const T& obj) {
      static std::ostringstream oss;
      oss.str("");
      oss << obj;
      return reinterpret_cast<const xmlChar*>(oss.str().c_str());
    }



    /**
     * Converts a C string to an xmlChar* array
     *
     * @param cstring C string to convert
     * 
     * @return An xmlChar pointer representation
     */
    inline const xmlChar* toXmlChar(const char* cstring) {
      return reinterpret_cast<const xmlChar*>(cstring);
    }



    /**
     * Converts a C++ string to an xmlChar* array
     *
     * @param str A string to convert
     * 
     * @return An xmlChar pointer representation
     */
    template<>
    inline const xmlChar* toXmlChar(const std::string& str) {
      return reinterpret_cast<const xmlChar*>(str.c_str());
    }



    /**
     * Converts an xmlChar* array to a C string
     *
     * @param xmlString An xmlChar* array to convert
     *
     * @return A C string representation
     */
    inline const char* toChar(const xmlChar* xmlString) {
      return reinterpret_cast<const char*>(xmlString);
    }



    /**
     * Converts an xmlChar* array to a string object
     *
     * @param xmlString An xmlChar* array to convert
     *
     * @return A new string object
     */
    inline std::string toString(const xmlChar* xmlString) {
      return std::string(reinterpret_cast<const char*>(xmlString));
    }



    /**
     * Returns the property as a string
     *
     * @param node The node
     * @param property The name of the property
     *
     * @return The content as a string, or an empty string in case the
     * property was not found
     */
    inline std::string getXmlProperty(xmlNodePtr node, const char* property) {
      char* temp = reinterpret_cast<char*>(xmlGetProp(node, reinterpret_cast<const xmlChar*>(property)));
      std::string temp2(temp ? temp : "");
      xmlFree(temp);
      return temp2;
    }



    /**
     * Finds a child element by name
     *
     * @return A pointer to the child element if it exists, or NULL if it could
     * not be found
     */
    xmlNodePtr getChildElementByName(xmlNodePtr parent,
                                     const std::string& name);



    /**
     * Finds a child element by the specified name. In case there are multiple
     * siblings by the same name, a pointer to the last one will be returned.
     *
     * @param parent The parent element
     * @param name Name of the element being sought
     *
     * @return A pointer to the last child element by the given name, or NULL if
     * no such element could be found.
     */
    xmlNodePtr getLastChildElementByName(xmlNodePtr parent, const std::string& name);



    /**
     * Dumps the element to the given C stream
     */
    void dumpElement(FILE* stream, const XmlDocument& doc, 
                     const XmlNode& element);



    /**
     * Dumps the element to the given C++ stream
     */
    void dumpElement(std::ostream& stream, const XmlDocument& doc, 
                     const XmlNode& element);
  }
}

#endif
