#ifndef SLMOTION_ADVANCED_STRING
#define SLMOTION_ADVANCED_STRING

#include "regex.hpp"
#include <boost/lexical_cast.hpp>

namespace slmotion {
  // class AdvancedString;
  // inline bool operator==(const char*, const AdvancedString&);
  // inline bool operator==(const std::string&, const AdvancedString&);

  /**
   * this class represents an advanced version of the string
   * it is
   * - immutable, better facilitating functional programming style
   * - contains more helper methods such as split(), join(), and regex methods
   * - implicitly convertible to/from std::string
   * - can be constructed explicitly from arbitrary types (lexical cast)
   * - mostly compatible with the std::string interface
   *
   * The class may suffer some performance penalty wrt. the regular
   * string class
   */
  class AdvancedString {
  public:
    typedef std::string::const_iterator         const_iterator;
    typedef std::string::const_reverse_iterator const_reverse_iterator;

    /**
     * Construct an empty string
     */
    AdvancedString() = default;



    /**
     * Construct from std::string
     */
    AdvancedString(const std::string& str) : internalString(str) {}



    /**
     * Construct from substring
     */
    AdvancedString(const std::string& str, size_t pos, size_t len = npos) :
      internalString(str, pos, len) {}



    /**
     * Construct from C string
     */
    AdvancedString(const char* c) : internalString(c) {}



    /**
     * Construct from character sequence
     */
    AdvancedString(const char* s, size_t n) : internalString(s, n) {}



    /**
     * Construct from iterator
     */
    template <typename InputIterator>
    AdvancedString(InputIterator first, InputIterator last) :
      internalString(first, last) {}



    /**
     * Copy constructor
     */
    AdvancedString(const AdvancedString& that) : 
      internalString(that.internalString) {}



    /**
     * Copy assignment
     */
    AdvancedString& operator=(const AdvancedString& that);



    /**
     * Implicit cast to std::string
     */
    inline operator std::string() const {
      return internalString;
    }



    /**
     * Regular read-only random-access iterator
     */ 
    inline std::string::const_iterator begin() const {
      return internalString.cbegin();
    }



    /**
     * Regular read-only random-access iterator
     */ 
    inline std::string::const_iterator end() const {
      return internalString.cend();
    }



    /**
     * Regular read-only random-access iterator
     */ 
    inline std::string::const_iterator cbegin() const {
      return internalString.cbegin();
    }



    /**
     * Regular read-only random-access iterator
     */ 
    inline std::string::const_iterator cend() const {
      return internalString.cend();
    }



    /**
     * Regular read-only reverse random-access iterator
     */ 
    inline std::string::const_reverse_iterator rbegin() const {
      return internalString.crbegin();
    }



    /**
     * Regular read-only reverse random-access iterator
     */ 
    inline std::string::const_reverse_iterator rend() const {
      return internalString.crend();
    }



    /**
     * Regular read-only reverse random-access iterator
     */ 
    inline std::string::const_reverse_iterator crbegin() const {
      return internalString.crbegin();
    }



    /**
     * Regular read-only reverse random-access iterator
     */ 
    inline std::string::const_reverse_iterator crend() const {
      return internalString.crend();
    }



    /**
     * Performs a lexical cast
     */
    template<typename T>
    explicit AdvancedString(const T& some) : 
      internalString(boost::lexical_cast<std::string>(some))
    {

    }



    /**
     * Performs a lexical cast
     */
    template<typename T>
    explicit operator T() const 
    {
      return boost::lexical_cast<T>(internalString);
    }



    /**
     * Returns the length of the string
     */
    inline size_t length() const {
      return internalString.length();
    }



    /**
     * Returns the length of the string
     */
    inline size_t size() const {
      return internalString.size();
    }



    /**
     * Returns the theoretical maximum size of the string
     */
    inline size_t max_size() const {
      return internalString.max_size();
    }



    /**
     * Returns true if the string is an empty string.
     */
    inline bool empty() const {
      return internalString.empty();
    }



    /**
     * Returns the character at the given position.
     */
    inline const char& operator[](size_t i) const {
      return internalString[i];
    }



    /**
     * Returns the character at the given position.
     */
    inline const char& at(size_t i) const {
      return internalString.at(i);
    }



    /**
     * Returns the last character.
     */
    const char& back() const {
      return internalString.back();
    }



    /**
     * Returns the first character.
     */
    const char& front() const {
      return internalString.front();
    }



    /**
     * String concatenation: returns a new string that consists of the
     * current string appended with the string that.
     */
    inline AdvancedString operator+ (const AdvancedString& that) const {
      return AdvancedString(internalString + that.internalString);
    }



    /**
     * String concatenation: returns a new string that consists of the
     * current string appended with the string that.
     */
    inline AdvancedString operator+ (const std::string& that) const {
      return AdvancedString(internalString + that);
    }



    /**
     * String concatenation: returns a new string that consists of the
     * current string appended with the string that.
     */
    inline AdvancedString operator+ (const char* that) const {
      return AdvancedString(internalString + that);
    }



    /**
     * String concatenation and assignment: concatenates this string
     * with the other string, replaces the current string with that
     * and returns a reference to the new string.
     */
    inline AdvancedString& operator+= (const AdvancedString& str) {
      return *this = *this + str;
    }



    /**
     * Constructs a new string by inserting the given string to the
     * given position. A new string is returned.
     */
    inline AdvancedString insert(size_t pos, const AdvancedString& str) const {
      return std::string(internalString).insert(pos, str);
    }



    /**
     * Returns a copy of the string without the given subsequence.
     */
    inline AdvancedString erase(size_t pos, size_t len = npos) const {
      return std::string(internalString).erase(pos,len);
    }



    /**
     * Replaces the given substring with the given string. Returns a new string.
     */
    inline AdvancedString replace(size_t pos, size_t len, 
                                  const AdvancedString& str) const {
      return std::string(internalString).replace(pos,len,str);
    }



    /**
     * Replaces the first occurrence of what after pos with
     * whatWith. If what is not a substring, a 1:1 copy is returned.
     */
    AdvancedString replace(const AdvancedString& what, 
                           const AdvancedString& whatWith, 
                           size_t pos = 0) const;



    /**
     * Replaces all occurrences of what after pos with whatWith.
     */
    AdvancedString replaceAll(const AdvancedString& what, 
                              const AdvancedString& whatWith,
                              size_t pos = 0) const;



    /**
     * Replaces the first occurrence of the regex after pos.
     */
    AdvancedString replaceRe(const AdvancedString& regex, 
                             const AdvancedString& str, size_t pos = 0) const;



    AdvancedString replaceAllRe(const AdvancedString& regex, 
                                const AdvancedString& str, 
                                size_t pos = 0) const;



    /**
     * Returns the C character array representation of the string
     */
    inline const char* c_str() const {
      return internalString.c_str(); 
    }



    /**
     * Returns a substring of up to len characters, beginning from the
     * given position.
     */
    inline AdvancedString substr (size_t pos = 0, size_t len = npos) const {
      return internalString.substr(pos, len);
    }



    /**
     * Locates the first occurrence of the subsequence, beginning from
     * the given position.
     */
    inline size_t find(const AdvancedString& str, size_t pos = 0) const {
      return internalString.find(str, pos);
    }



    /**
     * Locates the last occurrence of the subsequence, beginning
     * before the given position.
     */
    inline size_t rfind(const AdvancedString& str, size_t pos = npos) const {
      return internalString.rfind(str, pos);
    }



    /**
     * Returns all occurrences of the given string, beginning from pos.
     */
    std::vector<size_t> findAll(const AdvancedString& str, 
                                size_t pos = 0) const;



    /**
     * Returns the first occurrence of the regular expression after
     * pos, or npos if the regex could not be matched.
     */
    size_t findRe(const AdvancedString& regex, size_t pos = 0) const;



    /**
     * Returns true if the string matches the given regex beginnig at
     * pos
     */
    bool match(const AdvancedString& regex, size_t pos = 0) const;



    /**
     * Splits the string into substrings, using str as separator
     */
    std::vector<AdvancedString> split(const AdvancedString& str) const;
 


    /**
     * Joins the given strings by inserting this string as separator
     */
    AdvancedString join(const std::vector<AdvancedString>& strings) const;



    /**
     * Joins the given strings by inserting this string as separator
     */
    AdvancedString join(const std::vector<std::string>& strings) const;



    /**
     * Joins the strings while inserting str as separator.
     */
    inline static AdvancedString join(const AdvancedString& str,
                                      const std::vector<AdvancedString>& strings) {
      return str.join(strings);
    }



    /**
     * Joins the strings while inserting str as separator.
     */
    inline static AdvancedString join(const AdvancedString& str,
                                      const std::vector<std::string>& strings) {
      return str.join(strings);
    }



    /**
     * The same as strcmp
     */
    inline int compare(const AdvancedString& that) const {
      return internalString.compare(that.internalString);
    }



    /**
     * Value for non-existent string position
     */
    static const size_t npos = std::string::npos;



    /**
     * Returns true iff the two strings are identical
     */
    inline bool operator==(const char* c) const {
      return internalString == c;
    }



    /**
     * Returns true iff the two strings are identical
     */
    inline bool operator==(const std::string& s) const {
      return internalString == s;
    }



    /**
     * Returns true iff the two strings are identical
     */
    inline bool operator==(const AdvancedString& that) const {
      return internalString == that.internalString;
    }



    /**
     * Returns true iff the two strings are not identical
     */
    inline bool operator!=(const char* c) const {
      return !(internalString == c);
    }



    /**
     * Returns true iff the two strings are not identical
     */
    inline bool operator!=(const std::string& s) const {
      return !(internalString == s);
    }



    /**
     * Returns true iff the two strings are not identical
     */
    inline bool operator!=(const AdvancedString& that) const {
      return !(internalString == that.internalString);
    }



    /**
     * Returns the string concatenated with itself n times
     */
    AdvancedString operator*(size_t n) const;



  private:
    std::string internalString;
  };



  /**
   * Returns true iff the two strings are identical
   */
  inline bool operator==(const char* c, const AdvancedString& that) {
    return that == c;
  }



  /**
   * Returns true iff the two strings are identical
   */
  inline bool operator==(const std::string& s, const AdvancedString& that) {
    return that == s;
  }



  /**
   * Returns true iff the two strings are not identical
   */
  inline bool operator!=(const std::string& s, const AdvancedString& that) {
    return !(s == that);
  }



  /**
   * Returns true iff the two strings are not identical
   */
  inline bool operator!=(const char* c, const AdvancedString& that) {
    return !(c == that);
  }



  /**
   * Concatenates the two strings
   */
  inline AdvancedString operator+ (const std::string& lhs, const AdvancedString& rhs) {
    return AdvancedString(lhs) + rhs;
  }



  /**
   * Concatenates the two strings
   */
  inline AdvancedString operator+ (const char* lhs, const AdvancedString& rhs) {
    return AdvancedString(lhs) + rhs;
  }



  /**
   * Writes the string to the given outstream
   */
  inline std::ostream& operator<<(std::ostream& os, const AdvancedString& s) {
    return os << s.c_str();
  }



  /**
   * Returns the string concatenated with itself n times
   */
  inline AdvancedString operator*(size_t n, const AdvancedString& s) {
    return s * n;
  }
}

#endif
