#include "Analyser.hpp"
#include "util.hpp"
#include "BodyPartCollector.hpp"
#include "AsmTracker.hpp"
#include "Thread.hpp"

using std::cerr;
using std::endl;
using std::list;
using std::vector;
using std::string;
using std::for_each;
using std::deque;
using std::shared_ptr;
using std::make_pair;

using cv::Mat;
using cv::Point;
using cv::Scalar;
using cv::Rect;
using cv::Point2f;
using cv::Size;
using cv::Vec3b;
using cv::Vec4d;
using cv::Vec6d;
using cv::Vec3d;
using cv::Vec;
using cv::Vec2d;
using cv::Point2d;



namespace slmotion {
  namespace {
    /**
     * Returns true if all properties required by the component are on the 
     * black board as per the property set, and it does not provide any 
     * other properties that other components that are currently being 
     * processed will provide, and that no other component earlier in the 
     * sequence is going to provide a newer version of a property that is 
     * already present on the black board.
     */
    bool checkComponentRequirements(const Component& comp, 
                                    const Component::property_set_t& propertiesOnBlackBoard,
                                    const Component::property_set_t& propertiesComingToBlackBoard) {
      const Component::property_set_t& reqs = comp.getRequirements();
      for (auto it = reqs.cbegin(); it != reqs.end(); ++it)
        if (!propertiesOnBlackBoard.count(*it) || 
            propertiesComingToBlackBoard.count(*it))
          return false;
      const Component::property_set_t& provided = comp.getProvided();
      for (auto it = provided.cbegin(); it != provided.cend(); ++it)
        if (propertiesComingToBlackBoard.count(*it))
          return false;
      return true;
    }
  }
  


  extern int debug; ///< The debug option (defined elsewhere)



#ifdef SLMOTION_THREADING
  void Analyser::process(size_t first, size_t last) {
    // check what should and should not be on the blackboard, and create
    // threads for components that have everything they need
    // otherwise, wait for the oldest thread to finish, and then try again
    // (having updated the set of properties)
    auto componentIterator = preprocessComponents.begin();
    Component::property_set_t propertiesOnBlackBoard;
    Component::property_set_t propertiesComingToBlackBoard; // future provided properties

    // threads and properties provided by the thread
    std::list<std::pair<shared_ptr<Thread>,Component::property_set_t>> threads;
    while(componentIterator != preprocessComponents.end()) {
      if (checkComponentRequirements(**componentIterator, 
                                     propertiesOnBlackBoard,
                                     propertiesComingToBlackBoard)) {       
        if (debug > 0)
          cerr << "Processing '" << (*componentIterator)->getComponentName() << "' for frames ["
               << first << "," << last << ")" << endl;

        auto provided = (*componentIterator)->getProvided();        
       
        // only create a new thread if maximum number has not been exceeded,
        // otherwise process the component in this thread
        shared_ptr<Thread> thread(Thread::createThreadIfPossible(&Component::processRange, &**componentIterator, first, last, static_cast<UiCallback*>(NULL)));
        if (thread.get()) {
          propertiesComingToBlackBoard.insert(provided.cbegin(),
                                              provided.cend());
          threads.push_back(std::make_pair(thread, (*componentIterator)->getProvided()));
        }
        else {
          (*componentIterator)->processRange(first, last, NULL);
          propertiesOnBlackBoard.insert(provided.cbegin(), provided.cend());
        }
        ++componentIterator;
      }
      else if (threads.size() > 0) {
        // we arrive here if the current component cannot be processed now
        // (because of conflicting requirements and properties provided by
        // previous components), so wait for the first component to finish,
        // and try again
        if (threads.front().first->joinable())
          threads.front().first->join();
        auto provided = threads.front().second;
        propertiesOnBlackBoard.insert(provided.cbegin(), provided.cend());
        for (auto it = provided.cbegin(); it != provided.cend(); ++it)
          propertiesComingToBlackBoard.erase(*it);
        threads.pop_front();
      }
      else
        throw SLMotionException("Deadlock! Unsatisfied sequence of components; requirements cannot be fulfilled.");
    }
    // finally, wait for the threads to finish
    while (threads.size() > 0) {
      if (threads.front().first->joinable())
        threads.front().first->join();
      threads.pop_front();
    }
  }

#else

  void Analyser::process(size_t first, size_t last, UiCallback* callback) {
    if (debug > 1)
      std::cerr << "slmotion: Analyser::process" << std::endl;

    if (callback != nullptr)
      callback->setNComponents(components.size());
    size_t componentIdx = 0;
    
    // single-threaded case: just iterate over the components
    for (auto it = components.begin();
         it != components.end(); ++it) {
      if (callback != nullptr) {
        callback->setCurrentComponentName((*it)->getComponentName());
        callback->setCurrentComponentIndex(componentIdx);
      }

      if (debug > 1)
        cerr << "Processing '" << (*it)->getComponentName() << "' for frames ["
             << first << "," << last << ")" << endl;
      try {
        class ProcessTerminated : public SLMotionException {
        public:
          ProcessTerminated(const std::string& msg) : 
            SLMotionException(msg.c_str()) {}
          ~ProcessTerminated() throw() {}
        };

        if (!(*it)->processRange(first, last, callback))
          throw ProcessTerminated("User requested component chain process termination");
      }
      catch (std::exception& e) {
        throw AnalysisException("An exception occurred while processing " + 
                                (*it)->getComponentName() + ": " + e.what());
      }
      catch (...) {
        // throw AnalysisException("An exception of unhandled type (non-std::exception-derived) "
        //                        "was caught while processing " + (*it)->getComponentName());
        throw;
      }

      componentIdx++;
    }

    if (debug > 1)
      std::cerr << "slmotion: Analyser: Gone through all components" << std::endl;
  }
#endif



  cv::Mat Analyser::visualise(size_t frameNumber) const {
    return visualiser->visualise(/* (*frameSource)[frameNumber] */
                                 *frameSource, *blackboard,
                                 frameNumber);
  }
}
