#ifndef SLMOTION_ANALYSER
#define SLMOTION_ANALYSER

#include "KLTTracker.hpp"
#include "Visualiser.hpp"
#include "AnalysisResult.hpp"
#include "InterpretationResult.hpp"
#include "FrameSource.hpp"
#include "BlackBoard.hpp"
#include "Component.hpp"
#include "SkinDetector.hpp"
#include "FeatureCreator.hpp"



namespace slmotion {
  /**
   * Analyses the content framewise
   *
   * Currently supports simple motion descriptor analysis (i.e. number of
   * tracked features, x and y sums of motion vectors, sum of motion and
   * acceleration vector lengths), blob-based body part classification,
   * association of body part identities with the aforementioned features,
   * and an ASM-based model fitting for hands and the head
   */
  class Analyser {
  public:
    /**
     * Constructs a new Analyser object.
     *
     * @param io SLIO object reference
     */
    Analyser(std::shared_ptr<SLIO> io,
             std::shared_ptr<FrameSource> frameSource,
             std::shared_ptr<Visualiser> visualiser,
             std::shared_ptr<BlackBoard> blackboard) :
      slio(io),
      frameSource(frameSource),
      visualiser(visualiser),
      blackboard(blackboard)
    {}



    inline const BlackBoard& getBlackBoard() const {
      return *blackboard;
    }



    inline void clearBlackBoard() {
      blackboard->clear();
    }



    /**
     * @param first First frame to process (inclusive)
     * @param last Last frame to process (exclusive)
     */
    void process(size_t first, size_t last, UiCallback* callback = nullptr);



    /**
     * Performs the main processing phase
     *
     * @return The processed and visualised output. If the assembly
     * line is empty, and no more frames can be read in, returns an
     * unallocated NULL matrix.
     */
    cv::Mat visualise(size_t frameNumber) const;



    inline std::shared_ptr<Visualiser> getVisualiser() {
      return visualiser;
    }



    inline const Visualiser& getVisualiser() const {
      return *visualiser;
    }



    inline std::shared_ptr<SLIO> getSlio() {
      return slio;
    }



    inline const SLIO& getSlio() const {
      return *slio;
    }



    inline FrameSource& getFrameSource() {
      return *frameSource;
    }



    inline std::shared_ptr<FrameSource> getFrameSourcePtr() {
      return frameSource;
    }



    inline const FrameSource& getFrameSource() const {
      return *frameSource;
    }



    /**
     * Adds a preprocessor component. The class will take ownership of the 
     * pointer, and will eventually call delete upon it.
     */
    inline void addPreprocessComponent(Component* component) {
      std::shared_ptr<Component> p(component);
      components.push_back(p);
    }



    inline void addPreprocessComponents(std::vector<std::shared_ptr<Component>>&& comps) {
      components = comps;
    }


    /**
     * Returns the feature vector if it exists on the blackboard. Otherwise,
     * an empty vector is returned.
     */
    // inline FeatureVector getFeatureVector() const {
    //   if (blackboard->has(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY))
    //     return blackboard->get<FeatureVector>(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY);

    //   return FeatureVector();
    // }



    // Returns blobs
    // inline const std::vector<Blob>& getBlobs(size_t frnumber) const {
    //   if (blackboard->has(frnumber, SKINDETECTOR_BLACKBOARD_MASK_ENTRY))
    //     return blackboard->get<std::vector<Blob>>(frnumber, BLACKBOARD_BLOBS_ENTRY);
    //   const static std::vector<Blob> bogusblobs;
    //   return bogusblobs;
    // }
    


  private:
    // Returns the skin mask from the blackboard
    // inline const cv::Mat& getSkinMask(size_t frnumber) const {
    //   return blackboard->get<cv::Mat>(frnumber, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);
    // }



    std::vector<std::shared_ptr<Component>> components; ///< Components that are to be run in the processing phase in order (called simply through processRange)
    std::shared_ptr<SLIO> slio; ///< Handles all I/O operations, except reading frames
    std::shared_ptr<FrameSource> frameSource; ///< Reads in frames 
    std::shared_ptr<Visualiser> visualiser; ///< Visualisation object
    /** 
     * blackboard for storing and sharing processing results
     * that any process stage generates/needs
     */
    std::shared_ptr<BlackBoard> blackboard;
  };
}
#endif
