#ifndef SLMOTION_ASM
#define SLMOTION_ASM

#include <memory>
#include <list>
#include "Pdm.hpp"
#include "BodyPart.hpp"
#include "exceptions.hpp"

namespace slmotion {
  /**
   * This class represents an Active Shape Model whose shape is
   * statistically described by the associated PDM.
   */
  class Asm {
  public:
    /**
     * This struct simply represents a single ASM instance
     */
    class Instance {
    public:
      Instance() = default;

      bool operator==(const Instance& other) const;


      Instance(const Instance& other) : 
        pdm(other.pdm), bt(other.bt.clone()), pose(other.pose),
        landmarks(other.landmarks.clone()) {}



      explicit Instance(std::shared_ptr<const Pdm> pdm,
                        const cv::Mat& bt,
                        const Pdm::PoseParameter& pose,
                        const cv::Mat& landmarks) :
        pdm(pdm), bt(bt), pose(pose), landmarks(landmarks) {}



      // Instance(Instance&&) = default;

      inline Pdm::PoseParameter getPose() const {
        return pose;
      }

      inline cv::Mat getBt() const {
        return bt.clone();
      }

      inline cv::Mat getLandmarks() const {
        return landmarks.clone();
      }

      /**
       * Gets the minimum size for a matrix that can hold the, with 
       * the translation required (since some points may be located outside
       * the visible image; the minimal translation will move them into the
       * positive axes)
       */
      void getMinSize(cv::Size& outMinSize, cv::Point2d& outTranslation) const;
      

      void drawLine(cv::Mat& target, int thickness, 
                    const cv::Scalar& colour, 
                    const cv::Point2d& translation = cv::Point2d()) const;

      Instance& operator=(const Instance&);

      cv::Point2d computeCentroid() const;

      inline const Pdm& getPdm() const {
        return *pdm;
      }



    private:
      std::shared_ptr<const Pdm> pdm; ///< The associated point distribution model
      cv::Mat bt; ///< Shape parameter
      Pdm::PoseParameter pose; ///< Pose parameters
      /**
       * Instance landmarks in the usual representation (column vector, 
       * CV_64FC1): (x0,y0,x1,y1,...,xN-1,YN-1)^T
       */
      cv::Mat landmarks; 
    };



    /**
     * Returns an empty instance
     */
    static const Instance EMPTY;



    /**
     * Explicitly construct the ASM
     */
    Asm(std::shared_ptr<Pdm> pdm, size_t nComponents, 
        double maxShapeParameterDeviation) :
      pdm(pdm), nComponents(nComponents),
      maxShapeParameterDeviation(maxShapeParameterDeviation)
    {}



    Asm(const std::vector<Blob>& blobs, size_t nLandmarks, 
        size_t nComponents, double maxShapeParameterDeviation) :
      pdm(new Pdm(blobs, nLandmarks)), nComponents(nComponents),
      maxShapeParameterDeviation(maxShapeParameterDeviation)
    {}


    /**
     * The default copy constructor
     */
    Asm(const Asm&) = default;

    /**
     * The default copy-assignment
     */
    Asm& operator=(const Asm&) = default;



    /**
     * Fit the model with no historical knowledge
     */
    inline Instance fit(const cv::Mat& inFrame, 
                        const cv::Mat& initialMask) const {
      return fitImpl(inFrame, initialMask, NULL);
    }



    /**
     * Fit the model with the given knowledge about the previous frame
     */
    inline Instance fit(const cv::Mat& inFrame, 
                        const cv::Mat& initialMask,
                        const Instance& oldInstance) const {
      return fitImpl(inFrame, initialMask, &oldInstance);
    }

    inline const Pdm& getPdm() const {
      return *pdm;
    }
   
    inline size_t getNComponents() const {
      return nComponents;
    }

    inline double getMaxShapeParameterDeviation() const {
      return maxShapeParameterDeviation;
    }

  private:
    std::shared_ptr<const Pdm> pdm;
    size_t nComponents;
    double maxShapeParameterDeviation;



    /**
     * @param inFrame Input frame to fit the model into
     * @param initialMask Mask used for initialisation (shape rotation) 
     * @param oldInstance Old instance data or NULL
     * @param anchorGuess A guessed anchor point location (landmark zero 
     * location). This is only necessary if oldInstance is NULL.
     */
    Instance fitImpl(const cv::Mat& inFrame, 
                     const cv::Mat& initialMask,
                     const Instance* oldInstance) const;
  };



  class AsmException : public SLMotionException {
  public:
    explicit AsmException(const char* msg) :
      SLMotionException(msg) {}

    explicit AsmException(const std::string& msg) :
      SLMotionException(msg.c_str()) {}

    virtual ~AsmException() throw() {}
  };



  /**
   * Computes the area enclosed within the intersection of the two ASM 
   * instances
   */
  double intersectionArea(const Asm::Instance& a,
                          const Asm::Instance& b);

  double computeOrientation(const Asm::Instance&);
}
#endif
 
