#ifndef SLMOTION_BLACKBOARD
#define SLMOTION_BLACKBOARD

#include "BlackBoardEntry.hpp"
#include "BlackBoardPointer.hpp"
#include "exceptions.hpp"

#ifdef SLMOTION_ENABLE_PICSOM
#undef signals
#include <PicSOM.h>
#endif // SLMOTION_ENABLE_PICSOM

#include <map>
#include <string>
#include <memory>
#include <boost/any.hpp>
#include <set>

#ifdef SLMOTION_THREADING
#include <thread>
#define LOCK_WHEN_THREADING() std::lock_guard<std::mutex> lk(this->mutex)
#else
#define LOCK_WHEN_THREADING()
#endif

namespace slmotion {
  void timerOn();
  time_t timerOff();

  /**
   * This is a simple wrapper to make accesssing results from different 
   * frames easier.
   */
  class BlackBoard {
  public:
    typedef size_t frame_number_t;
    typedef std::string property_key_t;
    typedef std::shared_ptr<BlackBoardEntry> value_t;
    typedef std::pair<frame_number_t,property_key_t> frame_board_key_t;
    typedef std::map<frame_board_key_t,value_t> frame_board_t;
    typedef std::map<property_key_t,value_t> global_board_t;

    BlackBoard() = default;

#ifdef SLMOTION_ENABLE_PICSOM
    BlackBoard(std::shared_ptr<picsom::PicSOM> p, 
	       picsom::DataBase *db) :
    picsom(p), picsom_db(db) {}

    picsom::DataBase* getPicSOMdataBase() const {
      return picsom_db;
    }
#endif // SLMOTION_ENABLE_PICSOM


#ifdef SLMOTION_THREADING
    BlackBoard(const BlackBoard& other) {
      std::lock_guard<std::mutex> lk1(this->mutex);
      std::lock_guard<std::mutex> lk2(other.mutex);
      this->frameBoard = other.frameBoard;
      this->globalBoard = other.globalBoard;
    }

    BlackBoard& operator=(const BlackBoard& other) {
      std::lock_guard<std::mutex> lk1(this->mutex);
      std::lock_guard<std::mutex> lk2(other.mutex);
      
      frame_board_t temp_fb = other.frameBoard;
      global_board_t temp_gb = other.globalBoard;
      std::swap(this->frameBoard, temp_fb);
      std::swap(this->globalBoard, temp_gb);
      return *this;
    }

    BlackBoard(BlackBoard&& other) {
      std::lock_guard<std::mutex> lk1(this->mutex);
      std::lock_guard<std::mutex> lk2(other.mutex);
      this->frameBoard = std::move(other.frameBoard);
      this->globalBoard = std::move(other.globalBoard);
    }

    BlackBoard& operator=(BlackBoard&& other) {
      std::lock_guard<std::mutex> lk1(this->mutex);
      std::lock_guard<std::mutex> lk2(other.mutex);
      
      std::swap(this->frameBoard, other.frameBoard);
      std::swap(this->globalBoard, other.globalBoard);
      return *this;
    }
#else // SLMOTION_THREADING
    BlackBoard(const BlackBoard& other) = default;
    BlackBoard& operator=(const BlackBoard& other) = default;
#if (__GNUC__ == 4 && __GNUC_MINOR__ < 5)
    // empty
#else
    BlackBoard(BlackBoard&& other) = default;
#endif // (__GNUC__ == 4 && __GNUC_MINOR__ > 4))
    BlackBoard& operator=(BlackBoard&& other) {
      std::swap(this->frameBoard, other.frameBoard);
      std::swap(this->globalBoard, other.globalBoard);
      return *this;
    }
#endif // SLMOTION_THREADING

    /**
     * Returns true if the blackboard contains the given entry for the given
     * frame.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return True if the blackboard contains the entry
     */
    inline bool has(frame_number_t framenr, const property_key_t& property) const {
      LOCK_WHEN_THREADING();
      return frameBoard.count(std::make_pair(framenr,property));
    }



    /**
     * Returns true if the blackboard contains the given global entry
     *
     * @param property Property ID string
     *
     * @return True if the blackboard contains the entry
     */
    inline bool has(const property_key_t& property) const {
      LOCK_WHEN_THREADING();
      return globalBoard.count(property);
    }



#if BLACKBOARD_ENTRY_ALLOW_IMPLICIT_CONVERSIONS == true
    /**
     * Attempts to retrieve the entry from the given frame with the given 
     * property ID. Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline T& get(frame_number_t framenr, const property_key_t& property) {
      if (!has(framenr, property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      return *frameBoard[std::make_pair(framenr,property)];
    }



    /**
     * Attempts to retrieve the entry from the given frame with the given 
     * property ID. Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline const T& get(frame_number_t framenr, 
                        const property_key_t& property) const {
      if (!has(framenr, property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      return *frameBoard.find(std::make_pair(framenr,property))->second;
    }



    /**
     * Returns the plain and raw any object
     */
    inline const boost::any& get(frame_number_t framenr,
                                 const property_key_t& property) const {
      if (!has(framenr, property))
        throw BlackBoardException(property "+: No such element on the board!");
      LOCK_WHEN_THREADING();
      return frameBoard.find(std::make_pair(framenr,property))->second->getAnyReference();
    }

    /**
     * Returns the plain and raw any object
     */
    inline const boost::any& get(const property_key_t& property) const {
      if (!has(property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      return globalBoard.find(property)->second->getAnyReference();
    }



    /**
     * Attempts to retrieve the global entry with the given property ID. 
     * Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline const T& get(const property_key_t& property) const {
      if (!has(property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      return *boost::any_cast<T>(globalBoard.find(property)->second.get());
    }



    /**
     * Attempts to retrieve the global entry with the given property ID. 
     * Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline T& get(const property_key_t& property) {
      if (!has(property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      return *globalBoard[property];
    }



    /**
     * Attempts to retrieve the entry from the given frame with the given 
     * property ID. Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline const BlackBoardPointer<T> getPtr(frame_number_t framenr, 
                                             const property_key_t& property) const {
      if (!has(framenr, property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      auto p = frameBoard.find(std::make_pair(framenr,property))->second;
      return p->reference<T>(p);
    }



    /**
     * Attempts to retrieve the entry from the given frame with the given 
     * property ID. Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline BlackBoardPointer<T> getPtr(frame_number_t framenr, const property_key_t& property) {
      if (!has(framenr, property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      return frameBoard[std::make_pair(framenr,property)]->reference<T>();
    }



    /**
     * Attempts to retrieve the global entry with the given property ID. 
     * Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline const BlackBoardPointer<T> getPtr(const property_key_t& property) const {
      if (!has(property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      return globalBoard.find(property)->second->reference<T>();
    }



    /**
     * Attempts to retrieve the global entry with the given property ID. 
     * Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline BlackBoardPointer<T> getPtr(const property_key_t& property) {
      if (!has(property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      return globalBoard[property]->reference<T>();
    }
#else // BLACKBOARD_ENTRY_ALLOW_IMPLICIT_CONVERSIONS == true
    /**
     * Attempts to retrieve the entry from the given frame with the given 
     * property ID. Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline const BlackBoardPointer<T> get(frame_number_t framenr, 
                                          const property_key_t& property) const {
      if (!has(framenr, property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      auto p = frameBoard.find(std::make_pair(framenr,property))->second;
      return p->reference<T>(p);
    }



    /**
     * Attempts to retrieve the entry from the given frame with the given 
     * property ID. Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline BlackBoardPointer<T> get(frame_number_t framenr, const property_key_t& property) {
      if (!has(framenr, property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      auto p = frameBoard[std::make_pair(framenr,property)];
      return p->reference<T>(p);
    }



    /**
     * Attempts to retrieve the global entry with the given property ID. 
     * Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline const BlackBoardPointer<T> get(const property_key_t& property) const {
      if (!has(property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      auto p = globalBoard.find(property)->second;
      return p->reference<T>(p);
    }



    /**
     * Attempts to retrieve the global entry with the given property ID. 
     * Throws an exception upon failure.
     *
     * @param framenr Frame number
     * @param property Property ID string
     *
     * @return A reference to the object.
     */
    template<typename T>
    inline BlackBoardPointer<T> get(const property_key_t& property) {
      if (!has(property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      auto p = globalBoard[property];
      return p->reference<T>(p);
    }

    /**
     * Returns the plain and raw any object
     */
    inline const BlackBoardPointer<boost::any> get(const property_key_t& property) const {
      if (!has(property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      auto p = globalBoard.find(property)->second;
      return p->referenceAny(p);
    }

    /**
     * Returns the plain and raw any object
     */
    inline const BlackBoardPointer<boost::any> get(frame_number_t framenr,
                                                   const property_key_t& property) const {
      if (!has(framenr, property))
        throw BlackBoardException(property + ": No such element on the board!");
      LOCK_WHEN_THREADING();
      auto p = frameBoard.find(std::make_pair(framenr,property))->second;
      return p->referenceAny(p);
    }

#endif // BLACKBOARD_ENTRY_ALLOW_IMPLICIT_CONVERSIONS == true


    /**
     * Returns the type info for the given element on the board. This can be
     * used to deduce the actual type.
     */
    const std::type_info& typeOf(frame_number_t framenr, 
                                 const property_key_t& property) const {
      if (!has(framenr, property))
        throw BlackBoardException(property + ": No such element on the board!");
      return frameBoard.find(std::make_pair(framenr,property))->second->type();
    }



    /**
     * Returns the type info for the given element on the board. This can be
     * used to deduce the actual type.
     */
    const std::type_info& typeOf(const property_key_t& property) const {
      if (!has(property))
        throw BlackBoardException(property + ": No such element on the board!");
      return globalBoard.find(property)->second->type();
    }



    /**
     * Sets a property to the given value. The value must have valid copy 
     * semantics.
     *
     * @param framenr Frame number
     * @param property Property ID string
     * @param value Reference to a value
     */
    template<typename T>
    inline void set(frame_number_t framenr, const property_key_t& property, 
                    const T& value, 
                    BlackBoardEntry::BlackBoardEntryAttribute attribs = 
                    BlackBoardEntry::COMPRESS_WITHOUT_REFERENCES) {
      LOCK_WHEN_THREADING();
      // prevent self-assignment
      if (frameBoard.count(std::make_pair(framenr,property)))
        if (frameBoard[std::make_pair(framenr,property)]->pointsTo(&value))
          return;

      frameBoard[std::make_pair(framenr,property)] = value_t(new BlackBoardEntry(value, attribs));
    }



    /**
     * Sets a property to the given value. The value must have valid move 
     * semantics.
     *
     * @param framenr Frame number
     * @param property Property ID string
     * @param value Reference to a value
     */
    template<typename T>
    inline void set(frame_number_t framenr, const property_key_t& property, 
                    T&& value, 
                    BlackBoardEntry::BlackBoardEntryAttribute attribs = 
                    BlackBoardEntry::COMPRESS_WITHOUT_REFERENCES) {
      LOCK_WHEN_THREADING();
      // prevent self-assignment
      if (frameBoard.count(std::make_pair(framenr,property)))
        if (frameBoard[std::make_pair(framenr,property)]->pointsTo(&value))
          return;

      frameBoard[std::make_pair(framenr,property)] = value_t(new BlackBoardEntry(value, attribs));
    }



    /**
     * Copy-sets a global property (i.e. one that is not associated with any
     * particular frame)
     */
    template<typename T>
    inline void set(const property_key_t& property, const T& value,
                    BlackBoardEntry::BlackBoardEntryAttribute attribs = 
                    BlackBoardEntry::COMPRESS_WITHOUT_REFERENCES) {
      LOCK_WHEN_THREADING();
      // check for self-assignment
      if (globalBoard.count(property))
        if (globalBoard[property]->pointsTo(&value))
          return;
      globalBoard[property] = value_t(new BlackBoardEntry(value, attribs));
    }



    /**
     * Move-sets a global property (i.e. one that is not associated with any
     * particular frame)
     */
    template<typename T>
    inline void set(const property_key_t& property, T&& value, 
                    BlackBoardEntry::BlackBoardEntryAttribute attribs = 
                    BlackBoardEntry::COMPRESS_WITHOUT_REFERENCES) {
      LOCK_WHEN_THREADING();
      // check for self-assignment
      if (globalBoard.count(property))
        if (globalBoard[property]->pointsTo(&value))
          return;
      globalBoard[property] = value_t(new BlackBoardEntry(value, attribs));
    }



    /**
     * Returns all the keys associated with the frame board, i.e. 
     * property-frame number pairs
     */
    std::set<frame_board_key_t> getAllFrameboardKeys() const {
      std::set<frame_board_key_t> keys;
      std::transform(frameBoard.cbegin(), frameBoard.cend(), 
                     std::inserter(keys, keys.begin()), 
                     [](decltype(*frameBoard.cbegin()) it) {
                       return it.first;
                     });
      return keys;
    }

    

    inline void clear() {
      frameBoard.clear();
      globalBoard.clear();
    }

    const frame_board_t& getFrameBoard() const {
      return frameBoard;
    }

    const global_board_t& getGlobalBoard() const {
      return globalBoard;
    }

    /**
     * Returns the total number of elements on the black board (the sum of 
     * elements in the frame board over all frames plus the sum of elements
     * on the global board)
     */
    inline size_t size() const {
      return frameBoard.size() + globalBoard.size();
    }



    /**
     * Deletes the given property from the global board
     */
    inline void erase(const std::string& property) {
      if (!has(property))
        throw BlackBoardException(property + ": No such element on the board!");

      globalBoard.erase(property);
    }



    /**
     * Deletes the given property from the frame board
     */
    inline void erase(frame_number_t framenr, const std::string& property) {
      if (!has(framenr, property))
        throw BlackBoardException(property + ": No such element on the board!");

      frameBoard.erase(std::make_pair(framenr,property));
    }



    /**
     * Erases all instances of the given property
     */
    void eraseAll(const std::string& property);


  private:
    frame_board_t frameBoard; ///< Internal storage for frame-specific data
    global_board_t globalBoard; ///< Internal storage for global data

#ifdef SLMOTION_THREADING
    mutable std::mutex mutex; ///< If threaded access is used, this object will be used for mutual exclusion
#undef LOCK_WHEN_THREADING
#endif

#ifdef SLMOTION_ENABLE_PICSOM
    std::shared_ptr<picsom::PicSOM> picsom;
    picsom::DataBase *picsom_db;
#endif // SLMOTION_ENABLE_PICSOM
  };
}

#endif
