#include "BlackBoardDumpReader.hpp"
#include <fstream>
#include "Blob.hpp"
#include "BodyPart.hpp"
#include "TrackedPoint.hpp"
#include "Asm.hpp"
#include "TrackingSkinDetector.hpp"
#include "gzstream.hpp"
#include "util.hpp"

using std::string;
using std::vector;

namespace slmotion {
  extern int debug;

  static BlackBoardDumpReader DUMMY(true);

  // this is here to circumvent a bug in GCC 4.6
  static cv::Rect rectReaderFunction(BlackBoardDumpReader* const w, std::istream& ifs) {
    int32_t i[4];
    w->dumbRead(ifs, i);
    return cv::Rect(i[0], i[1], i[2], i[3]);
  }
  
  BlackBoardDumpReader::BlackBoardDumpReader(bool) : Component(true) { 
    registerUnDumper<cv::Rect>(&rectReaderFunction);
    registerUnDumper<cv::Point>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        int32_t i[2];
        w->dumbRead(ifs, i);
        return cv::Point(i[0], i[1]);
      });
    registerUnDumper<cv::Point2f>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        float f[2];
        w->dumbRead(ifs, f);
        return cv::Point2f(f[0], f[1]);
      });

    registerUnDumper<cv::Point2d>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        double d[2];
        w->dumbRead(ifs, d);
        return cv::Point2d(d[0], d[1]);
      });

    registerUnDumper<cv::Mat>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        int32_t t[3];
        w->dumbRead(ifs, t);
        cv::Mat m(t[2], t[1], t[0]);
        ifs.read(reinterpret_cast<char*>(m.data), 
                 m.total()*m.elemSize());
        return m;
      });

    registerUnDumper<std::string>([](BlackBoardDumpReader* const w, std::istream& ifs)->string {
        // format:
        // [uint64_t][char-array]
        //     |          |
        //     |          +--- content (NO terminating zero)
        //     +--- number of chars
        uint64_t len;
        w->dumbRead(ifs, len);
        vector<char> array(len+1);
        ifs.read(&array[0], len);
        array[len] = '\0';
        return &array[0];
      });
  }

  std::unordered_map<size_t, std::shared_ptr<BlackBoardDumpReader::UnDumperBase> >* BlackBoardDumpReader::undumpFunctions = nullptr;
  std::unordered_map<size_t, boost::any(*)(BlackBoardDumpReader* const, std::istream& )>* BlackBoardDumpReader::anyUndumpFunctions = nullptr;

  void BlackBoardDumpReader::process(frame_number_t) {
    assert(false && "NOT SUPPORTED");
  }



  Component* BlackBoardDumpReader::createComponentImpl(const boost::program_options::variables_map& opts, BlackBoard* blackBoard, FrameSource* frameSource) const {
    BlackBoardDumpReader b(blackBoard, frameSource);
    if (opts.count("in-dump-file"))
      parseCommaSeparatedStringToList(opts["in-dump-file"].as<std::string>(),b.filenames);
    //      b.filename = opts["in-dump-file"].as<std::string>();
    return new BlackBoardDumpReader(b);
  }



  boost::program_options::options_description BlackBoardDumpReader::getCommandLineOptionsDescription() const {
    boost::program_options::options_description opts("BlackBoard Dump Reader options");
    opts.add_options()("in-dump-file", 
                       boost::program_options::value<std::string>(),
                       "comma separated list of files to read the dumped black board contents from\nfiles ending in \".gz\" are handled as gzip compressed files");
    return opts;
  }



  boost::any BlackBoardDumpReader::undump(std::istream& ifs, uint64_t dataType) {
    if (anyUndumpFunctions && anyUndumpFunctions->count(dataType))
      return (*anyUndumpFunctions)[dataType](this, ifs);
    else if (undumpFunctions && undumpFunctions->count(dataType))
      return (*undumpFunctions)[dataType]->undumpAny(this, ifs);

    throw IOException(std::string("Cannot undump data with this hash code: ")
                      + boost::lexical_cast<std::string>(dataType) + 
                      " (type name: " + 
                      slmotion::hashCodeToTypeName((size_t)dataType) + ")");
    assert(false);
    return boost::any();
  }

  static void open(const std::string& filename, std::ifstream& ifs) {
    if (debug > 1) 
      std::cerr << "BlackBoard Dump Reader:" << std::endl
                << "Opening input file: \"" << filename << "\"...";

    ifs.open(filename);

    if (!ifs.good())
      throw IOException("Could not open \"" + filename + "\" for reading!");

    if (debug > 1)
      std::cerr << "OK!" << std::endl;
  }

  static void open(const std::string& filename, igzstream& ifs) {
    if (debug > 1) 
      std::cerr << "BlackBoard Dump Reader:" << std::endl
                << "gzOpening input file: \"" << filename << "\"...";

    ifs.open(filename.c_str());

    if (!ifs.good())
      throw IOException("Could not open \"" + filename + "\" for reading!");

    if (debug > 1)
      std::cerr << "OK!" << std::endl;
  }

  std::vector<DumpEntry> BlackBoardDumpReader::readHeaders(std::istream& ifs) const {
    if (debug > 1)
      std::cerr << "Reading header..." << std::endl;
    uint64_t totalSize = 0;
    ifs.read(reinterpret_cast<char*>(&totalSize), sizeof(totalSize));
    // dumbRead(ifs, totalSize);

    if (debug > 1)
      std::cerr << "Total number of entries: " << totalSize << std::endl
                << "Reading header entries..." << std::endl;
    std::vector<DumpEntry> entryHeaders(totalSize);
    // for (unsigned int j = 0; j < totalSize; ++j)
    //   BlackBoardDumpReader::dumbRead(ifs, entryHeaders[j]);
    for (unsigned int j = 0; j < totalSize; ++j)
      ifs.read(reinterpret_cast<char*>(&entryHeaders[j]), sizeof(DumpEntry));

    return entryHeaders;
  }

  bool BlackBoardDumpReader::processRangeImplementation(frame_number_t,
                                                        frame_number_t,
                                                        UiCallback* callback) {


    for(auto fnit=filenames.begin();fnit!=filenames.end();fnit++){
      std::ifstream ifs;
      igzstream igzs;

      bool input_file_compressed=false;

      std::istream *isptr=&ifs;

      if(fnit->substr(fnit->size()-3)==".gz"){
	open(*fnit,igzs);
	isptr=&igzs;
	input_file_compressed=true;
      } else{
	open(*fnit, ifs);
      }


      std::vector<DumpEntry> entryHeaders(readHeaders(*isptr));

      if (debug > 1) 
	std::cerr << "Reading data..." << std::endl;

      unsigned int i = 1;
      size_t totalSize = entryHeaders.size();
      for (auto it = entryHeaders.cbegin(); it != entryHeaders.cend(); ++it) {
	if (debug > 1) {
	  std::cerr << "Reading " << it->key;
	  if (it->frameNumber < uint64_t(-1))
	    std::cerr << '[' << it->frameNumber << ']';
	  std::cerr << '(' << i << "/" << totalSize  << ')' << std::endl;
	}

	if(!input_file_compressed){
	  // bypass this check for compressed input files
	  if (isptr->tellg() != static_cast<int64_t>(it->dataPointer))
	    throw IOException("Fatal error: data pointer mismatch.");
	}

	if (it->frameNumber < uint64_t(-1))
	  getBlackBoard().set(it->frameNumber, it->key, 
			      undump(*isptr, it->dataType));
	else
	  getBlackBoard().set(it->key, undump(*isptr, it->dataType));

        if (callback)
          (*callback)(i*100./totalSize);

        ++i;
      }


      oldToNewSharedPtrMap.clear();
    }
    
    return true;
  }



  Component::property_set_t BlackBoardDumpReader::getProvided() const {
    property_set_t properties;

    for(auto fit=filenames.begin();fit!=filenames.end();fit++){
      if (fit->length() == 0)
	continue;

      std::ifstream ifs;

      igzstream igzs;

      std::istream *isptr=&ifs;

      if(fit->substr(fit->size()-3)==".gz"){
	open(*fit,igzs);
	isptr=&igzs;
      } else{
	open(*fit, ifs);
      }

      std::vector<DumpEntry> entryHeaders(readHeaders(*isptr));

      for (auto it = entryHeaders.cbegin(); it != entryHeaders.cend(); ++it)
	properties.insert(it->key);
    }

    return properties;
  }
}
