#ifndef SLMOTION_BLACKBOARD_DUMP_WRITER
#define SLMOTION_BLACKBOARD_DUMP_WRITER

#include "Component.hpp"
#include <iostream>
#include "BlackBoardDumpReader.hpp"
#include "Asm.hpp"
#include "util.hpp"

namespace slmotion {
  /**
   * This component dumps all black board data into a file.
   *
   * The file format is as follows:
   * [header][dataentries]
   *
   * The header is stored as follows:
   *
   * [uint64t][entries...]
   *     |
   *     +---Number of entries
   *
   * NEW FORMAT:
   *
   * Each entry is stored sequentially as follows:
   * [uint64t][uint64t][char[32]][uint64t]
   *    |        |        |        |
   *    |        |        |        +---Byte position for the type specific
   *    |        |        |            data representation (after the header)
   *    |        |        +---Key string
   *    |        +---Data type (type_info hash)
   *    +---Frame number (or 2^64-1 for global board entries)
   *
   * Data entries are stored sequentially in a type-specific format.
   *
   * OLD FORMAT:
   * Each entry is stored sequentially as follows:
   * [uint64t][uint8t][char[32]][uint64t]
   *    |        |       |        |
   *    |        |       |        +---Byte position for the type specific
   *    |        |       |            data representation (after the header)
   *    |        |       +---Key string
   *    |        +---Data type (enum)
   *    +---Frame number (or 2^64-1 for global board entries)
   *
   * Data entries are stored sequentially in a type-specific format.
   */
  class BlackBoardDumpWriter : public Component {
  public:
    BlackBoardDumpWriter(bool);

    BlackBoardDumpWriter(BlackBoard* blackBoard, FrameSource* frameSource) :
      Component(blackBoard, frameSource) { }

    virtual void process(frame_number_t frameNumber);

    virtual std::string getShortDescription() const {
      return "Dumps blackboard data to a file";
    }

    virtual std::string getLongDescription() const {
      return "Dumps blackboard data to a file";
    }

    virtual property_set_t getRequirements() const {
      return property_set_t();
    }

    virtual property_set_t getProvided() const {
      return property_set_t();
    }

    virtual std::string getShortName() const {
      return "BlackBoardDumpWriter";
    }

    virtual std::string getComponentName() const {
      return "BlackBoard Dump Writer";
    }

    virtual void reset() {
    }

    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const {
      return boost::program_options::options_description();
    }

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const;

    inline void setFilename(const std::string& f) {
      filename = f;
    }
    
    /**
     * Registers a writer function that takes in a boost::any object,
     * and then serialises it into the ostream. The template parameter
     * controls the type it is registered for.
     */
    template<typename T>
    static void registerAnyWriter() {
      if (!anyWriterFunctions)
        anyWriterFunctions = new std::unordered_map<size_t, std::shared_ptr<AnyDumbWriterBase> >();

      if (!anyWriterFunctions->count(typeid(T).hash_code())) {
        (*anyWriterFunctions)[typeid(T).hash_code()] = std::shared_ptr<AnyDumbWriterBase>(new AnyDumbWriter<T>());
        registerAnySizeComputer<T>();
        BlackBoardDumpReader::registerAnyUnDumper<T>();
        addHashCodeToTypeName<T>();
      }
    }

    template<typename T>
    static void registerAnySizeComputer() {
      if (!anySizeComputers)
        anySizeComputers = new std::unordered_map<size_t, size_t(*)(const BlackBoardDumpWriter* const, 
                                                                    const boost::any&) >();

      if (!anySizeComputers->count(typeid(T).hash_code())) {
        (*anySizeComputers)[typeid(T).hash_code()] = [](const BlackBoardDumpWriter * const w, 
                                                        const boost::any& any)->size_t {
          return w->getSize(boost::any_cast<T>(any));
        };
      }
    }

    /**
     * Registers a "dumb writer" which writes a known custom type to
     * an output stream.
     */
    template<typename T>
    static void registerDumbWriter(void(*dumbWriterFunction)(std::ostream&, const T&)) {
      if (!dumbWriterFunctions)
        dumbWriterFunctions = new std::unordered_map<size_t,std::shared_ptr<DumbWriterBase> >();

      if (dumbWriterFunctions->count(typeid(T).hash_code())) {
        fprintf(stderr, "Fatal error: The type \"%s\" already has a writer "
                "associated to it!\n", string_demangle(typeid(T).name()).c_str());
        abort();
      }

      (*dumbWriterFunctions)[typeid(T).hash_code()] = std::shared_ptr<DumbWriterBase>(new SimpleDumbWriter<T>(dumbWriterFunction));

      addHashCodeToTypeName<T>();
    }

    /**
     * Registers a "dumb writer" which writes a known custom type to
     * an output stream.
     */
    template<typename T>
    static void registerDumbWriter(void(*dumbWriterFunction)(BlackBoardDumpWriter* const,
                                                             std::ostream&, const T&)) {
      if (!dumbWriterFunctions)
        dumbWriterFunctions = new std::unordered_map<size_t,std::shared_ptr<DumbWriterBase> >();

      if (dumbWriterFunctions->count(typeid(T).hash_code())) {
        fprintf(stderr, "Fatal error: The type \"%s\" already has a writer "
                "associated to it!\n", string_demangle(typeid(T).name()).c_str());
        abort();
      }

      class DumbWriter : public DumbWriterMidBase<T> {
      public:
        typedef void(*writer_function_t)(BlackBoardDumpWriter* const, std::ostream&, const T&);
        
        DumbWriter(writer_function_t writerFunction) : 
          function(writerFunction) {}
        
        void write(BlackBoardDumpWriter* const w, std::ostream& ofs, const T& data) {
          function(w, ofs, data);
      }
        
        virtual ~DumbWriter() { }
        
      private:
        writer_function_t function;
      };
      (*dumbWriterFunctions)[typeid(T).hash_code()] = std::shared_ptr<DumbWriterBase>(new DumbWriter(dumbWriterFunction));
    }



    /**
     * Simply writes the data in a dumb manner into the stream
     */
    template<typename T>
    void dumbWrite(std::ostream& ofs, const T& data) {
      DumbWriter<T>::write(this, ofs, data);
    }



    /**
     * Registers a size computation function for the complex type
     */
    template<typename T>
    static void registerSizeComputer(size_t(*sizeFunction)(const T&)) {
      if (!sizeComputers)
        sizeComputers = new std::unordered_map<size_t, std::shared_ptr<SizeComputerBase> >();

      if (sizeComputers->count(typeid(T).hash_code())) {
        fprintf(stderr, "Fatal error: The type \"%s\" already has a size "
                "computation function associated to it!\n", 
                string_demangle(typeid(T).name()).c_str());
        abort();
      }

      (*sizeComputers)[typeid(T).hash_code()] = std::shared_ptr<SizeComputerBase>(new SimpleSizeComputer<T>(sizeFunction));
    }

    /**
     * Registers a size computation function for the complex type
     */
    template<typename T>
    static void registerSizeComputer(size_t(*sizeFunction)(const BlackBoardDumpWriter* const, const T&)) {
      if (!sizeComputers)
        sizeComputers = new std::unordered_map<size_t, std::shared_ptr<SizeComputerBase> >();

      if (sizeComputers->count(typeid(T).hash_code())) {
        fprintf(stderr, "Fatal error: The type \"%s\" already has a size "
                "computation function associated to it!\n", 
                string_demangle(typeid(T).name()).c_str());
        abort();
      }

      class SizeComputer : public SizeComputerMidBase<T> {
      public:
        typedef size_t(*size_function)(const BlackBoardDumpWriter* const, const T&);
        SizeComputer(size_function func) : func(func) {}

        size_t getSize(const BlackBoardDumpWriter* const w, const T& t) {
          return func(w, t);
        }
      private:
        size_function func;
      };
      (*sizeComputers)[typeid(T).hash_code()] = std::shared_ptr<SizeComputerBase>(new SizeComputer(sizeFunction));
    }



    /**
     * Registers a function that will tell which pointers would be
     * stored if the object were to be written to a file. This is
     * mainly useful with conditional object storage. If the pointer
     * is going to be stored anyway, call to this function is unnecessary.
     */
    template<typename T>
    static void registerConditionalStoredPointersFunction(std::set<const void*>(*func)(BlackBoardDumpWriter const * const, 
                                                                                       const T&)) {
      if (!pointerStoreFunctions)
        pointerStoreFunctions = new std::unordered_map<size_t, std::shared_ptr<BlackBoardDumpWriter::AnyPointerSetProviderBase> > ();

      if (pointerStoreFunctions->count(typeid(T).hash_code())) {
        fprintf(stderr, "Fatal error: The type \"%s\" already has a conditional"
                " pointer storage function associated to it!\n", 
                string_demangle(typeid(T).name()).c_str());
        abort();
      }
      
      (*pointerStoreFunctions)[typeid(T).hash_code()] = std::shared_ptr<AnyPointerSetProviderBase >(new AnyPointerSetProvider<T>(func));
    }



    template<typename T, bool IS_POD = std::is_pod<T>::value>
    class DefaultSizeComputer {
    public:
      static size_t getSize();
    };

    template<typename T>
    size_t getSize(const T& data) const {
      if (sizeComputers->count(typeid(T).hash_code()))
        return dynamic_cast<SizeComputerMidBase<T>*>((*sizeComputers)[typeid(T).hash_code()].get())->getSize(this, data);
      else if (!std::is_pod<T>::value)
        throw IOException("Error: " + string_demangle(typeid(T).name()) + " is "
                          "not a POD type and there is no size computation "
                          "function / object associated with it!");
      return DefaultSizeComputer<T>::getSize();
    }



    /**
     * Returns true if the pointer has already been stored to the file
     */
    inline bool hasStoredPointer(const void* ptr) const {
      return storedPointers.count(ptr);
    }



    template<typename T>
    void dumbWrite(std::ostream& ofs, const T* data) {
      storedPointers.insert(data);
      uint64_t ptr = reinterpret_cast<uintptr_t>(data);
      dumbWrite<uint64_t>(ofs, ptr);
      dumbWrite(ofs, *data);
    }

    template<typename T>
    void dumbWrite(std::ostream& ofs, const std::shared_ptr<T>& data) {
      dumbWrite(ofs, data.get());
    }

    // std::vectors are stored as follows:
    // [uint64_t][uint64_t][data]
    //     |         |        |
    //     |         |        +---Each element dumped as usual
    //     |         +---Number of elements
    //     +---Size of data (not necessarily proportional to the number of
    //         elements)
    template<typename T>
    void dumbWrite(std::ostream& ofs, const std::vector<T>& data) {
      uint64_t r = getSize(data);
      ofs.write(reinterpret_cast<const char*>(&r), sizeof(r));
      r = data.size();
      ofs.write(reinterpret_cast<const char*>(&r), sizeof(r));
      for (auto it = data.cbegin(); it != data.cend(); ++it)
        dumbWrite(ofs, *it);
    }

    // std::lists are stored as follows:
    // [uint64_t][uint64_t][data]
    //     |         |        |
    //     |         |        +---Each element dumped as usual
    //     |         +---Number of elements
    //     +---Size of data (not necessarily proportional to the number of
    //         elements)
    template<typename T>
    void dumbWrite(std::ostream& ofs, const std::list<T>& data) {
      uint64_t r = getSize(data);
      ofs.write(reinterpret_cast<const char*>(&r), sizeof(r));
      r = data.size();
      ofs.write(reinterpret_cast<const char*>(&r), sizeof(r));
      for (auto it = data.cbegin(); it != data.cend(); ++it)
        dumbWrite(ofs, *it);
    }

    // std::sets are stored as follows:
    // [uint64_t][uint64_t][data]
    //     |         |        |
    //     |         |        +---Each element dumped as usual
    //     |         +---Number of elements
    //     +---Size of data (not necessarily proportional to the number of
    //         elements)
    template<typename T>
    void dumbWrite(std::ostream& ofs, const std::set<T>& data) {
      uint64_t r = getSize(data);
      ofs.write(reinterpret_cast<const char*>(&r), sizeof(r));
      r = data.size();
      ofs.write(reinterpret_cast<const char*>(&r), sizeof(r));
      for (auto it = data.cbegin(); it != data.cend(); ++it)
        dumbWrite(ofs, *it);
    }

    // std::maps are stored as follows:
    // [uint64_t][uint64_t]([key_i][value_i])*
    //     |         |        |
    //     |         |        +---Each (k,v) pair written after one another 
    //     |         +---Number of elements
    //     +---Size of data (not necessarily proportional to the number of
    //         elements)
    template<typename K, typename V>
    void dumbWrite(std::ostream& ofs, const std::map<K,V>& data) {
      uint64_t r = getSize(data);
      ofs.write(reinterpret_cast<const char*>(&r), sizeof(r));
      r = data.size();
      ofs.write(reinterpret_cast<const char*>(&r), sizeof(r));
      for (auto it = data.cbegin(); it != data.cend(); ++it){
        dumbWrite(ofs, it->first);
        dumbWrite(ofs, it->second);
      }
    }



    template<typename T>
    size_t getSize(const T* object) const {
      // the pointer itself is also stored as uint64_t
      return sizeof(uint64_t) + getSize(*object);
    }

    template<typename T>
    size_t getSize(const std::shared_ptr<T>& object) const {
      return getSize(object.get());
    }

    template<typename T>
    size_t getSize(const std::vector<T>& v) const {
      // format: [uint64_t][uint64_t][data]
      size_t size = 0;
      for (auto it = v.cbegin(); it != v.cend(); ++it)
        size += getSize(*it);
      return 2*sizeof(uint64_t) + size;
    }

    template<typename T>
    size_t getSize(const std::list<T>& v) const {
      // format: [uint64_t][uint64_t][data]
      size_t size = 0;
      for (auto it = v.cbegin(); it != v.cend(); ++it)
        size += getSize(*it);
      return 2*sizeof(uint64_t) + size;
    }
    
    template<typename T>
    size_t getSize(const std::set<T>& v) const {
      // format: [uint64_t][uint64_t][data]
      size_t size = 0;
      for (auto it = v.cbegin(); it != v.cend(); ++it)
        size += getSize(*it);
      return 2*sizeof(uint64_t) + size;
    }

    template<typename K, typename V>
    size_t getSize(const std::map<K,V>& v) const {
      // format: [uint64_t][uint64_t][data]
      // [data] is written as [key1][value1][key2][value2]...
      size_t size = 0;
      for (auto it = v.cbegin(); it != v.cend(); ++it){
        size += getSize(it->first);
	size += getSize(it->second);
      }
      return 2*sizeof(uint64_t) + size;
    }



  private:
    class DumbWriterBase {
    public:
      // virtual void write(BlackBoardDumpWriter* const w, std::ostream& ofs, const T& data) = 0;
      // every subclass should provide an implementation for this but it is 
      // impossible to have a virtual template function
      virtual ~DumbWriterBase() { }
    };

    template<typename T>
    class DumbWriterMidBase : public DumbWriterBase {
    public:
      // this class is here only to provide this virtual function
      virtual void write(BlackBoardDumpWriter* const w, std::ostream& ofs, const T& data) = 0;
      virtual ~DumbWriterMidBase() { }
    };

    template<typename T> 
    class SimpleDumbWriter : public DumbWriterMidBase<T> {
    public:
      typedef void(*writer_function_t)(std::ostream&, const T&);

      SimpleDumbWriter(writer_function_t writerFunction) : 
        function(writerFunction) {}

      void write(BlackBoardDumpWriter* const, std::ostream& ofs, const T& data) {
        function(ofs, data);
      }

      virtual ~SimpleDumbWriter() { }

    private:
      writer_function_t function;
    };    

    class AnyDumbWriterBase : public DumbWriterBase {
    public:
      virtual void write(BlackBoardDumpWriter* const w, std::ostream& ofs, const boost::any& data) = 0;

      virtual ~AnyDumbWriterBase() { }
    };    

    template<typename T>
    class AnyDumbWriter : public AnyDumbWriterBase {
    public:
      void write(BlackBoardDumpWriter* const w, std::ostream& ofs, const boost::any& data) {
        w->dumbWrite<T>(ofs, data);
      }

      virtual ~AnyDumbWriter() { }
    };    

    // a writer struct for partial template specialisation
    template<typename T, bool IS_POD = std::is_pod<T>::value>
    struct DumbWriter {
      static void write(BlackBoardDumpWriter* const w, std::ostream& ofs, const T& data);
    };

    // a wrapper around the user-supplied function
    class AnyPointerSetProviderBase {
    public:
      virtual std::set<const void*> operator()(BlackBoardDumpWriter const * const w, const boost::any& any) = 0;
      virtual ~AnyPointerSetProviderBase() {}
    };

    template<typename T>
    class AnyPointerSetProvider : public AnyPointerSetProviderBase {
    public:
      explicit AnyPointerSetProvider(std::set<const void*>(*func)(BlackBoardDumpWriter const * const, const T&)) :
      func(func) {}

      std::set<const void*> operator()(BlackBoardDumpWriter const * const w, const boost::any& any) {
        return func(w, boost::any_cast<T>(any));
      }

    private:
      std::set<const void*>(*func)(BlackBoardDumpWriter const * const, const T&);
    };



    // these compute data sizes
    class SizeComputerBase {
    public:
      virtual size_t getSize(const BlackBoardDumpWriter* const w, const boost::any& any) = 0;
      virtual ~SizeComputerBase() {}
    };

    template<typename T>
    class SizeComputerMidBase : public SizeComputerBase {
    public:
      virtual size_t getSize(const BlackBoardDumpWriter* const w, const boost::any& any) {
        const T& temp = boost::any_cast<const T>(any);
        return getSize(w, temp);
      }

      virtual size_t getSize(const BlackBoardDumpWriter* const w, const T& data) = 0;
    };

    // for partial specialization
    template<typename T>
    class SimpleSizeComputer : public SizeComputerMidBase<T> {
    public:
      SimpleSizeComputer(size_t(*function)(const T& data)) :
      sizeFunction(function) {}
      
      virtual size_t getSize(const BlackBoardDumpWriter* const, const T& data) {
        return sizeFunction(data);
      }

    private:
      size_t (*sizeFunction)(const T& data);
    };

    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;

    /**
     * Dumps the content to a file
     */
    virtual bool processRangeImplementation(frame_number_t first, 
                                            frame_number_t last, 
                                            UiCallback* uiCallback);

    void dump(std::ostream& ofs, const boost::any& data);
    
    /**
     * Simply writes the data in a dumb manner into the stream
     */
    template<typename T>
    void dumbWrite(std::ostream& ofs, const boost::any& data) {
      dumbWrite(ofs, boost::any_cast<const T&>(data));
    }

    /**
     * POD-type specialization
     */ 
    template<typename T>
    void dumbWritePod(std::ostream& ofs, const T& data) {
      static_assert(std::is_pod<T>::value, 
                    "Only POD types can be dumbly dumped!");
      ofs.write(reinterpret_cast<const char*>(&data), sizeof(data));
    }

    template<typename T>
    void dumbWriteCustom(std::ostream& ofs, const T& data) {
      if (dumbWriterFunctions->count(typeid(T).hash_code()))
        dynamic_cast<DumbWriterMidBase<T>*>((*dumbWriterFunctions)[typeid(T).hash_code()].get())->write(this, ofs, data);
      else
        throw IOException("No writer function is registered for " + string_demangle(typeid(T).name()));
    }



    /**
     * Constructs a dump entry. The data pointer is incremented the required
     * amount (should start at zero'ish). If necessary, a pointer is stored
     * to avoid double-storing of PDMs and such
     */
  DumpEntry constructDumpEntry(uint64_t frameNumber,
                                 const std::string& key,
                                 uint64_t& dataPointer,
                                 const boost::any& data);


    
    // map typehash --> writer function
    static std::unordered_map<size_t, std::shared_ptr<AnyDumbWriterBase> >* anyWriterFunctions;
    static std::unordered_map<size_t, std::shared_ptr<DumbWriterBase> >* dumbWriterFunctions;
    static std::unordered_map<size_t, std::shared_ptr<SizeComputerBase> >* sizeComputers;
    static std::unordered_map<size_t, size_t(*)(const BlackBoardDumpWriter* const,
                                                const boost::any&)>* anySizeComputers;
    static std::unordered_map<size_t, std::shared_ptr<AnyPointerSetProviderBase> >* pointerStoreFunctions;

    // this is here to just free the variable above. do not use for anything else
    static struct Foo {
      ~Foo() {
        delete anyWriterFunctions;
        delete dumbWriterFunctions;
        delete sizeComputers;
        delete anySizeComputers;
        delete pointerStoreFunctions;
      }
    } anyWriterFunctionsDeleter;



     /**
     * Dump filename
     */
    std::string filename;

    /**
     * Pointers to (shared) pointer objects, members of some black board 
     * objects or possibly black board objects themselves, will be stored 
     * here.
     */
    std::set<const void*> storedPointers;
  };

  template<typename T>
  struct BlackBoardDumpWriter::DumbWriter<T,true> {
    static void write(BlackBoardDumpWriter* w, std::ostream& ofs, const T& data) {
      w->dumbWritePod(ofs, data);      
    }
  };

  template<typename T>
  struct BlackBoardDumpWriter::DumbWriter<T,false> {
    static void write(BlackBoardDumpWriter* w, std::ostream& ofs, const T& data) {
      w->dumbWriteCustom(ofs, data);      
    }
  };

  template<typename T>
  class BlackBoardDumpWriter::DefaultSizeComputer<T, true> {
  public:
    static size_t getSize() {
      static_assert(std::is_pod<T>::value,
                    "Only POD types can be dealt with this way");
      return sizeof(T);
    }
  };

  template<typename T>
  class BlackBoardDumpWriter::DefaultSizeComputer<T, false> {
  public:
    static size_t getSize() {
      throw IOException("Default size computation is only applicable on POD types");
      return 0;
    }
  };

  template<>
  class BlackBoardDumpWriter::SizeComputerMidBase<boost::any> : public BlackBoardDumpWriter::SizeComputerBase {
  public:
    virtual size_t getSize(const BlackBoardDumpWriter* const w, const boost::any& data) = 0;
  };
}

#endif

