#include "BlackBoardEntry.hpp"
#include "BlackBoardPointer.hpp"
#include "opencv.hpp"
extern "C" {
#include <zlib.h>
}

namespace slmotion {
  /**
   * Compressed the matrix data. PLEASE NOTE THAT THIS WILL MAKE THE
   * MATRIX INVALID. It must be decompresed before any other
   * operations are attempted on it.
   *
   * @return True on success.
   */
  static bool compressMatrix(cv::Mat& matrix,
                             std::vector<unsigned char>& compressedData) {
    if (!matrix.isContinuous())
      return false;

    uLong sourceLen = matrix.elemSize();
    if (matrix.dims == 2)
      sourceLen *= matrix.cols * matrix.rows;
    else
      for (int d = 0; d < matrix.dims; ++d)
        sourceLen *= matrix.size[d];
    uLong bound = compressBound(sourceLen);
    std::vector<unsigned char> temp(bound);
    uLong actualSize = bound;
    int status = compress(&temp[0], &actualSize, matrix.data, sourceLen);
    if (status == Z_OK) {
      compressedData = std::vector<unsigned char>(temp.cbegin(), 
                                                  temp.cbegin() + actualSize);
      if (matrix.dims == 2)
        matrix = cv::Mat(matrix.size(), matrix.type(), nullptr);
      else
        matrix = cv::Mat(matrix.dims, matrix.size, matrix.type(), nullptr);
      return true;
    }
    else if (status == Z_MEM_ERROR)
      throw BlackBoardException("Compression failed: not enough memory.");
    else if (status == Z_BUF_ERROR)
      throw BlackBoardException("Compression failed: There was not enough room in the output buffer.");
    
    return false;
  }

  static bool decompressMatrix(cv::Mat& matrix,
                               std::vector<unsigned char>& compressedData) {
    if (matrix.dims == 2)
      matrix = cv::Mat(matrix.size(), matrix.type());
    else
      matrix = cv::Mat(matrix.dims, matrix.size, matrix.type());

    uLong uncompressedSize = matrix.elemSize();
    if (matrix.dims == 2)
      uncompressedSize *= matrix.cols * matrix.rows;
    else 
      for (int d = 0; d < matrix.dims; ++d)
        uncompressedSize *= matrix.size[d];
    unsigned long desiredSize = uncompressedSize; // uncompressed size is overwritten by uncompress()

    int status = uncompress(matrix.data, &uncompressedSize,
                            &compressedData[0], compressedData.size());

    if (status == Z_OK) {
      if (uncompressedSize == desiredSize) {
        compressedData = std::vector<unsigned char>();
        return true;
      }
      else
        throw BlackBoardException("Decompressing the matrix resulted in a wrong-sized matrix!");
    }
    else if (status == Z_MEM_ERROR)
      throw BlackBoardException("Decompression failed: not enough memory.");
    else if (status == Z_BUF_ERROR)
      throw BlackBoardException("Decompression failed: There was not enough room in the output buffer.");
    else
      throw BlackBoardException("Failed to decompress a matrix (unknown error)!");
    
    return false;
  }

  bool BlackBoardEntry::decompress() {
    if (status == COMPRESSED) {
      if (value->type() == typeid(cv::Mat)) {
        if (decompressMatrix(boost::any_cast<cv::Mat&>(*value), 
                             compressedData)) {
          status = UNCOMPRESSED;
          return true;
        }
        else
          return false;
      }
      else
        throw BlackBoardException("Cannot decompress a value of type " +
                                  string_demangle(value->type().name()));
    }

    return false;
  }

  bool BlackBoardEntry::compressIfPossible() {
    if (attribs == COMPRESS_WITHOUT_REFERENCES && refCount == 0 && 
        status == UNCOMPRESSED) {
      if (value->type() == typeid(cv::Mat)) {
        if (compressMatrix(boost::any_cast<cv::Mat&>(*value),
                           compressedData)) {
          status = COMPRESSED;
          return true;
        }
      }
    }
    
    return false;
  }



  void BlackBoardEntry::unReference() {
    if (refCount > 0)
      --refCount;

    if (refCount == 0 && status == UNCOMPRESSED && 
        attribs == COMPRESS_WITHOUT_REFERENCES)
      compressIfPossible();
  }



  BlackBoardPointer<boost::any> BlackBoardEntry::referenceAny(std::shared_ptr<BlackBoardEntry> thisPtr) {
    if (status == COMPRESSED)
      if (!decompress())
        throw BlackBoardException("Decompression failed!");
    
    ++refCount;
    return BlackBoardPointer<boost::any>(value.get(), thisPtr);
  }
}
