#ifndef SLMOTION_BLACKBOARD_POINTER
#define SLMOTION_BLACKBOARD_POINTER

#include "BlackBoardEntry.hpp"

namespace slmotion {
  /**
   * This class represents a shared_ptr-style smart pointer to an
   * entry on the black board
   */
  template<typename T>
  class BlackBoardPointer {
  public:
    /**
     * Initialises a null-pointer
     */
    BlackBoardPointer() : value(nullptr), parent(nullptr) {}

    BlackBoardPointer(T* v, std::shared_ptr<BlackBoardEntry> parent) :
      value(v), parent(parent) {

    }

    BlackBoardPointer(const BlackBoardPointer<T>& other) :
      value(nullptr), parent(nullptr)
    {
      BlackBoardPointer<T> temp(other.parent->template reference<T>(other.parent));
      std::swap(value,temp.value);
      std::swap(parent,temp.parent);
    }

    BlackBoardPointer<T>& operator=(const BlackBoardPointer<T>& other) {
      if (this != &other) {
        BlackBoardPointer<T> temp(other.parent->template reference<T>(other.parent));
        std::swap(value,temp.value);
        std::swap(parent,temp.parent);
      }
      return *this;
    }

    ~BlackBoardPointer() {
      if (parent)
        parent->unReference();
    }


    // standard dereference operators
    
    T* operator->() {
      return value;
    }

    T& operator*() {
      return *value;
    }

    const T* operator->() const {
      return value;
    }

    const T& operator*() const {
      return *value;
    }

  private:
    T* value;
    std::shared_ptr<BlackBoardEntry> parent;
  };



  // this specialisation is here to prevent funny things from
  // happening in the case we're dealing with any pointers
  template<>
  class BlackBoardPointer<boost::any> {
  public:
    // take care of the special case that the pointer is an any pointer
    BlackBoardPointer(const BlackBoardPointer<boost::any>& other) :
      value(nullptr), parent(nullptr)
    {
      BlackBoardPointer<boost::any> temp(other.parent->value->type() == typeid(boost::any) ?
                                         other.parent->reference<boost::any>(other.parent) :
                                         other.parent->referenceAny(other.parent));
      std::swap(value,temp.value);
      std::swap(parent,temp.parent);
    }

    BlackBoardPointer() : value(nullptr), parent(nullptr) {}

    BlackBoardPointer(boost::any* v, std::shared_ptr<BlackBoardEntry> parent) :
      value(v), parent(parent) { }

    BlackBoardPointer<boost::any>& operator=(const BlackBoardPointer<boost::any>& other) {
      if (this != &other) {
        BlackBoardPointer<boost::any> temp(other.parent->value->type() == typeid(boost::any) ?
                                           other.parent->reference<boost::any>(other.parent) :
                                           other.parent->referenceAny(other.parent));
        std::swap(value,temp.value);
        std::swap(parent,temp.parent);
      }
      return *this;
    }

    ~BlackBoardPointer() {
      if (parent)
        parent->unReference();
    }


    // standard dereference operators
    
    boost::any* operator->() {
      return value;
    }

    boost::any& operator*() {
      return *value;
    }

    const boost::any* operator->() const {
      return value;
    }

    const boost::any& operator*() const {
      return *value;
    }

  private:
    boost::any* value;
    std::shared_ptr<BlackBoardEntry> parent;
  };
}

#endif
