#ifndef SLMOTION_BLOB
#define SLMOTION_BLOB

#include <vector>

#include "opencv.hpp"



namespace slmotion {
  /**
   * A class that represents binary blobs in a binary image
   */
  class Blob {
  public:
    typedef std::vector<cv::Point> PointContainerType;



    const static cv::Point INVALID_POINT;



    /**
     * Reads a binary image, and returns a vector of blob objects
     *
     * @param mask A binary mask
     * @param minSize Minimum size for a blob
     * @param minDistance Minimum distance between blobs
     * @param maxBlobs Maximum number of blobs (largest ones will be 
     * returned)
     *
     * @return A vector of Blobs that meet the requirements
     */
    static std::vector<Blob> extractBlobsFromMask(const cv::Mat& mask, 
                                                  size_t minSize,
                                                  int maxBlobs);



    /**
     * The default constructor. Constructs an empty blob.
     */
    Blob() :
      centroid(INVALID_POINT),
      leftMost(INVALID_POINT),
      rightMost(INVALID_POINT),
      topMost(INVALID_POINT),
      bottomMost(INVALID_POINT),
      blobSize(0)
    {}



    /**
     * Returns the bottommost point
     */
    inline cv::Point getBottomMost() const {
      return bottomMost;
    }



    /**
     * Returns the topmost point
     */
    inline cv::Point getTopMost() const {
      return topMost;
    }



    /**
     * Returns the topmost point
     */
    inline cv::Point getLeftMost() const {
      return leftMost;
    }



    /**
     * Returns the topmost point
     */
    inline cv::Point getRightMost() const {
      return rightMost;
    }



    /**
     * Converts the blob to a binary matrix of type CV_8UC1 and desired size
     * where elements corresponding to blob pixels are non-zero.
     *
     * @param size Desired size. The size should be large enough to 
     * accomodate even extreme points.
     * @param invert By default, blob pixels are marked 255 and non-blob 
     * pixels 0. If invert is true, the values will be inverted.
     * @param translation Move pixels somewehere before drawing
     *
     * @return A new matrix
     */
    cv::Mat toMatrix(const cv::Size& size, 
                     bool invert = false,
                     const cv::Point& translation = cv::Point()) const;



    /**
     * A getter for the centroid point
     *
     * @return A reference to the centroid point
     */
    inline cv::Point getCentroid() const {
      return centroid;
    }



    /**
     * Returns the bounding box for the blob
     */
    cv::Rect getBoundingBox() const;



    /**
     * Removes any points that are not contained within the box.
     *
     * If the blob is split into several disjoint parts as a result, only
     * the largest part is preserved.
     *
     * @param box Limiter rectangle
     */
    void cutPointsOutside(const cv::Rect& box);



    /**
     * Returns the size of the blob (number of points)
     *
     * @return The size
     */
    inline size_t size() const {
      return blobSize;
    }



    /**
     * Converts the blob to a matrix that is just large enough to contain 
     * all its points.
     *
     * @param invert By default, blob pixels are marked 255 and non-blob 
     * pixels 0. If invert is true, the values will be inverted.
     * 
     * @return A new matrix
     */
    inline cv::Mat toMatrix(bool invert = false) const {
      return toMatrix(cv::Size(rightMost.x-leftMost.x + 1, 
                               bottomMost.y-topMost.y + 1),
                      invert,
                      cv::Point(-leftMost.x, -topMost.y));
    }



    /**
     * draws the blob on the image in a fancy user-supplied colour
     *
     * @param img Target image
     * @param colour The colour that the blob should be drawn in
     */
    void draw(cv::Mat& img, const cv::Scalar& colour) const;



    /**
     * Copy-assignment
     */
    Blob& operator=(const Blob& another); 



    /**
     * Move-assignment
     */
    Blob& operator=(Blob&& another); 



    /**
     * Standard copy constructor
     */
    Blob(const Blob& another) :
      centroid(another.centroid), points(another.points), 
      leftMost(another.leftMost), rightMost(another.rightMost),
      topMost(another.topMost), bottomMost(another.bottomMost),
      blobSize(another.blobSize)
    {}



    /**
     * Standard move constructor
     */
    Blob(Blob&& another) :
      centroid(std::move(another.centroid)), 
      points(std::move(another.points)), 
      leftMost(std::move(another.leftMost)), 
      rightMost(std::move(another.rightMost)),
      topMost(std::move(another.topMost)), 
      bottomMost(std::move(another.bottomMost)),
      blobSize(std::move(another.blobSize))
    {
      another = Blob();
    }



    bool operator==(const Blob&) const;



    /**
     * Returns an explicit representation of the closed blob contour loop.
     *
     * @return A vector of points (p0,p1,...,pn) such that all
     * (pi,pi+1) and (pn,p0) are 8-adjacent.
     */
    std::vector<cv::Point> getContour() const;



    /**
     * Returns the internal chain coded representation
     */
    inline const std::vector<cv::Point>& getChainCodedContour() const {
      return points;
    }

    /**
     * Explicitly constructs the blob. Be careful!
     */
    Blob(const cv::Point& centroid, 
         const std::vector<cv::Point>& chainCodedContour,
         const cv::Point& leftMost,
         const cv::Point& rightMost,
         const cv::Point& topMost,
         const cv::Point& bottomMost,
         size_t size) :
      centroid(centroid),
      points(chainCodedContour),
      leftMost(leftMost),
      rightMost(rightMost),
      topMost(topMost),
      bottomMost(bottomMost),
      blobSize(size)
    {}

 /**
     * Constructs a blob from a point list
     *
     * @param points Points to construct from
     */
    explicit Blob(const std::vector<cv::Point>& points) :
      points(points) {
      computeCachedValues();
    }



    /**
     * Constructs a blob from a point list
     *
     * @param points Points to construct from
     */
    explicit Blob(std::vector<cv::Point>&& points):
      points(std::move(points)) {
      computeCachedValues();
    }




  private:
    /**
     * Locates blobs using cv::findContours. Points are stored using the 
     * "simple" chain code.
     */
    static std::vector<Blob> getContourBlobs(const cv::Mat& inMask);



   

    /**
     * Compares the sizes of two blobs
     *
     * @param a One of the objects of comparison
     * @param b Another object of comparison
     *
     * @return true if a.getSize() > b.getSize(), false otherwise
     */
    inline static bool compare(const Blob& a, const Blob& b) {
      return a.size() > b.size();
    }



    /**
     * updates the cached centroid (the centroid information is kept
     * cached because it might be needed often but it only changes
     * when points change and it is relatively expensive to compute)
     * also updates extreme points (and width and height, which depend
     * on these points
     *
     */
    void computeCachedValues();



    cv::Point centroid; ///< The cached centroid
    std::vector<cv::Point> points; ///< Simple chain-coded list of points that make up the blob. 
    cv::Point leftMost; ///< The extreme left point
    cv::Point rightMost; ///< The extreme right point
    cv::Point topMost; ///< The extreme top point
    cv::Point bottomMost; ///< The extreme bottom point
    size_t blobSize; ///< Size of the blob
  };
}

#endif
