#include <boost/math/constants/constants.hpp>
#include "ColourSpace.hpp"
#include "util.hpp"

using cv::Mat;
using cv::Vec3b;
using cv::Vec3f;

namespace slmotion {
  const std::unique_ptr<const ColourSpace> ColourSpace::YCRCB(new YCrCbColourSpace());
  const std::unique_ptr<const ColourSpace> ColourSpace::RGB(new RGBColourSpace());
  const std::unique_ptr<const ColourSpace> ColourSpace::BGR(new BGRColourSpace());
  const std::unique_ptr<const ColourSpace> ColourSpace::HSV(new HSVColourSpace());
  const std::unique_ptr<const ColourSpace> ColourSpace::GREY(new GreyColourSpace());
  const std::unique_ptr<const ColourSpace> ColourSpace::XYZ(new XYZColourSpace());
  //  const std::unique_ptr<const ColourSpace> ColourSpace::CGZ(new PerceptualColourSpace());
  const std::unique_ptr<const ColourSpace> ColourSpace::IHLS(new IHLSColourSpace());



  void ColourSpace::convertColour(const cv::Mat& in, cv::Mat& out) const {
    Mat YCbCr;
    Mat HSV;
    Mat XYZ;

    out = Mat(in.size(), CV_8UC(components.size()));

    for(int i = 0; i < out.rows; ++i) {
      for (int j = 0; j < out.cols; ++j) {
        size_t cn = 0;
        for (auto it = components.cbegin(); it != components.end(); ++it) {
          // make sure the components are available
          const Mat* src;

          switch(*it) {
          case ColourSpace::Y:
          case ColourSpace::CB:
          case ColourSpace::CR:
            if (YCbCr.empty())
              cvtColor(in, YCbCr, CV_BGR2YCrCb);
            src = &YCbCr;
            break;

          case ColourSpace::H:
          case ColourSpace::S:
          case ColourSpace::V:
            if (HSV.empty())
              cvtColor(in, HSV, CV_BGR2HSV);
            src = &HSV;
            break;

          case ColourSpace::R:
          case ColourSpace::G:
          case ColourSpace::B:
            src = &in;
            break;

          case ColourSpace::X:
          case ColourSpace::XYZ_Y:
          case ColourSpace::Z:
            if (XYZ.empty())
              cvtColor(in, XYZ, CV_BGR2XYZ);
            src = &XYZ;
            break;

          // case ColourSpace::CHONG:
          // case ColourSpace::GORTLER:
          // case ColourSpace::ZICKLER:
          case ColourSpace::IHLS_Y:
          case ColourSpace::IHLS_S:
          case ColourSpace::IHLS_H:
            assert(false && "These components cannot be used");
            break;
          }        

          // then, perform assignment
          uchar* p = out.ptr(i) + j*out.channels();
          const uchar* q = src->ptr(i) + j*src->channels();
          switch(*it) {
          case ColourSpace::Y:
            p[cn] = q[0];
            break;
          case ColourSpace::CB:
            p[cn] = q[2];
            break;
          case ColourSpace::CR:
            p[cn] = q[1];
            break;

          case ColourSpace::H:
            p[cn] = q[0];
            break;
          case ColourSpace::S:
            p[cn] = q[1];
            break;
          case ColourSpace::V:
            p[cn] = q[2];
            break;

          case ColourSpace::R:
            p[cn] = q[2];
            break;
          case ColourSpace::G:
            p[cn] = q[1];
            break;
          case ColourSpace::B:
            p[cn] = q[0];
            break;

          case ColourSpace::X:
            p[cn] = q[0];
            break;
          case ColourSpace::XYZ_Y:
            p[cn] = q[1];
            break;
          case ColourSpace::Z:
            p[cn] = q[2];
            break;

          // case ColourSpace::CHONG:
          // case ColourSpace::GORTLER:
          // case ColourSpace::ZICKLER:
          case ColourSpace::IHLS_Y:
          case ColourSpace::IHLS_S:
          case ColourSpace::IHLS_H:
            assert(false && "These components cannot be used");
            break;
          }
          cn++;
        }
      }
    }
  }



  ColourSpace::~ColourSpace() {
  }



  namespace {
    Mat createAMatrix() {
      float m[3][3] = {{27.07439, -22.80783, -1.806681},
                        {-5.646736, -7.722125, 12.86503},
                        {-4.163133, -4.579428, -4.576049}};
      return Mat(3,3,CV_32FC1, m).clone();      
    }

    Mat createBMatrix() {
      float m[3][3] = {{0.9465229, 0.2946927, -0.1313419},
                        {-0.1179179, 0.9929960, 0.007371554},
                        {0.09230461, -0.04645794, 0.9946464}};

      return Mat(3,3,CV_32FC1, m).clone();
    }

    Mat AMATRIX = createAMatrix(); // CGZ A matrix
    Mat BMATRIX = createBMatrix(); // CGZ B matrix
  }



  namespace {
    Vec3f computeCGZPixelValue(const Vec3f& v) {
      // first, create the column vector
      Mat temp;
      Mat colVec(3,1,CV_32FC1);
      for (int k=0; k<3; ++k)
        colVec.at<float>(k,0) = v[k];
      
      // Then compute!
      cv::log(BMATRIX*colVec, temp);
      temp = AMATRIX*temp;

      Vec3f rv;
      for (int k=0; k<3; ++k)
        rv[k] = temp.at<float>(k,0);
      return rv;
    }

    struct less {
      bool operator()(const Vec3b& a, const Vec3b& b) {
        if (a[0] != b[0])
          return a[0] < b[0];
        if (a[1] != b[1])
          return a[1] < b[1];
        return a[2] < b[2];
      }
    };
  }



#if 0
  void PerceptualColourSpace::convertColour(const cv::Mat& inImg, cv::Mat& out) const {
    static std::map<cv::Vec3b,cv::Vec3f,less> cacheMap; // caches results
    // Conversion is defined as
    // F(x) = A(ln(Bx))
    //  where x is a 3-column-vector
    //        A, B 3x3 matrices
    //        ln is componentwise natural logarithm
    
    assert(inImg.type() == CV_8UC3);
    out = Mat(inImg.size(), CV_32FC3);

    // Preparation: perform colour space conversion
    // (input is assumed to be in XYZ)
    Mat temp;
    inImg.convertTo(temp, CV_32FC3);
    temp *= 1./255;
    Mat in;
    cvtColor(temp, in, CV_BGR2XYZ);

    for (int i = 0; i < in.rows; ++i) {
      for (int j = 0; j < in.cols; ++j) {
        const Vec3f& v = in.at<const Vec3f>(i,j);
        Vec3f& v2 = out.at<Vec3f>(i,j);

        auto it = cacheMap.find(v);
        if (it != cacheMap.end())
          v2 = it->second;
        else 
          v2 = cacheMap[v] = computeCGZPixelValue(v);       
        v2 = computeCGZPixelValue(v);
      }
    } 
  }
#endif



  void IHLSColourSpace::convertColour(const cv::Mat& in, cv::Mat& out) 
    const {
    static const double pi = boost::math::constants::pi<double>();
    assert(in.type() == CV_8UC3);
    out = Mat(in.size(), CV_8UC3);
    for (int i = 0; i < in.rows; ++i) {
      for (int j = 0; j < in.cols; ++j) {
        const Vec3b& v = in.at<const Vec3b>(i,j);
        uchar R = v[2], G = v[1], B = v[0];
        uchar Y = round(0.2126*R + 0.7152*G + 0.0722*B);
        uchar S = std::max(std::max(R,G),B) - std::min(std::min(R,G),B);
        double HPrime = acos(static_cast<double>(R-0.5*G-0.5*B) /
                             sqrt(R*R + G*G + B*B - R*G - R*B - B*G));
        uchar H = round((HPrime/(2.0*pi)) * 255);
        if (B > G)
          H = 255 - H;
        out.at<Vec3b>(i,j) = Vec3b(H,Y,S);
      }
    }
  }
}
