#include <boost/algorithm/string.hpp>

#include "Component.hpp"
#include "Thread.hpp"
#include "configuration.hpp"
#include "debug.hpp"

using std::cerr;
using std::vector;
using std::pair;
using std::make_pair;
using std::endl;
using std::shared_ptr;
using std::string;

namespace slmotion {
  const Component* COMPONENT_LIST = NULL;



  Component::~Component() { }



  Component& Component::operator=(const Component& another) {
    if (this != &another) {
      blackBoard = another.blackBoard;
      frameSource = another.frameSource;
      this->reset();
    }
    return *this;
  }



#ifdef SLMOTION_THREADING

  // Takes in a range of numbers (frame indices), and splits it into n 
  // chunks and returns corresponding ranges.
  // If n == 0, returns a single element containing the original range.
  static vector<pair<size_t,size_t>> splitRangeToChunks(size_t first,
                                                        size_t last,
                                                        size_t n) {
    assert(last > first);
    if (n == 0)
      return vector<pair<size_t,size_t>> { make_pair(first, last) };

    if (n > last-first)
      n = last - first; // limit the number of chunks to the number of elements

    vector<pair<size_t,size_t>> chunks;
    for (size_t i = 0; i < n; ++i)
      chunks.push_back(make_pair(first + ((last-first)*i)/n,
                                 first + ((last-first)*(i+1))/n));

    // in case last - first is not divisible by n, ensure that the very last
    // element will be last
    chunks.back().second = last;
    return chunks;
  }



  // this multithreaded implementation will split the range into chunks and 
  // concurrently call process for each chunk 
  bool Component::processRangeImplementation(frame_number_t first, 
                                             frame_number_t last, 
                                             UiCallback*) {
    auto chunks = splitRangeToChunks(first, last, Thread::getMaxThreads());
    std::list<shared_ptr<Thread>> threads;

    // a simple function that just iterates over the range and calls process
    auto iterateAndProcess = [&](size_t first, size_t last) {
      for (size_t i = first; i < last; ++i) {
        if (slmotion::debug > 0) {
          if (slmotion::debug == 1)
            cerr << '\r';
          cerr << i + 1 - first << '/' << last - first;
        }
        this->process(i);
      }
    };

    
    for (auto it = chunks.cbegin(); it != chunks.end(); ) {
      if (slmotion::debug > 0)
        cerr << "Processing chunk [" << it->first << "," << it->second 
             << ")" << endl;

      // three cases:
      //  - can create thread: then do so
      //  - cannot but local threads still running:
      //    -> wait for one to finish, then try again
      //  - no room for threads and no threads running
      //    -> run the chunk in this thread
      shared_ptr<Thread> thread(Thread::createThreadIfPossible(iterateAndProcess, it->first, it->second));
      if (thread.get()) {
        threads.push_back(thread);
        ++it;
      }
      else if (threads.size() > 0) {
        if (threads.front()->joinable())
          threads.front()->join();
        threads.erase(threads.begin());
      }
      else {
        iterateAndProcess(it->first, it->second);
        ++it;
      }
    }

    // cleanup: wait for any remaining threads to finish
    while (threads.size() > 0) {
      if (threads.front()->joinable())
        threads.front()->join();
      threads.erase(threads.begin());
    }

    // if (uiCallback != NULL && !(*uiCallback)(last))
    //   return false;

    return true;
  }

#else

  // single-threaded implementation
  bool Component::processRangeImplementation(frame_number_t first, 
                                             frame_number_t last, UiCallback* uiCallback) {
    for (size_t i = first; i < last; ++i) {
      if (slmotion::debug > 1) {
        // if (slmotion::debug == 1)
        //   cerr << '\r';
        cerr << i + 1 - first << '/' << last - first;
        if (slmotion::debug > 1)
          cerr << endl;
      }
      this->process(i);
      if (uiCallback != NULL && !(*uiCallback)((i+1-first)*100./(last-first)))
        return false;
    }
    // if (slmotion::debug == 1)
    //   cerr << endl;
    return true;
  }
#endif



  bool Component::processRange(frame_number_t first, 
                               frame_number_t last,
                               UiCallback* uiCallback) {
    if (last > getFrameSource().size())
      last = getFrameSource().size();
    
    if (first > last)
      throw std::invalid_argument("Invalid arguments provided for processRange: last must go after first.");
    
    return processRangeImplementation(first, last, uiCallback);
  }



  /**
   * Takes in an arbitrary number of string vectors, and a corresponding
   * number of field widths, and then outputs a string in following format:
   * v1e1 - v2e1 - ... - vne1\n
   * v1e2 - v2e2 - ... - vne2\n
   *  .      .     .      .
   *  .      .      .     .
   *  .      .       .    .
   * v1en - v2en - ... - vmen\n
   *
   * where each field is at most fieldWidths long
   */
  static std::string fieldify(const vector<vector<string>>& strings,
                              const vector<size_t>& fieldWidths) {
    assert(strings.size() == fieldWidths.size());
    for (auto it = strings.cbegin(); it != strings.cend(); ++it)
      assert(strings.front().size() == it->size());

    size_t numEntries = strings.front().size();

    // compute line width
    size_t lineWidth = 0;
    for (auto it = fieldWidths.cbegin(); it != fieldWidths.cend(); ++it) {
      if (it != fieldWidths.cbegin())
        lineWidth += 3;
      lineWidth += *it;
    }

    std::string result;
    vector<char> buffer(lineWidth + 1);

    for (size_t i = 0; i < numEntries; ++i) {
      // entry elements split by lines
      vector<vector<string>> splitEntries(strings.size());
      for (size_t j = 0; j < strings.size(); ++j)
        boost::algorithm::split(splitEntries[j], strings[j][i], 
                                boost::is_any_of(" "));

      bool first = true;
      while (true) {
        bool empty = true;
        for (size_t j = 0; j < strings.size(); ++j) 
          if (splitEntries[j].size() > 0)
            empty = false;
        if (empty)
          break;

        for (size_t j = 0; j < strings.size(); ++j) {
          if (first && j > 0)
            result += " - ";
          else if (j > 0)
            result += "   ";

          for (size_t k = 0; k < fieldWidths[j]; ++k)
            buffer[k] = ' ';
          buffer[fieldWidths[j]] = '\0';

          size_t lineLength = 0;
          while (splitEntries[j].size() > 0 && 
                 (lineLength == 0 || 
                  (lineLength < fieldWidths[j] && 
                   (splitEntries[j].front().length() <= (fieldWidths[j] - lineLength))))) {
            strncpy(&buffer[lineLength], splitEntries[j].front().c_str(), 
                    std::min(fieldWidths[j], splitEntries[j].front().length()));
            lineLength += splitEntries[j].front().length();
            splitEntries[j].erase(splitEntries[j].begin());
            if (lineLength < fieldWidths[j] && splitEntries[j].size() > 0) {
              buffer[lineLength] = ' ';
              lineLength++;
            }
          }
          result += &buffer[0];
        }
        result += '\n';
        first = false;
      }
    }
    return result;
  }



#if __GNUC__ == 4 && __GNUC_MINOR__ < 5
  static bool strLenCmp(const std::string& a, const std::string& b) {
      return a.length() < b.length();
  };
#endif // __GNUC__ === 4 && __GNUC_MINOR__ < 5



  std::string Component::getRegisteredComponents(size_t maxLineWidth) {
    // components and their descriptions
    std::vector<std::string> registeredComponents;
    std::vector<std::string> componentDescriptions;
    const Component* currentComponent = COMPONENT_LIST;
    while (currentComponent != NULL) {
      registeredComponents.push_back(currentComponent->getShortName());
      componentDescriptions.push_back(currentComponent->getShortDescription());
      currentComponent = currentComponent->nextComponent;
    }

#if __GNUC__ == 4 && __GNUC_MINOR__ >= 5
    auto strLenCmp = [](const std::string& a, const std::string& b) {
      return a.length() < b.length();
    };
#endif // __GNUC__ === 4 && __GNUC_MINOR__ >= 5

    // find field widths
    size_t nameFieldWidth = std::max_element(registeredComponents.begin(),
                                             registeredComponents.end(),
                                             strLenCmp)->length();
    size_t descriptionFieldWidth = std::min(std::max_element(componentDescriptions.begin(),
                                                             componentDescriptions.end(),
                                                             strLenCmp)->length(), maxLineWidth-3-nameFieldWidth);

    return fieldify(vector<vector<string>> { registeredComponents,
          componentDescriptions }, vector<size_t> { nameFieldWidth, 
                                       descriptionFieldWidth });
  }



  // configuration::ConfigurationFileOptionsDescription 
  boost::program_options::options_description Component::getConfigurationFileOptionsDescriptionByComponent(const std::string& name) {
    const Component* c = COMPONENT_LIST;
    while (c != NULL) {
      if (c->getShortName() == name)
        return c->getConfigurationFileOptionsDescription();
      c = c->nextComponent;
    }
    return boost::program_options::options_description();
  }




  std::string Component::getLongDescriptionForComponent(const std::string& name) {
    const Component* c = COMPONENT_LIST;
    while (c != NULL) {
      if (c->getShortName() == name)
        return c->getLongDescription();
      c = c->nextComponent;
    }
    return "";
  }



  boost::program_options::options_description Component::getCompleteConfigurationFileOptionsDescription() {
    boost::program_options::options_description opts;
    const Component* c = COMPONENT_LIST;
    while (c != NULL) {
      opts.add(c->getConfigurationFileOptionsDescription());
      c = c->nextComponent;      
    }
    return opts;
  }

  boost::program_options::options_description Component::getCompleteCommandLineOptionsDescription() {
    boost::program_options::options_description opts;
    const Component* c = COMPONENT_LIST;
    while (c != NULL) {
      auto cmdOpts = c->getCommandLineOptionsDescription();
      if (cmdOpts.options().size() > 0)
        opts.add(cmdOpts);
      c = c->nextComponent;      
    }
    return opts;
  }



  Component::Component(bool) :
    nextComponent(COMPONENT_LIST),
    blackBoard(NULL), frameSource(NULL)
  {
    COMPONENT_LIST = this;
  }



  Component* Component::createComponent(const std::string& componentName,
                                        const boost::program_options::variables_map& configuration, 
                                        BlackBoard* blackBoard, 
                                        FrameSource* frameSource) const {
    const Component* c = COMPONENT_LIST;
    while (c) {
      if (c->getShortName() == componentName)
        return c->createComponentImpl(configuration, blackBoard, frameSource);
      c = c->nextComponent;
    }
    return NULL;
  }



  Component* Component::createComponent(const std::string& componentName,
                                        const std::map<std::string, boost::any>& configuration,
                                        const boost::program_options::variables_map* commandLineOptions,
                                        BlackBoard* blackBoard, 
                                        FrameSource* frameSource) const {
    const Component* c = COMPONENT_LIST;
    while (c) {
      if (c->getShortName() == componentName)
        return c->createComponentImpl(configuration, commandLineOptions, blackBoard, frameSource);
      c = c->nextComponent;
    }
    return NULL;
  }



  Component* Component::createComponentImpl(const std::map<std::string, boost::any>& configuration, 
                                            const boost::program_options::variables_map* commandLineOptions,
                                            BlackBoard* blackBoard, FrameSource* frameSource) const {
    // this is the default implementation
    // it just prefixes the options and uses the older, configuration-file based method
    boost::program_options::variables_map vm = commandLineOptions ? 
      *commandLineOptions : boost::program_options::variables_map();
    for (auto it = configuration.cbegin(); it != configuration.cend(); ++it) {
      boost::program_options::variable_value v(it->second, false);
      vm.insert(std::make_pair(getShortName() + "." + it->first, v));
    }

    return createComponentImpl(vm, blackBoard, frameSource);
  }

#if !SLMOTION_DISABLE_DEFAULT_COMPONENT_MEMBER_DEFINITIONS
 boost::program_options::options_description Component::getConfigurationFileOptionsDescription() const { 
   return  boost::program_options::options_description();
 }
#endif



  std::vector<string> Component::findProviders(const std::string& property) {
    std::vector<string> comps;
    for (const Component* c = COMPONENT_LIST; c != NULL; c = c->nextComponent) 
      if (c->getProvided().count(property))
        comps.push_back(c->getShortName());
    return comps;
  }
}
